import { useEffect, useRef } from "react";
import JsBarcode from "jsbarcode";
import "./receipt.css";

interface ServiceTicketProps {
  orderId: number;
  orderNumber: string;
  lineId: number;
  serviceName: string;
  customerName: string;
  employeeName: string | null;
  executionTime: string | null;
}

const formatTime = (timeStr: string | null): string => {
  if (!timeStr) return "—";
  try {
    const date = new Date(timeStr);
    return date.toLocaleTimeString("ar-SA", {
      hour: "2-digit",
      minute: "2-digit",
      hour12: true,
    });
  } catch {
    return timeStr;
  }
};

const formatDate = (timeStr: string | null): string => {
  if (!timeStr) return "—";
  try {
    const date = new Date(timeStr);
    return date.toLocaleDateString("ar-SA", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  } catch {
    return "—";
  }
};

export function ServiceTicket({
  orderId,
  orderNumber,
  lineId,
  serviceName,
  customerName,
  employeeName,
  executionTime,
}: ServiceTicketProps) {
  const barcodeRef = useRef<SVGSVGElement>(null);
  const barcodeValue = `${orderNumber}-${lineId}`;

  useEffect(() => {
    if (barcodeRef.current) {
      try {
        JsBarcode(barcodeRef.current, barcodeValue, {
          format: "CODE128",
          width: 1.5,
          height: 40,
          displayValue: false,
          margin: 0,
        });
      } catch (error) {
        console.error("Error generating barcode:", error);
      }
    }
  }, [barcodeValue]);

  return (
    <div className="service-ticket" dir="rtl">
      <div className="ticket-header">
        قصاصة خدمة - Service Ticket
      </div>

      <div className="ticket-info">
        <div className="ticket-row">
          <span className="ticket-label">رقم الطلب:</span>
          <span>{orderNumber}</span>
        </div>
        <div className="ticket-row">
          <span className="ticket-label">العميل:</span>
          <span>{customerName}</span>
        </div>
        {employeeName && (
          <div className="ticket-row">
            <span className="ticket-label">الخبيرة:</span>
            <span>{employeeName}</span>
          </div>
        )}
        {executionTime && (
          <>
            <div className="ticket-row">
              <span className="ticket-label">التاريخ:</span>
              <span>{formatDate(executionTime)}</span>
            </div>
            <div className="ticket-row">
              <span className="ticket-label">الوقت:</span>
              <span>{formatTime(executionTime)}</span>
            </div>
          </>
        )}
      </div>

      <div className="service-name">
        {serviceName}
      </div>

      <div className="barcode-container">
        <svg ref={barcodeRef}></svg>
        <div className="barcode-text">{barcodeValue}</div>
      </div>
    </div>
  );
}

export type { ServiceTicketProps };
