import { QRCodeSVG } from "qrcode.react";
import { generateZatcaQrPayload } from "@/lib/zatcaQr";
import "./receipt.css";

interface CompanyInfo {
  companyNameAr: string;
  phone: string;
  branchName: string;
  crNumber: string;
  vatNumber: string;
  addressLine: string;
  city: string;
  extraNote: string;
}

interface OrderLine {
  type: "service" | "product";
  name: string;
  executionTime: string | null;
  employeeName: string | null;
  discount: number;
  price: number;
  quantity: number;
}

interface PaymentInfo {
  methodName: string;
  amount: number;
}

interface CreditNoteInfo {
  creditNoteNumber: string;
  originalInvoiceNumber: string;
  refundPaymentMethodName?: string;
}

interface PrintData {
  companyInfo: CompanyInfo;
  invoiceNumber: string;
  status: string;
  statusLabel: string;
  invoiceDate: string;
  invoiceTime: string;
  invoiceTimestampISO: string;
  bookingDate: string | null;
  bookingTime: string | null;
  printDate: string;
  printTime: string;
  customerName: string;
  payments: PaymentInfo[];
  cashierName: string;
  lines: OrderLine[];
  subtotalBeforeDiscount: number;
  discountAmount: number;
  vatRate: number;
  vatAmount: number;
  totalWithVat: number;
  paidAmount: number;
  remainingAmount: number;
  creditNoteInfo?: CreditNoteInfo | null;
  isCancelled?: boolean;
}

interface ThermalReceiptProps {
  data: PrintData;
}

const formatNumber = (num: number): string => {
  return num.toFixed(2);
};

const formatTime = (timeStr: string | null): string => {
  if (!timeStr) return "—";
  try {
    const date = new Date(timeStr);
    return date.toLocaleTimeString("ar-SA", {
      hour: "2-digit",
      minute: "2-digit",
      hour12: true,
    });
  } catch {
    return timeStr;
  }
};

export function ThermalReceipt({ data }: ThermalReceiptProps) {
  const zatcaPayload = generateZatcaQrPayload({
    sellerName: data.companyInfo.companyNameAr || "—",
    vatNumber: data.companyInfo.vatNumber || "—",
    invoiceTimestamp: data.invoiceTimestampISO || new Date().toISOString(),
    invoiceTotal: data.totalWithVat,
    vatTotal: data.vatAmount,
  });

  const discountPercentage = data.subtotalBeforeDiscount > 0 
    ? ((data.discountAmount / data.subtotalBeforeDiscount) * 100).toFixed(1)
    : "0";

  const isCreditNote = data.isCancelled && data.creditNoteInfo;

  return (
    <div className="thermal-receipt" dir="rtl">
      <div className="receipt-header">
        <div className="receipt-title">
          {isCreditNote ? "إشعار دائن" : "فاتورة ضريبية مبسطة"}
        </div>
        {isCreditNote && (
          <div className="credit-note-subtitle">Credit Note</div>
        )}
        <div className="company-name">{data.companyInfo.companyNameAr || "—"}</div>
        {data.companyInfo.phone && (
          <div className="company-phone">الجوال: {data.companyInfo.phone}</div>
        )}
        {data.companyInfo.branchName && (
          <div className="branch-name">{data.companyInfo.branchName}</div>
        )}
      </div>

      <div className="receipt-divider" />

      <div className="receipt-info">
        <div className="info-row">
          <span className="info-label">الرقم الضريبي:</span>
          <span className="info-value">{data.companyInfo.vatNumber || "—"}</span>
        </div>
        <div className="info-row">
          <span className="info-label">السجل التجاري:</span>
          <span className="info-value">{data.companyInfo.crNumber || "—"}</span>
        </div>
        {isCreditNote ? (
          <>
            <div className="info-row">
              <span className="info-label">رقم إشعار الدائن:</span>
              <span className="info-value">{data.creditNoteInfo?.creditNoteNumber}</span>
            </div>
            <div className="info-row">
              <span className="info-label">رقم الفاتورة الأصلية:</span>
              <span className="info-value">{data.creditNoteInfo?.originalInvoiceNumber}</span>
            </div>
            {data.creditNoteInfo?.refundPaymentMethodName && (
              <div className="info-row">
                <span className="info-label">طريقة الاسترداد:</span>
                <span className="info-value">{data.creditNoteInfo.refundPaymentMethodName}</span>
              </div>
            )}
          </>
        ) : (
          <div className="info-row">
            <span className="info-label">رقم الفاتورة:</span>
            <span className="info-value">{data.invoiceNumber}</span>
          </div>
        )}
        <div className="info-row">
          <span className="info-label">{isCreditNote ? "حالة الإشعار:" : "حالة الفاتورة:"}</span>
          <span className="info-value">{data.statusLabel}</span>
        </div>
        <div className="info-row">
          <span className="info-label">تاريخ الفاتورة:</span>
          <span className="info-value">{data.invoiceDate} {data.invoiceTime}</span>
        </div>
        {data.bookingDate && (
          <div className="info-row">
            <span className="info-label">تاريخ الحجز:</span>
            <span className="info-value">{data.bookingDate} {data.bookingTime || ""}</span>
          </div>
        )}
        <div className="info-row">
          <span className="info-label">تاريخ الطباعة:</span>
          <span className="info-value">{data.printDate} {data.printTime}</span>
        </div>
        <div className="info-row">
          <span className="info-label">العميل:</span>
          <span className="info-value">{data.customerName}</span>
        </div>
      </div>

      <div className="receipt-divider" />

      <div className="payment-info">
        {data.payments.length > 0 ? (
          data.payments.map((payment, index) => (
            <div key={index} className="info-row">
              <span className="info-label">طريقة الدفع:</span>
              <span className="info-value">{payment.methodName} - {formatNumber(payment.amount)} ر.س</span>
            </div>
          ))
        ) : (
          <div className="info-row">
            <span className="info-label">طريقة الدفع:</span>
            <span className="info-value">—</span>
          </div>
        )}
      </div>

      <div className="receipt-divider" />

      <table className="items-table">
        <thead>
          <tr>
            <th className="col-service">Service</th>
            <th className="col-price">Price</th>
            <th className="col-discount">.Dist</th>
            <th className="col-expert">Expert</th>
            <th className="col-time">Time</th>
          </tr>
        </thead>
        <tbody>
          {data.lines.map((line, index) => (
            <tr key={index}>
              <td className="col-service">{line.name}</td>
              <td className="col-price">{formatNumber(line.price)}</td>
              <td className="col-discount">{formatNumber(line.discount)}</td>
              <td className="col-expert">{line.type === "service" && line.employeeName ? line.employeeName : "—"}</td>
              <td className="col-time">{line.type === "service" ? formatTime(line.executionTime) : "—"}</td>
            </tr>
          ))}
        </tbody>
      </table>

      <div className="receipt-divider" />

      <div className="summary-section">
        <table className="summary-table">
          <tbody>
            <tr>
              <td className="summary-label">سعر الخدمات/المنتجات</td>
              <td className="summary-value">{formatNumber(data.subtotalBeforeDiscount)} ر.س</td>
            </tr>
            {data.discountAmount > 0 && (
              <tr>
                <td className="summary-label">Discount ({discountPercentage}%)</td>
                <td className="summary-value discount-value">-{formatNumber(data.discountAmount)} ر.س</td>
              </tr>
            )}
            <tr>
              <td className="summary-label">الضريبة {(data.vatRate * 100).toFixed(0)}%</td>
              <td className="summary-value">{formatNumber(data.vatAmount)} ر.س</td>
            </tr>
            <tr className="summary-total">
              <td className="summary-label">المجموع Total</td>
              <td className="summary-value">{formatNumber(data.totalWithVat)} ر.س</td>
            </tr>
            <tr>
              <td className="summary-label">المدفوع Paid</td>
              <td className="summary-value">{formatNumber(data.paidAmount)} ر.س</td>
            </tr>
            <tr>
              <td className="summary-label">المتبقي Remain</td>
              <td className="summary-value">{formatNumber(data.remainingAmount)} ر.س</td>
            </tr>
            {data.remainingAmount < 0 && (
              <tr>
                <td className="summary-label">مبالغ مستردة</td>
                <td className="summary-value refund-value">{formatNumber(Math.abs(data.remainingAmount))} ر.س</td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      <div className="receipt-divider" />

      <div className="receipt-footer">
        <div className="cashier-info">طبع بواسطة: {data.cashierName}</div>
        {data.companyInfo.extraNote && (
          <div className="extra-note">{data.companyInfo.extraNote}</div>
        )}
        <div className="qr-container">
          <QRCodeSVG
            value={zatcaPayload}
            size={100}
            level="M"
            includeMargin={true}
          />
        </div>
        <div className="zatca-note">رمز الاستجابة السريعة - هيئة الزكاة والضريبة والجمارك</div>
      </div>
    </div>
  );
}

export type { PrintData, CompanyInfo, OrderLine, PaymentInfo };
