/**
 * ZATCA Phase 1 QR Code Generator
 * Generates TLV (Tag-Length-Value) encoded data for simplified tax invoices
 */

interface ZatcaData {
  sellerName: string;       // اسم البائع
  vatNumber: string;        // الرقم الضريبي
  invoiceTimestamp: string; // تاريخ ووقت الفاتورة (ISO 8601)
  invoiceTotal: number;     // إجمالي الفاتورة
  vatTotal: number;         // إجمالي الضريبة
}

function toTlv(tag: number, value: string): Uint8Array {
  const encoder = new TextEncoder();
  const valueBytes = encoder.encode(value);
  const length = valueBytes.length;
  
  const result = new Uint8Array(2 + length);
  result[0] = tag;
  result[1] = length;
  result.set(valueBytes, 2);
  
  return result;
}

function concatenateArrays(...arrays: Uint8Array[]): Uint8Array {
  const totalLength = arrays.reduce((sum, arr) => sum + arr.length, 0);
  const result = new Uint8Array(totalLength);
  let offset = 0;
  for (const arr of arrays) {
    result.set(arr, offset);
    offset += arr.length;
  }
  return result;
}

function uint8ArrayToBase64(bytes: Uint8Array): string {
  let binary = '';
  for (let i = 0; i < bytes.length; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return btoa(binary);
}

export function generateZatcaQrPayload(data: ZatcaData): string {
  const tlv1 = toTlv(1, data.sellerName);
  const tlv2 = toTlv(2, data.vatNumber);
  const tlv3 = toTlv(3, data.invoiceTimestamp);
  const tlv4 = toTlv(4, data.invoiceTotal.toFixed(2));
  const tlv5 = toTlv(5, data.vatTotal.toFixed(2));
  
  const combined = concatenateArrays(tlv1, tlv2, tlv3, tlv4, tlv5);
  return uint8ArrayToBase64(combined);
}

export function formatInvoiceTimestamp(date: Date): string {
  return date.toISOString();
}
