import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Link, useParams, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  Calendar, 
  ArrowRight, 
  Clock, 
  CheckCircle2, 
  XCircle,
  User,
  MapPin,
  Phone,
  Home,
  Building2,
  Edit,
  Save,
  ExternalLink,
  AlertTriangle,
  FileText,
  DollarSign,
} from "lucide-react";

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
  email: string | null;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  price: number;
}

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
}

interface AppointmentService {
  id: number;
  appointmentId: number;
  serviceId: number;
  quantity: number;
  basePrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  service?: Service;
  preferredEmployee?: Employee;
}

interface Order {
  id: number;
  orderNumber: string;
  status: string;
  totalAmount: number;
}

interface Appointment {
  id: number;
  appointmentNumber: string;
  source: string;
  clientId: number | null;
  branchId: number | null;
  appointmentType: string;
  scheduledAt: string;
  status: string;
  visitFee: number;
  locationLat: number | null;
  locationLng: number | null;
  addressText: string | null;
  notes: string | null;
  cancelledReason: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  createdAt: string;
  updatedAt: string | null;
  client?: Customer;
  services: AppointmentService[];
  relatedOrder?: Order;
  relatedOrderId?: number | null;
}

const statusLabels: Record<string, string> = {
  pending: "قيد الانتظار",
  confirmed: "مؤكد",
  in_progress: "جاري التنفيذ",
  completed: "مكتمل",
  canceled: "ملغي",
  no_show: "لم يحضر",
  postponed: "مؤجل",
};

const statusColors: Record<string, string> = {
  pending: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
  confirmed: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
  in_progress: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
  completed: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
  canceled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
  no_show: "bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300",
  postponed: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
};

const appointmentTypeLabels: Record<string, string> = {
  in_salon: "في الصالون",
  home_service: "خدمة منزلية",
};

const formatDateForInput = (dateStr: string): string => {
  const date = new Date(dateStr);
  return date.toISOString().slice(0, 16);
};

export default function AppointmentDetailPage() {
  const params = useParams<{ id: string }>();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const appointmentId = parseInt(params.id || "0");

  const [isEditing, setIsEditing] = useState(false);
  const [editedNotes, setEditedNotes] = useState("");
  const [editedScheduledAt, setEditedScheduledAt] = useState("");
  const [editedVisitFee, setEditedVisitFee] = useState("");
  const [showCancelDialog, setShowCancelDialog] = useState(false);
  const [cancelReason, setCancelReason] = useState("");
  const [showStatusDialog, setShowStatusDialog] = useState(false);
  const [newStatus, setNewStatus] = useState("");

  const { data: appointment, isLoading, error } = useQuery<Appointment>({
    queryKey: ["/api/appointments", appointmentId],
    enabled: appointmentId > 0,
  });

  const updateMutation = useMutation({
    mutationFn: async (updates: any) => {
      return apiRequest("PUT", `/api/appointments/${appointmentId}`, updates);
    },
    onSuccess: () => {
      toast({ title: "تم تحديث الحجز بنجاح" });
      setIsEditing(false);
      queryClient.invalidateQueries({ queryKey: ["/api/appointments", appointmentId] });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في تحديث الحجز",
        variant: "destructive",
      });
    },
  });

  const confirmMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/appointments/${appointmentId}/confirm`);
    },
    onSuccess: () => {
      toast({ title: "تم تأكيد الحجز بنجاح" });
      queryClient.invalidateQueries({ queryKey: ["/api/appointments", appointmentId] });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في تأكيد الحجز",
        variant: "destructive",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async (reason?: string) => {
      return apiRequest("POST", `/api/appointments/${appointmentId}/cancel`, { reason });
    },
    onSuccess: () => {
      toast({ title: "تم إلغاء الحجز بنجاح" });
      setShowCancelDialog(false);
      setCancelReason("");
      queryClient.invalidateQueries({ queryKey: ["/api/appointments", appointmentId] });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في إلغاء الحجز",
        variant: "destructive",
      });
    },
  });

  const changeStatusMutation = useMutation({
    mutationFn: async (status: string) => {
      return apiRequest("PUT", `/api/appointments/${appointmentId}`, { status });
    },
    onSuccess: () => {
      toast({ title: "تم تغيير حالة الحجز بنجاح" });
      setShowStatusDialog(false);
      queryClient.invalidateQueries({ queryKey: ["/api/appointments", appointmentId] });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في تغيير الحالة",
        variant: "destructive",
      });
    },
  });

  const handleStartEdit = () => {
    if (appointment) {
      setEditedNotes(appointment.notes || "");
      setEditedScheduledAt(formatDateForInput(appointment.scheduledAt));
      setEditedVisitFee(String(appointment.visitFee || 0));
      setIsEditing(true);
    }
  };

  const handleSaveEdit = () => {
    const updates: any = {};
    if (editedNotes !== (appointment?.notes || "")) {
      updates.notes = editedNotes || null;
    }
    if (editedScheduledAt !== formatDateForInput(appointment?.scheduledAt || "")) {
      updates.scheduledAt = new Date(editedScheduledAt);
    }
    const newVisitFee = parseFloat(editedVisitFee) || 0;
    if (newVisitFee !== (appointment?.visitFee || 0)) {
      updates.visitFee = newVisitFee;
    }
    
    if (Object.keys(updates).length > 0) {
      updateMutation.mutate(updates);
    } else {
      setIsEditing(false);
    }
  };

  const handleConfirm = () => {
    confirmMutation.mutate();
  };

  const handleCancelConfirm = () => {
    cancelMutation.mutate(cancelReason);
  };

  const handleStatusChange = () => {
    if (newStatus) {
      changeStatusMutation.mutate(newStatus);
    }
  };

  if (isLoading) {
    return (
      <div className="p-6 space-y-6" dir="rtl">
        <Skeleton className="h-10 w-48" />
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <div className="lg:col-span-2 space-y-6">
            <Skeleton className="h-64 w-full" />
            <Skeleton className="h-48 w-full" />
          </div>
          <div className="space-y-6">
            <Skeleton className="h-40 w-full" />
            <Skeleton className="h-40 w-full" />
          </div>
        </div>
      </div>
    );
  }

  if (error || !appointment) {
    return (
      <div className="p-6" dir="rtl">
        <Card>
          <CardContent className="p-8 text-center">
            <AlertTriangle className="h-12 w-12 mx-auto mb-4 text-destructive" />
            <h2 className="text-xl font-bold mb-2">الحجز غير موجود</h2>
            <p className="text-muted-foreground mb-4">لم يتم العثور على الحجز المطلوب</p>
            <Link href="/appointments">
              <Button>العودة للحجوزات</Button>
            </Link>
          </CardContent>
        </Card>
      </div>
    );
  }

  const canConfirm = appointment.status === "pending";
  const canCancel = ["pending", "confirmed"].includes(appointment.status);
  const canEdit = !["completed", "canceled"].includes(appointment.status);
  const canChangeStatus = !["completed", "canceled"].includes(appointment.status);

  return (
    <div className="p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-3">
          <Link href="/appointments">
            <Button variant="ghost" size="icon">
              <ArrowRight className="h-5 w-5" />
            </Button>
          </Link>
          <Calendar className="h-8 w-8 text-primary" />
          <div>
            <h1 className="text-2xl font-bold">الحجز #{appointment.appointmentNumber}</h1>
            <p className="text-sm text-muted-foreground">
              تاريخ الإنشاء: {formatDate(appointment.createdAt)}
            </p>
          </div>
        </div>
        <div className="flex items-center gap-2">
          <Badge className={`text-sm ${statusColors[appointment.status]}`}>
            {statusLabels[appointment.status] || appointment.status}
          </Badge>
          {canEdit && !isEditing && (
            <Button variant="outline" onClick={handleStartEdit} data-testid="button-edit">
              <Edit className="h-4 w-4 ml-2" />
              تعديل
            </Button>
          )}
          {isEditing && (
            <>
              <Button variant="outline" onClick={() => setIsEditing(false)}>
                إلغاء
              </Button>
              <Button onClick={handleSaveEdit} disabled={updateMutation.isPending} data-testid="button-save">
                <Save className="h-4 w-4 ml-2" />
                {updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
              </Button>
            </>
          )}
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="h-5 w-5" />
                معلومات العميل والموعد
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-4">
                  <div>
                    <Label className="text-muted-foreground">العميل</Label>
                    <div className="flex items-center gap-2 mt-1">
                      <User className="h-4 w-4 text-muted-foreground" />
                      <span className="font-medium">{appointment.client?.nameAr || "غير محدد"}</span>
                    </div>
                  </div>
                  {appointment.client?.phone && (
                    <div>
                      <Label className="text-muted-foreground">الهاتف</Label>
                      <div className="flex items-center gap-2 mt-1">
                        <Phone className="h-4 w-4 text-muted-foreground" />
                        <span>{appointment.client.phone}</span>
                      </div>
                    </div>
                  )}
                  {appointment.client && (
                    <Link href={`/customers/${appointment.client.id}`}>
                      <Button variant="ghost" size="sm" className="px-0 text-primary">
                        <ExternalLink className="h-4 w-4 ml-2" />
                        عرض ملف العميل
                      </Button>
                    </Link>
                  )}
                </div>
                <div className="space-y-4">
                  <div>
                    <Label className="text-muted-foreground">نوع الحجز</Label>
                    <div className="flex items-center gap-2 mt-1">
                      {appointment.appointmentType === "home_service" ? (
                        <Home className="h-4 w-4 text-orange-500" />
                      ) : (
                        <Building2 className="h-4 w-4 text-blue-500" />
                      )}
                      <span className="font-medium">
                        {appointmentTypeLabels[appointment.appointmentType] || appointment.appointmentType}
                      </span>
                    </div>
                  </div>
                  <div>
                    <Label className="text-muted-foreground">الموعد</Label>
                    {isEditing ? (
                      <Input
                        type="datetime-local"
                        value={editedScheduledAt}
                        onChange={(e) => setEditedScheduledAt(e.target.value)}
                        className="mt-1"
                        data-testid="input-scheduled-at"
                      />
                    ) : (
                      <div className="flex items-center gap-2 mt-1">
                        <Clock className="h-4 w-4 text-muted-foreground" />
                        <span className="font-medium">{formatDate(appointment.scheduledAt)}</span>
                      </div>
                    )}
                  </div>
                </div>
              </div>

              {appointment.appointmentType === "home_service" && (
                <div className="pt-4 border-t">
                  <Label className="text-muted-foreground">العنوان</Label>
                  <div className="flex items-start gap-2 mt-1">
                    <MapPin className="h-4 w-4 text-orange-500 mt-1" />
                    <span>{appointment.addressText || "غير محدد"}</span>
                  </div>
                  {appointment.locationLat && appointment.locationLng && (
                    <a 
                      href={`https://www.google.com/maps?q=${appointment.locationLat},${appointment.locationLng}`}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="text-sm text-blue-600 hover:underline mt-2 inline-flex items-center gap-1"
                    >
                      <ExternalLink className="h-3 w-3" />
                      عرض على الخريطة
                    </a>
                  )}
                </div>
              )}

              <div className="pt-4 border-t">
                <Label className="text-muted-foreground">ملاحظات</Label>
                {isEditing ? (
                  <Textarea
                    value={editedNotes}
                    onChange={(e) => setEditedNotes(e.target.value)}
                    placeholder="أضف ملاحظات..."
                    className="mt-1"
                    rows={3}
                    data-testid="input-notes"
                  />
                ) : (
                  <p className="mt-1">{appointment.notes || "لا توجد ملاحظات"}</p>
                )}
              </div>

              {appointment.cancelledReason && (
                <div className="pt-4 border-t">
                  <Label className="text-destructive">سبب الإلغاء</Label>
                  <p className="mt-1 text-destructive">{appointment.cancelledReason}</p>
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="h-5 w-5" />
                الخدمات المحجوزة
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="border rounded-lg divide-y">
                {appointment.services.map((svc, idx) => (
                  <div key={idx} className="p-4 flex items-center justify-between">
                    <div>
                      <p className="font-medium">{svc.service?.nameAr || "خدمة"}</p>
                      <div className="text-sm text-muted-foreground">
                        <span>الكمية: {svc.quantity}</span>
                        <span className="mx-2">|</span>
                        <span>السعر: {formatNumber(svc.basePrice)} ر.س</span>
                        {svc.preferredEmployee && (
                          <>
                            <span className="mx-2">|</span>
                            <span>الموظف المفضل: {svc.preferredEmployee.nameAr}</span>
                          </>
                        )}
                      </div>
                    </div>
                    <div className="text-left">
                      <p className="font-medium">{formatNumber(svc.lineTotal)} ر.س</p>
                      {svc.vatAmount > 0 && (
                        <p className="text-xs text-muted-foreground">
                          ض.ق.م: {formatNumber(svc.vatAmount)} ر.س
                        </p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <DollarSign className="h-5 w-5" />
                ملخص المبالغ
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex justify-between">
                <span className="text-muted-foreground">المجموع الفرعي</span>
                <span>{formatNumber(appointment.subtotalAmount)} ر.س</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">ضريبة القيمة المضافة</span>
                <span>{formatNumber(appointment.vatAmount)} ر.س</span>
              </div>
              {appointment.visitFee > 0 && (
                <div className="flex justify-between">
                  <span className="text-muted-foreground">رسوم الزيارة</span>
                  {isEditing ? (
                    <Input
                      type="number"
                      value={editedVisitFee}
                      onChange={(e) => setEditedVisitFee(e.target.value)}
                      className="w-24 text-left"
                      step="0.01"
                      data-testid="input-visit-fee"
                    />
                  ) : (
                    <span>{formatNumber(appointment.visitFee)} ر.س</span>
                  )}
                </div>
              )}
              <div className="flex justify-between pt-3 border-t text-lg font-bold">
                <span>الإجمالي</span>
                <span>{formatNumber(appointment.totalAmount)} ر.س</span>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>الإجراءات</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              {canConfirm && (
                <Button 
                  className="w-full" 
                  onClick={handleConfirm}
                  disabled={confirmMutation.isPending}
                  data-testid="button-confirm-appointment"
                >
                  <CheckCircle2 className="h-4 w-4 ml-2" />
                  {confirmMutation.isPending ? "جاري التأكيد..." : "تأكيد الحجز"}
                </Button>
              )}
              {canChangeStatus && (
                <Button 
                  variant="outline" 
                  className="w-full"
                  onClick={() => setShowStatusDialog(true)}
                  data-testid="button-change-status"
                >
                  تغيير الحالة
                </Button>
              )}
              {canCancel && (
                <Button 
                  variant="destructive" 
                  className="w-full"
                  onClick={() => setShowCancelDialog(true)}
                  data-testid="button-cancel-appointment"
                >
                  <XCircle className="h-4 w-4 ml-2" />
                  إلغاء الحجز
                </Button>
              )}
              {appointment.relatedOrder && (
                <Link href={`/orders/${appointment.relatedOrder.id}`}>
                  <Button variant="outline" className="w-full">
                    <ExternalLink className="h-4 w-4 ml-2" />
                    عرض الطلب المرتبط
                  </Button>
                </Link>
              )}
            </CardContent>
          </Card>

          {appointment.updatedAt && (
            <div className="text-sm text-muted-foreground text-center">
              آخر تحديث: {formatDate(appointment.updatedAt)}
            </div>
          )}
        </div>
      </div>

      <Dialog open={showCancelDialog} onOpenChange={setShowCancelDialog}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>إلغاء الحجز</DialogTitle>
            <DialogDescription>
              هل أنت متأكد من إلغاء الحجز رقم {appointment.appointmentNumber}؟
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>سبب الإلغاء (اختياري)</Label>
              <Textarea
                value={cancelReason}
                onChange={(e) => setCancelReason(e.target.value)}
                placeholder="أدخل سبب الإلغاء..."
                rows={3}
                data-testid="input-cancel-reason"
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setShowCancelDialog(false)}>
              تراجع
            </Button>
            <Button 
              variant="destructive" 
              onClick={handleCancelConfirm}
              disabled={cancelMutation.isPending}
              data-testid="button-confirm-cancel"
            >
              {cancelMutation.isPending ? "جاري الإلغاء..." : "تأكيد الإلغاء"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={showStatusDialog} onOpenChange={setShowStatusDialog}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تغيير حالة الحجز</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>الحالة الجديدة</Label>
              <Select value={newStatus} onValueChange={setNewStatus}>
                <SelectTrigger data-testid="select-new-status">
                  <SelectValue placeholder="اختر الحالة" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="pending">قيد الانتظار</SelectItem>
                  <SelectItem value="confirmed">مؤكد</SelectItem>
                  <SelectItem value="in_progress">جاري التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                  <SelectItem value="no_show">لم يحضر</SelectItem>
                  <SelectItem value="postponed">مؤجل</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setShowStatusDialog(false)}>
              تراجع
            </Button>
            <Button 
              onClick={handleStatusChange}
              disabled={!newStatus || changeStatusMutation.isPending}
              data-testid="button-confirm-status"
            >
              {changeStatusMutation.isPending ? "جاري التغيير..." : "تأكيد"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
