import { useQuery, useMutation } from "@tanstack/react-query";
import { useState, useEffect } from "react";
import { Link, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  Calendar, 
  Plus, 
  Eye, 
  Clock, 
  CheckCircle2, 
  XCircle,
  User,
  Filter,
  MapPin,
  Phone,
  Home,
  Building2,
  RefreshCw,
  CalendarCheck,
  CalendarX,
  AlertCircle,
} from "lucide-react";

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  price: number;
}

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
}

interface AppointmentService {
  id: number;
  appointmentId: number;
  serviceId: number;
  quantity: number;
  basePrice: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  service?: Service;
  preferredEmployee?: Employee;
}

interface Appointment {
  id: number;
  appointmentNumber: string;
  source: string;
  clientId: number | null;
  branchId: number | null;
  appointmentType: string;
  scheduledAt: string;
  status: string;
  visitFee: number;
  locationLat: number | null;
  locationLng: number | null;
  addressText: string | null;
  notes: string | null;
  cancelledReason: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  createdAt: string;
  client?: Customer;
  services: AppointmentService[];
  relatedOrderId?: number | null;
}

const statusLabels: Record<string, string> = {
  pending: "قيد الانتظار",
  confirmed: "مؤكد",
  in_progress: "جاري التنفيذ",
  completed: "مكتمل",
  canceled: "ملغي",
  no_show: "لم يحضر",
  postponed: "مؤجل",
};

const statusColors: Record<string, string> = {
  pending: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
  confirmed: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
  in_progress: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
  completed: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
  canceled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
  no_show: "bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300",
  postponed: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
};

const appointmentTypeLabels: Record<string, string> = {
  in_salon: "في الصالون",
  home_service: "خدمة منزلية",
};

const formatDateOnly = (dateStr: string): string => {
  const date = new Date(dateStr);
  return new Intl.DateTimeFormat("ar-SA", {
    year: "numeric",
    month: "short",
    day: "numeric",
  }).format(date);
};

const getTodayDate = (): string => {
  const today = new Date();
  return today.toISOString().split('T')[0];
};

const pageTitles: Record<string, string> = {
  "/appointments": "الحجوزات",
  "/appointments/today": "حجوزات اليوم",
  "/appointments/app": "حجوزات التطبيق",
  "/appointments/pending": "الحجوزات المعلقة",
};

export default function AppointmentsPage() {
  const [location] = useLocation();
  const { toast } = useToast();
  const [dateFilter, setDateFilter] = useState<string>("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [typeFilter, setTypeFilter] = useState<string>("all");
  const [sourceFilter, setSourceFilter] = useState<string>("all");
  const [selectedAppointment, setSelectedAppointment] = useState<Appointment | null>(null);
  const [showCancelDialog, setShowCancelDialog] = useState(false);
  const [cancelReason, setCancelReason] = useState("");
  const [appointmentToCancel, setAppointmentToCancel] = useState<Appointment | null>(null);

  useEffect(() => {
    if (location === "/appointments/today") {
      setDateFilter(getTodayDate());
      setStatusFilter("all");
      setSourceFilter("all");
    } else if (location === "/appointments/app") {
      setDateFilter("");
      setStatusFilter("all");
      setSourceFilter("app");
    } else if (location === "/appointments/pending") {
      setDateFilter("");
      setStatusFilter("pending");
      setSourceFilter("all");
    } else {
      setDateFilter("");
      setStatusFilter("all");
      setSourceFilter("all");
    }
  }, [location]);

  const pageTitle = pageTitles[location] || "الحجوزات";

  const buildQueryString = () => {
    const params = new URLSearchParams();
    if (dateFilter) params.append("date", dateFilter);
    if (statusFilter && statusFilter !== "all") params.append("status", statusFilter);
    if (typeFilter && typeFilter !== "all") params.append("type", typeFilter);
    if (sourceFilter && sourceFilter !== "all") params.append("source", sourceFilter);
    return params.toString();
  };

  const { data: appointments = [], isLoading, refetch } = useQuery<Appointment[]>({
    queryKey: ["/api/appointments", dateFilter, statusFilter, typeFilter, sourceFilter],
    queryFn: async () => {
      const qs = buildQueryString();
      const url = qs ? `/api/appointments?${qs}` : "/api/appointments";
      const response = await fetch(url);
      if (!response.ok) throw new Error("فشل في جلب الحجوزات");
      return response.json();
    },
  });

  const confirmMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("POST", `/api/appointments/${id}/confirm`);
    },
    onSuccess: () => {
      toast({ title: "تم تأكيد الحجز بنجاح" });
      queryClient.invalidateQueries({ queryKey: ["/api/appointments"] });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في تأكيد الحجز",
        variant: "destructive",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async ({ id, reason }: { id: number; reason?: string }) => {
      return apiRequest("POST", `/api/appointments/${id}/cancel`, { reason });
    },
    onSuccess: () => {
      toast({ title: "تم إلغاء الحجز بنجاح" });
      setShowCancelDialog(false);
      setCancelReason("");
      setAppointmentToCancel(null);
      queryClient.invalidateQueries({ queryKey: ["/api/appointments"] });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في إلغاء الحجز",
        variant: "destructive",
      });
    },
  });

  const handleConfirm = (appointment: Appointment) => {
    confirmMutation.mutate(appointment.id);
  };

  const handleCancelClick = (appointment: Appointment) => {
    setAppointmentToCancel(appointment);
    setShowCancelDialog(true);
  };

  const handleCancelConfirm = () => {
    if (appointmentToCancel) {
      cancelMutation.mutate({ id: appointmentToCancel.id, reason: cancelReason });
    }
  };

  const pendingCount = appointments.filter(a => a.status === "pending").length;
  const confirmedCount = appointments.filter(a => a.status === "confirmed").length;
  const completedCount = appointments.filter(a => a.status === "completed").length;
  const canceledCount = appointments.filter(a => a.status === "canceled").length;

  return (
    <div className="p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-3">
          <Calendar className="h-8 w-8 text-primary" />
          <h1 className="text-2xl font-bold">{pageTitle}</h1>
        </div>
        <Link href="/appointments/new">
          <Button data-testid="button-new-appointment">
            <Plus className="h-4 w-4 ml-2" />
            حجز جديد
          </Button>
        </Link>
      </div>

      <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="pt-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">قيد الانتظار</p>
                <p className="text-2xl font-bold text-yellow-600">{pendingCount}</p>
              </div>
              <Clock className="h-8 w-8 text-yellow-500 opacity-50" />
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="pt-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">مؤكدة</p>
                <p className="text-2xl font-bold text-blue-600">{confirmedCount}</p>
              </div>
              <CalendarCheck className="h-8 w-8 text-blue-500 opacity-50" />
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="pt-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">مكتملة</p>
                <p className="text-2xl font-bold text-green-600">{completedCount}</p>
              </div>
              <CheckCircle2 className="h-8 w-8 text-green-500 opacity-50" />
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="pt-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">ملغاة</p>
                <p className="text-2xl font-bold text-red-600">{canceledCount}</p>
              </div>
              <CalendarX className="h-8 w-8 text-red-500 opacity-50" />
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            فلترة الحجوزات
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="space-y-2">
              <Label>التاريخ</Label>
              <Input
                type="date"
                value={dateFilter}
                onChange={(e) => setDateFilter(e.target.value)}
                data-testid="input-date-filter"
              />
            </div>
            <div className="space-y-2">
              <Label>الحالة</Label>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger data-testid="select-status-filter">
                  <SelectValue placeholder="جميع الحالات" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">جميع الحالات</SelectItem>
                  <SelectItem value="pending">قيد الانتظار</SelectItem>
                  <SelectItem value="confirmed">مؤكد</SelectItem>
                  <SelectItem value="in_progress">جاري التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                  <SelectItem value="canceled">ملغي</SelectItem>
                  <SelectItem value="no_show">لم يحضر</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>نوع الحجز</Label>
              <Select value={typeFilter} onValueChange={setTypeFilter}>
                <SelectTrigger data-testid="select-type-filter">
                  <SelectValue placeholder="جميع الأنواع" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">جميع الأنواع</SelectItem>
                  <SelectItem value="in_salon">في الصالون</SelectItem>
                  <SelectItem value="home_service">خدمة منزلية</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="flex items-end gap-2">
              <Button 
                variant="outline" 
                onClick={() => {
                  setDateFilter("");
                  setStatusFilter("all");
                  setTypeFilter("all");
                }}
                data-testid="button-clear-filters"
              >
                مسح الفلاتر
              </Button>
              <Button variant="outline" onClick={() => refetch()} data-testid="button-refresh">
                <RefreshCw className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardContent className="p-0">
          {isLoading ? (
            <div className="p-6 space-y-4">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : appointments.length === 0 ? (
            <div className="p-8 text-center text-muted-foreground">
              <Calendar className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد حجوزات</p>
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رقم الحجز</TableHead>
                  <TableHead className="text-right">العميل</TableHead>
                  <TableHead className="text-right">نوع الحجز</TableHead>
                  <TableHead className="text-right">الموعد</TableHead>
                  <TableHead className="text-right">الخدمات</TableHead>
                  <TableHead className="text-right">المبلغ</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-right">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {appointments.map((appointment) => (
                  <TableRow key={appointment.id} data-testid={`row-appointment-${appointment.id}`}>
                    <TableCell className="font-mono text-sm">
                      {appointment.appointmentNumber}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <User className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <div className="font-medium">{appointment.client?.nameAr || "عميل غير محدد"}</div>
                          {appointment.client?.phone && (
                            <div className="text-xs text-muted-foreground flex items-center gap-1">
                              <Phone className="h-3 w-3" />
                              {appointment.client.phone}
                            </div>
                          )}
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        {appointment.appointmentType === "home_service" ? (
                          <Home className="h-4 w-4 text-orange-500" />
                        ) : (
                          <Building2 className="h-4 w-4 text-blue-500" />
                        )}
                        <span>{appointmentTypeLabels[appointment.appointmentType] || appointment.appointmentType}</span>
                      </div>
                      {appointment.appointmentType === "home_service" && appointment.addressText && (
                        <div className="text-xs text-muted-foreground mt-1 flex items-center gap-1">
                          <MapPin className="h-3 w-3" />
                          {appointment.addressText.substring(0, 30)}...
                        </div>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Clock className="h-4 w-4 text-muted-foreground" />
                        {formatDate(appointment.scheduledAt)}
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="space-y-1">
                        {appointment.services.slice(0, 2).map((svc, idx) => (
                          <div key={idx} className="text-sm">
                            {svc.service?.nameAr || "خدمة"}
                            {svc.quantity > 1 && ` (×${svc.quantity})`}
                          </div>
                        ))}
                        {appointment.services.length > 2 && (
                          <span className="text-xs text-muted-foreground">
                            +{appointment.services.length - 2} خدمات أخرى
                          </span>
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="font-medium">
                      {formatNumber(appointment.totalAmount)} ر.س
                      {appointment.visitFee > 0 && (
                        <div className="text-xs text-muted-foreground">
                          (رسوم الزيارة: {formatNumber(appointment.visitFee)})
                        </div>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge className={statusColors[appointment.status]}>
                        {statusLabels[appointment.status] || appointment.status}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Link href={`/appointments/${appointment.id}`}>
                          <Button size="sm" variant="ghost" data-testid={`button-view-${appointment.id}`}>
                            <Eye className="h-4 w-4" />
                          </Button>
                        </Link>
                        {appointment.status === "pending" && (
                          <>
                            <Button 
                              size="sm" 
                              variant="ghost" 
                              className="text-green-600 hover:text-green-700"
                              onClick={() => handleConfirm(appointment)}
                              disabled={confirmMutation.isPending}
                              data-testid={`button-confirm-${appointment.id}`}
                            >
                              <CheckCircle2 className="h-4 w-4" />
                            </Button>
                            <Button 
                              size="sm" 
                              variant="ghost" 
                              className="text-red-600 hover:text-red-700"
                              onClick={() => handleCancelClick(appointment)}
                              data-testid={`button-cancel-${appointment.id}`}
                            >
                              <XCircle className="h-4 w-4" />
                            </Button>
                          </>
                        )}
                        {appointment.status === "confirmed" && (
                          <Button 
                            size="sm" 
                            variant="ghost" 
                            className="text-red-600 hover:text-red-700"
                            onClick={() => handleCancelClick(appointment)}
                            data-testid={`button-cancel-${appointment.id}`}
                          >
                            <XCircle className="h-4 w-4" />
                          </Button>
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={showCancelDialog} onOpenChange={setShowCancelDialog}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>إلغاء الحجز</DialogTitle>
            <DialogDescription>
              هل أنت متأكد من إلغاء الحجز رقم {appointmentToCancel?.appointmentNumber}؟
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>سبب الإلغاء (اختياري)</Label>
              <Input
                value={cancelReason}
                onChange={(e) => setCancelReason(e.target.value)}
                placeholder="أدخل سبب الإلغاء..."
                data-testid="input-cancel-reason"
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setShowCancelDialog(false)}>
              تراجع
            </Button>
            <Button 
              variant="destructive" 
              onClick={handleCancelConfirm}
              disabled={cancelMutation.isPending}
              data-testid="button-confirm-cancel"
            >
              {cancelMutation.isPending ? "جاري الإلغاء..." : "تأكيد الإلغاء"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={!!selectedAppointment} onOpenChange={() => setSelectedAppointment(null)}>
        {selectedAppointment && (
          <DialogContent dir="rtl" className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>تفاصيل الحجز #{selectedAppointment.appointmentNumber}</DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="text-muted-foreground">العميل</Label>
                  <p className="font-medium">{selectedAppointment.client?.nameAr || "غير محدد"}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">الموعد</Label>
                  <p className="font-medium">{formatDate(selectedAppointment.scheduledAt)}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">نوع الحجز</Label>
                  <p className="font-medium">{appointmentTypeLabels[selectedAppointment.appointmentType]}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">الحالة</Label>
                  <Badge className={statusColors[selectedAppointment.status]}>
                    {statusLabels[selectedAppointment.status]}
                  </Badge>
                </div>
              </div>
              
              {selectedAppointment.appointmentType === "home_service" && selectedAppointment.addressText && (
                <div>
                  <Label className="text-muted-foreground">العنوان</Label>
                  <p className="font-medium flex items-center gap-2">
                    <MapPin className="h-4 w-4" />
                    {selectedAppointment.addressText}
                  </p>
                </div>
              )}

              <div>
                <Label className="text-muted-foreground mb-2 block">الخدمات</Label>
                <div className="border rounded-lg divide-y">
                  {selectedAppointment.services.map((svc, idx) => (
                    <div key={idx} className="p-3 flex items-center justify-between">
                      <div>
                        <p className="font-medium">{svc.service?.nameAr || "خدمة"}</p>
                        <p className="text-sm text-muted-foreground">الكمية: {svc.quantity}</p>
                      </div>
                      <p className="font-medium">{formatNumber(svc.lineTotal)} ر.س</p>
                    </div>
                  ))}
                </div>
              </div>

              <div className="border-t pt-4 space-y-2">
                <div className="flex justify-between">
                  <span>المجموع الفرعي</span>
                  <span>{formatNumber(selectedAppointment.subtotalAmount)} ر.س</span>
                </div>
                <div className="flex justify-between">
                  <span>الضريبة</span>
                  <span>{formatNumber(selectedAppointment.vatAmount)} ر.س</span>
                </div>
                {selectedAppointment.visitFee > 0 && (
                  <div className="flex justify-between">
                    <span>رسوم الزيارة</span>
                    <span>{formatNumber(selectedAppointment.visitFee)} ر.س</span>
                  </div>
                )}
                <div className="flex justify-between text-lg font-bold">
                  <span>الإجمالي</span>
                  <span>{formatNumber(selectedAppointment.totalAmount)} ر.س</span>
                </div>
              </div>
            </div>
          </DialogContent>
        )}
      </Dialog>
    </div>
  );
}
