import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { format } from "date-fns";
import { ar } from "date-fns/locale";
import {
  ScanLine,
  LogIn,
  LogOut,
  Clock,
  Users,
  CheckCircle,
  XCircle,
} from "lucide-react";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

interface Employee {
  id: number;
  nameAr: string;
  phone?: string;
}

interface AttendanceRaw {
  id: number;
  employeeId: number;
  timestamp: string;
  direction: "in" | "out";
  source: string;
}

export default function AttendanceCheckPage() {
  const { toast } = useToast();
  const [selectedEmployeeId, setSelectedEmployeeId] = useState<string>("");
  const today = format(new Date(), "yyyy-MM-dd");

  const { data: employees = [] } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: todayAttendance = [] } = useQuery<AttendanceRaw[]>({
    queryKey: ["/api/hr/attendance-raw", { date: today }],
  });

  const checkInMutation = useMutation({
    mutationFn: async (employeeId: number) => {
      return apiRequest("POST", "/api/hr/attendance-raw", {
        employeeId,
        direction: "in",
        source: "manual",
      });
    },
    onSuccess: () => {
      toast({ title: "تم تسجيل الحضور بنجاح" });
      queryClient.invalidateQueries({ queryKey: ["/api/hr/attendance-raw"] });
      setSelectedEmployeeId("");
    },
    onError: (error: Error) => {
      toast({ title: error.message, variant: "destructive" });
    },
  });

  const checkOutMutation = useMutation({
    mutationFn: async (employeeId: number) => {
      return apiRequest("POST", "/api/hr/attendance-raw", {
        employeeId,
        direction: "out",
        source: "manual",
      });
    },
    onSuccess: () => {
      toast({ title: "تم تسجيل الانصراف بنجاح" });
      queryClient.invalidateQueries({ queryKey: ["/api/hr/attendance-raw"] });
      setSelectedEmployeeId("");
    },
    onError: (error: Error) => {
      toast({ title: error.message, variant: "destructive" });
    },
  });

  const getEmployeeLastAction = (employeeId: number) => {
    const empRecords = todayAttendance
      .filter((r) => r.employeeId === employeeId)
      .sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime());
    return empRecords[0];
  };

  const getEmployeeStatus = (employeeId: number) => {
    const lastAction = getEmployeeLastAction(employeeId);
    if (!lastAction) return "absent";
    return lastAction.direction === "in" ? "present" : "left";
  };

  const presentCount = employees.filter((e) => getEmployeeStatus(e.id) === "present").length;
  const leftCount = employees.filter((e) => getEmployeeStatus(e.id) === "left").length;
  const absentCount = employees.filter((e) => getEmployeeStatus(e.id) === "absent").length;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold">الحضور</h1>
          <p className="text-muted-foreground">
            تسجيل حضور وانصراف الموظفين - {format(new Date(), "EEEE، d MMMM yyyy", { locale: ar })}
          </p>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي الموظفين</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-employees">
              {employees.length}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">حاضر</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600" data-testid="text-present-count">
              {presentCount}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">انصرف</CardTitle>
            <LogOut className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600" data-testid="text-left-count">
              {leftCount}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">غائب</CardTitle>
            <XCircle className="h-4 w-4 text-red-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600" data-testid="text-absent-count">
              {absentCount}
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ScanLine className="h-5 w-5" />
            تسجيل حضور / انصراف
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-end">
            <div className="flex-1 w-full">
              <label className="text-sm font-medium mb-2 block">اختر الموظف</label>
              <Select value={selectedEmployeeId} onValueChange={setSelectedEmployeeId}>
                <SelectTrigger data-testid="select-check-employee">
                  <SelectValue placeholder="اختر الموظف..." />
                </SelectTrigger>
                <SelectContent>
                  {employees.map((emp) => (
                    <SelectItem key={emp.id} value={emp.id.toString()}>
                      {emp.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="flex gap-2">
              <Button
                onClick={() => selectedEmployeeId && checkInMutation.mutate(parseInt(selectedEmployeeId))}
                disabled={!selectedEmployeeId || checkInMutation.isPending}
                className="bg-green-600 hover:bg-green-700"
                data-testid="button-check-in"
              >
                <LogIn className="h-4 w-4 ml-2" />
                تسجيل حضور
              </Button>
              <Button
                onClick={() => selectedEmployeeId && checkOutMutation.mutate(parseInt(selectedEmployeeId))}
                disabled={!selectedEmployeeId || checkOutMutation.isPending}
                variant="outline"
                className="border-red-300 text-red-600 hover:bg-red-50"
                data-testid="button-check-out"
              >
                <LogOut className="h-4 w-4 ml-2" />
                تسجيل انصراف
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Clock className="h-5 w-5" />
            حالة الموظفين اليوم
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-3">
            {employees.map((emp) => {
              const status = getEmployeeStatus(emp.id);
              const lastAction = getEmployeeLastAction(emp.id);

              return (
                <div
                  key={emp.id}
                  className="flex items-center justify-between p-3 border rounded-lg"
                  data-testid={`card-employee-status-${emp.id}`}
                >
                  <div>
                    <div className="font-medium">{emp.nameAr}</div>
                    {lastAction && (
                      <div className="text-xs text-muted-foreground">
                        آخر تسجيل: {format(new Date(lastAction.timestamp), "HH:mm")}
                      </div>
                    )}
                  </div>
                  <Badge
                    variant={
                      status === "present"
                        ? "default"
                        : status === "left"
                        ? "secondary"
                        : "destructive"
                    }
                    className={
                      status === "present"
                        ? "bg-green-100 text-green-800 hover:bg-green-100"
                        : status === "left"
                        ? "bg-blue-100 text-blue-800 hover:bg-blue-100"
                        : ""
                    }
                  >
                    {status === "present" ? "حاضر" : status === "left" ? "انصرف" : "غائب"}
                  </Badge>
                </div>
              );
            })}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
