import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Fingerprint, Plus, Pencil, Trash2, Users, Clock, CheckCircle2, XCircle, AlertTriangle } from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

interface Employee {
  id: number;
  nameAr: string;
}

interface AttendanceLog {
  id: number;
  employeeId: number;
  date: string;
  workShiftId: number | null;
  scheduledStartTime: string | null;
  scheduledEndTime: string | null;
  actualCheckIn: string | null;
  actualCheckOut: string | null;
  lateMinutes: number;
  overtimeMinutes: number;
  totalWorkMinutes: number;
  status: string;
  notes: string | null;
  createdAt: string;
  employee?: Employee;
}

interface AttendanceFormData {
  employeeId: number | null;
  date: string;
  actualCheckIn: string;
  actualCheckOut: string;
  status: string;
  notes: string;
}

const statusOptions = [
  { value: "present", label: "حاضر", color: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" },
  { value: "absent", label: "غائب", color: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200" },
  { value: "late", label: "متأخر", color: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200" },
  { value: "early_leave", label: "مغادرة مبكرة", color: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200" },
  { value: "on_leave", label: "إجازة", color: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200" },
  { value: "holiday", label: "عطلة", color: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200" },
];

export default function AttendancePage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingLog, setEditingLog] = useState<AttendanceLog | null>(null);
  const [filterDate, setFilterDate] = useState(format(new Date(), "yyyy-MM-dd"));
  const [formData, setFormData] = useState<AttendanceFormData>({
    employeeId: null,
    date: format(new Date(), "yyyy-MM-dd"),
    actualCheckIn: "",
    actualCheckOut: "",
    status: "present",
    notes: "",
  });
  const { toast } = useToast();

  const { data: logs, isLoading } = useQuery<AttendanceLog[]>({
    queryKey: ["/api/hr/attendance-logs", filterDate],
    queryFn: async () => {
      const res = await fetch(`/api/hr/attendance-logs?date=${filterDate}`);
      if (!res.ok) throw new Error("Failed to fetch attendance logs");
      return res.json();
    },
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const createMutation = useMutation({
    mutationFn: (data: AttendanceFormData) =>
      apiRequest("POST", "/api/hr/attendance-logs", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/attendance-logs"] });
      toast({ title: "تم إنشاء سجل الحضور بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء سجل الحضور",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<AttendanceFormData> }) =>
      apiRequest("PUT", `/api/hr/attendance-logs/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/attendance-logs"] });
      toast({ title: "تم تحديث سجل الحضور بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث سجل الحضور",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/attendance-logs/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/attendance-logs"] });
      toast({ title: "تم حذف سجل الحضور بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف سجل الحضور",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      employeeId: null,
      date: format(new Date(), "yyyy-MM-dd"),
      actualCheckIn: "",
      actualCheckOut: "",
      status: "present",
      notes: "",
    });
    setEditingLog(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (log?: AttendanceLog) => {
    if (log) {
      setEditingLog(log);
      setFormData({
        employeeId: log.employeeId,
        date: log.date,
        actualCheckIn: log.actualCheckIn || "",
        actualCheckOut: log.actualCheckOut || "",
        status: log.status,
        notes: log.notes || "",
      });
    } else {
      setFormData({
        employeeId: null,
        date: filterDate,
        actualCheckIn: "",
        actualCheckOut: "",
        status: "present",
        notes: "",
      });
      setEditingLog(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.employeeId) {
      toast({ title: "الموظف مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.date) {
      toast({ title: "التاريخ مطلوب", variant: "destructive" });
      return;
    }

    if (editingLog) {
      updateMutation.mutate({ id: editingLog.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getEmployeeName = (id: number) => {
    return employees?.find(e => e.id === id)?.nameAr || `موظف #${id}`;
  };

  const getStatusBadge = (status: string) => {
    const statusOption = statusOptions.find(s => s.value === status);
    return statusOption || { label: status, color: "bg-gray-100 text-gray-800" };
  };

  const formatMinutes = (minutes: number) => {
    if (!minutes) return "-";
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    if (hours > 0) {
      return `${hours}س ${mins > 0 ? mins + "د" : ""}`.trim();
    }
    return `${mins}د`;
  };

  const presentCount = logs?.filter(l => l.status === "present").length || 0;
  const absentCount = logs?.filter(l => l.status === "absent").length || 0;
  const lateCount = logs?.filter(l => l.status === "late").length || 0;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">سجل الحضور</h1>
          <p className="text-muted-foreground">تتبع حضور وانصراف الموظفين</p>
        </div>
        <div className="flex items-center gap-2 flex-wrap">
          <Input
            type="date"
            value={filterDate}
            onChange={(e) => setFilterDate(e.target.value)}
            className="w-auto"
            data-testid="input-filter-date"
          />
          <Button onClick={() => handleOpenDialog()} data-testid="button-add-attendance">
            <Plus className="h-4 w-4 ml-2" />
            إضافة سجل جديد
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Fingerprint className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي السجلات</p>
                <p className="text-xl font-bold" data-testid="text-total-logs">
                  {logs?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">حاضرين</p>
                <p className="text-xl font-bold" data-testid="text-present-count">
                  {presentCount}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">غائبين</p>
                <p className="text-xl font-bold" data-testid="text-absent-count">
                  {absentCount}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">
                <AlertTriangle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">متأخرين</p>
                <p className="text-xl font-bold" data-testid="text-late-count">
                  {lateCount}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Fingerprint className="h-5 w-5" />
            سجلات الحضور - {format(new Date(filterDate), "d MMMM yyyy", { locale: ar })}
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الموظف</TableHead>
                  <TableHead className="text-right">الحضور</TableHead>
                  <TableHead className="text-right">الانصراف</TableHead>
                  <TableHead className="text-right">التأخير</TableHead>
                  <TableHead className="text-right">إضافي</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {logs?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                      لا توجد سجلات حضور لهذا اليوم
                    </TableCell>
                  </TableRow>
                ) : (
                  logs?.map((log) => (
                    <TableRow key={log.id} data-testid={`row-attendance-${log.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          {log.employee?.nameAr || getEmployeeName(log.employeeId)}
                        </div>
                      </TableCell>
                      <TableCell>
                        {log.actualCheckIn ? (
                          <div className="flex items-center gap-1 text-sm">
                            <Clock className="h-3 w-3 text-green-600" />
                            <span dir="ltr">{log.actualCheckIn}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {log.actualCheckOut ? (
                          <div className="flex items-center gap-1 text-sm">
                            <Clock className="h-3 w-3 text-red-600" />
                            <span dir="ltr">{log.actualCheckOut}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {log.lateMinutes > 0 ? (
                          <Badge variant="outline" className="text-yellow-700">
                            {formatMinutes(log.lateMinutes)}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {log.overtimeMinutes > 0 ? (
                          <Badge variant="outline" className="text-blue-700">
                            {formatMinutes(log.overtimeMinutes)}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <Badge className={getStatusBadge(log.status).color}>
                          {getStatusBadge(log.status).label}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(log)}
                            data-testid={`button-edit-attendance-${log.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(log.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-attendance-${log.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingLog ? "تعديل سجل الحضور" : "إضافة سجل حضور جديد"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="employeeId">الموظف <span className="text-destructive">*</span></Label>
                <Select
                  value={formData.employeeId?.toString() || ""}
                  onValueChange={(value) => setFormData({ ...formData, employeeId: parseInt(value) })}
                >
                  <SelectTrigger data-testid="select-attendance-employee">
                    <SelectValue placeholder="اختر الموظف" />
                  </SelectTrigger>
                  <SelectContent>
                    {employees?.map((employee) => (
                      <SelectItem key={employee.id} value={employee.id.toString()}>
                        {employee.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="date">التاريخ <span className="text-destructive">*</span></Label>
                <Input
                  id="date"
                  type="date"
                  value={formData.date}
                  onChange={(e) => setFormData({ ...formData, date: e.target.value })}
                  data-testid="input-attendance-date"
                />
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="actualCheckIn">وقت الحضور</Label>
                <Input
                  id="actualCheckIn"
                  type="time"
                  value={formData.actualCheckIn}
                  onChange={(e) => setFormData({ ...formData, actualCheckIn: e.target.value })}
                  data-testid="input-check-in"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="actualCheckOut">وقت الانصراف</Label>
                <Input
                  id="actualCheckOut"
                  type="time"
                  value={formData.actualCheckOut}
                  onChange={(e) => setFormData({ ...formData, actualCheckOut: e.target.value })}
                  data-testid="input-check-out"
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="status">الحالة <span className="text-destructive">*</span></Label>
              <Select
                value={formData.status}
                onValueChange={(value) => setFormData({ ...formData, status: value })}
              >
                <SelectTrigger data-testid="select-attendance-status">
                  <SelectValue placeholder="اختر الحالة" />
                </SelectTrigger>
                <SelectContent>
                  {statusOptions.map((status) => (
                    <SelectItem key={status.value} value={status.value}>
                      {status.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="notes">ملاحظات</Label>
              <Textarea
                id="notes"
                value={formData.notes}
                onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                placeholder="ملاحظات إضافية"
                rows={2}
                data-testid="input-attendance-notes"
              />
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-attendance"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-attendance"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
