import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
  TableFooter,
} from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { 
  Scale, 
  Building2, 
  CreditCard, 
  Users,
  RefreshCw,
  Printer,
  CheckCircle2,
  XCircle,
} from "lucide-react";
import { format } from "date-fns";
import { formatCurrency, formatNumber } from "@/lib/formatters";

interface BalanceSheetItem {
  accountCode: string;
  accountName: string;
  amount: number;
}

interface BalanceSheetData {
  assets: BalanceSheetItem[];
  liabilities: BalanceSheetItem[];
  equity: BalanceSheetItem[];
  retainedEarnings: number;
  totalAssets: number;
  totalLiabilities: number;
  totalEquity: number;
  totalLiabilitiesAndEquity: number;
  isBalanced: boolean;
  asOfDate?: string;
}

export default function BalanceSheetPage() {
  const [asOfDate, setAsOfDate] = useState<string>(
    format(new Date(), "yyyy-MM-dd")
  );

  const { data: balanceSheet, isLoading, refetch } = useQuery<BalanceSheetData>({
    queryKey: [`/api/reports/balance-sheet?asOfDate=${asOfDate}`],
  });

  const handlePrint = () => {
    window.print();
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap print:hidden">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">الميزانية العمومية</h1>
          <p className="text-muted-foreground">قائمة المركز المالي</p>
        </div>
        <div className="flex items-center gap-4 flex-wrap">
          <div className="flex items-center gap-2">
            <Label htmlFor="asOfDate">حتى تاريخ:</Label>
            <Input
              id="asOfDate"
              type="date"
              value={asOfDate}
              onChange={(e) => setAsOfDate(e.target.value)}
              className="w-40"
              data-testid="input-as-of-date"
            />
          </div>
          <Button variant="outline" onClick={() => refetch()} data-testid="button-refresh">
            <RefreshCw className="h-4 w-4 ml-2" />
            تحديث
          </Button>
          <Button variant="outline" onClick={handlePrint} data-testid="button-print">
            <Printer className="h-4 w-4 ml-2" />
            طباعة
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 print:hidden">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <Building2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الأصول</p>
                <div className="text-xl font-bold text-blue-600" data-testid="text-total-assets">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(balanceSheet?.totalAssets || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <CreditCard className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الخصوم</p>
                <div className="text-xl font-bold text-red-600" data-testid="text-total-liabilities">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(balanceSheet?.totalLiabilities || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <Users className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">حقوق الملكية</p>
                <div className="text-xl font-bold text-green-600" data-testid="text-total-equity">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(balanceSheet?.totalEquity || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className={`p-2 rounded-lg ${balanceSheet?.isBalanced ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'}`}>
                {balanceSheet?.isBalanced ? <CheckCircle2 className="h-5 w-5" /> : <XCircle className="h-5 w-5" />}
              </div>
              <div>
                <p className="text-sm text-muted-foreground">حالة التوازن</p>
                <div className={`text-lg font-bold ${balanceSheet?.isBalanced ? 'text-green-600' : 'text-red-600'}`}>
                  {isLoading ? <Skeleton className="h-6 w-16" /> : (balanceSheet?.isBalanced ? 'متوازنة' : 'غير متوازنة')}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card className="print:shadow-none print:border-0">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-blue-700 dark:text-blue-400">
              <Building2 className="h-5 w-5" />
              الأصول
            </CardTitle>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="space-y-2">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-10 w-full" />
                ))}
              </div>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">رقم الحساب</TableHead>
                    <TableHead className="text-right">اسم الحساب</TableHead>
                    <TableHead className="text-left">المبلغ</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {balanceSheet?.assets && balanceSheet.assets.length > 0 ? (
                    balanceSheet.assets.map((item) => (
                      <TableRow key={item.accountCode} data-testid={`row-asset-${item.accountCode}`}>
                        <TableCell className="font-mono">{item.accountCode}</TableCell>
                        <TableCell>{item.accountName}</TableCell>
                        <TableCell className="font-mono text-left">
                          {formatNumber(item.amount)}
                        </TableCell>
                      </TableRow>
                    ))
                  ) : (
                    <TableRow>
                      <TableCell colSpan={3} className="text-center text-muted-foreground">
                        لا توجد أصول مسجلة
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
                <TableFooter>
                  <TableRow className="bg-blue-50 dark:bg-blue-950">
                    <TableCell colSpan={2} className="font-bold">إجمالي الأصول</TableCell>
                    <TableCell className="font-mono font-bold text-left text-blue-600">
                      {formatCurrency(balanceSheet?.totalAssets || 0)}
                    </TableCell>
                  </TableRow>
                </TableFooter>
              </Table>
            )}
          </CardContent>
        </Card>

        <div className="space-y-6">
          <Card className="print:shadow-none print:border-0">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-red-700 dark:text-red-400">
                <CreditCard className="h-5 w-5" />
                الخصوم
              </CardTitle>
            </CardHeader>
            <CardContent>
              {isLoading ? (
                <div className="space-y-2">
                  {[1, 2].map((i) => (
                    <Skeleton key={i} className="h-10 w-full" />
                  ))}
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">رقم الحساب</TableHead>
                      <TableHead className="text-right">اسم الحساب</TableHead>
                      <TableHead className="text-left">المبلغ</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {balanceSheet?.liabilities && balanceSheet.liabilities.length > 0 ? (
                      balanceSheet.liabilities.map((item) => (
                        <TableRow key={item.accountCode} data-testid={`row-liability-${item.accountCode}`}>
                          <TableCell className="font-mono">{item.accountCode}</TableCell>
                          <TableCell>{item.accountName}</TableCell>
                          <TableCell className="font-mono text-left">
                            {formatNumber(item.amount)}
                          </TableCell>
                        </TableRow>
                      ))
                    ) : (
                      <TableRow>
                        <TableCell colSpan={3} className="text-center text-muted-foreground">
                          لا توجد خصوم مسجلة
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                  <TableFooter>
                    <TableRow className="bg-red-50 dark:bg-red-950">
                      <TableCell colSpan={2} className="font-bold">إجمالي الخصوم</TableCell>
                      <TableCell className="font-mono font-bold text-left text-red-600">
                        {formatCurrency(balanceSheet?.totalLiabilities || 0)}
                      </TableCell>
                    </TableRow>
                  </TableFooter>
                </Table>
              )}
            </CardContent>
          </Card>

          <Card className="print:shadow-none print:border-0">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-green-700 dark:text-green-400">
                <Users className="h-5 w-5" />
                حقوق الملكية
              </CardTitle>
            </CardHeader>
            <CardContent>
              {isLoading ? (
                <div className="space-y-2">
                  {[1, 2].map((i) => (
                    <Skeleton key={i} className="h-10 w-full" />
                  ))}
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">رقم الحساب</TableHead>
                      <TableHead className="text-right">اسم الحساب</TableHead>
                      <TableHead className="text-left">المبلغ</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {balanceSheet?.equity && balanceSheet.equity.length > 0 ? (
                      balanceSheet.equity.map((item) => (
                        <TableRow key={item.accountCode} data-testid={`row-equity-${item.accountCode}`}>
                          <TableCell className="font-mono">{item.accountCode}</TableCell>
                          <TableCell>{item.accountName}</TableCell>
                          <TableCell className="font-mono text-left">
                            {formatNumber(item.amount)}
                          </TableCell>
                        </TableRow>
                      ))
                    ) : (
                      <TableRow>
                        <TableCell colSpan={3} className="text-center text-muted-foreground">
                          لا توجد حسابات ملكية مسجلة
                        </TableCell>
                      </TableRow>
                    )}
                    {balanceSheet?.retainedEarnings !== undefined && balanceSheet.retainedEarnings !== 0 && (
                      <TableRow data-testid="row-retained-earnings">
                        <TableCell className="font-mono">-</TableCell>
                        <TableCell>الأرباح المحتجزة (صافي الدخل)</TableCell>
                        <TableCell className={`font-mono text-left ${balanceSheet.retainedEarnings >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                          {formatNumber(balanceSheet.retainedEarnings)}
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                  <TableFooter>
                    <TableRow className="bg-green-50 dark:bg-green-950">
                      <TableCell colSpan={2} className="font-bold">إجمالي حقوق الملكية</TableCell>
                      <TableCell className="font-mono font-bold text-left text-green-600">
                        {formatCurrency(balanceSheet?.totalEquity || 0)}
                      </TableCell>
                    </TableRow>
                  </TableFooter>
                </Table>
              )}
            </CardContent>
          </Card>
        </div>
      </div>

      <Card className={`${balanceSheet?.isBalanced ? 'bg-green-50 dark:bg-green-950 border-green-200' : 'bg-red-50 dark:bg-red-950 border-red-200'} print:shadow-none`}>
        <CardContent className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center p-4 rounded-lg bg-background">
              <p className="text-sm text-muted-foreground mb-1">إجمالي الأصول</p>
              <p className="text-2xl font-bold text-blue-600">
                {formatCurrency(balanceSheet?.totalAssets || 0)}
              </p>
            </div>
            <div className="flex items-center justify-center">
              <Scale className={`h-12 w-12 ${balanceSheet?.isBalanced ? 'text-green-600' : 'text-red-600'}`} />
            </div>
            <div className="text-center p-4 rounded-lg bg-background">
              <p className="text-sm text-muted-foreground mb-1">الخصوم + حقوق الملكية</p>
              <p className="text-2xl font-bold text-green-600">
                {formatCurrency(balanceSheet?.totalLiabilitiesAndEquity || 0)}
              </p>
            </div>
          </div>
          {!balanceSheet?.isBalanced && (
            <div className="mt-4 text-center text-red-600">
              <p className="font-semibold">الميزانية غير متوازنة!</p>
              <p className="text-sm">الفرق: {formatCurrency(Math.abs((balanceSheet?.totalAssets || 0) - (balanceSheet?.totalLiabilitiesAndEquity || 0)))}</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
