import { useQuery, useMutation } from "@tanstack/react-query";
import { useState, useRef } from "react";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { Avatar, AvatarImage, AvatarFallback } from "@/components/ui/avatar";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Award, Plus, Pencil, Trash2, CheckCircle2, XCircle, Upload, X, ImageIcon } from "lucide-react";

interface Brand {
  id: number;
  nameAr: string;
  nameEn: string | null;
  descriptionAr: string | null;
  imageUrl: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface BrandFormData {
  nameAr: string;
  nameEn: string;
  descriptionAr: string;
  imageUrl: string;
  isActive: boolean;
}

export default function BrandsPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingBrand, setEditingBrand] = useState<Brand | null>(null);
  const [formData, setFormData] = useState<BrandFormData>({
    nameAr: "",
    nameEn: "",
    descriptionAr: "",
    imageUrl: "",
    isActive: true,
  });
  const [uploadingImage, setUploadingImage] = useState(false);
  const [previewImage, setPreviewImage] = useState<string | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const { data: brands, isLoading } = useQuery<Brand[]>({
    queryKey: ["/api/brands"],
  });

  const createMutation = useMutation({
    mutationFn: (data: BrandFormData) =>
      apiRequest("POST", "/api/brands", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/brands"] });
      toast({ title: "تم إنشاء العلامة التجارية بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء العلامة التجارية",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<BrandFormData> }) =>
      apiRequest("PUT", `/api/brands/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/brands"] });
      toast({ title: "تم تحديث العلامة التجارية بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث العلامة التجارية",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/brands/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/brands"] });
      toast({ title: "تم تعطيل العلامة التجارية بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف العلامة التجارية",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({ nameAr: "", nameEn: "", descriptionAr: "", imageUrl: "", isActive: true });
    setEditingBrand(null);
    setIsDialogOpen(false);
    setPreviewImage(null);
  };

  const handleOpenDialog = (brand?: Brand) => {
    if (brand) {
      setEditingBrand(brand);
      setFormData({
        nameAr: brand.nameAr,
        nameEn: brand.nameEn || "",
        descriptionAr: brand.descriptionAr || "",
        imageUrl: brand.imageUrl || "",
        isActive: brand.isActive,
      });
      setPreviewImage(brand.imageUrl);
    } else {
      setFormData({ nameAr: "", nameEn: "", descriptionAr: "", imageUrl: "", isActive: true });
      setEditingBrand(null);
      setPreviewImage(null);
    }
    setIsDialogOpen(true);
  };

  const handleImageUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!allowedTypes.includes(file.type)) {
      toast({
        title: "نوع ملف غير مدعوم",
        description: "يرجى اختيار صورة بصيغة JPEG, PNG, GIF أو WebP",
        variant: "destructive",
      });
      return;
    }

    const maxSize = 5 * 1024 * 1024;
    if (file.size > maxSize) {
      toast({
        title: "حجم الملف كبير جداً",
        description: "الحد الأقصى لحجم الصورة هو 5 ميجابايت",
        variant: "destructive",
      });
      return;
    }

    setUploadingImage(true);

    try {
      const formDataUpload = new FormData();
      formDataUpload.append("image", file);

      const response = await fetch("/api/upload/brand-image", {
        method: "POST",
        body: formDataUpload,
      });

      if (!response.ok) {
        throw new Error("فشل في رفع الصورة");
      }

      const result = await response.json();
      setFormData({ ...formData, imageUrl: result.url });
      setPreviewImage(result.url);
      toast({ title: "تم رفع الصورة بنجاح" });
    } catch (error) {
      const reader = new FileReader();
      reader.onload = (e) => {
        const base64 = e.target?.result as string;
        setFormData({ ...formData, imageUrl: base64 });
        setPreviewImage(base64);
      };
      reader.readAsDataURL(file);
      toast({ title: "تم تحميل الصورة محلياً" });
    } finally {
      setUploadingImage(false);
    }
  };

  const handleRemoveImage = () => {
    setFormData({ ...formData, imageUrl: "" });
    setPreviewImage(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = "";
    }
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم العلامة التجارية مطلوب", variant: "destructive" });
      return;
    }

    if (editingBrand) {
      updateMutation.mutate({ id: editingBrand.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const activeBrands = brands?.filter(b => b.isActive) || [];
  const inactiveBrands = brands?.filter(b => !b.isActive) || [];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">العلامات التجارية</h1>
          <p className="text-muted-foreground">إدارة العلامات التجارية للمنتجات</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-brand">
          <Plus className="h-4 w-4 ml-2" />
          إضافة علامة تجارية
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Award className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي العلامات</p>
                <p className="text-xl font-bold" data-testid="text-total-brands">
                  {brands?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">علامات مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-active-brands">
                  {activeBrands.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">علامات غير مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-inactive-brands">
                  {inactiveBrands.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Award className="h-5 w-5" />
            قائمة العلامات التجارية
          </CardTitle>
          <CardDescription>
            جميع العلامات التجارية المعرّفة في النظام
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الصورة</TableHead>
                  <TableHead className="text-right">الاسم بالعربية</TableHead>
                  <TableHead className="text-right">الاسم بالإنجليزية</TableHead>
                  <TableHead className="text-right">الوصف</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {brands?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                      لا توجد علامات تجارية مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  brands?.map((brand) => (
                    <TableRow key={brand.id} data-testid={`row-brand-${brand.id}`}>
                      <TableCell>
                        <Avatar className="h-10 w-10">
                          {brand.imageUrl ? (
                            <AvatarImage src={brand.imageUrl} alt={brand.nameAr} />
                          ) : null}
                          <AvatarFallback>
                            <Award className="h-5 w-5" />
                          </AvatarFallback>
                        </Avatar>
                      </TableCell>
                      <TableCell className="font-medium">
                        {brand.nameAr}
                      </TableCell>
                      <TableCell className="text-muted-foreground">
                        {brand.nameEn || "-"}
                      </TableCell>
                      <TableCell className="max-w-xs truncate">
                        {brand.descriptionAr || "-"}
                      </TableCell>
                      <TableCell>
                        {brand.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            مفعّل
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير مفعّل
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(brand)}
                            data-testid={`button-edit-brand-${brand.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(brand.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-brand-${brand.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingBrand ? "تعديل العلامة التجارية" : "إضافة علامة تجارية جديدة"}
            </DialogTitle>
            <DialogDescription>
              أدخل بيانات العلامة التجارية
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>صورة العلامة التجارية</Label>
              <div className="flex items-center gap-4">
                {previewImage ? (
                  <div className="relative">
                    <Avatar className="h-20 w-20">
                      <AvatarImage src={previewImage} alt="Preview" />
                      <AvatarFallback>
                        <Award className="h-8 w-8" />
                      </AvatarFallback>
                    </Avatar>
                    <Button
                      variant="destructive"
                      size="icon"
                      className="absolute -top-2 -left-2 h-6 w-6"
                      onClick={handleRemoveImage}
                      type="button"
                    >
                      <X className="h-3 w-3" />
                    </Button>
                  </div>
                ) : (
                  <div className="h-20 w-20 rounded-full border-2 border-dashed flex items-center justify-center bg-muted">
                    <ImageIcon className="h-8 w-8 text-muted-foreground" />
                  </div>
                )}
                <div className="flex-1">
                  <input
                    ref={fileInputRef}
                    type="file"
                    accept="image/jpeg,image/png,image/gif,image/webp"
                    className="hidden"
                    onChange={handleImageUpload}
                    data-testid="input-brand-image"
                  />
                  <Button
                    variant="outline"
                    onClick={() => fileInputRef.current?.click()}
                    disabled={uploadingImage}
                    type="button"
                    className="w-full"
                    data-testid="button-upload-image"
                  >
                    <Upload className="h-4 w-4 ml-2" />
                    {uploadingImage ? "جاري الرفع..." : "رفع صورة"}
                  </Button>
                  <p className="text-xs text-muted-foreground mt-1">
                    JPEG, PNG, GIF, WebP - حد أقصى 5 ميجابايت
                  </p>
                </div>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="nameAr">الاسم بالعربية <span className="text-destructive">*</span></Label>
              <Input
                id="nameAr"
                value={formData.nameAr}
                onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                placeholder="مثال: سامسونج"
                data-testid="input-brand-name-ar"
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="nameEn">الاسم بالإنجليزية</Label>
              <Input
                id="nameEn"
                value={formData.nameEn}
                onChange={(e) => setFormData({ ...formData, nameEn: e.target.value })}
                placeholder="مثال: Samsung"
                dir="ltr"
                className="text-left"
                data-testid="input-brand-name-en"
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="descriptionAr">الوصف</Label>
              <Textarea
                id="descriptionAr"
                value={formData.descriptionAr}
                onChange={(e) => setFormData({ ...formData, descriptionAr: e.target.value })}
                placeholder="وصف مختصر للعلامة التجارية"
                rows={3}
                data-testid="input-brand-description"
              />
            </div>
            
            <div className="flex items-center justify-between">
              <Label htmlFor="isActive">الحالة</Label>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">
                  {formData.isActive ? "مفعّل" : "غير مفعّل"}
                </span>
                <Switch
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                  data-testid="switch-brand-active"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-brand"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-brand"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
