import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { formatDate, formatCurrency } from "@/lib/formatters";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import {
  Plus,
  Search,
  Package,
  User,
  Calendar,
  DollarSign,
  X,
  Check,
  Trash2,
  Eye,
} from "lucide-react";
import type {
  CafeteriaEmployeeCustodyWithDetails,
  CafeteriaProduct,
  Employee,
} from "@shared/schema";

const statusLabels: Record<string, string> = {
  open: "مفتوحة",
  returned: "مرتجعة",
  consumed: "مستهلكة",
  partially_returned: "مرتجعة جزئياً",
  lost: "مفقودة",
  damaged: "تالفة",
};

const statusColors: Record<string, string> = {
  open: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200",
  returned: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200",
  consumed: "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200",
  partially_returned: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200",
  lost: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
  damaged: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200",
};

const closureReasons = [
  { value: "returned", label: "مرتجعة" },
  { value: "consumed", label: "مستهلكة" },
  { value: "lost", label: "مفقودة" },
  { value: "damaged", label: "تالفة" },
];

export default function CafeteriaCustodiesPage() {
  const { toast } = useToast();
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [isViewDialogOpen, setIsViewDialogOpen] = useState(false);
  const [isCloseDialogOpen, setIsCloseDialogOpen] = useState(false);
  const [selectedCustody, setSelectedCustody] = useState<CafeteriaEmployeeCustodyWithDetails | null>(null);

  const [newCustody, setNewCustody] = useState({
    employeeId: "",
    cafeteriaProductId: "",
    quantity: "",
    custodyDate: new Date().toISOString().split("T")[0],
    notes: "",
  });

  const [closureData, setClosureData] = useState({
    closedByEmployeeId: "",
    closureReason: "",
    returnedQuantity: "",
    closureNotes: "",
  });

  const { data: custodies, isLoading } = useQuery<CafeteriaEmployeeCustodyWithDetails[]>({
    queryKey: ["/api/cafeteria/custodies"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: products } = useQuery<CafeteriaProduct[]>({
    queryKey: ["/api/cafeteria/products"],
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", "/api/cafeteria/custodies", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/custodies"] });
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/stock-batches"] });
      setIsCreateDialogOpen(false);
      resetNewCustody();
      toast({ title: "تم إنشاء العهدة بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في إنشاء العهدة", variant: "destructive" });
    },
  });

  const closeMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      apiRequest("PUT", `/api/cafeteria/custodies/${id}/close`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/custodies"] });
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/stock-batches"] });
      setIsCloseDialogOpen(false);
      setSelectedCustody(null);
      resetClosureData();
      toast({ title: "تم إغلاق العهدة بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في إغلاق العهدة", variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/cafeteria/custodies/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/custodies"] });
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/stock-batches"] });
      toast({ title: "تم حذف العهدة بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في حذف العهدة", variant: "destructive" });
    },
  });

  const resetNewCustody = () => {
    setNewCustody({
      employeeId: "",
      cafeteriaProductId: "",
      quantity: "",
      custodyDate: new Date().toISOString().split("T")[0],
      notes: "",
    });
  };

  const resetClosureData = () => {
    setClosureData({
      closedByEmployeeId: "",
      closureReason: "",
      returnedQuantity: "",
      closureNotes: "",
    });
  };

  const handleCreate = () => {
    if (!newCustody.employeeId || !newCustody.cafeteriaProductId || !newCustody.quantity) {
      toast({ title: "خطأ", description: "يرجى ملء جميع الحقول المطلوبة", variant: "destructive" });
      return;
    }

    createMutation.mutate({
      employeeId: parseInt(newCustody.employeeId),
      cafeteriaProductId: parseInt(newCustody.cafeteriaProductId),
      quantity: parseFloat(newCustody.quantity),
      custodyDate: newCustody.custodyDate,
      notes: newCustody.notes || null,
      unitCost: 0,
      totalCost: 0,
    });
  };

  const handleClose = () => {
    if (!selectedCustody || !closureData.closedByEmployeeId || !closureData.closureReason) {
      toast({ title: "خطأ", description: "يرجى ملء جميع الحقول المطلوبة", variant: "destructive" });
      return;
    }

    closeMutation.mutate({
      id: selectedCustody.id,
      data: {
        closedByEmployeeId: parseInt(closureData.closedByEmployeeId),
        closureReason: closureData.closureReason,
        returnedQuantity: closureData.returnedQuantity ? parseFloat(closureData.returnedQuantity) : undefined,
        closureNotes: closureData.closureNotes || undefined,
      },
    });
  };

  const filteredCustodies = custodies?.filter((custody) => {
    const matchesSearch =
      custody.employee?.nameAr?.toLowerCase().includes(search.toLowerCase()) ||
      custody.cafeteriaProduct?.nameAr?.toLowerCase().includes(search.toLowerCase());
    const matchesStatus = statusFilter === "all" || custody.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const openCustodies = custodies?.filter((c) => c.status === "open") || [];
  const closedCustodies = custodies?.filter((c) => c.status !== "open") || [];
  const totalValue = openCustodies.reduce((sum, c) => sum + c.totalCost, 0);

  return (
    <div className="p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold">عهد موظفي الكافتيريا</h1>
          <p className="text-muted-foreground">إدارة صرف المنتجات للموظفين</p>
        </div>
        <Button onClick={() => setIsCreateDialogOpen(true)} data-testid="button-create-custody">
          <Plus className="ml-2 h-4 w-4" />
          عهدة جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي العهد</CardTitle>
            <Package className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-custodies">{custodies?.length || 0}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">عهد مفتوحة</CardTitle>
            <User className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600" data-testid="text-open-custodies">{openCustodies.length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">عهد مغلقة</CardTitle>
            <Check className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600" data-testid="text-closed-custodies">{closedCustodies.length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">قيمة العهد المفتوحة</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-value">{formatCurrency(totalValue)}</div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <div className="flex flex-col md:flex-row gap-4 items-start md:items-center justify-between">
            <CardTitle>قائمة العهد</CardTitle>
            <div className="flex flex-col md:flex-row gap-2 w-full md:w-auto">
              <div className="relative flex-1 md:w-64">
                <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="بحث..."
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  className="pr-10"
                  data-testid="input-search"
                />
              </div>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger className="w-full md:w-40" data-testid="select-status-filter">
                  <SelectValue placeholder="الحالة" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">جميع الحالات</SelectItem>
                  <SelectItem value="open">مفتوحة</SelectItem>
                  <SelectItem value="returned">مرتجعة</SelectItem>
                  <SelectItem value="consumed">مستهلكة</SelectItem>
                  <SelectItem value="partially_returned">مرتجعة جزئياً</SelectItem>
                  <SelectItem value="lost">مفقودة</SelectItem>
                  <SelectItem value="damaged">تالفة</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8">جاري التحميل...</div>
          ) : filteredCustodies?.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">لا توجد عهد</div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>الموظف</TableHead>
                  <TableHead>المنتج</TableHead>
                  <TableHead>الكمية</TableHead>
                  <TableHead>التكلفة</TableHead>
                  <TableHead>التاريخ</TableHead>
                  <TableHead>الحالة</TableHead>
                  <TableHead>الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredCustodies?.map((custody) => (
                  <TableRow key={custody.id} data-testid={`row-custody-${custody.id}`}>
                    <TableCell className="font-medium">{custody.employee?.nameAr || "-"}</TableCell>
                    <TableCell>{custody.cafeteriaProduct?.nameAr || "-"}</TableCell>
                    <TableCell>{custody.quantity}</TableCell>
                    <TableCell>{formatCurrency(custody.totalCost)}</TableCell>
                    <TableCell>{formatDate(custody.custodyDate)}</TableCell>
                    <TableCell>
                      <Badge className={statusColors[custody.status] || ""}>
                        {statusLabels[custody.status] || custody.status}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => {
                            setSelectedCustody(custody);
                            setIsViewDialogOpen(true);
                          }}
                          data-testid={`button-view-${custody.id}`}
                        >
                          <Eye className="h-4 w-4" />
                        </Button>
                        {custody.status === "open" && (
                          <>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                setSelectedCustody(custody);
                                setIsCloseDialogOpen(true);
                              }}
                              data-testid={`button-close-${custody.id}`}
                            >
                              <X className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                if (confirm("هل أنت متأكد من حذف هذه العهدة؟")) {
                                  deleteMutation.mutate(custody.id);
                                }
                              }}
                              data-testid={`button-delete-${custody.id}`}
                            >
                              <Trash2 className="h-4 w-4 text-red-500" />
                            </Button>
                          </>
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
        <DialogContent className="max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>إنشاء عهدة جديدة</DialogTitle>
            <DialogDescription>صرف منتج كافتيريا لموظف</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label>الموظف *</Label>
              <Select value={newCustody.employeeId} onValueChange={(v) => setNewCustody({ ...newCustody, employeeId: v })}>
                <SelectTrigger data-testid="select-employee">
                  <SelectValue placeholder="اختر الموظف" />
                </SelectTrigger>
                <SelectContent>
                  {employees?.map((emp) => (
                    <SelectItem key={emp.id} value={emp.id.toString()}>
                      {emp.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>المنتج *</Label>
              <Select
                value={newCustody.cafeteriaProductId}
                onValueChange={(v) => setNewCustody({ ...newCustody, cafeteriaProductId: v })}
              >
                <SelectTrigger data-testid="select-product">
                  <SelectValue placeholder="اختر المنتج" />
                </SelectTrigger>
                <SelectContent>
                  {products
                    ?.filter((p) => p.isActive && (p.productType === "consume" || p.productType === "both"))
                    .map((product) => (
                      <SelectItem key={product.id} value={product.id.toString()}>
                        {product.nameAr}
                      </SelectItem>
                    ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>الكمية *</Label>
              <Input
                type="number"
                min="0.01"
                step="0.01"
                value={newCustody.quantity}
                onChange={(e) => setNewCustody({ ...newCustody, quantity: e.target.value })}
                data-testid="input-quantity"
              />
            </div>
            <div>
              <Label>تاريخ العهدة</Label>
              <Input
                type="date"
                value={newCustody.custodyDate}
                onChange={(e) => setNewCustody({ ...newCustody, custodyDate: e.target.value })}
                data-testid="input-date"
              />
            </div>
            <div>
              <Label>ملاحظات</Label>
              <Textarea
                value={newCustody.notes}
                onChange={(e) => setNewCustody({ ...newCustody, notes: e.target.value })}
                rows={2}
                data-testid="input-notes"
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsCreateDialogOpen(false)}>
              إلغاء
            </Button>
            <Button onClick={handleCreate} disabled={createMutation.isPending} data-testid="button-submit-create">
              {createMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isViewDialogOpen} onOpenChange={setIsViewDialogOpen}>
        <DialogContent className="max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>تفاصيل العهدة</DialogTitle>
          </DialogHeader>
          {selectedCustody && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="text-muted-foreground">الموظف</Label>
                  <p className="font-medium">{selectedCustody.employee?.nameAr || "-"}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">المنتج</Label>
                  <p className="font-medium">{selectedCustody.cafeteriaProduct?.nameAr || "-"}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">الكمية</Label>
                  <p className="font-medium">{selectedCustody.quantity}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">تكلفة الوحدة</Label>
                  <p className="font-medium">{formatCurrency(selectedCustody.unitCost)}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">إجمالي التكلفة</Label>
                  <p className="font-medium">{formatCurrency(selectedCustody.totalCost)}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">تاريخ العهدة</Label>
                  <p className="font-medium">{formatDate(selectedCustody.custodyDate)}</p>
                </div>
                <div>
                  <Label className="text-muted-foreground">الحالة</Label>
                  <Badge className={statusColors[selectedCustody.status] || ""}>
                    {statusLabels[selectedCustody.status] || selectedCustody.status}
                  </Badge>
                </div>
                {selectedCustody.createdByEmployee && (
                  <div>
                    <Label className="text-muted-foreground">أنشأها</Label>
                    <p className="font-medium">{selectedCustody.createdByEmployee.nameAr}</p>
                  </div>
                )}
              </div>
              {selectedCustody.notes && (
                <div>
                  <Label className="text-muted-foreground">ملاحظات</Label>
                  <p className="mt-1">{selectedCustody.notes}</p>
                </div>
              )}
              {selectedCustody.status !== "open" && (
                <div className="border-t pt-4 space-y-2">
                  <h4 className="font-medium">تفاصيل الإغلاق</h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label className="text-muted-foreground">تاريخ الإغلاق</Label>
                      <p className="font-medium">{selectedCustody.closedDate ? formatDate(selectedCustody.closedDate) : "-"}</p>
                    </div>
                    <div>
                      <Label className="text-muted-foreground">سبب الإغلاق</Label>
                      <p className="font-medium">
                        {closureReasons.find((r) => r.value === selectedCustody.closureReason)?.label ||
                          selectedCustody.closureReason}
                      </p>
                    </div>
                    {selectedCustody.returnedQuantity !== null && selectedCustody.returnedQuantity > 0 && (
                      <div>
                        <Label className="text-muted-foreground">الكمية المرتجعة</Label>
                        <p className="font-medium">{selectedCustody.returnedQuantity}</p>
                      </div>
                    )}
                    {selectedCustody.closedByEmployee && (
                      <div>
                        <Label className="text-muted-foreground">أغلقها</Label>
                        <p className="font-medium">{selectedCustody.closedByEmployee.nameAr}</p>
                      </div>
                    )}
                  </div>
                  {selectedCustody.closureNotes && (
                    <div>
                      <Label className="text-muted-foreground">ملاحظات الإغلاق</Label>
                      <p className="mt-1">{selectedCustody.closureNotes}</p>
                    </div>
                  )}
                </div>
              )}
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsViewDialogOpen(false)}>
              إغلاق
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isCloseDialogOpen} onOpenChange={setIsCloseDialogOpen}>
        <DialogContent className="max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>إغلاق العهدة</DialogTitle>
            <DialogDescription>
              إغلاق عهدة {selectedCustody?.cafeteriaProduct?.nameAr} للموظف {selectedCustody?.employee?.nameAr}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label>الموظف المغلق *</Label>
              <Select
                value={closureData.closedByEmployeeId}
                onValueChange={(v) => setClosureData({ ...closureData, closedByEmployeeId: v })}
              >
                <SelectTrigger data-testid="select-closed-by">
                  <SelectValue placeholder="اختر الموظف" />
                </SelectTrigger>
                <SelectContent>
                  {employees?.map((emp) => (
                    <SelectItem key={emp.id} value={emp.id.toString()}>
                      {emp.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>سبب الإغلاق *</Label>
              <Select
                value={closureData.closureReason}
                onValueChange={(v) => setClosureData({ ...closureData, closureReason: v })}
              >
                <SelectTrigger data-testid="select-closure-reason">
                  <SelectValue placeholder="اختر السبب" />
                </SelectTrigger>
                <SelectContent>
                  {closureReasons.map((reason) => (
                    <SelectItem key={reason.value} value={reason.value}>
                      {reason.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            {closureData.closureReason === "returned" && (
              <div>
                <Label>الكمية المرتجعة</Label>
                <Input
                  type="number"
                  min="0"
                  max={selectedCustody?.quantity || 0}
                  step="0.01"
                  value={closureData.returnedQuantity}
                  onChange={(e) => setClosureData({ ...closureData, returnedQuantity: e.target.value })}
                  placeholder={`الحد الأقصى: ${selectedCustody?.quantity || 0}`}
                  data-testid="input-returned-quantity"
                />
              </div>
            )}
            <div>
              <Label>ملاحظات الإغلاق</Label>
              <Textarea
                value={closureData.closureNotes}
                onChange={(e) => setClosureData({ ...closureData, closureNotes: e.target.value })}
                rows={2}
                data-testid="input-closure-notes"
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsCloseDialogOpen(false)}>
              إلغاء
            </Button>
            <Button onClick={handleClose} disabled={closeMutation.isPending} data-testid="button-submit-close">
              {closeMutation.isPending ? "جاري الإغلاق..." : "إغلاق العهدة"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
