import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import {
  Search,
  Package,
  TrendingUp,
  TrendingDown,
  Eye,
  AlertTriangle,
} from "lucide-react";
import type { CafeteriaInventorySummaryItem, CafeteriaStockMovementReportItem } from "@shared/schema";

function InventorySkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-40" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-8 w-8" /></TableCell>
    </TableRow>
  );
}

function EmptyState() {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <Package className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا توجد منتجات في المخزون</h3>
      <p className="text-muted-foreground">قم بإضافة منتجات وفواتير مشتريات لعرض المخزون</p>
    </div>
  );
}

function getMovementIcon(type: string) {
  switch (type) {
    case "purchase":
      return <TrendingUp className="h-4 w-4 text-green-500" />;
    case "sale":
      return <TrendingDown className="h-4 w-4 text-red-500" />;
    default:
      return <Package className="h-4 w-4 text-muted-foreground" />;
  }
}

function getMovementTypeName(type: string) {
  switch (type) {
    case "purchase":
      return "شراء";
    case "sale":
      return "بيع";
    case "adjustment":
      return "تعديل";
    default:
      return type;
  }
}

export default function CafeteriaInventoryPage() {
  const [searchQuery, setSearchQuery] = useState("");
  const [movementsDialogOpen, setMovementsDialogOpen] = useState(false);
  const [selectedProductId, setSelectedProductId] = useState<number | null>(null);
  const [selectedProductName, setSelectedProductName] = useState<string>("");

  const { data: inventory, isLoading } = useQuery<CafeteriaInventorySummaryItem[]>({
    queryKey: ["/api/cafeteria/inventory/summary"],
  });

  const { data: movements, isLoading: movementsLoading } = useQuery<CafeteriaStockMovementReportItem[]>({
    queryKey: ["/api/cafeteria/inventory/movements", selectedProductId],
    enabled: !!selectedProductId,
  });

  const filteredInventory = inventory?.filter((item) =>
    item.productNameAr.includes(searchQuery) ||
    item.sku?.includes(searchQuery)
  );

  const handleViewMovements = (item: CafeteriaInventorySummaryItem) => {
    setSelectedProductId(item.cafeteriaProductId);
    setSelectedProductName(item.productNameAr);
    setMovementsDialogOpen(true);
  };

  const totalItems = inventory?.length || 0;
  const totalQuantity = inventory?.reduce((sum, item) => sum + item.quantityOnHand, 0) || 0;
  const totalValue = inventory?.reduce((sum, item) => sum + item.totalCost, 0) || 0;
  const lowStockItems = inventory?.filter((item) => item.quantityOnHand <= 5).length || 0;

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">مخزون الكافتيريا</h1>
          <p className="text-muted-foreground">تقارير مخزون منتجات الكافتيريا</p>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">عدد المنتجات</CardTitle>
            <Package className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalItems}</div>
            <p className="text-xs text-muted-foreground">منتج مسجل</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي الكميات</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalQuantity.toLocaleString("ar-SA")}</div>
            <p className="text-xs text-muted-foreground">وحدة</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">قيمة المخزون</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalValue.toLocaleString("ar-SA")}</div>
            <p className="text-xs text-muted-foreground">ر.س</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">منتجات منخفضة</CardTitle>
            <AlertTriangle className="h-4 w-4 text-yellow-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600">{lowStockItems}</div>
            <p className="text-xs text-muted-foreground">تحتاج إعادة طلب</p>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">ملخص المخزون</CardTitle>
            <div className="relative w-full sm:w-72">
              <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="بحث بالاسم أو الكود..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pr-10"
                data-testid="input-search-inventory"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">المنتج</TableHead>
                    <TableHead className="text-right">الكود</TableHead>
                    <TableHead className="text-right">الكمية المتوفرة</TableHead>
                    <TableHead className="text-right">القيمة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {[1, 2, 3, 4, 5].map((i) => (
                    <InventorySkeleton key={i} />
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : !filteredInventory || filteredInventory.length === 0 ? (
            <EmptyState />
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">المنتج</TableHead>
                    <TableHead className="text-right">الكود</TableHead>
                    <TableHead className="text-right">الكمية المتوفرة</TableHead>
                    <TableHead className="text-right">القيمة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredInventory.map((item) => (
                    <TableRow
                      key={item.cafeteriaProductId}
                      className="hover-elevate"
                      data-testid={`inventory-row-${item.cafeteriaProductId}`}
                    >
                      <TableCell>
                        <div className="flex items-center gap-3">
                          <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10 text-primary">
                            <Package className="h-5 w-5" />
                          </div>
                          <div>
                            <p className="font-medium">{item.productNameAr}</p>
                            {item.unitNameAr && (
                              <p className="text-xs text-muted-foreground">{item.unitNameAr}</p>
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell className="font-mono text-sm">
                        {item.sku || "-"}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <span className="font-mono font-medium tabular-nums">
                            {item.quantityOnHand.toLocaleString("ar-SA")}
                          </span>
                          {item.quantityOnHand <= 5 && (
                            <Badge variant="outline" className="text-yellow-600 border-yellow-600">
                              منخفض
                            </Badge>
                          )}
                        </div>
                      </TableCell>
                      <TableCell className="font-mono tabular-nums">
                        {item.totalCost.toLocaleString("ar-SA")} ر.س
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleViewMovements(item)}
                          data-testid={`view-movements-${item.cafeteriaProductId}`}
                        >
                          <Eye className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={movementsDialogOpen} onOpenChange={setMovementsDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>حركة المخزون - {selectedProductName}</DialogTitle>
            <DialogDescription>عرض جميع حركات المخزون للمنتج</DialogDescription>
          </DialogHeader>
          {movementsLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : !movements || movements.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">
              لا توجد حركات مخزون لهذا المنتج
            </div>
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">النوع</TableHead>
                    <TableHead className="text-right">وارد</TableHead>
                    <TableHead className="text-right">صادر</TableHead>
                    <TableHead className="text-right">الرصيد</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {movements.map((movement) => (
                    <TableRow key={movement.id}>
                      <TableCell>{movement.movementDate}</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {getMovementIcon(movement.movementType)}
                          <span>{getMovementTypeName(movement.movementType)}</span>
                        </div>
                      </TableCell>
                      <TableCell className="font-mono text-green-600">
                        {movement.quantityIn > 0 ? `+${movement.quantityIn.toLocaleString("ar-SA")}` : "-"}
                      </TableCell>
                      <TableCell className="font-mono text-red-600">
                        {movement.quantityOut > 0 ? `-${movement.quantityOut.toLocaleString("ar-SA")}` : "-"}
                      </TableCell>
                      <TableCell className="font-mono font-medium">
                        {movement.balance.toLocaleString("ar-SA")}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
