import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useForm, useFieldArray } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import {
  Plus,
  Search,
  ShoppingCart,
  Trash2,
  Eye,
} from "lucide-react";
import type {
  CafeteriaPurchaseInvoiceWithDetails,
  CafeteriaProductWithUnit,
  SupplierWithBankAccounts,
  PaymentMethod,
  CreateCafeteriaPurchaseInvoiceRequest,
} from "@shared/schema";

const invoiceFormSchema = z.object({
  supplierId: z.number({ required_error: "المورد مطلوب" }),
  invoiceDate: z.string({ required_error: "تاريخ الفاتورة مطلوب" }),
  dueDate: z.string().optional(),
  notes: z.string().optional(),
  lines: z.array(z.object({
    cafeteriaProductId: z.number({ required_error: "المنتج مطلوب" }),
    quantity: z.number().min(0.01, "الكمية يجب أن تكون أكبر من صفر"),
    purchasePricePerUnit: z.number().min(0, "السعر يجب أن يكون صفر أو أكثر"),
    vatType: z.enum(["inclusive", "exclusive", "exempt"]),
    vatRate: z.number().min(0).max(1),
    expiryDate: z.string().optional(),
  })).min(1, "يجب إضافة منتج واحد على الأقل"),
});

type InvoiceFormData = z.infer<typeof invoiceFormSchema>;

function InvoiceSkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-4 w-40" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-6 w-16" /></TableCell>
      <TableCell><Skeleton className="h-8 w-8" /></TableCell>
    </TableRow>
  );
}

function EmptyState({ onAdd }: { onAdd: () => void }) {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <ShoppingCart className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا توجد فواتير مشتريات</h3>
      <p className="text-muted-foreground mb-4">ابدأ بإضافة أول فاتورة مشتريات للكافتيريا</p>
      <Button onClick={onAdd} data-testid="empty-add-cafeteria-invoice">
        إضافة فاتورة
        <Plus className="h-4 w-4 mr-2" />
      </Button>
    </div>
  );
}

function getPaymentStatusBadge(status: string) {
  switch (status) {
    case "paid":
      return <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">مدفوعة</Badge>;
    case "partially_paid":
      return <Badge className="bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">مدفوعة جزئياً</Badge>;
    default:
      return <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">غير مدفوعة</Badge>;
  }
}

export default function CafeteriaPurchasesPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [dialogOpen, setDialogOpen] = useState(false);
  const [detailDialogOpen, setDetailDialogOpen] = useState(false);
  const [selectedInvoice, setSelectedInvoice] = useState<CafeteriaPurchaseInvoiceWithDetails | null>(null);

  const { data: invoices, isLoading } = useQuery<CafeteriaPurchaseInvoiceWithDetails[]>({
    queryKey: ["/api/cafeteria/purchase-invoices"],
  });

  const { data: products } = useQuery<CafeteriaProductWithUnit[]>({
    queryKey: ["/api/cafeteria/products"],
  });

  const { data: suppliers } = useQuery<SupplierWithBankAccounts[]>({
    queryKey: ["/api/suppliers"],
  });

  const form = useForm<InvoiceFormData>({
    resolver: zodResolver(invoiceFormSchema),
    defaultValues: {
      supplierId: undefined,
      invoiceDate: new Date().toISOString().split("T")[0],
      dueDate: "",
      notes: "",
      lines: [{ cafeteriaProductId: 0, quantity: 1, purchasePricePerUnit: 0, vatType: "exclusive", vatRate: 0.15, expiryDate: "" }],
    },
  });

  const { fields, append, remove } = useFieldArray({
    control: form.control,
    name: "lines",
  });

  const createMutation = useMutation({
    mutationFn: async (data: CreateCafeteriaPurchaseInvoiceRequest) => {
      await apiRequest("POST", "/api/cafeteria/purchase-invoices", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/purchase-invoices"] });
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/inventory/summary"] });
      toast({
        title: "تم الإضافة بنجاح",
        description: "تم إنشاء فاتورة الشراء وتحديث المخزون",
      });
      setDialogOpen(false);
      form.reset();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء إنشاء الفاتورة",
        variant: "destructive",
      });
    },
  });

  const filteredInvoices = invoices?.filter((invoice) =>
    invoice.supplier?.nameAr?.includes(searchQuery)
  );

  const handleAdd = () => {
    form.reset({
      supplierId: undefined,
      invoiceDate: new Date().toISOString().split("T")[0],
      dueDate: "",
      notes: "",
      lines: [{ cafeteriaProductId: 0, quantity: 1, purchasePricePerUnit: 0, vatType: "exclusive", vatRate: 0.15, expiryDate: "" }],
    });
    setDialogOpen(true);
  };

  const handleViewDetails = (invoice: CafeteriaPurchaseInvoiceWithDetails) => {
    setSelectedInvoice(invoice);
    setDetailDialogOpen(true);
  };

  const onSubmit = (data: InvoiceFormData) => {
    const request: CreateCafeteriaPurchaseInvoiceRequest = {
      supplierId: data.supplierId,
      invoiceDate: data.invoiceDate,
      dueDate: data.dueDate || undefined,
      notes: data.notes || undefined,
      lines: data.lines.map((line) => ({
        cafeteriaProductId: line.cafeteriaProductId,
        quantity: line.quantity,
        purchasePricePerUnit: line.purchasePricePerUnit,
        vatType: line.vatType,
        vatRate: line.vatRate,
        expiryDate: line.expiryDate || undefined,
      })),
    };
    createMutation.mutate(request);
  };

  const handleProductChange = (index: number, productId: number) => {
    const product = products?.find((p) => p.id === productId);
    if (product) {
      form.setValue(`lines.${index}.cafeteriaProductId`, productId);
      form.setValue(`lines.${index}.purchasePricePerUnit`, product.defaultPurchasePrice || 0);
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">مشتريات الكافتيريا</h1>
          <p className="text-muted-foreground">إدارة فواتير مشتريات الكافتيريا</p>
        </div>
        <Button onClick={handleAdd} data-testid="button-add-cafeteria-invoice">
          فاتورة جديدة
          <Plus className="h-4 w-4 mr-2" />
        </Button>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">فواتير المشتريات</CardTitle>
            <div className="relative w-full sm:w-72">
              <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="بحث بالمورد..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pr-10"
                data-testid="input-search-invoices"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">رقم</TableHead>
                    <TableHead className="text-right">المورد</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">الإجمالي</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {[1, 2, 3, 4, 5].map((i) => (
                    <InvoiceSkeleton key={i} />
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : !filteredInvoices || filteredInvoices.length === 0 ? (
            <EmptyState onAdd={handleAdd} />
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">رقم</TableHead>
                    <TableHead className="text-right">المورد</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">الإجمالي</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredInvoices.map((invoice) => (
                    <TableRow
                      key={invoice.id}
                      className="hover-elevate cursor-pointer"
                      onClick={() => handleViewDetails(invoice)}
                      data-testid={`cafeteria-invoice-row-${invoice.id}`}
                    >
                      <TableCell className="font-mono">#{invoice.id}</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-3">
                          <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10 text-primary">
                            <ShoppingCart className="h-5 w-5" />
                          </div>
                          <span className="font-medium">{invoice.supplier?.nameAr || "-"}</span>
                        </div>
                      </TableCell>
                      <TableCell>{invoice.invoiceDate}</TableCell>
                      <TableCell className="font-mono tabular-nums">
                        {invoice.totalAmount.toLocaleString("ar-SA")} ر.س
                      </TableCell>
                      <TableCell>
                        {getPaymentStatusBadge(invoice.paymentStatus)}
                      </TableCell>
                      <TableCell>
                        <Button variant="ghost" size="icon" data-testid={`view-invoice-${invoice.id}`}>
                          <Eye className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>فاتورة مشتريات جديدة</DialogTitle>
            <DialogDescription>أدخل بيانات فاتورة مشتريات الكافتيريا</DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="supplierId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>المورد *</FormLabel>
                      <Select onValueChange={(v) => field.onChange(parseInt(v))} value={field.value?.toString()}>
                        <FormControl>
                          <SelectTrigger data-testid="select-supplier">
                            <SelectValue placeholder="اختر المورد" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {suppliers?.map((supplier) => (
                            <SelectItem key={supplier.id} value={supplier.id.toString()}>
                              {supplier.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="invoiceDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>تاريخ الفاتورة *</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} data-testid="input-invoice-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="space-y-2">
                <div className="flex items-center justify-between">
                  <FormLabel>المنتجات *</FormLabel>
                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={() => append({ cafeteriaProductId: 0, quantity: 1, purchasePricePerUnit: 0, vatType: "exclusive", vatRate: 0.15, expiryDate: "" })}
                    data-testid="button-add-line"
                  >
                    إضافة منتج
                    <Plus className="h-4 w-4 mr-1" />
                  </Button>
                </div>

                <div className="rounded-lg border p-4 space-y-4">
                  {fields.map((field, index) => (
                    <div key={field.id} className="grid grid-cols-12 gap-2 items-end">
                      <div className="col-span-4">
                        <FormField
                          control={form.control}
                          name={`lines.${index}.cafeteriaProductId`}
                          render={({ field }) => (
                            <FormItem>
                              {index === 0 && <FormLabel>المنتج</FormLabel>}
                              <Select
                                onValueChange={(v) => handleProductChange(index, parseInt(v))}
                                value={field.value?.toString()}
                              >
                                <FormControl>
                                  <SelectTrigger data-testid={`select-product-${index}`}>
                                    <SelectValue placeholder="اختر المنتج" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  {products?.filter(p => p.isActive).map((product) => (
                                    <SelectItem key={product.id} value={product.id.toString()}>
                                      {product.nameAr}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="col-span-2">
                        <FormField
                          control={form.control}
                          name={`lines.${index}.quantity`}
                          render={({ field }) => (
                            <FormItem>
                              {index === 0 && <FormLabel>الكمية</FormLabel>}
                              <FormControl>
                                <Input
                                  type="number"
                                  step="0.01"
                                  {...field}
                                  onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                                  data-testid={`input-quantity-${index}`}
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="col-span-2">
                        <FormField
                          control={form.control}
                          name={`lines.${index}.purchasePricePerUnit`}
                          render={({ field }) => (
                            <FormItem>
                              {index === 0 && <FormLabel>السعر</FormLabel>}
                              <FormControl>
                                <Input
                                  type="number"
                                  step="0.01"
                                  {...field}
                                  onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                                  data-testid={`input-price-${index}`}
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="col-span-3">
                        <FormField
                          control={form.control}
                          name={`lines.${index}.vatType`}
                          render={({ field }) => (
                            <FormItem>
                              {index === 0 && <FormLabel>الضريبة</FormLabel>}
                              <Select onValueChange={field.onChange} value={field.value}>
                                <FormControl>
                                  <SelectTrigger data-testid={`select-vat-${index}`}>
                                    <SelectValue />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="exclusive">خارج السعر 15%</SelectItem>
                                  <SelectItem value="inclusive">داخل السعر 15%</SelectItem>
                                  <SelectItem value="exempt">معفى</SelectItem>
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="col-span-1">
                        {fields.length > 1 && (
                          <Button
                            type="button"
                            variant="ghost"
                            size="icon"
                            onClick={() => remove(index)}
                            data-testid={`remove-line-${index}`}
                          >
                            <Trash2 className="h-4 w-4 text-destructive" />
                          </Button>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>ملاحظات</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="ملاحظات إضافية..." data-testid="input-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2 pt-4">
                <Button type="button" variant="outline" onClick={() => setDialogOpen(false)}>
                  إلغاء
                </Button>
                <Button type="submit" disabled={createMutation.isPending} data-testid="button-save-invoice">
                  {createMutation.isPending ? "جاري الحفظ..." : "حفظ الفاتورة"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      <Dialog open={detailDialogOpen} onOpenChange={setDetailDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>تفاصيل فاتورة المشتريات #{selectedInvoice?.id}</DialogTitle>
            <DialogDescription>عرض تفاصيل فاتورة المشتريات</DialogDescription>
          </DialogHeader>
          {selectedInvoice && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">المورد:</span>
                  <span className="mr-2 font-medium">{selectedInvoice.supplier?.nameAr}</span>
                </div>
                <div>
                  <span className="text-muted-foreground">التاريخ:</span>
                  <span className="mr-2 font-medium">{selectedInvoice.invoiceDate}</span>
                </div>
                <div>
                  <span className="text-muted-foreground">الحالة:</span>
                  <span className="mr-2">{getPaymentStatusBadge(selectedInvoice.paymentStatus)}</span>
                </div>
              </div>

              <div className="rounded-lg border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">المنتج</TableHead>
                      <TableHead className="text-right">الكمية</TableHead>
                      <TableHead className="text-right">السعر</TableHead>
                      <TableHead className="text-right">الضريبة</TableHead>
                      <TableHead className="text-right">الإجمالي</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {selectedInvoice.lines.map((line) => (
                      <TableRow key={line.id}>
                        <TableCell>{line.cafeteriaProduct?.nameAr || "-"}</TableCell>
                        <TableCell className="font-mono">{line.quantity}</TableCell>
                        <TableCell className="font-mono">{line.purchasePricePerUnit.toLocaleString("ar-SA")}</TableCell>
                        <TableCell className="font-mono">{line.vatAmount.toLocaleString("ar-SA")}</TableCell>
                        <TableCell className="font-mono font-medium">{line.lineTotal.toLocaleString("ar-SA")}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>

              <div className="flex justify-between items-center text-lg font-bold border-t pt-4">
                <span>الإجمالي:</span>
                <span className="font-mono">{selectedInvoice.totalAmount.toLocaleString("ar-SA")} ر.س</span>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
