import { useQuery } from "@tanstack/react-query";
import { useRoute, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { ThermalReceipt, PrintData } from "@/components/receipt/ThermalReceipt";
import { Printer, ArrowRight, AlertCircle } from "lucide-react";

interface CafeteriaPrintData extends PrintData {
  isCafeteria?: boolean;
}

export default function CafeteriaSalesPrint() {
  const [, params] = useRoute("/cafeteria/sales/:id/print");
  const [, navigate] = useLocation();
  const orderId = params?.id ? parseInt(params.id) : 0;

  const { data: printData, isLoading, error } = useQuery<CafeteriaPrintData>({
    queryKey: ["/api/cafeteria/sales-orders", orderId, "print-data"],
    enabled: orderId > 0,
  });

  const handlePrintReceipt = () => {
    const printWindow = window.open("", "_blank");
    if (!printWindow) {
      alert("يرجى السماح بالنوافذ المنبثقة لطباعة الفاتورة");
      return;
    }
    
    const receiptElement = document.getElementById("invoice-section");
    if (!receiptElement) return;
    
    printWindow.document.write(`
      <!DOCTYPE html>
      <html dir="rtl" lang="ar">
      <head>
        <meta charset="UTF-8">
        <title>فاتورة كافتيريا ${printData?.invoiceNumber}</title>
        <style>
          @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap');
          
          * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
          }
          
          body {
            font-family: 'Cairo', 'Courier New', monospace;
            background: white;
            color: black;
          }
          
          @page {
            size: 58mm auto;
            margin: 0;
          }
          
          @media print {
            body {
              width: 58mm;
            }
          }
          
          ${getReceiptStyles()}
        </style>
      </head>
      <body>
        ${receiptElement.innerHTML}
      </body>
      </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    setTimeout(() => {
      printWindow.print();
      printWindow.close();
    }, 500);
  };

  if (error) {
    return (
      <div className="flex flex-col items-center justify-center min-h-[60vh] text-center">
        <div className="rounded-full bg-destructive/10 p-4 mb-4">
          <AlertCircle className="h-8 w-8 text-destructive" />
        </div>
        <h2 className="text-xl font-semibold mb-2">حدث خطأ</h2>
        <p className="text-muted-foreground mb-4">تعذر تحميل بيانات الفاتورة</p>
        <Button variant="outline" onClick={() => navigate("/cafeteria/sales")}>
          العودة للمبيعات
          <ArrowRight className="h-4 w-4 mr-2" />
        </Button>
      </div>
    );
  }

  if (isLoading || !printData) {
    return (
      <div className="space-y-6">
        <div className="flex items-center gap-4">
          <Skeleton className="h-10 w-32" />
          <Skeleton className="h-10 w-32" />
        </div>
        <div className="flex justify-center">
          <Skeleton className="h-[600px] w-[240px]" />
        </div>
      </div>
    );
  }

  return (
    <>
      <style>{getPrintMediaStyles()}</style>
      
      <div className="print-hide space-y-6">
        <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
          <div>
            <h1 className="text-2xl font-bold" data-testid="page-title">طباعة فاتورة الكافتيريا</h1>
            <p className="text-muted-foreground">معاينة وطباعة فاتورة مبيعات الكافتيريا</p>
          </div>
          <div className="flex gap-2">
            <Button variant="outline" onClick={() => navigate("/cafeteria/sales")} data-testid="button-back">
              العودة للمبيعات
              <ArrowRight className="h-4 w-4 mr-2" />
            </Button>
            <Button onClick={handlePrintReceipt} data-testid="button-print-receipt">
              طباعة الفاتورة
              <Printer className="h-4 w-4 mr-2" />
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Card>
            <CardContent className="p-6">
              <h3 className="font-semibold mb-4">معاينة الفاتورة الحرارية</h3>
              <div className="bg-gray-100 dark:bg-gray-800 rounded-lg p-4 flex justify-center">
                <div className="bg-white dark:bg-gray-900 shadow-lg">
                  <ThermalReceipt data={printData} />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <h3 className="font-semibold mb-4">معلومات الطلب</h3>
              <div className="space-y-3 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">رقم الطلب:</span>
                  <span className="font-mono font-medium">{printData.invoiceNumber}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">الحالة:</span>
                  <span className="font-medium">{printData.statusLabel}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">التاريخ:</span>
                  <span className="font-medium">{printData.invoiceDate} {printData.invoiceTime}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">العميل:</span>
                  <span className="font-medium">{printData.customerName}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">الموظف:</span>
                  <span className="font-medium">{printData.cashierName}</span>
                </div>
                <hr className="my-3" />
                <div className="flex justify-between">
                  <span className="text-muted-foreground">المجموع قبل الضريبة:</span>
                  <span className="font-mono">{printData.subtotalBeforeDiscount.toFixed(2)} ر.س</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">ضريبة القيمة المضافة (15%):</span>
                  <span className="font-mono">{printData.vatAmount.toFixed(2)} ر.س</span>
                </div>
                <div className="flex justify-between text-lg font-bold border-t pt-2">
                  <span>الإجمالي:</span>
                  <span className="font-mono">{printData.totalWithVat.toFixed(2)} ر.س</span>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      <div id="print-root" className="print-only" dir="rtl">
        <div id="invoice-section">
          <ThermalReceipt data={printData} />
        </div>
      </div>
    </>
  );
}

function getPrintMediaStyles(): string {
  return `
    @media screen {
      .print-only {
        display: none !important;
      }
    }
    
    @media print {
      @page {
        size: 58mm auto;
        margin: 0;
      }
      
      body {
        margin: 0;
        padding: 0;
        background: white !important;
        -webkit-print-color-adjust: exact;
        print-color-adjust: exact;
      }
      
      .print-hide,
      .navbar,
      .sidebar,
      .print-actions,
      header,
      nav,
      aside,
      [data-sidebar],
      .toaster {
        display: none !important;
      }
      
      .print-only {
        display: block !important;
      }
      
      #print-root {
        display: block;
        width: 58mm;
        margin: 0 auto;
      }
      
      #invoice-section {
        width: 58mm;
        margin: 0 auto;
      }
      
      ${getReceiptStyles()}
    }
  `;
}

function getReceiptStyles(): string {
  return `
    .thermal-receipt {
      width: 58mm;
      max-width: 230px;
      margin: 0 auto;
      padding: 4mm;
      background: white;
      color: #000;
      font-family: 'Cairo', 'Courier New', monospace;
      font-size: 10px;
      line-height: 1.4;
      text-align: right;
    }

    .receipt-header {
      text-align: center;
      margin-bottom: 3mm;
    }

    .receipt-title {
      font-size: 11px;
      font-weight: normal;
      margin-bottom: 2mm;
    }

    .company-name {
      font-size: 14px;
      font-weight: bold;
      margin-bottom: 2mm;
    }

    .company-phone,
    .branch-name {
      font-size: 10px;
      margin-bottom: 1mm;
    }

    .credit-note-subtitle {
      font-size: 9px;
      color: #666;
      margin-bottom: 2mm;
    }

    .receipt-divider {
      border-top: 1px dashed #000;
      margin: 2mm 0;
    }

    .receipt-info,
    .payment-info {
      margin-bottom: 2mm;
    }

    .info-row {
      display: flex;
      justify-content: space-between;
      margin-bottom: 1mm;
      font-size: 9px;
    }

    .info-label {
      font-weight: bold;
    }

    .info-value {
      text-align: left;
      direction: ltr;
    }

    .items-table {
      width: 100%;
      border-collapse: collapse;
      margin: 2mm 0;
      font-size: 8px;
    }

    .items-table th {
      border-bottom: 1px solid #000;
      padding: 1mm;
      text-align: center;
      font-weight: bold;
      font-size: 8px;
    }

    .items-table td {
      padding: 1mm;
      text-align: center;
      font-size: 8px;
    }

    .col-service {
      text-align: right;
      width: 35%;
    }

    .col-price,
    .col-discount {
      width: 15%;
    }

    .col-expert {
      width: 20%;
    }

    .col-time {
      width: 15%;
    }

    .summary-section {
      margin: 2mm 0;
    }

    .summary-table {
      width: 100%;
      font-size: 9px;
    }

    .summary-table td {
      padding: 1mm 0;
    }

    .summary-label {
      text-align: right;
    }

    .summary-value {
      text-align: left;
      direction: ltr;
      font-family: 'Courier New', monospace;
    }

    .discount-value {
      color: #c00;
    }

    .refund-value {
      color: #c00;
    }

    .summary-total {
      font-weight: bold;
      font-size: 11px;
      border-top: 1px solid #000;
      border-bottom: 1px solid #000;
    }

    .receipt-footer {
      text-align: center;
      margin-top: 3mm;
    }

    .cashier-info {
      font-size: 9px;
      margin-bottom: 2mm;
    }

    .extra-note {
      font-size: 8px;
      color: #666;
      margin-bottom: 2mm;
    }

    .qr-container {
      display: flex;
      justify-content: center;
      margin: 2mm 0;
    }

    .zatca-note {
      font-size: 7px;
      color: #666;
      margin-top: 1mm;
    }
  `;
}
