import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useForm, useFieldArray } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import {
  Plus,
  Search,
  Receipt,
  Trash2,
  Eye,
  CheckCircle,
  XCircle,
  Printer,
} from "lucide-react";
import type {
  CafeteriaSalesOrderWithDetails,
  CafeteriaProductWithUnit,
  EmployeeWithRole,
  Customer,
  CreateCafeteriaSalesOrderRequest,
} from "@shared/schema";

const orderFormSchema = z.object({
  createdByEmployeeId: z.number({ required_error: "الموظف مطلوب" }),
  clientId: z.number().optional(),
  branchId: z.number().optional(),
  notes: z.string().optional(),
  lines: z.array(z.object({
    cafeteriaProductId: z.number({ required_error: "المنتج مطلوب" }),
    quantity: z.number().min(1, "الكمية يجب أن تكون 1 أو أكثر"),
  })).min(1, "يجب إضافة منتج واحد على الأقل"),
});

type OrderFormData = z.infer<typeof orderFormSchema>;

function OrderSkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-32" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-6 w-16" /></TableCell>
      <TableCell><Skeleton className="h-8 w-8" /></TableCell>
    </TableRow>
  );
}

function EmptyState({ onAdd }: { onAdd: () => void }) {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <Receipt className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا توجد طلبات مبيعات</h3>
      <p className="text-muted-foreground mb-4">ابدأ بإنشاء أول طلب مبيعات للكافتيريا</p>
      <Button onClick={onAdd} data-testid="empty-add-cafeteria-order">
        طلب جديد
        <Plus className="h-4 w-4 mr-2" />
      </Button>
    </div>
  );
}

function getStatusBadge(status: string) {
  switch (status) {
    case "completed":
      return <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">مكتمل</Badge>;
    case "canceled":
      return <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">ملغي</Badge>;
    default:
      return <Badge className="bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">جديد</Badge>;
  }
}

export default function CafeteriaSalesPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [dialogOpen, setDialogOpen] = useState(false);
  const [detailDialogOpen, setDetailDialogOpen] = useState(false);
  const [cancelDialogOpen, setCancelDialogOpen] = useState(false);
  const [cancelReason, setCancelReason] = useState("");
  const [selectedOrder, setSelectedOrder] = useState<CafeteriaSalesOrderWithDetails | null>(null);

  const { data: orders, isLoading } = useQuery<CafeteriaSalesOrderWithDetails[]>({
    queryKey: ["/api/cafeteria/sales-orders"],
  });

  const { data: products } = useQuery<CafeteriaProductWithUnit[]>({
    queryKey: ["/api/cafeteria/products"],
  });

  const { data: employees } = useQuery<EmployeeWithRole[]>({
    queryKey: ["/api/employees"],
  });

  const { data: customers } = useQuery<Customer[]>({
    queryKey: ["/api/customers"],
  });

  const form = useForm<OrderFormData>({
    resolver: zodResolver(orderFormSchema),
    defaultValues: {
      createdByEmployeeId: undefined,
      clientId: undefined,
      branchId: undefined,
      notes: "",
      lines: [{ cafeteriaProductId: 0, quantity: 1 }],
    },
  });

  const { fields, append, remove } = useFieldArray({
    control: form.control,
    name: "lines",
  });

  const createMutation = useMutation({
    mutationFn: async (data: CreateCafeteriaSalesOrderRequest) => {
      await apiRequest("POST", "/api/cafeteria/sales-orders", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/sales-orders"] });
      toast({
        title: "تم الإضافة بنجاح",
        description: "تم إنشاء طلب المبيعات",
      });
      setDialogOpen(false);
      form.reset();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء إنشاء الطلب",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, status }: { id: number; status: string }) => {
      await apiRequest("PUT", `/api/cafeteria/sales-orders/${id}`, { status });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/sales-orders"] });
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/inventory/summary"] });
      toast({
        title: "تم التحديث بنجاح",
        description: "تم تحديث حالة الطلب",
      });
      setDetailDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء تحديث الطلب",
        variant: "destructive",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async ({ id, reason }: { id: number; reason: string }) => {
      await apiRequest("POST", `/api/cafeteria/sales-orders/${id}/cancel`, { reason });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/sales-orders"] });
      queryClient.invalidateQueries({ queryKey: ["/api/cafeteria/inventory/summary"] });
      toast({
        title: "تم الإلغاء بنجاح",
        description: "تم إلغاء الطلب وإنشاء إشعار دائن",
      });
      setCancelDialogOpen(false);
      setDetailDialogOpen(false);
      setCancelReason("");
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء إلغاء الطلب",
        variant: "destructive",
      });
    },
  });

  const filteredOrders = orders?.filter((order) => {
    const matchesSearch = order.orderNumber.includes(searchQuery) ||
      order.createdByEmployee?.nameAr?.includes(searchQuery);
    const matchesStatus = statusFilter === "all" || order.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const handleAdd = () => {
    form.reset({
      createdByEmployeeId: undefined,
      clientId: undefined,
      branchId: undefined,
      notes: "",
      lines: [{ cafeteriaProductId: 0, quantity: 1 }],
    });
    setDialogOpen(true);
  };

  const handleViewDetails = (order: CafeteriaSalesOrderWithDetails) => {
    setSelectedOrder(order);
    setDetailDialogOpen(true);
  };

  const handleComplete = (order: CafeteriaSalesOrderWithDetails) => {
    updateMutation.mutate({ id: order.id, status: "completed" });
  };

  const handleOpenCancelDialog = (order: CafeteriaSalesOrderWithDetails) => {
    setSelectedOrder(order);
    setCancelReason("");
    setCancelDialogOpen(true);
  };

  const handleConfirmCancel = () => {
    if (!selectedOrder || !cancelReason.trim()) return;
    cancelMutation.mutate({ id: selectedOrder.id, reason: cancelReason.trim() });
  };

  const onSubmit = (data: OrderFormData) => {
    const request: CreateCafeteriaSalesOrderRequest = {
      createdByEmployeeId: data.createdByEmployeeId,
      clientId: data.clientId,
      branchId: data.branchId,
      notes: data.notes,
      lines: data.lines.map((line) => ({
        cafeteriaProductId: line.cafeteriaProductId,
        quantity: line.quantity,
      })),
    };
    createMutation.mutate(request);
  };

  const calculateTotal = () => {
    const lines = form.watch("lines");
    return lines.reduce((sum, line) => {
      const product = products?.find((p) => p.id === line.cafeteriaProductId);
      return sum + (product?.defaultSellPrice || 0) * line.quantity;
    }, 0);
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">مبيعات الكافتيريا</h1>
          <p className="text-muted-foreground">إدارة طلبات مبيعات الكافتيريا</p>
        </div>
        <Button onClick={handleAdd} data-testid="button-add-cafeteria-order">
          طلب جديد
          <Plus className="h-4 w-4 mr-2" />
        </Button>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">طلبات المبيعات</CardTitle>
            <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger className="w-36" data-testid="select-status-filter">
                  <SelectValue placeholder="الحالة" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  <SelectItem value="new">جديد</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                  <SelectItem value="canceled">ملغي</SelectItem>
                </SelectContent>
              </Select>
              <div className="relative w-full sm:w-72">
                <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                  placeholder="بحث بالرقم أو الموظف..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pr-10"
                  data-testid="input-search-orders"
                />
              </div>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">رقم الطلب</TableHead>
                    <TableHead className="text-right">الموظف</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">الإجمالي</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {[1, 2, 3, 4, 5].map((i) => (
                    <OrderSkeleton key={i} />
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : !filteredOrders || filteredOrders.length === 0 ? (
            <EmptyState onAdd={handleAdd} />
          ) : (
            <div className="rounded-lg border">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">رقم الطلب</TableHead>
                    <TableHead className="text-right">الموظف</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">الإجمالي</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="w-12"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredOrders.map((order) => (
                    <TableRow
                      key={order.id}
                      className="hover-elevate cursor-pointer"
                      onClick={() => handleViewDetails(order)}
                      data-testid={`cafeteria-order-row-${order.id}`}
                    >
                      <TableCell className="font-mono font-medium">{order.orderNumber}</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-3">
                          <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10 text-primary">
                            <Receipt className="h-5 w-5" />
                          </div>
                          <span>{order.createdByEmployee?.nameAr || "-"}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {new Date(order.orderDate).toLocaleDateString("ar-SA")}
                      </TableCell>
                      <TableCell className="font-mono tabular-nums">
                        {order.totalAmount.toLocaleString("ar-SA")} ر.س
                      </TableCell>
                      <TableCell>
                        {getStatusBadge(order.status)}
                      </TableCell>
                      <TableCell>
                        <Button variant="ghost" size="icon" data-testid={`view-order-${order.id}`}>
                          <Eye className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>طلب مبيعات جديد</DialogTitle>
            <DialogDescription>إنشاء طلب مبيعات جديد للكافتيريا</DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="createdByEmployeeId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الموظف *</FormLabel>
                      <Select onValueChange={(v) => field.onChange(parseInt(v))} value={field.value?.toString()}>
                        <FormControl>
                          <SelectTrigger data-testid="select-employee">
                            <SelectValue placeholder="اختر الموظف" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {employees?.filter(e => e.isActive).map((employee) => (
                            <SelectItem key={employee.id} value={employee.id.toString()}>
                              {employee.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="clientId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>العميل (اختياري)</FormLabel>
                      <Select onValueChange={(v) => field.onChange(v ? parseInt(v) : undefined)} value={field.value?.toString()}>
                        <FormControl>
                          <SelectTrigger data-testid="select-client">
                            <SelectValue placeholder="اختر العميل" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {customers?.map((customer) => (
                            <SelectItem key={customer.id} value={customer.id.toString()}>
                              {customer.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="space-y-2">
                <div className="flex items-center justify-between">
                  <FormLabel>المنتجات *</FormLabel>
                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={() => append({ cafeteriaProductId: 0, quantity: 1 })}
                    data-testid="button-add-line"
                  >
                    إضافة منتج
                    <Plus className="h-4 w-4 mr-1" />
                  </Button>
                </div>

                <div className="rounded-lg border p-4 space-y-4">
                  {fields.map((field, index) => {
                    const productId = form.watch(`lines.${index}.cafeteriaProductId`);
                    const product = products?.find((p) => p.id === productId);
                    const quantity = form.watch(`lines.${index}.quantity`);
                    const lineTotal = (product?.defaultSellPrice || 0) * quantity;

                    return (
                      <div key={field.id} className="grid grid-cols-12 gap-2 items-end">
                        <div className="col-span-5">
                          <FormField
                            control={form.control}
                            name={`lines.${index}.cafeteriaProductId`}
                            render={({ field }) => (
                              <FormItem>
                                {index === 0 && <FormLabel>المنتج</FormLabel>}
                                <Select
                                  onValueChange={(v) => field.onChange(parseInt(v))}
                                  value={field.value?.toString()}
                                >
                                  <FormControl>
                                    <SelectTrigger data-testid={`select-product-${index}`}>
                                      <SelectValue placeholder="اختر المنتج" />
                                    </SelectTrigger>
                                  </FormControl>
                                  <SelectContent>
                                    {products?.filter(p => p.isActive).map((product) => (
                                      <SelectItem key={product.id} value={product.id.toString()}>
                                        {product.nameAr} - {product.defaultSellPrice?.toLocaleString("ar-SA")} ر.س
                                      </SelectItem>
                                    ))}
                                  </SelectContent>
                                </Select>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>
                        <div className="col-span-2">
                          <FormField
                            control={form.control}
                            name={`lines.${index}.quantity`}
                            render={({ field }) => (
                              <FormItem>
                                {index === 0 && <FormLabel>الكمية</FormLabel>}
                                <FormControl>
                                  <Input
                                    type="number"
                                    min="1"
                                    {...field}
                                    onChange={(e) => field.onChange(parseInt(e.target.value) || 1)}
                                    data-testid={`input-quantity-${index}`}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>
                        <div className="col-span-3 text-left">
                          {index === 0 && <FormLabel className="block mb-2">الإجمالي</FormLabel>}
                          <span className="font-mono font-medium">
                            {lineTotal.toLocaleString("ar-SA")} ر.س
                          </span>
                        </div>
                        <div className="col-span-2">
                          {fields.length > 1 && (
                            <Button
                              type="button"
                              variant="ghost"
                              size="icon"
                              onClick={() => remove(index)}
                              data-testid={`remove-line-${index}`}
                            >
                              <Trash2 className="h-4 w-4 text-destructive" />
                            </Button>
                          )}
                        </div>
                      </div>
                    );
                  })}
                </div>

                <div className="flex justify-between items-center text-lg font-bold pt-2">
                  <span>الإجمالي الكلي:</span>
                  <span className="font-mono">{calculateTotal().toLocaleString("ar-SA")} ر.س</span>
                </div>
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>ملاحظات</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="ملاحظات إضافية..." data-testid="input-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2 pt-4">
                <Button type="button" variant="outline" onClick={() => setDialogOpen(false)}>
                  إلغاء
                </Button>
                <Button type="submit" disabled={createMutation.isPending} data-testid="button-save-order">
                  {createMutation.isPending ? "جاري الحفظ..." : "حفظ الطلب"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      <Dialog open={detailDialogOpen} onOpenChange={setDetailDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>تفاصيل الطلب {selectedOrder?.orderNumber}</DialogTitle>
            <DialogDescription>عرض تفاصيل طلب المبيعات</DialogDescription>
          </DialogHeader>
          {selectedOrder && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">الموظف:</span>
                  <span className="mr-2 font-medium">{selectedOrder.createdByEmployee?.nameAr}</span>
                </div>
                <div>
                  <span className="text-muted-foreground">التاريخ:</span>
                  <span className="mr-2 font-medium">{new Date(selectedOrder.orderDate).toLocaleString("ar-SA")}</span>
                </div>
                <div>
                  <span className="text-muted-foreground">الحالة:</span>
                  <span className="mr-2">{getStatusBadge(selectedOrder.status)}</span>
                </div>
                {selectedOrder.client && (
                  <div>
                    <span className="text-muted-foreground">العميل:</span>
                    <span className="mr-2 font-medium">{selectedOrder.client.nameAr}</span>
                  </div>
                )}
              </div>

              <div className="rounded-lg border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">المنتج</TableHead>
                      <TableHead className="text-right">الكمية</TableHead>
                      <TableHead className="text-right">السعر</TableHead>
                      <TableHead className="text-right">الإجمالي</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {selectedOrder.lines.map((line) => (
                      <TableRow key={line.id}>
                        <TableCell>{line.cafeteriaProduct?.nameAr || "-"}</TableCell>
                        <TableCell className="font-mono">{line.quantity}</TableCell>
                        <TableCell className="font-mono">{line.unitPrice.toLocaleString("ar-SA")}</TableCell>
                        <TableCell className="font-mono font-medium">{line.lineTotal.toLocaleString("ar-SA")}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>

              <div className="flex justify-between items-center text-lg font-bold border-t pt-4">
                <span>الإجمالي:</span>
                <span className="font-mono">{selectedOrder.totalAmount.toLocaleString("ar-SA")} ر.س</span>
              </div>

              <div className="flex justify-between items-center pt-4">
                <Button
                  variant="outline"
                  onClick={() => navigate(`/cafeteria/sales/${selectedOrder.id}/print`)}
                  data-testid="button-print-order"
                >
                  طباعة الفاتورة
                  <Printer className="h-4 w-4 mr-2" />
                </Button>
                
                {selectedOrder.status === "new" && (
                  <div className="flex gap-2">
                    <Button
                      variant="outline"
                      onClick={() => handleOpenCancelDialog(selectedOrder)}
                      disabled={updateMutation.isPending || cancelMutation.isPending}
                      data-testid="button-cancel-order"
                    >
                      إلغاء الطلب
                      <XCircle className="h-4 w-4 mr-2" />
                    </Button>
                    <Button
                      onClick={() => handleComplete(selectedOrder)}
                      disabled={updateMutation.isPending}
                      data-testid="button-complete-order"
                    >
                      إكمال الطلب
                      <CheckCircle className="h-4 w-4 mr-2" />
                    </Button>
                  </div>
                )}
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Cancel Order Dialog with Reason */}
      <Dialog open={cancelDialogOpen} onOpenChange={setCancelDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>إلغاء الطلب</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <p className="text-sm text-muted-foreground">
              هل أنت متأكد من إلغاء الطلب؟ سيتم إنشاء إشعار دائن متوافق مع الزكاة والضريبة.
            </p>
            {selectedOrder && (
              <div className="rounded-lg bg-muted p-3">
                <p className="font-medium">رقم الطلب: {selectedOrder.orderNumber}</p>
                <p className="text-sm text-muted-foreground">
                  المبلغ: {selectedOrder.totalAmount.toLocaleString("en-SA")} ر.س
                </p>
              </div>
            )}
            <div className="space-y-2">
              <Label htmlFor="cancelReason">سبب الإلغاء *</Label>
              <Textarea
                id="cancelReason"
                placeholder="أدخل سبب الإلغاء..."
                value={cancelReason}
                onChange={(e) => setCancelReason(e.target.value)}
                className="min-h-[80px]"
                data-testid="input-cancel-reason"
              />
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button
              variant="outline"
              onClick={() => setCancelDialogOpen(false)}
              disabled={cancelMutation.isPending}
              data-testid="button-cancel-dialog-close"
            >
              إغلاق
            </Button>
            <Button
              variant="destructive"
              onClick={handleConfirmCancel}
              disabled={cancelMutation.isPending || !cancelReason.trim()}
              data-testid="button-confirm-cancel"
            >
              {cancelMutation.isPending ? "جاري الإلغاء..." : "تأكيد الإلغاء"}
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
