import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Plus, Pencil, Trash2, Loader2, Filter, Coins, Calendar, User, Car, Wallet, Gift, CircleMinus, Landmark, Eye } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

interface Employee {
  id: number;
  fullNameAr: string | null;
}

interface Driver {
  id: number;
  fullNameAr: string | null;
}

interface PaymentMethod {
  id: number;
  nameAr: string;
}

interface AdvanceInstallment {
  id: number;
  cashDisbursementId: number;
  installmentNumber: number;
  amount: number;
  dueDate: string;
  status: string;
  deductedAt: string | null;
  payrollLineId: number | null;
}

interface CashDisbursement {
  id: number;
  disbursementNumber: string;
  disbursementType: string;
  employeeId: number | null;
  driverId: number | null;
  paymentMethodId: number;
  amount: number;
  disbursementDate: string;
  notes: string | null;
  isInstallment: boolean;
  installmentMonths: number | null;
  installmentAmount: number | null;
  employee?: Employee;
  driver?: Driver;
  paymentMethod?: PaymentMethod;
  installments?: AdvanceInstallment[];
  createdAt: string;
}

const disbursementTypes = [
  { value: "advance", label: "سلفة", icon: Wallet },
  { value: "gift", label: "هدية", icon: Gift },
  { value: "deduction", label: "خصم", icon: CircleMinus },
  { value: "owner_withdrawal", label: "سحب مالك", icon: Landmark },
];

const getTypeLabel = (type: string) => {
  return disbursementTypes.find(t => t.value === type)?.label || type;
};

const getTypeBadge = (type: string) => {
  const colors: Record<string, string> = {
    advance: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200",
    gift: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200",
    deduction: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
    owner_withdrawal: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200",
  };
  return colors[type] || "";
};

export default function CashDisbursementsPage() {
  const { toast } = useToast();
  const [activeTab, setActiveTab] = useState("all");
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [isViewDialogOpen, setIsViewDialogOpen] = useState(false);
  const [editingDisbursement, setEditingDisbursement] = useState<CashDisbursement | null>(null);
  const [viewingDisbursement, setViewingDisbursement] = useState<CashDisbursement | null>(null);
  const [filters, setFilters] = useState({
    from_date: "",
    to_date: "",
    type: "",
    employee_id: "",
    driver_id: "",
  });
  const [formData, setFormData] = useState({
    disbursementType: "advance",
    recipientType: "employee",
    employeeId: "",
    driverId: "",
    paymentMethodId: "",
    amount: "",
    disbursementDate: format(new Date(), "yyyy-MM-dd"),
    notes: "",
    isInstallment: false,
    installmentMonths: "3",
  });

  const buildQueryString = () => {
    const params: string[] = [];
    if (activeTab !== "all") {
      params.push(`type=${activeTab}`);
    } else if (filters.type) {
      params.push(`type=${filters.type}`);
    }
    if (filters.from_date) params.push(`from_date=${filters.from_date}`);
    if (filters.to_date) params.push(`to_date=${filters.to_date}`);
    if (filters.employee_id) params.push(`employee_id=${filters.employee_id}`);
    if (filters.driver_id) params.push(`driver_id=${filters.driver_id}`);
    return params.length > 0 ? `?${params.join("&")}` : "";
  };

  const queryString = buildQueryString();

  const { data: disbursements = [], isLoading } = useQuery<CashDisbursement[]>({
    queryKey: ["/api/cash-disbursements", queryString],
    queryFn: async () => {
      const res = await fetch(`/api/cash-disbursements${queryString}`);
      if (!res.ok) throw new Error("فشل في جلب سجلات الصرف");
      return res.json();
    },
  });

  const { data: employees = [] } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: drivers = [] } = useQuery<Driver[]>({
    queryKey: ["/api/drivers"],
  });

  const { data: paymentMethods = [] } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof formData) => {
      const payload: any = {
        disbursementType: data.disbursementType,
        paymentMethodId: parseInt(data.paymentMethodId),
        amount: parseFloat(data.amount),
        disbursementDate: data.disbursementDate,
        notes: data.notes || null,
        isInstallment: data.isInstallment,
        installmentMonths: data.isInstallment ? parseInt(data.installmentMonths) : null,
      };
      
      if (data.disbursementType !== "owner_withdrawal") {
        if (data.recipientType === "employee" && data.employeeId) {
          payload.employeeId = parseInt(data.employeeId);
        } else if (data.recipientType === "driver" && data.driverId) {
          payload.driverId = parseInt(data.driverId);
        }
      }

      const response = await apiRequest("POST", "/api/cash-disbursements", payload);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cash-disbursements"] });
      toast({ title: "تم إنشاء سجل الصرف بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "فشل في إنشاء سجل الصرف",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: typeof formData }) => {
      const payload: any = {
        disbursementType: data.disbursementType,
        paymentMethodId: parseInt(data.paymentMethodId),
        amount: parseFloat(data.amount),
        disbursementDate: data.disbursementDate,
        notes: data.notes || null,
      };
      
      if (data.disbursementType !== "owner_withdrawal") {
        if (data.recipientType === "employee" && data.employeeId) {
          payload.employeeId = parseInt(data.employeeId);
          payload.driverId = null;
        } else if (data.recipientType === "driver" && data.driverId) {
          payload.driverId = parseInt(data.driverId);
          payload.employeeId = null;
        }
      }

      const response = await apiRequest("PUT", `/api/cash-disbursements/${id}`, payload);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cash-disbursements"] });
      toast({ title: "تم تحديث سجل الصرف بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "فشل في تحديث سجل الصرف",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/cash-disbursements/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/cash-disbursements"] });
      toast({ title: "تم حذف سجل الصرف بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "فشل في حذف سجل الصرف",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      disbursementType: "advance",
      recipientType: "employee",
      employeeId: "",
      driverId: "",
      paymentMethodId: "",
      amount: "",
      disbursementDate: format(new Date(), "yyyy-MM-dd"),
      notes: "",
      isInstallment: false,
      installmentMonths: "3",
    });
    setEditingDisbursement(null);
    setIsDialogOpen(false);
  };

  const handleEdit = (disbursement: CashDisbursement) => {
    setEditingDisbursement(disbursement);
    setFormData({
      disbursementType: disbursement.disbursementType,
      recipientType: disbursement.employeeId ? "employee" : disbursement.driverId ? "driver" : "employee",
      employeeId: disbursement.employeeId?.toString() || "",
      driverId: disbursement.driverId?.toString() || "",
      paymentMethodId: disbursement.paymentMethodId.toString(),
      amount: disbursement.amount.toString(),
      disbursementDate: disbursement.disbursementDate,
      notes: disbursement.notes || "",
      isInstallment: disbursement.isInstallment,
      installmentMonths: disbursement.installmentMonths?.toString() || "3",
    });
    setIsDialogOpen(true);
  };

  const handleView = (disbursement: CashDisbursement) => {
    setViewingDisbursement(disbursement);
    setIsViewDialogOpen(true);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.paymentMethodId || !formData.amount) {
      toast({
        title: "خطأ في البيانات",
        description: "يرجى ملء جميع الحقول المطلوبة",
        variant: "destructive",
      });
      return;
    }

    if (formData.disbursementType !== "owner_withdrawal") {
      if (formData.recipientType === "employee" && !formData.employeeId) {
        toast({
          title: "خطأ في البيانات",
          description: "يرجى اختيار الموظف",
          variant: "destructive",
        });
        return;
      }
      if (formData.recipientType === "driver" && !formData.driverId) {
        toast({
          title: "خطأ في البيانات",
          description: "يرجى اختيار السائق",
          variant: "destructive",
        });
        return;
      }
    }

    if (editingDisbursement) {
      updateMutation.mutate({ id: editingDisbursement.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const handleDelete = (id: number) => {
    if (confirm("هل أنت متأكد من حذف هذا السجل؟")) {
      deleteMutation.mutate(id);
    }
  };

  const getRecipientName = (d: CashDisbursement) => {
    if (d.disbursementType === "owner_withdrawal") return "---";
    if (d.employee) return d.employee.fullNameAr || "---";
    if (d.driver) return d.driver.fullNameAr || "---";
    return "---";
  };

  const totalAmount = disbursements.reduce((sum, d) => sum + d.amount, 0);

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4">
        <div className="flex items-center gap-2">
          <Coins className="h-6 w-6" />
          <h1 className="text-2xl font-bold">صرف نقدي</h1>
        </div>
        <Button onClick={() => setIsDialogOpen(true)} data-testid="button-new-disbursement">
          <Plus className="h-4 w-4 ml-2" />
          صرف جديد
        </Button>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="mb-4" data-testid="tabs-disbursement-types">
          <TabsTrigger value="all" data-testid="tab-all">الكل</TabsTrigger>
          <TabsTrigger value="advance" data-testid="tab-advance">السلف</TabsTrigger>
          <TabsTrigger value="gift" data-testid="tab-gift">الهدايا</TabsTrigger>
          <TabsTrigger value="deduction" data-testid="tab-deduction">الخصومات</TabsTrigger>
          <TabsTrigger value="owner_withdrawal" data-testid="tab-owner-withdrawal">سحب المالك</TabsTrigger>
        </TabsList>

        <Card className="mb-4">
          <CardHeader className="pb-3">
            <CardTitle className="flex items-center gap-2 text-lg">
              <Filter className="h-4 w-4" />
              البحث والفلترة
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <Label>من تاريخ</Label>
                <Input
                  type="date"
                  value={filters.from_date}
                  onChange={(e) => setFilters({ ...filters, from_date: e.target.value })}
                  data-testid="input-filter-from-date"
                />
              </div>
              <div>
                <Label>إلى تاريخ</Label>
                <Input
                  type="date"
                  value={filters.to_date}
                  onChange={(e) => setFilters({ ...filters, to_date: e.target.value })}
                  data-testid="input-filter-to-date"
                />
              </div>
              <div>
                <Label>الموظف</Label>
                <Select
                  value={filters.employee_id}
                  onValueChange={(v) => setFilters({ ...filters, employee_id: v, driver_id: "" })}
                >
                  <SelectTrigger data-testid="select-filter-employee">
                    <SelectValue placeholder="الكل" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">الكل</SelectItem>
                    {employees.map((emp) => (
                      <SelectItem key={emp.id} value={emp.id.toString()}>
                        {emp.fullNameAr || `موظف ${emp.id}`}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>السائق</Label>
                <Select
                  value={filters.driver_id}
                  onValueChange={(v) => setFilters({ ...filters, driver_id: v, employee_id: "" })}
                >
                  <SelectTrigger data-testid="select-filter-driver">
                    <SelectValue placeholder="الكل" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">الكل</SelectItem>
                    {drivers.map((drv) => (
                      <SelectItem key={drv.id} value={drv.id.toString()}>
                        {drv.fullNameAr || `سائق ${drv.id}`}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="mb-4">
          <CardContent className="pt-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <div className="text-center p-4 bg-muted rounded-lg">
                <div className="text-sm text-muted-foreground">عدد السجلات</div>
                <div className="text-2xl font-bold">{disbursements.length}</div>
              </div>
              <div className="text-center p-4 bg-muted rounded-lg">
                <div className="text-sm text-muted-foreground">إجمالي المبالغ</div>
                <div className="text-2xl font-bold">{totalAmount.toFixed(2)} ر.س</div>
              </div>
              <div className="text-center p-4 bg-blue-50 dark:bg-blue-950 rounded-lg">
                <div className="text-sm text-muted-foreground">السلف</div>
                <div className="text-2xl font-bold text-blue-600">
                  {disbursements.filter(d => d.disbursementType === "advance").reduce((sum, d) => sum + d.amount, 0).toFixed(2)} ر.س
                </div>
              </div>
              <div className="text-center p-4 bg-purple-50 dark:bg-purple-950 rounded-lg">
                <div className="text-sm text-muted-foreground">سحب المالك</div>
                <div className="text-2xl font-bold text-purple-600">
                  {disbursements.filter(d => d.disbursementType === "owner_withdrawal").reduce((sum, d) => sum + d.amount, 0).toFixed(2)} ر.س
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <TabsContent value={activeTab}>
          <Card>
            <CardContent className="pt-6">
              {isLoading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="h-8 w-8 animate-spin" />
                </div>
              ) : disbursements.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  لا توجد سجلات صرف
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">رقم الصرف</TableHead>
                      <TableHead className="text-right">النوع</TableHead>
                      <TableHead className="text-right">المستفيد</TableHead>
                      <TableHead className="text-right">المبلغ</TableHead>
                      <TableHead className="text-right">طريقة الدفع</TableHead>
                      <TableHead className="text-right">التاريخ</TableHead>
                      <TableHead className="text-right">أقساط</TableHead>
                      <TableHead className="text-right">الإجراءات</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {disbursements.map((d) => (
                      <TableRow key={d.id} data-testid={`row-disbursement-${d.id}`}>
                        <TableCell className="font-medium">{d.disbursementNumber}</TableCell>
                        <TableCell>
                          <Badge className={getTypeBadge(d.disbursementType)}>
                            {getTypeLabel(d.disbursementType)}
                          </Badge>
                        </TableCell>
                        <TableCell>{getRecipientName(d)}</TableCell>
                        <TableCell className="font-bold">{d.amount.toFixed(2)} ر.س</TableCell>
                        <TableCell>{d.paymentMethod?.nameAr || "---"}</TableCell>
                        <TableCell>
                          {format(new Date(d.disbursementDate), "dd/MM/yyyy", { locale: ar })}
                        </TableCell>
                        <TableCell>
                          {d.isInstallment && d.installmentMonths ? (
                            <Badge variant="outline">
                              {d.installmentMonths} شهر
                            </Badge>
                          ) : "---"}
                        </TableCell>
                        <TableCell>
                          <div className="flex items-center gap-1">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleView(d)}
                              data-testid={`button-view-${d.id}`}
                            >
                              <Eye className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleEdit(d)}
                              data-testid={`button-edit-${d.id}`}
                            >
                              <Pencil className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleDelete(d.id)}
                              data-testid={`button-delete-${d.id}`}
                            >
                              <Trash2 className="h-4 w-4 text-destructive" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>
              {editingDisbursement ? "تعديل سجل الصرف" : "صرف نقدي جديد"}
            </DialogTitle>
            <DialogDescription>
              أدخل بيانات الصرف النقدي
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="col-span-2">
                <Label>نوع الصرف</Label>
                <Select
                  value={formData.disbursementType}
                  onValueChange={(v) => setFormData({ ...formData, disbursementType: v })}
                  disabled={!!editingDisbursement}
                >
                  <SelectTrigger data-testid="select-disbursement-type">
                    <SelectValue placeholder="اختر النوع" />
                  </SelectTrigger>
                  <SelectContent>
                    {disbursementTypes.map((type) => (
                      <SelectItem key={type.value} value={type.value}>
                        {type.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {formData.disbursementType !== "owner_withdrawal" && (
                <>
                  <div className="col-span-2">
                    <Label>المستفيد</Label>
                    <Select
                      value={formData.recipientType}
                      onValueChange={(v) => setFormData({ ...formData, recipientType: v, employeeId: "", driverId: "" })}
                    >
                      <SelectTrigger data-testid="select-recipient-type">
                        <SelectValue placeholder="اختر نوع المستفيد" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="employee">موظف</SelectItem>
                        <SelectItem value="driver">سائق</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  {formData.recipientType === "employee" && (
                    <div className="col-span-2">
                      <Label>الموظف</Label>
                      <Select
                        value={formData.employeeId}
                        onValueChange={(v) => setFormData({ ...formData, employeeId: v })}
                      >
                        <SelectTrigger data-testid="select-employee">
                          <SelectValue placeholder="اختر الموظف" />
                        </SelectTrigger>
                        <SelectContent>
                          {employees.map((emp) => (
                            <SelectItem key={emp.id} value={emp.id.toString()}>
                              {emp.fullNameAr || `موظف ${emp.id}`}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                  )}

                  {formData.recipientType === "driver" && (
                    <div className="col-span-2">
                      <Label>السائق</Label>
                      <Select
                        value={formData.driverId}
                        onValueChange={(v) => setFormData({ ...formData, driverId: v })}
                      >
                        <SelectTrigger data-testid="select-driver">
                          <SelectValue placeholder="اختر السائق" />
                        </SelectTrigger>
                        <SelectContent>
                          {drivers.map((drv) => (
                            <SelectItem key={drv.id} value={drv.id.toString()}>
                              {drv.fullNameAr || `سائق ${drv.id}`}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                  )}
                </>
              )}

              <div>
                <Label>المبلغ</Label>
                <Input
                  type="number"
                  step="0.01"
                  value={formData.amount}
                  onChange={(e) => setFormData({ ...formData, amount: e.target.value })}
                  placeholder="0.00"
                  data-testid="input-amount"
                />
              </div>

              <div>
                <Label>طريقة الدفع</Label>
                <Select
                  value={formData.paymentMethodId}
                  onValueChange={(v) => setFormData({ ...formData, paymentMethodId: v })}
                >
                  <SelectTrigger data-testid="select-payment-method">
                    <SelectValue placeholder="اختر طريقة الدفع" />
                  </SelectTrigger>
                  <SelectContent>
                    {paymentMethods.map((pm) => (
                      <SelectItem key={pm.id} value={pm.id.toString()}>
                        {pm.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>تاريخ الصرف</Label>
                <Input
                  type="date"
                  value={formData.disbursementDate}
                  onChange={(e) => setFormData({ ...formData, disbursementDate: e.target.value })}
                  data-testid="input-date"
                />
              </div>

              {formData.disbursementType === "advance" && !editingDisbursement && (
                <>
                  <div className="flex items-center gap-2">
                    <input
                      type="checkbox"
                      id="isInstallment"
                      checked={formData.isInstallment}
                      onChange={(e) => setFormData({ ...formData, isInstallment: e.target.checked })}
                      className="h-4 w-4"
                      data-testid="checkbox-installment"
                    />
                    <Label htmlFor="isInstallment" className="mb-0">تقسيط على أشهر</Label>
                  </div>

                  {formData.isInstallment && (
                    <div>
                      <Label>عدد الأشهر</Label>
                      <Input
                        type="number"
                        min="1"
                        max="24"
                        value={formData.installmentMonths}
                        onChange={(e) => setFormData({ ...formData, installmentMonths: e.target.value })}
                        data-testid="input-installment-months"
                      />
                    </div>
                  )}
                </>
              )}

              <div className="col-span-2">
                <Label>ملاحظات</Label>
                <Textarea
                  value={formData.notes}
                  onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                  placeholder="ملاحظات إضافية..."
                  data-testid="input-notes"
                />
              </div>

              {formData.isInstallment && formData.amount && formData.installmentMonths && (
                <div className="col-span-2 p-3 bg-muted rounded-lg">
                  <div className="text-sm">
                    <strong>قيمة القسط الشهري:</strong>{" "}
                    {(parseFloat(formData.amount) / parseInt(formData.installmentMonths)).toFixed(2)} ر.س
                  </div>
                </div>
              )}
            </div>

            <DialogFooter>
              <Button type="button" variant="outline" onClick={resetForm}>
                إلغاء
              </Button>
              <Button
                type="submit"
                disabled={createMutation.isPending || updateMutation.isPending}
                data-testid="button-submit"
              >
                {(createMutation.isPending || updateMutation.isPending) && (
                  <Loader2 className="h-4 w-4 ml-2 animate-spin" />
                )}
                {editingDisbursement ? "تحديث" : "حفظ"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={isViewDialogOpen} onOpenChange={setIsViewDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>تفاصيل سجل الصرف</DialogTitle>
            <DialogDescription>
              رقم الصرف: {viewingDisbursement?.disbursementNumber}
            </DialogDescription>
          </DialogHeader>
          {viewingDisbursement && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="text-muted-foreground">النوع</Label>
                  <div>
                    <Badge className={getTypeBadge(viewingDisbursement.disbursementType)}>
                      {getTypeLabel(viewingDisbursement.disbursementType)}
                    </Badge>
                  </div>
                </div>
                <div>
                  <Label className="text-muted-foreground">المبلغ</Label>
                  <div className="font-bold text-lg">{viewingDisbursement.amount.toFixed(2)} ر.س</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">المستفيد</Label>
                  <div>{getRecipientName(viewingDisbursement)}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">طريقة الدفع</Label>
                  <div>{viewingDisbursement.paymentMethod?.nameAr || "---"}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">تاريخ الصرف</Label>
                  <div>{format(new Date(viewingDisbursement.disbursementDate), "dd/MM/yyyy", { locale: ar })}</div>
                </div>
                {viewingDisbursement.notes && (
                  <div className="col-span-2">
                    <Label className="text-muted-foreground">ملاحظات</Label>
                    <div>{viewingDisbursement.notes}</div>
                  </div>
                )}
              </div>

              {viewingDisbursement.isInstallment && viewingDisbursement.installments && viewingDisbursement.installments.length > 0 && (
                <div className="mt-4">
                  <Label className="text-muted-foreground mb-2 block">الأقساط</Label>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="text-right">رقم القسط</TableHead>
                        <TableHead className="text-right">المبلغ</TableHead>
                        <TableHead className="text-right">تاريخ الاستحقاق</TableHead>
                        <TableHead className="text-right">الحالة</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {viewingDisbursement.installments.map((inst) => (
                        <TableRow key={inst.id}>
                          <TableCell>{inst.installmentNumber}</TableCell>
                          <TableCell>{inst.amount.toFixed(2)} ر.س</TableCell>
                          <TableCell>{format(new Date(inst.dueDate), "dd/MM/yyyy", { locale: ar })}</TableCell>
                          <TableCell>
                            <Badge variant={inst.status === "deducted" ? "default" : "outline"}>
                              {inst.status === "pending" ? "قيد الانتظار" : inst.status === "deducted" ? "تم الخصم" : inst.status}
                            </Badge>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </div>
              )}
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsViewDialogOpen(false)}>
              إغلاق
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
