import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
  TableFooter,
} from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { 
  Wallet, 
  TrendingUp, 
  TrendingDown,
  RefreshCw,
  Printer,
  ArrowUpCircle,
  ArrowDownCircle,
  Briefcase,
  Building,
  Landmark,
} from "lucide-react";
import { format, startOfMonth, endOfMonth, startOfYear, endOfYear } from "date-fns";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";

interface CashMovement {
  date: string;
  entryNumber: string;
  description: string;
  accountCode: string;
  accountName: string;
  debit: number;
  credit: number;
  netChange: number;
  referenceType: string;
  referenceId: number | null;
}

interface CashFlowData {
  operatingActivities: CashMovement[];
  investingActivities: CashMovement[];
  financingActivities: CashMovement[];
  totalOperating: number;
  totalInvesting: number;
  totalFinancing: number;
  netCashChange: number;
  openingBalance: number;
  closingBalance: number;
  startDate?: string;
  endDate?: string;
}

function CashMovementTable({ movements, title, icon: Icon, color }: { 
  movements: CashMovement[]; 
  title: string; 
  icon: any; 
  color: string;
}) {
  const total = movements.reduce((sum, m) => sum + m.netChange, 0);
  
  return (
    <Card>
      <CardHeader>
        <CardTitle className={`flex items-center gap-2 ${color}`}>
          <Icon className="h-5 w-5" />
          {title}
        </CardTitle>
      </CardHeader>
      <CardContent>
        {movements.length > 0 ? (
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="text-right">التاريخ</TableHead>
                <TableHead className="text-right">رقم القيد</TableHead>
                <TableHead className="text-right">البيان</TableHead>
                <TableHead className="text-left">تدفق داخل</TableHead>
                <TableHead className="text-left">تدفق خارج</TableHead>
                <TableHead className="text-left">الصافي</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {movements.map((m, idx) => (
                <TableRow key={`${m.entryNumber}-${idx}`}>
                  <TableCell className="font-mono text-sm">{formatDate(m.date)}</TableCell>
                  <TableCell className="font-mono text-sm">{m.entryNumber}</TableCell>
                  <TableCell className="text-sm">{m.description}</TableCell>
                  <TableCell className="font-mono text-green-600">
                    {m.debit > 0 ? formatNumber(m.debit) : '-'}
                  </TableCell>
                  <TableCell className="font-mono text-red-600">
                    {m.credit > 0 ? formatNumber(m.credit) : '-'}
                  </TableCell>
                  <TableCell className={`font-mono ${m.netChange >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {formatNumber(m.netChange)}
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
            <TableFooter>
              <TableRow className="bg-muted/50">
                <TableCell colSpan={5} className="font-bold">الإجمالي</TableCell>
                <TableCell className={`font-mono font-bold ${total >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                  {formatCurrency(total)}
                </TableCell>
              </TableRow>
            </TableFooter>
          </Table>
        ) : (
          <div className="text-center py-8 text-muted-foreground">
            لا توجد حركات في هذه الفئة
          </div>
        )}
      </CardContent>
    </Card>
  );
}

export default function CashFlowPage() {
  const today = new Date();
  const [startDate, setStartDate] = useState<string>(
    format(startOfMonth(today), "yyyy-MM-dd")
  );
  const [endDate, setEndDate] = useState<string>(
    format(endOfMonth(today), "yyyy-MM-dd")
  );

  const { data: cashFlow, isLoading, refetch } = useQuery<CashFlowData>({
    queryKey: [`/api/reports/cash-flow?startDate=${startDate}&endDate=${endDate}`],
  });

  const setThisMonth = () => {
    setStartDate(format(startOfMonth(today), "yyyy-MM-dd"));
    setEndDate(format(endOfMonth(today), "yyyy-MM-dd"));
  };

  const setThisYear = () => {
    setStartDate(format(startOfYear(today), "yyyy-MM-dd"));
    setEndDate(format(endOfYear(today), "yyyy-MM-dd"));
  };

  const handlePrint = () => {
    window.print();
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap print:hidden">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">قائمة التدفقات النقدية</h1>
          <p className="text-muted-foreground">تقرير حركة النقد والبنك</p>
        </div>
        <div className="flex items-center gap-4 flex-wrap">
          <div className="flex items-center gap-2">
            <Label htmlFor="startDate">من:</Label>
            <Input
              id="startDate"
              type="date"
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
              className="w-40"
              data-testid="input-start-date"
            />
          </div>
          <div className="flex items-center gap-2">
            <Label htmlFor="endDate">إلى:</Label>
            <Input
              id="endDate"
              type="date"
              value={endDate}
              onChange={(e) => setEndDate(e.target.value)}
              className="w-40"
              data-testid="input-end-date"
            />
          </div>
          <div className="flex gap-2">
            <Button variant="outline" size="sm" onClick={setThisMonth} data-testid="button-this-month">
              هذا الشهر
            </Button>
            <Button variant="outline" size="sm" onClick={setThisYear} data-testid="button-this-year">
              هذه السنة
            </Button>
          </div>
          <Button variant="outline" onClick={() => refetch()} data-testid="button-refresh">
            <RefreshCw className="h-4 w-4 ml-2" />
            تحديث
          </Button>
          <Button variant="outline" onClick={handlePrint} data-testid="button-print">
            <Printer className="h-4 w-4 ml-2" />
            طباعة
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 print:hidden">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <Wallet className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الرصيد الافتتاحي</p>
                <div className="text-xl font-bold" data-testid="text-opening-balance">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(cashFlow?.openingBalance || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className={`p-2 rounded-lg ${(cashFlow?.netCashChange || 0) >= 0 ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'}`}>
                {(cashFlow?.netCashChange || 0) >= 0 ? <ArrowUpCircle className="h-5 w-5" /> : <ArrowDownCircle className="h-5 w-5" />}
              </div>
              <div>
                <p className="text-sm text-muted-foreground">صافي التغير</p>
                <div className={`text-xl font-bold ${(cashFlow?.netCashChange || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`} data-testid="text-net-change">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(cashFlow?.netCashChange || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">
                <Wallet className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الرصيد الختامي</p>
                <div className="text-xl font-bold text-purple-600" data-testid="text-closing-balance">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(cashFlow?.closingBalance || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200">
                <Briefcase className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الأنشطة التشغيلية</p>
                <div className={`text-xl font-bold ${(cashFlow?.totalOperating || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`} data-testid="text-operating">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(cashFlow?.totalOperating || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {isLoading ? (
        <div className="space-y-4">
          {[1, 2, 3].map((i) => (
            <Card key={i}>
              <CardContent className="p-6">
                <Skeleton className="h-40 w-full" />
              </CardContent>
            </Card>
          ))}
        </div>
      ) : (
        <div className="space-y-6">
          <CashMovementTable
            movements={cashFlow?.operatingActivities || []}
            title="التدفقات من الأنشطة التشغيلية"
            icon={Briefcase}
            color="text-orange-700 dark:text-orange-400"
          />

          <CashMovementTable
            movements={cashFlow?.investingActivities || []}
            title="التدفقات من الأنشطة الاستثمارية"
            icon={Building}
            color="text-blue-700 dark:text-blue-400"
          />

          <CashMovementTable
            movements={cashFlow?.financingActivities || []}
            title="التدفقات من الأنشطة التمويلية"
            icon={Landmark}
            color="text-green-700 dark:text-green-400"
          />

          <Card className="bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-950 dark:to-purple-950">
            <CardContent className="p-6">
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <Wallet className="h-5 w-5" />
                ملخص التدفقات النقدية
              </h3>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div className="p-4 rounded-lg bg-background">
                  <p className="text-sm text-muted-foreground">من الأنشطة التشغيلية</p>
                  <p className={`text-xl font-bold ${(cashFlow?.totalOperating || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {formatCurrency(cashFlow?.totalOperating || 0)}
                  </p>
                </div>
                <div className="p-4 rounded-lg bg-background">
                  <p className="text-sm text-muted-foreground">من الأنشطة الاستثمارية</p>
                  <p className={`text-xl font-bold ${(cashFlow?.totalInvesting || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {formatCurrency(cashFlow?.totalInvesting || 0)}
                  </p>
                </div>
                <div className="p-4 rounded-lg bg-background">
                  <p className="text-sm text-muted-foreground">من الأنشطة التمويلية</p>
                  <p className={`text-xl font-bold ${(cashFlow?.totalFinancing || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {formatCurrency(cashFlow?.totalFinancing || 0)}
                  </p>
                </div>
                <div className="p-4 rounded-lg bg-background border-2 border-primary">
                  <p className="text-sm text-muted-foreground">صافي التغير في النقدية</p>
                  <p className={`text-xl font-bold ${(cashFlow?.netCashChange || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {formatCurrency(cashFlow?.netCashChange || 0)}
                  </p>
                </div>
              </div>

              <div className="mt-6 p-4 rounded-lg bg-background">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-muted-foreground">الرصيد الافتتاحي</p>
                    <p className="text-xl font-bold">{formatCurrency(cashFlow?.openingBalance || 0)}</p>
                  </div>
                  <div className="text-center">
                    <p className={`text-lg ${(cashFlow?.netCashChange || 0) >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                      {(cashFlow?.netCashChange || 0) >= 0 ? '+' : ''}{formatNumber(cashFlow?.netCashChange || 0)}
                    </p>
                  </div>
                  <div className="text-left">
                    <p className="text-sm text-muted-foreground">الرصيد الختامي</p>
                    <p className="text-xl font-bold text-purple-600">{formatCurrency(cashFlow?.closingBalance || 0)}</p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  );
}
