import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
} from "@/components/ui/dialog";
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/ui/accordion";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import {
  Plus,
  ChevronLeft,
  Edit,
  Trash2,
  FolderOpen,
  FileText,
  DollarSign,
  TrendingUp,
  TrendingDown,
  Scale,
  PiggyBank,
} from "lucide-react";

interface Account {
  id: number;
  code: string;
  nameAr: string;
  accountType: string;
  normalBalance: string;
  parentId: number | null;
  level: number;
  isPostable: boolean;
  isActive: boolean;
  description?: string;
}

interface AccountWithChildren extends Account {
  children: AccountWithChildren[];
}

const accountTypeLabels: Record<string, string> = {
  asset: "أصول",
  liability: "خصوم",
  equity: "حقوق ملكية",
  revenue: "إيرادات",
  expense: "مصروفات",
};

const accountTypeColors: Record<string, string> = {
  asset: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200",
  liability: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
  equity: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200",
  revenue: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200",
  expense: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200",
};

const accountTypeIcons: Record<string, typeof DollarSign> = {
  asset: PiggyBank,
  liability: TrendingDown,
  equity: Scale,
  revenue: TrendingUp,
  expense: DollarSign,
};

function AccountTreeItem({ account, level = 0, allAccounts }: { account: AccountWithChildren; level?: number; allAccounts: Account[] }) {
  const { toast } = useToast();
  const [isEditing, setIsEditing] = useState(false);
  const [editForm, setEditForm] = useState({
    code: account.code,
    nameAr: account.nameAr,
    description: account.description || "",
  });

  const updateMutation = useMutation({
    mutationFn: async (data: Partial<Account>) => {
      return apiRequest("PATCH", `/api/accounts/${account.id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accounts"] });
      setIsEditing(false);
      toast({ title: "تم تحديث الحساب بنجاح" });
    },
    onError: () => {
      toast({ title: "فشل تحديث الحساب", variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("DELETE", `/api/accounts/${account.id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accounts"] });
      toast({ title: "تم حذف الحساب بنجاح" });
    },
    onError: () => {
      toast({ title: "لا يمكن حذف حساب له حسابات فرعية أو قيود محاسبية", variant: "destructive" });
    },
  });

  const Icon = account.isPostable ? FileText : FolderOpen;
  const TypeIcon = accountTypeIcons[account.accountType] || DollarSign;

  if (account.children.length === 0) {
    return (
      <div
        className="flex items-center justify-between py-2 px-3 hover-elevate rounded-md"
        style={{ marginRight: `${level * 1.5}rem` }}
        data-testid={`account-item-${account.code}`}
      >
        <div className="flex items-center gap-2">
          <Icon className="h-4 w-4 text-muted-foreground" />
          <span className="font-mono text-sm text-muted-foreground">{account.code}</span>
          <span className="text-sm">{account.nameAr}</span>
          {account.isPostable && (
            <Badge variant="outline" className="text-xs">
              قابل للترحيل
            </Badge>
          )}
        </div>
        <div className="flex items-center gap-2">
          <Badge className={accountTypeColors[account.accountType]}>
            <TypeIcon className="h-3 w-3 ml-1" />
            {accountTypeLabels[account.accountType]}
          </Badge>
          <Dialog open={isEditing} onOpenChange={setIsEditing}>
            <DialogTrigger asChild>
              <Button size="icon" variant="ghost" data-testid={`button-edit-account-${account.code}`}>
                <Edit className="h-4 w-4" />
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>تعديل الحساب</DialogTitle>
                <DialogDescription>تعديل بيانات الحساب المحاسبي</DialogDescription>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label>رمز الحساب</Label>
                  <Input
                    value={editForm.code}
                    onChange={(e) => setEditForm({ ...editForm, code: e.target.value })}
                    data-testid="input-edit-account-code"
                  />
                </div>
                <div>
                  <Label>اسم الحساب</Label>
                  <Input
                    value={editForm.nameAr}
                    onChange={(e) => setEditForm({ ...editForm, nameAr: e.target.value })}
                    data-testid="input-edit-account-name"
                  />
                </div>
                <div>
                  <Label>الوصف</Label>
                  <Input
                    value={editForm.description}
                    onChange={(e) => setEditForm({ ...editForm, description: e.target.value })}
                    data-testid="input-edit-account-description"
                  />
                </div>
                <Button
                  onClick={() => updateMutation.mutate(editForm)}
                  disabled={updateMutation.isPending}
                  className="w-full"
                  data-testid="button-save-account"
                >
                  حفظ التغييرات
                </Button>
              </div>
            </DialogContent>
          </Dialog>
          <Button
            size="icon"
            variant="ghost"
            onClick={() => deleteMutation.mutate()}
            disabled={deleteMutation.isPending}
            data-testid={`button-delete-account-${account.code}`}
          >
            <Trash2 className="h-4 w-4" />
          </Button>
        </div>
      </div>
    );
  }

  return (
    <AccordionItem value={account.code} className="border-none">
      <AccordionTrigger
        className="py-2 px-3 hover:no-underline hover-elevate rounded-md"
        style={{ marginRight: `${level * 1.5}rem` }}
        data-testid={`account-folder-${account.code}`}
      >
        <div className="flex items-center gap-2 flex-1">
          <FolderOpen className="h-4 w-4 text-muted-foreground" />
          <span className="font-mono text-sm text-muted-foreground">{account.code}</span>
          <span className="text-sm font-medium">{account.nameAr}</span>
          <Badge className={accountTypeColors[account.accountType]}>
            <TypeIcon className="h-3 w-3 ml-1" />
            {accountTypeLabels[account.accountType]}
          </Badge>
        </div>
      </AccordionTrigger>
      <AccordionContent className="pb-0">
        <Accordion type="multiple" className="w-full">
          {account.children.map((child) => (
            <AccountTreeItem
              key={child.id}
              account={child}
              level={level + 1}
              allAccounts={allAccounts}
            />
          ))}
        </Accordion>
      </AccordionContent>
    </AccordionItem>
  );
}

export default function ChartOfAccountsPage() {
  const { toast } = useToast();
  const [isAddOpen, setIsAddOpen] = useState(false);
  const [newAccount, setNewAccount] = useState({
    code: "",
    nameAr: "",
    accountType: "asset",
    parentId: null as number | null,
    isPostable: true,
    description: "",
  });

  const { data: accountsTree, isLoading: isLoadingTree } = useQuery<AccountWithChildren[]>({
    queryKey: ["/api/accounts", "tree"],
    queryFn: async () => {
      const res = await fetch("/api/accounts?tree=true");
      if (!res.ok) throw new Error("Failed to fetch accounts");
      return res.json();
    },
  });

  const { data: accountsList } = useQuery<Account[]>({
    queryKey: ["/api/accounts", "list"],
    queryFn: async () => {
      const res = await fetch("/api/accounts");
      if (!res.ok) throw new Error("Failed to fetch accounts");
      return res.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof newAccount) => {
      return apiRequest("POST", "/api/accounts", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accounts"] });
      setIsAddOpen(false);
      setNewAccount({
        code: "",
        nameAr: "",
        accountType: "asset",
        parentId: null,
        isPostable: true,
        description: "",
      });
      toast({ title: "تم إضافة الحساب بنجاح" });
    },
    onError: () => {
      toast({ title: "فشل إضافة الحساب", variant: "destructive" });
    },
  });

  const countAccounts = (tree: AccountWithChildren[] | undefined): number => {
    if (!tree) return 0;
    return tree.reduce((count, acc) => count + 1 + countAccounts(acc.children), 0);
  };

  const countByType = (list: Account[] | undefined, type: string): number => {
    if (!list) return 0;
    return list.filter(acc => acc.accountType === type).length;
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">دليل الحسابات</h1>
          <p className="text-muted-foreground">إدارة شجرة الحسابات المحاسبية</p>
        </div>
        <Dialog open={isAddOpen} onOpenChange={setIsAddOpen}>
          <DialogTrigger asChild>
            <Button data-testid="button-add-account">
              <Plus className="h-4 w-4 ml-2" />
              إضافة حساب
            </Button>
          </DialogTrigger>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>إضافة حساب جديد</DialogTitle>
              <DialogDescription>إضافة حساب محاسبي جديد إلى دليل الحسابات</DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <Label>رمز الحساب</Label>
                <Input
                  value={newAccount.code}
                  onChange={(e) => setNewAccount({ ...newAccount, code: e.target.value })}
                  placeholder="مثال: 1111"
                  data-testid="input-new-account-code"
                />
              </div>
              <div>
                <Label>اسم الحساب</Label>
                <Input
                  value={newAccount.nameAr}
                  onChange={(e) => setNewAccount({ ...newAccount, nameAr: e.target.value })}
                  placeholder="اسم الحساب بالعربية"
                  data-testid="input-new-account-name"
                />
              </div>
              <div>
                <Label>نوع الحساب</Label>
                <Select
                  value={newAccount.accountType}
                  onValueChange={(v) => setNewAccount({ ...newAccount, accountType: v })}
                >
                  <SelectTrigger data-testid="select-account-type">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="asset">أصول</SelectItem>
                    <SelectItem value="liability">خصوم</SelectItem>
                    <SelectItem value="equity">حقوق ملكية</SelectItem>
                    <SelectItem value="revenue">إيرادات</SelectItem>
                    <SelectItem value="expense">مصروفات</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>الحساب الأب</Label>
                <Select
                  value={newAccount.parentId?.toString() || "none"}
                  onValueChange={(v) => setNewAccount({ ...newAccount, parentId: v === "none" ? null : parseInt(v) })}
                >
                  <SelectTrigger data-testid="select-parent-account">
                    <SelectValue placeholder="اختر الحساب الأب (اختياري)" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">بدون (حساب رئيسي)</SelectItem>
                    {accountsList?.filter(a => !a.isPostable).map((acc) => (
                      <SelectItem key={acc.id} value={acc.id.toString()}>
                        {acc.code} - {acc.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="flex items-center gap-2">
                <input
                  type="checkbox"
                  id="isPostable"
                  checked={newAccount.isPostable}
                  onChange={(e) => setNewAccount({ ...newAccount, isPostable: e.target.checked })}
                  className="rounded border-input"
                  data-testid="checkbox-is-postable"
                />
                <Label htmlFor="isPostable">حساب قابل للترحيل (يمكن القيد عليه مباشرة)</Label>
              </div>
              <div>
                <Label>الوصف (اختياري)</Label>
                <Input
                  value={newAccount.description}
                  onChange={(e) => setNewAccount({ ...newAccount, description: e.target.value })}
                  placeholder="وصف إضافي للحساب"
                  data-testid="input-new-account-description"
                />
              </div>
              <Button
                onClick={() => createMutation.mutate(newAccount)}
                disabled={createMutation.isPending || !newAccount.code || !newAccount.nameAr}
                className="w-full"
                data-testid="button-create-account"
              >
                إنشاء الحساب
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
        {[
          { type: "asset", label: "أصول", icon: PiggyBank },
          { type: "liability", label: "خصوم", icon: TrendingDown },
          { type: "equity", label: "حقوق ملكية", icon: Scale },
          { type: "revenue", label: "إيرادات", icon: TrendingUp },
          { type: "expense", label: "مصروفات", icon: DollarSign },
        ].map(({ type, label, icon: Icon }) => (
          <Card key={type}>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <div className={`p-2 rounded-lg ${accountTypeColors[type]}`}>
                  <Icon className="h-5 w-5" />
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">{label}</p>
                  <p className="text-xl font-bold" data-testid={`text-count-${type}`}>
                    {isLoadingTree ? "-" : countByType(accountsList, type)}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ChevronLeft className="h-5 w-5" />
            شجرة الحسابات
            <Badge variant="secondary" data-testid="text-total-accounts">
              {countAccounts(accountsTree)} حساب
            </Badge>
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoadingTree ? (
            <div className="space-y-2">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-10 w-full" />
              ))}
            </div>
          ) : accountsTree && accountsTree.length > 0 ? (
            <Accordion type="multiple" className="w-full" defaultValue={accountsTree.map(a => a.code)}>
              {accountsTree.map((account) => (
                <AccountTreeItem
                  key={account.id}
                  account={account}
                  allAccounts={accountsList || []}
                />
              ))}
            </Accordion>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <FolderOpen className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد حسابات بعد</p>
              <Button
                variant="outline"
                className="mt-4"
                onClick={() => setIsAddOpen(true)}
                data-testid="button-add-first-account"
              >
                إضافة أول حساب
              </Button>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
