import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Percent, Plus, Edit, Trash2, TrendingUp, Target,
  Users, ChartBar, CheckCircle, Settings
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency } from "@/lib/formatters";

interface Employee {
  id: number;
  nameAr: string | null;
  fullNameAr: string | null;
  isActive: boolean;
}

interface CommissionTargetSource {
  id: number;
  profileId: number;
  sourceType: string;
  isEnabled: boolean;
}

interface CommissionProfile {
  id: number;
  employeeId: number;
  monthlyTarget: number;
  commissionRate: number;
  isActive: boolean;
  employee?: Employee;
  targetSources: CommissionTargetSource[];
  createdAt: string;
  updatedAt: string;
}

const SOURCE_TYPE_LABELS: Record<string, string> = {
  executed_services: "الخدمات المنفذة",
  created_orders: "الطلبات المنشأة",
  cafeteria_sales: "مبيعات الكافتيريا",
  product_sales: "مبيعات المنتجات",
};

export default function CommissionsPage() {
  const { toast } = useToast();
  const [isCreateOpen, setIsCreateOpen] = useState(false);
  const [editingProfile, setEditingProfile] = useState<CommissionProfile | null>(null);
  
  const [formData, setFormData] = useState({
    employeeId: "",
    monthlyTarget: "",
    commissionRate: "",
    targetSources: [] as string[],
  });

  const { data: profiles, isLoading: isLoadingProfiles } = useQuery<CommissionProfile[]>({
    queryKey: ["/api/commissions/profiles"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const activeEmployees = employees?.filter(e => e.isActive) || [];

  const employeesWithoutProfile = activeEmployees.filter(
    emp => !profiles?.some(p => p.employeeId === emp.id)
  );

  const createMutation = useMutation({
    mutationFn: async (data: any) => {
      return apiRequest("POST", "/api/commissions/profiles", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/commissions/profiles"] });
      setIsCreateOpen(false);
      resetForm();
      toast({
        title: "تم بنجاح",
        description: "تم إنشاء ملف العمولات بنجاح",
      });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في إنشاء ملف العمولات",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: any }) => {
      return apiRequest("PUT", `/api/commissions/profiles/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/commissions/profiles"] });
      setEditingProfile(null);
      resetForm();
      toast({
        title: "تم بنجاح",
        description: "تم تحديث ملف العمولات بنجاح",
      });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في تحديث ملف العمولات",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/commissions/profiles/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/commissions/profiles"] });
      toast({
        title: "تم بنجاح",
        description: "تم حذف ملف العمولات",
      });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في حذف ملف العمولات",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      employeeId: "",
      monthlyTarget: "",
      commissionRate: "",
      targetSources: [],
    });
  };

  const handleOpenEdit = (profile: CommissionProfile) => {
    setEditingProfile(profile);
    setFormData({
      employeeId: profile.employeeId.toString(),
      monthlyTarget: profile.monthlyTarget.toString(),
      commissionRate: profile.commissionRate.toString(),
      targetSources: profile.targetSources.filter(s => s.isEnabled).map(s => s.sourceType),
    });
  };

  const handleSubmit = () => {
    if (!formData.employeeId || !formData.monthlyTarget || !formData.commissionRate) {
      toast({
        title: "خطأ",
        description: "يرجى ملء جميع الحقول المطلوبة",
        variant: "destructive",
      });
      return;
    }

    if (formData.targetSources.length === 0) {
      toast({
        title: "خطأ",
        description: "يرجى اختيار مصدر هدف واحد على الأقل",
        variant: "destructive",
      });
      return;
    }

    const data = {
      employeeId: parseInt(formData.employeeId),
      monthlyTarget: parseFloat(formData.monthlyTarget),
      commissionRate: parseFloat(formData.commissionRate),
      targetSources: formData.targetSources,
      isActive: true,
    };

    if (editingProfile) {
      updateMutation.mutate({ id: editingProfile.id, data });
    } else {
      createMutation.mutate(data);
    }
  };

  const handleSourceToggle = (source: string) => {
    setFormData(prev => ({
      ...prev,
      targetSources: prev.targetSources.includes(source)
        ? prev.targetSources.filter(s => s !== source)
        : [...prev.targetSources, source],
    }));
  };

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex justify-between items-center gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold flex items-center gap-2">
            <Percent className="h-6 w-6" />
            إدارة العمولات
          </h1>
          <p className="text-muted-foreground">
            إعداد ملفات العمولات وتتبع الأهداف الشهرية للموظفين
          </p>
        </div>
        <Dialog open={isCreateOpen} onOpenChange={(open) => {
          setIsCreateOpen(open);
          if (!open) resetForm();
        }}>
          <DialogTrigger asChild>
            <Button data-testid="button-add-profile">
              <Plus className="ml-2 h-4 w-4" />
              إنشاء ملف عمولات
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-lg">
            <DialogHeader>
              <DialogTitle>إنشاء ملف عمولات جديد</DialogTitle>
              <DialogDescription>
                إعداد ملف عمولات لموظف مع تحديد الهدف الشهري ومصادر الإيراد
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label>الموظف *</Label>
                <Select
                  value={formData.employeeId}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, employeeId: value }))}
                >
                  <SelectTrigger data-testid="select-employee">
                    <SelectValue placeholder="اختر الموظف" />
                  </SelectTrigger>
                  <SelectContent>
                    {employeesWithoutProfile.map(emp => (
                      <SelectItem key={emp.id} value={emp.id.toString()}>
                        {emp.fullNameAr || emp.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>الهدف الشهري (ريال) *</Label>
                  <Input
                    type="number"
                    value={formData.monthlyTarget}
                    onChange={(e) => setFormData(prev => ({ ...prev, monthlyTarget: e.target.value }))}
                    placeholder="مثال: 10000"
                    data-testid="input-monthly-target"
                  />
                </div>
                <div className="space-y-2">
                  <Label>نسبة العمولة (%) *</Label>
                  <Input
                    type="number"
                    value={formData.commissionRate}
                    onChange={(e) => setFormData(prev => ({ ...prev, commissionRate: e.target.value }))}
                    placeholder="مثال: 5"
                    min="0"
                    max="100"
                    step="0.1"
                    data-testid="input-commission-rate"
                  />
                </div>
              </div>

              <div className="space-y-3">
                <Label>مصادر الهدف *</Label>
                <p className="text-sm text-muted-foreground">
                  اختر مصادر الإيراد التي تحتسب ضمن الهدف الشهري للموظف
                </p>
                <div className="grid grid-cols-2 gap-3">
                  {Object.entries(SOURCE_TYPE_LABELS).map(([key, label]) => (
                    <div
                      key={key}
                      className={`flex items-center gap-2 p-3 border rounded-lg cursor-pointer transition-colors ${
                        formData.targetSources.includes(key) 
                          ? "border-primary bg-primary/5" 
                          : "hover:bg-muted/50"
                      }`}
                      onClick={() => handleSourceToggle(key)}
                      data-testid={`checkbox-source-${key}`}
                    >
                      <Checkbox
                        checked={formData.targetSources.includes(key)}
                        onCheckedChange={() => handleSourceToggle(key)}
                      />
                      <span className="text-sm">{label}</span>
                    </div>
                  ))}
                </div>
              </div>
            </div>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsCreateOpen(false)}>
                إلغاء
              </Button>
              <Button
                onClick={handleSubmit}
                disabled={createMutation.isPending}
                data-testid="button-submit-profile"
              >
                {createMutation.isPending ? "جارٍ الحفظ..." : "حفظ"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="pt-6">
            <div className="flex items-center gap-4">
              <div className="p-3 rounded-full bg-blue-100 dark:bg-blue-900">
                <Users className="h-6 w-6 text-blue-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الملفات</p>
                <p className="text-2xl font-bold">{profiles?.length || 0}</p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="pt-6">
            <div className="flex items-center gap-4">
              <div className="p-3 rounded-full bg-green-100 dark:bg-green-900">
                <CheckCircle className="h-6 w-6 text-green-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الملفات النشطة</p>
                <p className="text-2xl font-bold">{profiles?.filter(p => p.isActive).length || 0}</p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="pt-6">
            <div className="flex items-center gap-4">
              <div className="p-3 rounded-full bg-purple-100 dark:bg-purple-900">
                <Target className="h-6 w-6 text-purple-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">متوسط الهدف</p>
                <p className="text-2xl font-bold">
                  {profiles && profiles.length > 0
                    ? formatCurrency(profiles.reduce((sum, p) => sum + p.monthlyTarget, 0) / profiles.length)
                    : formatCurrency(0)}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="pt-6">
            <div className="flex items-center gap-4">
              <div className="p-3 rounded-full bg-orange-100 dark:bg-orange-900">
                <Percent className="h-6 w-6 text-orange-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">متوسط نسبة العمولة</p>
                <p className="text-2xl font-bold">
                  {profiles && profiles.length > 0
                    ? (profiles.reduce((sum, p) => sum + p.commissionRate, 0) / profiles.length).toFixed(1)
                    : 0}%
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Profiles Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Settings className="h-5 w-5" />
            ملفات العمولات
          </CardTitle>
          <CardDescription>
            قائمة ملفات العمولات المُعدة للموظفين
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoadingProfiles ? (
            <div className="space-y-2">
              <Skeleton className="h-12 w-full" />
              <Skeleton className="h-12 w-full" />
              <Skeleton className="h-12 w-full" />
            </div>
          ) : profiles && profiles.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>الموظف</TableHead>
                  <TableHead>الهدف الشهري</TableHead>
                  <TableHead>نسبة العمولة</TableHead>
                  <TableHead>مصادر الهدف</TableHead>
                  <TableHead>الحالة</TableHead>
                  <TableHead>الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {profiles.map(profile => (
                  <TableRow key={profile.id} data-testid={`row-profile-${profile.id}`}>
                    <TableCell>
                      <Link href={`/employees/${profile.employeeId}`}>
                        <span className="text-primary hover:underline cursor-pointer">
                          {profile.employee?.fullNameAr || profile.employee?.nameAr || "موظف غير معروف"}
                        </span>
                      </Link>
                    </TableCell>
                    <TableCell className="font-bold">{formatCurrency(profile.monthlyTarget)}</TableCell>
                    <TableCell>{profile.commissionRate}%</TableCell>
                    <TableCell>
                      <div className="flex flex-wrap gap-1">
                        {profile.targetSources.filter(s => s.isEnabled).map(source => (
                          <Badge key={source.id} variant="secondary" className="text-xs">
                            {SOURCE_TYPE_LABELS[source.sourceType] || source.sourceType}
                          </Badge>
                        ))}
                      </div>
                    </TableCell>
                    <TableCell>
                      <Badge variant={profile.isActive ? "default" : "secondary"}>
                        {profile.isActive ? "نشط" : "غير نشط"}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex gap-2">
                        <Dialog 
                          open={editingProfile?.id === profile.id}
                          onOpenChange={(open) => {
                            if (!open) {
                              setEditingProfile(null);
                              resetForm();
                            }
                          }}
                        >
                          <DialogTrigger asChild>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleOpenEdit(profile)}
                              data-testid={`button-edit-${profile.id}`}
                            >
                              <Edit className="h-4 w-4" />
                            </Button>
                          </DialogTrigger>
                          <DialogContent className="max-w-lg">
                            <DialogHeader>
                              <DialogTitle>تعديل ملف العمولات</DialogTitle>
                              <DialogDescription>
                                تعديل إعدادات ملف عمولات {profile.employee?.fullNameAr || profile.employee?.nameAr}
                              </DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4">
                              <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                  <Label>الهدف الشهري (ريال) *</Label>
                                  <Input
                                    type="number"
                                    value={formData.monthlyTarget}
                                    onChange={(e) => setFormData(prev => ({ ...prev, monthlyTarget: e.target.value }))}
                                    placeholder="مثال: 10000"
                                    data-testid="input-edit-monthly-target"
                                  />
                                </div>
                                <div className="space-y-2">
                                  <Label>نسبة العمولة (%) *</Label>
                                  <Input
                                    type="number"
                                    value={formData.commissionRate}
                                    onChange={(e) => setFormData(prev => ({ ...prev, commissionRate: e.target.value }))}
                                    placeholder="مثال: 5"
                                    min="0"
                                    max="100"
                                    step="0.1"
                                    data-testid="input-edit-commission-rate"
                                  />
                                </div>
                              </div>

                              <div className="space-y-3">
                                <Label>مصادر الهدف *</Label>
                                <div className="grid grid-cols-2 gap-3">
                                  {Object.entries(SOURCE_TYPE_LABELS).map(([key, label]) => (
                                    <div
                                      key={key}
                                      className={`flex items-center gap-2 p-3 border rounded-lg cursor-pointer transition-colors ${
                                        formData.targetSources.includes(key) 
                                          ? "border-primary bg-primary/5" 
                                          : "hover:bg-muted/50"
                                      }`}
                                      onClick={() => handleSourceToggle(key)}
                                      data-testid={`checkbox-edit-source-${key}`}
                                    >
                                      <Checkbox
                                        checked={formData.targetSources.includes(key)}
                                        onCheckedChange={() => handleSourceToggle(key)}
                                      />
                                      <span className="text-sm">{label}</span>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                            <DialogFooter>
                              <Button variant="outline" onClick={() => setEditingProfile(null)}>
                                إلغاء
                              </Button>
                              <Button
                                onClick={handleSubmit}
                                disabled={updateMutation.isPending}
                                data-testid="button-update-profile"
                              >
                                {updateMutation.isPending ? "جارٍ الحفظ..." : "حفظ التعديلات"}
                              </Button>
                            </DialogFooter>
                          </DialogContent>
                        </Dialog>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => {
                            if (confirm("هل أنت متأكد من حذف ملف العمولات؟")) {
                              deleteMutation.mutate(profile.id);
                            }
                          }}
                          data-testid={`button-delete-${profile.id}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                        <Link href={`/employees/${profile.employeeId}`}>
                          <Button variant="ghost" size="icon" data-testid={`button-view-${profile.id}`}>
                            <TrendingUp className="h-4 w-4" />
                          </Button>
                        </Link>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="text-center py-12">
              <Percent className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
              <h3 className="text-lg font-semibold mb-2">لا توجد ملفات عمولات</h3>
              <p className="text-muted-foreground mb-4">
                ابدأ بإنشاء ملف عمولات لتتبع أهداف الموظفين
              </p>
              <Button onClick={() => setIsCreateOpen(true)} data-testid="button-empty-add">
                <Plus className="ml-2 h-4 w-4" />
                إنشاء ملف عمولات
              </Button>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
