import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Ticket, Plus, Pencil, Trash2, Percent, DollarSign, CheckCircle2, XCircle, Copy } from "lucide-react";
import { format } from "date-fns";

interface Service {
  id: number;
  nameAr: string;
  price: number;
  isActive: boolean;
}

interface Product {
  id: number;
  nameAr: string;
  defaultSellPrice: number | null;
  isActive: boolean;
}

interface CouponServiceItem {
  id: number;
  couponId: number;
  serviceId: number;
  service?: Service;
}

interface CouponProductItem {
  id: number;
  couponId: number;
  productId: number;
  product?: Product;
}

interface CouponData {
  id: number;
  nameAr: string;
  code: string;
  descriptionAr: string | null;
  discountType: "percent" | "amount";
  discountValue: number;
  appliesToServices: boolean;
  appliesToProducts: boolean;
  startDate: string;
  endDate: string;
  isActive: boolean;
  allowedServices: CouponServiceItem[];
  allowedProducts: CouponProductItem[];
  createdAt: string;
  updatedAt: string;
}

interface CouponFormData {
  nameAr: string;
  code: string;
  descriptionAr: string;
  discountType: "percent" | "amount";
  discountValue: number;
  appliesToServices: boolean;
  appliesToProducts: boolean;
  startDate: string;
  endDate: string;
  isActive: boolean;
  serviceIds: number[];
  productIds: number[];
}

export default function CouponsPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingCoupon, setEditingCoupon] = useState<CouponData | null>(null);
  const [formData, setFormData] = useState<CouponFormData>({
    nameAr: "",
    code: "",
    descriptionAr: "",
    discountType: "percent",
    discountValue: 0,
    appliesToServices: true,
    appliesToProducts: false,
    startDate: "",
    endDate: "",
    isActive: true,
    serviceIds: [],
    productIds: [],
  });
  const { toast } = useToast();

  const { data: coupons, isLoading } = useQuery<CouponData[]>({
    queryKey: ["/api/marketing/coupons"],
  });

  const { data: servicesData } = useQuery<Service[]>({
    queryKey: ["/api/services"],
  });

  const { data: productsData } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const createMutation = useMutation({
    mutationFn: (data: CouponFormData) =>
      apiRequest("POST", "/api/marketing/coupons", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/marketing/coupons"] });
      toast({ title: "تم إنشاء الكوبون بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الكوبون",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<CouponFormData> }) =>
      apiRequest("PUT", `/api/marketing/coupons/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/marketing/coupons"] });
      toast({ title: "تم تحديث الكوبون بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الكوبون",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/marketing/coupons/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/marketing/coupons"] });
      toast({ title: "تم إلغاء تفعيل الكوبون بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الكوبون",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      nameAr: "",
      code: "",
      descriptionAr: "",
      discountType: "percent",
      discountValue: 0,
      appliesToServices: true,
      appliesToProducts: false,
      startDate: "",
      endDate: "",
      isActive: true,
      serviceIds: [],
      productIds: [],
    });
    setEditingCoupon(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (coupon?: CouponData) => {
    if (coupon) {
      setEditingCoupon(coupon);
      setFormData({
        nameAr: coupon.nameAr,
        code: coupon.code,
        descriptionAr: coupon.descriptionAr || "",
        discountType: coupon.discountType,
        discountValue: coupon.discountValue,
        appliesToServices: coupon.appliesToServices,
        appliesToProducts: coupon.appliesToProducts,
        startDate: coupon.startDate,
        endDate: coupon.endDate,
        isActive: coupon.isActive,
        serviceIds: coupon.allowedServices.map(s => s.serviceId),
        productIds: coupon.allowedProducts.map(p => p.productId),
      });
    } else {
      setFormData({
        nameAr: "",
        code: "",
        descriptionAr: "",
        discountType: "percent",
        discountValue: 0,
        appliesToServices: true,
        appliesToProducts: false,
        startDate: "",
        endDate: "",
        isActive: true,
        serviceIds: [],
        productIds: [],
      });
      setEditingCoupon(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الكوبون مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.code.trim()) {
      toast({ title: "رمز الكوبون مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.discountValue || formData.discountValue <= 0) {
      toast({ title: "قيمة الخصم مطلوبة", variant: "destructive" });
      return;
    }
    if (!formData.startDate || !formData.endDate) {
      toast({ title: "تاريخ البداية والنهاية مطلوبان", variant: "destructive" });
      return;
    }

    if (editingCoupon) {
      updateMutation.mutate({ id: editingCoupon.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const copyCode = (code: string) => {
    navigator.clipboard.writeText(code);
    toast({ title: "تم نسخ الرمز" });
  };

  const toggleServiceId = (serviceId: number) => {
    setFormData(prev => ({
      ...prev,
      serviceIds: prev.serviceIds.includes(serviceId)
        ? prev.serviceIds.filter(id => id !== serviceId)
        : [...prev.serviceIds, serviceId],
    }));
  };

  const toggleProductId = (productId: number) => {
    setFormData(prev => ({
      ...prev,
      productIds: prev.productIds.includes(productId)
        ? prev.productIds.filter(id => id !== productId)
        : [...prev.productIds, productId],
    }));
  };

  const isCouponValid = (coupon: CouponData) => {
    const today = new Date().toISOString().split('T')[0];
    return coupon.isActive && coupon.startDate <= today && coupon.endDate >= today;
  };

  const activeCoupons = coupons?.filter(c => c.isActive) || [];
  const availableServices = servicesData?.filter(s => s.isActive) || [];
  const availableProducts = productsData?.filter(p => p.isActive) || [];

  const formatDiscount = (coupon: CouponData) => {
    if (coupon.discountType === "percent") {
      return `${coupon.discountValue}%`;
    }
    return `${coupon.discountValue} ر.س`;
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">الكوبونات</h1>
          <p className="text-muted-foreground">إدارة كوبونات الخصم</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-coupon">
          <Plus className="h-4 w-4 ml-2" />
          إضافة كوبون جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Ticket className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الكوبونات</p>
                <p className="text-xl font-bold" data-testid="text-total-coupons">
                  {coupons?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-500/10 text-green-600">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الكوبونات النشطة</p>
                <p className="text-xl font-bold" data-testid="text-active-coupons">
                  {activeCoupons.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-500/10 text-blue-600">
                <Percent className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">الكوبونات الصالحة اليوم</p>
                <p className="text-xl font-bold" data-testid="text-valid-coupons">
                  {coupons?.filter(c => isCouponValid(c)).length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Ticket className="h-5 w-5" />
            قائمة الكوبونات
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              <Skeleton className="h-10 w-full" />
              <Skeleton className="h-10 w-full" />
              <Skeleton className="h-10 w-full" />
            </div>
          ) : coupons && coupons.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>اسم الكوبون</TableHead>
                  <TableHead>الرمز</TableHead>
                  <TableHead>الخصم</TableHead>
                  <TableHead>يطبق على</TableHead>
                  <TableHead>تاريخ البداية</TableHead>
                  <TableHead>تاريخ النهاية</TableHead>
                  <TableHead>الحالة</TableHead>
                  <TableHead>إجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {coupons.map((coupon) => (
                  <TableRow key={coupon.id} data-testid={`row-coupon-${coupon.id}`}>
                    <TableCell className="font-medium">{coupon.nameAr}</TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <code className="bg-muted px-2 py-1 rounded text-sm">{coupon.code}</code>
                        <Button
                          variant="ghost"
                          size="icon"
                          className="h-6 w-6"
                          onClick={() => copyCode(coupon.code)}
                          data-testid={`button-copy-code-${coupon.id}`}
                        >
                          <Copy className="h-3 w-3" />
                        </Button>
                      </div>
                    </TableCell>
                    <TableCell>
                      <Badge variant="secondary" className="flex items-center gap-1 w-fit">
                        {coupon.discountType === "percent" ? (
                          <Percent className="h-3 w-3" />
                        ) : (
                          <DollarSign className="h-3 w-3" />
                        )}
                        {formatDiscount(coupon)}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex gap-1">
                        {coupon.appliesToServices && (
                          <Badge variant="outline" className="text-xs">خدمات</Badge>
                        )}
                        {coupon.appliesToProducts && (
                          <Badge variant="outline" className="text-xs">منتجات</Badge>
                        )}
                      </div>
                    </TableCell>
                    <TableCell>{format(new Date(coupon.startDate), 'yyyy-MM-dd')}</TableCell>
                    <TableCell>{format(new Date(coupon.endDate), 'yyyy-MM-dd')}</TableCell>
                    <TableCell>
                      {isCouponValid(coupon) ? (
                        <Badge className="bg-green-500">صالح</Badge>
                      ) : coupon.isActive ? (
                        <Badge variant="secondary">غير صالح</Badge>
                      ) : (
                        <Badge variant="destructive">معطل</Badge>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleOpenDialog(coupon)}
                          data-testid={`button-edit-coupon-${coupon.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => deleteMutation.mutate(coupon.id)}
                          disabled={deleteMutation.isPending}
                          data-testid={`button-delete-coupon-${coupon.id}`}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              لا توجد كوبونات. اضغط على "إضافة كوبون جديد" للبدء.
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingCoupon ? "تعديل الكوبون" : "إضافة كوبون جديد"}
            </DialogTitle>
          </DialogHeader>

          <div className="space-y-6 py-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="nameAr">اسم الكوبون *</Label>
                <Input
                  id="nameAr"
                  value={formData.nameAr}
                  onChange={(e) => setFormData(prev => ({ ...prev, nameAr: e.target.value }))}
                  placeholder="مثال: خصم افتتاح الفرع"
                  data-testid="input-coupon-name"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="code">رمز الكوبون *</Label>
                <Input
                  id="code"
                  value={formData.code}
                  onChange={(e) => setFormData(prev => ({ ...prev, code: e.target.value.toUpperCase() }))}
                  placeholder="مثال: OPEN10"
                  className="uppercase"
                  data-testid="input-coupon-code"
                />
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="discountType">نوع الخصم *</Label>
                <Select
                  value={formData.discountType}
                  onValueChange={(value: "percent" | "amount") =>
                    setFormData(prev => ({ ...prev, discountType: value }))
                  }
                >
                  <SelectTrigger data-testid="select-discount-type">
                    <SelectValue placeholder="اختر نوع الخصم" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="percent">نسبة مئوية (%)</SelectItem>
                    <SelectItem value="amount">مبلغ ثابت (ر.س)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="discountValue">قيمة الخصم *</Label>
                <Input
                  id="discountValue"
                  type="number"
                  min="0"
                  step={formData.discountType === "percent" ? "1" : "0.01"}
                  max={formData.discountType === "percent" ? "100" : undefined}
                  value={formData.discountValue}
                  onChange={(e) => setFormData(prev => ({
                    ...prev,
                    discountValue: parseFloat(e.target.value) || 0
                  }))}
                  placeholder={formData.discountType === "percent" ? "10" : "50"}
                  data-testid="input-discount-value"
                />
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="startDate">تاريخ البداية *</Label>
                <Input
                  id="startDate"
                  type="date"
                  value={formData.startDate}
                  onChange={(e) => setFormData(prev => ({ ...prev, startDate: e.target.value }))}
                  data-testid="input-coupon-start-date"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="endDate">تاريخ النهاية *</Label>
                <Input
                  id="endDate"
                  type="date"
                  value={formData.endDate}
                  onChange={(e) => setFormData(prev => ({ ...prev, endDate: e.target.value }))}
                  data-testid="input-coupon-end-date"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="descriptionAr">الوصف</Label>
              <Textarea
                id="descriptionAr"
                value={formData.descriptionAr}
                onChange={(e) => setFormData(prev => ({ ...prev, descriptionAr: e.target.value }))}
                placeholder="وصف الكوبون..."
                data-testid="input-coupon-description"
              />
            </div>

            <div className="space-y-4">
              <Label className="text-lg font-semibold">يطبق على</Label>
              <div className="flex flex-wrap gap-4">
                <div className="flex items-center gap-2">
                  <Switch
                    id="appliesToServices"
                    checked={formData.appliesToServices}
                    onCheckedChange={(checked) => setFormData(prev => ({
                      ...prev,
                      appliesToServices: checked,
                      serviceIds: checked ? prev.serviceIds : [],
                    }))}
                    data-testid="switch-applies-services"
                  />
                  <Label htmlFor="appliesToServices">الخدمات</Label>
                </div>
                <div className="flex items-center gap-2">
                  <Switch
                    id="appliesToProducts"
                    checked={formData.appliesToProducts}
                    onCheckedChange={(checked) => setFormData(prev => ({
                      ...prev,
                      appliesToProducts: checked,
                      productIds: checked ? prev.productIds : [],
                    }))}
                    data-testid="switch-applies-products"
                  />
                  <Label htmlFor="appliesToProducts">المنتجات</Label>
                </div>
              </div>
            </div>

            {formData.appliesToServices && (
              <div className="space-y-2">
                <Label>تقييد على خدمات محددة (اتركه فارغاً ليطبق على الكل)</Label>
                <div className="flex flex-wrap gap-2 p-3 border rounded-lg max-h-40 overflow-y-auto">
                  {availableServices.map(service => (
                    <Badge
                      key={service.id}
                      variant={formData.serviceIds.includes(service.id) ? "default" : "outline"}
                      className="cursor-pointer"
                      onClick={() => toggleServiceId(service.id)}
                      data-testid={`badge-service-${service.id}`}
                    >
                      {service.nameAr}
                    </Badge>
                  ))}
                </div>
                {formData.serviceIds.length > 0 && (
                  <p className="text-sm text-muted-foreground">
                    تم اختيار {formData.serviceIds.length} خدمة
                  </p>
                )}
              </div>
            )}

            {formData.appliesToProducts && (
              <div className="space-y-2">
                <Label>تقييد على منتجات محددة (اتركه فارغاً ليطبق على الكل)</Label>
                <div className="flex flex-wrap gap-2 p-3 border rounded-lg max-h-40 overflow-y-auto">
                  {availableProducts.map(product => (
                    <Badge
                      key={product.id}
                      variant={formData.productIds.includes(product.id) ? "default" : "outline"}
                      className="cursor-pointer"
                      onClick={() => toggleProductId(product.id)}
                      data-testid={`badge-product-${product.id}`}
                    >
                      {product.nameAr}
                    </Badge>
                  ))}
                </div>
                {formData.productIds.length > 0 && (
                  <p className="text-sm text-muted-foreground">
                    تم اختيار {formData.productIds.length} منتج
                  </p>
                )}
              </div>
            )}

            <div className="flex items-center gap-2">
              <Switch
                id="isActive"
                checked={formData.isActive}
                onCheckedChange={(checked) => setFormData(prev => ({ ...prev, isActive: checked }))}
                data-testid="switch-coupon-active"
              />
              <Label htmlFor="isActive">الكوبون نشط</Label>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={resetForm} data-testid="button-cancel">
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-submit"
            >
              {editingCoupon ? "حفظ التغييرات" : "إنشاء الكوبون"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
