import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, Link } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Textarea } from "@/components/ui/textarea";
import {
  ArrowRight,
  User,
  Phone,
  Mail,
  MapPin,
  Star,
  Gift,
  ShoppingCart,
  Calendar,
  Plus,
  Minus,
  Eye,
  TrendingUp,
  TrendingDown,
} from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

type Customer = {
  id: number;
  nameAr: string;
  phone: string | null;
  phoneCountryCode: string | null;
  email: string | null;
  cityAr: string | null;
  branchId: number | null;
  address: string | null;
  profileImageUrl: string | null;
  loyaltyPointsBalance: number;
  isActive: boolean;
  notes: string | null;
  createdAt: string;
  updatedAt: string;
};

type CustomerSummary = {
  customer: Customer;
  ordersCount: number;
  totalSpent: number;
  ratingsCount: number;
  averageRating: number;
  loyaltyPoints: number;
  scheduledOrdersCount: number;
};

type OrderWithDetails = {
  id: number;
  orderNumber: string;
  orderDate: string;
  status: string;
  totalAmount: number;
  source: string;
  services: any[];
  products: any[];
  rating?: { rating: number; comment: string | null };
};

type OrderRating = {
  id: number;
  orderId: number;
  rating: number;
  comment: string | null;
  createdAt: string;
  order?: OrderWithDetails;
};

type LoyaltyTransaction = {
  id: number;
  customerId: number;
  transactionType: string;
  points: number;
  balanceAfter: number;
  referenceType: string | null;
  referenceId: number | null;
  notes: string | null;
  expiresAt: string | null;
  createdAt: string;
};

const addPointsSchema = z.object({
  points: z.coerce.number().min(1, "يجب أن تكون النقاط أكبر من صفر"),
  notes: z.string().optional(),
});

type AddPointsData = z.infer<typeof addPointsSchema>;

export default function CustomerDetailPage() {
  const { id } = useParams<{ id: string }>();
  const customerId = parseInt(id || "0");
  const { toast } = useToast();
  const [activeTab, setActiveTab] = useState("overview");
  const [isAddPointsOpen, setIsAddPointsOpen] = useState(false);
  const [isRedeemPointsOpen, setIsRedeemPointsOpen] = useState(false);

  const { data: summary, isLoading: loadingSummary } = useQuery<CustomerSummary>({
    queryKey: ["/api/customers", customerId, "summary"],
  });

  const { data: orders, isLoading: loadingOrders } = useQuery<OrderWithDetails[]>({
    queryKey: ["/api/customers", customerId, "orders"],
    enabled: activeTab === "orders" || activeTab === "scheduled",
  });

  const { data: ratings, isLoading: loadingRatings } = useQuery<OrderRating[]>({
    queryKey: ["/api/customers", customerId, "ratings"],
    enabled: activeTab === "ratings",
  });

  const { data: loyaltyHistory, isLoading: loadingLoyalty } = useQuery<LoyaltyTransaction[]>({
    queryKey: ["/api/customers", customerId, "loyalty"],
    enabled: activeTab === "loyalty",
  });

  const addPointsForm = useForm<AddPointsData>({
    resolver: zodResolver(addPointsSchema),
    defaultValues: { points: 0, notes: "" },
  });

  const redeemPointsForm = useForm<AddPointsData>({
    resolver: zodResolver(addPointsSchema),
    defaultValues: { points: 0, notes: "" },
  });

  const addPointsMutation = useMutation({
    mutationFn: (data: { points: number; notes?: string }) =>
      apiRequest("POST", `/api/customers/${customerId}/loyalty`, {
        points: data.points,
        transactionType: "earn",
        referenceType: "manual",
        notes: data.notes,
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers", customerId] });
      toast({ title: "تمت إضافة النقاط بنجاح" });
      setIsAddPointsOpen(false);
      addPointsForm.reset();
    },
    onError: (error: any) => {
      toast({ title: "فشل في إضافة النقاط", description: error.message, variant: "destructive" });
    },
  });

  const redeemPointsMutation = useMutation({
    mutationFn: (data: { points: number; notes?: string }) =>
      apiRequest("POST", `/api/customers/${customerId}/loyalty/redeem`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers", customerId] });
      toast({ title: "تم استبدال النقاط بنجاح" });
      setIsRedeemPointsOpen(false);
      redeemPointsForm.reset();
    },
    onError: (error: any) => {
      toast({ title: "فشل في استبدال النقاط", description: error.message, variant: "destructive" });
    },
  });

  const getStatusLabel = (status: string) => {
    const labels: Record<string, string> = {
      new: "جديد",
      in_progress: "قيد التنفيذ",
      completed: "مكتمل",
      canceled: "ملغي",
      scheduled: "مجدول",
      postponed: "مؤجل",
      deferred: "مؤجل",
      returned: "مرتجع",
    };
    return labels[status] || status;
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      new: "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400",
      in_progress: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400",
      completed: "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400",
      canceled: "bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400",
      scheduled: "bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400",
    };
    return colors[status] || "bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400";
  };

  const getTransactionTypeLabel = (type: string) => {
    const labels: Record<string, string> = {
      earn: "اكتساب",
      redeem: "استبدال",
      expire: "انتهاء صلاحية",
      adjust: "تعديل",
    };
    return labels[type] || type;
  };

  const getTransactionTypeColor = (type: string) => {
    if (type === "earn") return "text-green-600";
    if (type === "redeem" || type === "expire") return "text-red-600";
    return "text-gray-600";
  };

  const scheduledOrders = orders?.filter((o) => o.status === "scheduled") || [];

  if (loadingSummary) {
    return (
      <div className="container mx-auto p-6" dir="rtl">
        <div className="text-center py-12 text-muted-foreground">جاري التحميل...</div>
      </div>
    );
  }

  if (!summary) {
    return (
      <div className="container mx-auto p-6" dir="rtl">
        <div className="text-center py-12 text-muted-foreground">العميل غير موجود</div>
      </div>
    );
  }

  const customer = summary.customer;

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center gap-2 text-muted-foreground mb-4">
        <Link href="/customers">
          <Button variant="ghost" size="sm" data-testid="link-back">
            العملاء
            <ArrowRight className="mr-2 h-4 w-4" />
          </Button>
        </Link>
        <span>/</span>
        <span>{customer.nameAr}</span>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <Card className="lg:col-span-1">
          <CardHeader className="text-center">
            {customer.profileImageUrl ? (
              <img
                src={customer.profileImageUrl}
                alt={customer.nameAr}
                className="h-24 w-24 rounded-full mx-auto object-cover"
              />
            ) : (
              <div className="h-24 w-24 rounded-full bg-primary/10 flex items-center justify-center mx-auto">
                <User className="h-12 w-12 text-primary" />
              </div>
            )}
            <CardTitle className="mt-4" data-testid="text-customer-name">{customer.nameAr}</CardTitle>
            {customer.isActive ? (
              <Badge className="bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400 w-fit mx-auto">نشط</Badge>
            ) : (
              <Badge variant="secondary" className="w-fit mx-auto">غير نشط</Badge>
            )}
          </CardHeader>
          <CardContent className="space-y-4">
            {customer.phone && (
              <div className="flex items-center gap-2">
                <Phone className="h-4 w-4 text-muted-foreground" />
                <span dir="ltr">{customer.phoneCountryCode && `+${customer.phoneCountryCode}`} {customer.phone}</span>
              </div>
            )}
            {customer.email && (
              <div className="flex items-center gap-2">
                <Mail className="h-4 w-4 text-muted-foreground" />
                <span>{customer.email}</span>
              </div>
            )}
            {customer.cityAr && (
              <div className="flex items-center gap-2">
                <MapPin className="h-4 w-4 text-muted-foreground" />
                <span>{customer.cityAr}</span>
              </div>
            )}
            {customer.address && (
              <div className="flex items-center gap-2 text-sm text-muted-foreground">
                <span>{customer.address}</span>
              </div>
            )}

            <div className="border-t pt-4 mt-4">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm text-muted-foreground">نقاط الولاء</span>
                <div className="flex items-center gap-1">
                  <Star className="h-4 w-4 text-amber-500" />
                  <span className="font-bold text-lg" data-testid="text-loyalty-points">{formatNumber(customer.loyaltyPointsBalance)}</span>
                </div>
              </div>
              <div className="flex gap-2">
                <Button size="sm" className="flex-1" onClick={() => setIsAddPointsOpen(true)} data-testid="button-add-points">
                  إضافة
                  <Plus className="mr-1 h-4 w-4" />
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  className="flex-1"
                  onClick={() => setIsRedeemPointsOpen(true)}
                  disabled={customer.loyaltyPointsBalance === 0}
                  data-testid="button-redeem-points"
                >
                  استبدال
                  <Minus className="mr-1 h-4 w-4" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="lg:col-span-2">
          <CardHeader>
            <CardTitle>ملخص العميل</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <div className="text-center p-4 bg-muted/50 rounded-lg">
                <ShoppingCart className="h-6 w-6 mx-auto mb-2 text-primary" />
                <div className="text-2xl font-bold" data-testid="stat-orders">{formatNumber(summary.ordersCount)}</div>
                <div className="text-sm text-muted-foreground">الطلبات</div>
              </div>
              <div className="text-center p-4 bg-muted/50 rounded-lg">
                <TrendingUp className="h-6 w-6 mx-auto mb-2 text-green-600" />
                <div className="text-2xl font-bold" data-testid="stat-spent">{formatCurrency(summary.totalSpent)}</div>
                <div className="text-sm text-muted-foreground">إجمالي الإنفاق</div>
              </div>
              <div className="text-center p-4 bg-muted/50 rounded-lg">
                <Star className="h-6 w-6 mx-auto mb-2 text-amber-500" />
                <div className="text-2xl font-bold" data-testid="stat-rating">
                  {summary.averageRating > 0 ? summary.averageRating.toFixed(1) : "-"}
                </div>
                <div className="text-sm text-muted-foreground">متوسط التقييم ({summary.ratingsCount})</div>
              </div>
              <div className="text-center p-4 bg-muted/50 rounded-lg">
                <Calendar className="h-6 w-6 mx-auto mb-2 text-purple-600" />
                <div className="text-2xl font-bold" data-testid="stat-scheduled">{formatNumber(summary.scheduledOrdersCount)}</div>
                <div className="text-sm text-muted-foreground">الطلبات المجدولة</div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardContent className="p-0">
          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="w-full justify-start border-b rounded-none h-auto p-0">
              <TabsTrigger value="overview" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary" data-testid="tab-overview">
                نظرة عامة
              </TabsTrigger>
              <TabsTrigger value="orders" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary" data-testid="tab-orders">
                الطلبات
              </TabsTrigger>
              <TabsTrigger value="ratings" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary" data-testid="tab-ratings">
                التقييمات
              </TabsTrigger>
              <TabsTrigger value="loyalty" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary" data-testid="tab-loyalty">
                نقاط الولاء
              </TabsTrigger>
              <TabsTrigger value="scheduled" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary" data-testid="tab-scheduled">
                الطلبات المجدولة
              </TabsTrigger>
            </TabsList>

            <TabsContent value="overview" className="p-6">
              <div className="space-y-4">
                <h3 className="font-semibold">معلومات إضافية</h3>
                {customer.notes && (
                  <div className="p-4 bg-muted/50 rounded-lg">
                    <p className="text-sm text-muted-foreground mb-1">ملاحظات</p>
                    <p>{customer.notes}</p>
                  </div>
                )}
                <div className="grid grid-cols-2 gap-4 text-sm">
                  <div>
                    <span className="text-muted-foreground">تاريخ التسجيل: </span>
                    <span>{formatDate(customer.createdAt)}</span>
                  </div>
                  <div>
                    <span className="text-muted-foreground">آخر تحديث: </span>
                    <span>{formatDate(customer.updatedAt)}</span>
                  </div>
                </div>
              </div>
            </TabsContent>

            <TabsContent value="orders" className="p-6">
              {loadingOrders ? (
                <div className="text-center py-8 text-muted-foreground">جاري التحميل...</div>
              ) : orders?.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">لا توجد طلبات</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">رقم الطلب</TableHead>
                      <TableHead className="text-right">التاريخ</TableHead>
                      <TableHead className="text-right">الحالة</TableHead>
                      <TableHead className="text-right">المبلغ</TableHead>
                      <TableHead className="text-center">الإجراءات</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {orders?.map((order) => (
                      <TableRow key={order.id} data-testid={`row-order-${order.id}`}>
                        <TableCell className="font-medium">{order.orderNumber}</TableCell>
                        <TableCell>{formatDate(order.orderDate)}</TableCell>
                        <TableCell>
                          <Badge className={getStatusColor(order.status)}>{getStatusLabel(order.status)}</Badge>
                        </TableCell>
                        <TableCell>{formatCurrency(order.totalAmount)}</TableCell>
                        <TableCell className="text-center">
                          <Link href={`/orders/${order.id}`}>
                            <Button size="icon" variant="ghost" data-testid={`button-view-order-${order.id}`}>
                              <Eye className="h-4 w-4" />
                            </Button>
                          </Link>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </TabsContent>

            <TabsContent value="ratings" className="p-6">
              {loadingRatings ? (
                <div className="text-center py-8 text-muted-foreground">جاري التحميل...</div>
              ) : ratings?.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">لا توجد تقييمات</div>
              ) : (
                <div className="space-y-4">
                  {ratings?.map((rating) => (
                    <Card key={rating.id} data-testid={`card-rating-${rating.id}`}>
                      <CardContent className="p-4">
                        <div className="flex items-start justify-between">
                          <div>
                            <div className="flex items-center gap-2 mb-2">
                              {[1, 2, 3, 4, 5].map((star) => (
                                <Star
                                  key={star}
                                  className={`h-5 w-5 ${star <= rating.rating ? "text-amber-500 fill-amber-500" : "text-gray-300"}`}
                                />
                              ))}
                              <span className="font-bold">{rating.rating}/5</span>
                            </div>
                            {rating.comment && <p className="text-muted-foreground">{rating.comment}</p>}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            <div>{formatDate(rating.createdAt)}</div>
                            {rating.order && (
                              <Link href={`/orders/${rating.orderId}`}>
                                <span className="text-primary hover:underline">طلب #{rating.order.orderNumber}</span>
                              </Link>
                            )}
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </TabsContent>

            <TabsContent value="loyalty" className="p-6">
              {loadingLoyalty ? (
                <div className="text-center py-8 text-muted-foreground">جاري التحميل...</div>
              ) : loyaltyHistory?.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">لا توجد معاملات</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">التاريخ</TableHead>
                      <TableHead className="text-right">النوع</TableHead>
                      <TableHead className="text-right">النقاط</TableHead>
                      <TableHead className="text-right">الرصيد بعد</TableHead>
                      <TableHead className="text-right">ملاحظات</TableHead>
                      <TableHead className="text-right">تاريخ الانتهاء</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {loyaltyHistory?.map((transaction) => (
                      <TableRow key={transaction.id} data-testid={`row-loyalty-${transaction.id}`}>
                        <TableCell>{formatDate(transaction.createdAt)}</TableCell>
                        <TableCell>
                          <Badge variant="outline">{getTransactionTypeLabel(transaction.transactionType)}</Badge>
                        </TableCell>
                        <TableCell>
                          <span className={`font-bold ${getTransactionTypeColor(transaction.transactionType)}`}>
                            {transaction.points > 0 ? (
                              <span className="flex items-center gap-1">
                                <TrendingUp className="h-4 w-4" />+{formatNumber(transaction.points)}
                              </span>
                            ) : (
                              <span className="flex items-center gap-1">
                                <TrendingDown className="h-4 w-4" />{formatNumber(transaction.points)}
                              </span>
                            )}
                          </span>
                        </TableCell>
                        <TableCell>{formatNumber(transaction.balanceAfter)}</TableCell>
                        <TableCell className="text-muted-foreground">{transaction.notes || "-"}</TableCell>
                        <TableCell>
                          {transaction.expiresAt ? formatDate(transaction.expiresAt) : "-"}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </TabsContent>

            <TabsContent value="scheduled" className="p-6">
              {loadingOrders ? (
                <div className="text-center py-8 text-muted-foreground">جاري التحميل...</div>
              ) : scheduledOrders.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">لا توجد طلبات مجدولة</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">رقم الطلب</TableHead>
                      <TableHead className="text-right">التاريخ</TableHead>
                      <TableHead className="text-right">الخدمات</TableHead>
                      <TableHead className="text-right">المبلغ</TableHead>
                      <TableHead className="text-center">الإجراءات</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {scheduledOrders.map((order) => (
                      <TableRow key={order.id} data-testid={`row-scheduled-${order.id}`}>
                        <TableCell className="font-medium">{order.orderNumber}</TableCell>
                        <TableCell>{formatDate(order.orderDate)}</TableCell>
                        <TableCell>{order.services.length} خدمة</TableCell>
                        <TableCell>{formatCurrency(order.totalAmount)}</TableCell>
                        <TableCell className="text-center">
                          <Link href={`/orders/${order.id}`}>
                            <Button size="icon" variant="ghost">
                              <Eye className="h-4 w-4" />
                            </Button>
                          </Link>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>

      <Dialog open={isAddPointsOpen} onOpenChange={setIsAddPointsOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>إضافة نقاط ولاء</DialogTitle>
            <DialogDescription>أضف نقاط ولاء يدوياً للعميل</DialogDescription>
          </DialogHeader>
          <Form {...addPointsForm}>
            <form onSubmit={addPointsForm.handleSubmit((data) => addPointsMutation.mutate(data))} className="space-y-4">
              <FormField
                control={addPointsForm.control}
                name="points"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>عدد النقاط</FormLabel>
                    <FormControl>
                      <Input {...field} type="number" min="1" data-testid="input-add-points" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={addPointsForm.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>ملاحظات (اختياري)</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="سبب إضافة النقاط" data-testid="input-add-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <div className="flex justify-end gap-2">
                <Button type="button" variant="outline" onClick={() => setIsAddPointsOpen(false)}>
                  إلغاء
                </Button>
                <Button type="submit" disabled={addPointsMutation.isPending} data-testid="button-confirm-add">
                  {addPointsMutation.isPending ? "جاري الإضافة..." : "إضافة"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      <Dialog open={isRedeemPointsOpen} onOpenChange={setIsRedeemPointsOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>استبدال نقاط ولاء</DialogTitle>
            <DialogDescription>
              الرصيد الحالي: {formatNumber(customer.loyaltyPointsBalance)} نقطة
            </DialogDescription>
          </DialogHeader>
          <Form {...redeemPointsForm}>
            <form onSubmit={redeemPointsForm.handleSubmit((data) => redeemPointsMutation.mutate(data))} className="space-y-4">
              <FormField
                control={redeemPointsForm.control}
                name="points"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>عدد النقاط للاستبدال</FormLabel>
                    <FormControl>
                      <Input {...field} type="number" min="1" max={customer.loyaltyPointsBalance} data-testid="input-redeem-points" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={redeemPointsForm.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>ملاحظات (اختياري)</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="سبب الاستبدال" data-testid="input-redeem-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <div className="flex justify-end gap-2">
                <Button type="button" variant="outline" onClick={() => setIsRedeemPointsOpen(false)}>
                  إلغاء
                </Button>
                <Button type="submit" disabled={redeemPointsMutation.isPending} data-testid="button-confirm-redeem">
                  {redeemPointsMutation.isPending ? "جاري الاستبدال..." : "استبدال"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
