import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { MessageSquare, Eye, CheckCircle2, Clock, AlertTriangle, Star, ThumbsUp, ThumbsDown, Search } from "lucide-react";
import { formatDate } from "@/lib/formatters";

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface Employee {
  id: number;
  nameAr: string;
}

interface ReviewReason {
  id: number;
  nameAr: string;
  reasonType: string;
  sortOrder: number;
  isActive: boolean;
}

interface CustomerReview {
  id: number;
  customerId: number;
  orderId: number | null;
  reviewReasonId: number;
  rating: number | null;
  additionalNotes: string | null;
  status: string;
  reviewedByEmployeeId: number | null;
  reviewedAt: string | null;
  responseNotes: string | null;
  createdAt: string;
  updatedAt: string;
  customer?: Customer;
  reviewReason?: ReviewReason;
  reviewedByEmployee?: Employee;
}

const statusOptions = [
  { value: "pending", label: "قيد المراجعة", color: "bg-yellow-500" },
  { value: "reviewed", label: "تمت المراجعة", color: "bg-blue-500" },
  { value: "resolved", label: "تم الحل", color: "bg-green-500" },
];

export default function CustomerReviewsPage() {
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedReview, setSelectedReview] = useState<CustomerReview | null>(null);
  const [isViewDialogOpen, setIsViewDialogOpen] = useState(false);
  const [responseNotes, setResponseNotes] = useState("");
  const [newStatus, setNewStatus] = useState("reviewed");
  const [selectedReviewerId, setSelectedReviewerId] = useState<string>("");
  const { toast } = useToast();

  const { data: reviews, isLoading } = useQuery<CustomerReview[]>({
    queryKey: ["/api/customer-reviews"],
  });

  const { data: reviewReasons } = useQuery<ReviewReason[]>({
    queryKey: ["/api/review-reasons"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const updateStatusMutation = useMutation({
    mutationFn: async ({ id, status, responseNotes, reviewedByEmployeeId }: { id: number; status: string; responseNotes?: string; reviewedByEmployeeId?: number }) => {
      return apiRequest("PUT", `/api/customer-reviews/${id}/status`, {
        status,
        reviewedByEmployeeId,
        responseNotes,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/customer-reviews"] });
      toast({
        title: "تم تحديث الحالة",
        description: "تم تحديث حالة التقييم بنجاح",
      });
      setIsViewDialogOpen(false);
      setSelectedReview(null);
      setResponseNotes("");
    },
    onError: () => {
      toast({
        title: "خطأ",
        description: "فشل في تحديث حالة التقييم",
        variant: "destructive",
      });
    },
  });

  const handleViewReview = (review: CustomerReview) => {
    setSelectedReview(review);
    setResponseNotes(review.responseNotes || "");
    setNewStatus(review.status);
    setSelectedReviewerId(review.reviewedByEmployeeId?.toString() || "");
    setIsViewDialogOpen(true);
  };

  const handleUpdateStatus = () => {
    if (!selectedReview) return;
    updateStatusMutation.mutate({
      id: selectedReview.id,
      status: newStatus,
      responseNotes,
      reviewedByEmployeeId: selectedReviewerId ? parseInt(selectedReviewerId) : undefined,
    });
  };

  const getReasonTypeIcon = (type: string) => {
    switch (type) {
      case "positive":
        return <ThumbsUp className="h-4 w-4 text-green-500" />;
      case "negative":
        return <ThumbsDown className="h-4 w-4 text-red-500" />;
      default:
        return <MessageSquare className="h-4 w-4 text-gray-500" />;
    }
  };

  const getStatusBadge = (status: string) => {
    const statusInfo = statusOptions.find(s => s.value === status);
    if (!statusInfo) return null;
    return (
      <Badge className={`${statusInfo.color} text-white`}>
        {status === "pending" && <Clock className="h-3 w-3 ml-1" />}
        {status === "reviewed" && <Eye className="h-3 w-3 ml-1" />}
        {status === "resolved" && <CheckCircle2 className="h-3 w-3 ml-1" />}
        {statusInfo.label}
      </Badge>
    );
  };

  const getRatingStars = (rating: number | null) => {
    if (!rating) return <span className="text-muted-foreground">-</span>;
    return (
      <div className="flex gap-0.5">
        {[1, 2, 3, 4, 5].map((star) => (
          <Star
            key={star}
            className={`h-4 w-4 ${star <= rating ? "text-yellow-500 fill-yellow-500" : "text-gray-300"}`}
          />
        ))}
      </div>
    );
  };

  const filteredReviews = reviews?.filter((review) => {
    const matchesStatus = statusFilter === "all" || review.status === statusFilter;
    const matchesSearch = searchTerm === "" || 
      review.customer?.nameAr.includes(searchTerm) ||
      review.reviewReason?.nameAr.includes(searchTerm) ||
      review.additionalNotes?.includes(searchTerm);
    return matchesStatus && matchesSearch;
  });

  const pendingCount = reviews?.filter(r => r.status === "pending").length || 0;
  const positiveCount = reviews?.filter(r => r.reviewReason?.reasonType === "positive").length || 0;
  const negativeCount = reviews?.filter(r => r.reviewReason?.reasonType === "negative").length || 0;

  return (
    <div className="p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 bg-primary/10 rounded-lg">
            <MessageSquare className="h-6 w-6 text-primary" />
          </div>
          <div>
            <h1 className="text-2xl font-bold">تقييمات العملاء</h1>
            <p className="text-muted-foreground">إدارة ومتابعة تقييمات وملاحظات العملاء</p>
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي التقييمات</CardTitle>
            <MessageSquare className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{reviews?.length || 0}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">قيد المراجعة</CardTitle>
            <Clock className="h-4 w-4 text-yellow-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600">{pendingCount}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">تقييمات إيجابية</CardTitle>
            <ThumbsUp className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{positiveCount}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">تقييمات سلبية</CardTitle>
            <ThumbsDown className="h-4 w-4 text-red-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{negativeCount}</div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between gap-4 flex-wrap">
            <CardTitle>قائمة التقييمات</CardTitle>
            <div className="flex items-center gap-4 flex-wrap">
              <div className="relative">
                <Search className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="بحث..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pr-10 w-64"
                  data-testid="input-search"
                />
              </div>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger className="w-40" data-testid="select-status-filter">
                  <SelectValue placeholder="الحالة" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  {statusOptions.map((status) => (
                    <SelectItem key={status.value} value={status.value}>
                      {status.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-4">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : filteredReviews?.length === 0 ? (
            <div className="text-center py-12">
              <MessageSquare className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
              <p className="text-muted-foreground">لا توجد تقييمات</p>
            </div>
          ) : (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">العميل</TableHead>
                    <TableHead className="text-right">سبب التقييم</TableHead>
                    <TableHead className="text-right">التقييم</TableHead>
                    <TableHead className="text-right">رقم الطلب</TableHead>
                    <TableHead className="text-right">الملاحظات</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">الإجراءات</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredReviews?.map((review) => (
                    <TableRow key={review.id} data-testid={`row-review-${review.id}`}>
                      <TableCell className="font-medium">
                        {review.customer?.nameAr || "-"}
                        {review.customer?.phone && (
                          <span className="text-xs text-muted-foreground block">{review.customer.phone}</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {getReasonTypeIcon(review.reviewReason?.reasonType || "")}
                          <span>{review.reviewReason?.nameAr || "-"}</span>
                        </div>
                      </TableCell>
                      <TableCell>{getRatingStars(review.rating)}</TableCell>
                      <TableCell>
                        {review.orderId ? (
                          <Badge variant="outline">#{review.orderId}</Badge>
                        ) : (
                          "-"
                        )}
                      </TableCell>
                      <TableCell className="max-w-[200px] truncate">
                        {review.additionalNotes || "-"}
                      </TableCell>
                      <TableCell>{getStatusBadge(review.status)}</TableCell>
                      <TableCell className="text-muted-foreground">
                        {formatDate(review.createdAt)}
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleViewReview(review)}
                          data-testid={`button-view-${review.id}`}
                        >
                          <Eye className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={isViewDialogOpen} onOpenChange={setIsViewDialogOpen}>
        <DialogContent className="max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>تفاصيل التقييم</DialogTitle>
          </DialogHeader>
          {selectedReview && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">العميل</p>
                  <p className="font-medium">{selectedReview.customer?.nameAr || "-"}</p>
                  {selectedReview.customer?.phone && (
                    <p className="text-sm text-muted-foreground">{selectedReview.customer.phone}</p>
                  )}
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">رقم الطلب</p>
                  <p className="font-medium">
                    {selectedReview.orderId ? `#${selectedReview.orderId}` : "-"}
                  </p>
                </div>
              </div>

              <div>
                <p className="text-sm text-muted-foreground">سبب التقييم</p>
                <div className="flex items-center gap-2 mt-1">
                  {getReasonTypeIcon(selectedReview.reviewReason?.reasonType || "")}
                  <span className="font-medium">{selectedReview.reviewReason?.nameAr || "-"}</span>
                </div>
              </div>

              {selectedReview.rating && (
                <div>
                  <p className="text-sm text-muted-foreground">التقييم</p>
                  <div className="mt-1">{getRatingStars(selectedReview.rating)}</div>
                </div>
              )}

              {selectedReview.additionalNotes && (
                <div>
                  <p className="text-sm text-muted-foreground">ملاحظات العميل</p>
                  <p className="bg-muted p-3 rounded-lg mt-1">{selectedReview.additionalNotes}</p>
                </div>
              )}

              <div>
                <p className="text-sm text-muted-foreground">تاريخ التقييم</p>
                <p className="font-medium">{formatDate(selectedReview.createdAt)}</p>
              </div>

              {selectedReview.reviewedByEmployee && (
                <div>
                  <p className="text-sm text-muted-foreground">تمت المراجعة بواسطة</p>
                  <p className="font-medium">
                    {selectedReview.reviewedByEmployee.nameAr}
                    {selectedReview.reviewedAt && (
                      <span className="text-sm text-muted-foreground mr-2">
                        ({formatDate(selectedReview.reviewedAt)})
                      </span>
                    )}
                  </p>
                </div>
              )}

              <div className="border-t pt-4 space-y-4">
                <div>
                  <label className="text-sm font-medium">تغيير الحالة</label>
                  <Select value={newStatus} onValueChange={setNewStatus}>
                    <SelectTrigger className="mt-1" data-testid="select-new-status">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {statusOptions.map((status) => (
                        <SelectItem key={status.value} value={status.value}>
                          {status.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <label className="text-sm font-medium">المراجع</label>
                  <Select value={selectedReviewerId} onValueChange={setSelectedReviewerId}>
                    <SelectTrigger className="mt-1" data-testid="select-reviewer">
                      <SelectValue placeholder="اختر المراجع (اختياري)" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="">-- لم يحدد --</SelectItem>
                      {employees?.map((employee) => (
                        <SelectItem key={employee.id} value={employee.id.toString()}>
                          {employee.nameAr}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <label className="text-sm font-medium">ملاحظات الرد</label>
                  <Textarea
                    value={responseNotes}
                    onChange={(e) => setResponseNotes(e.target.value)}
                    placeholder="أضف ملاحظاتك أو الرد على العميل..."
                    className="mt-1"
                    rows={3}
                    data-testid="textarea-response"
                  />
                </div>
              </div>
            </div>
          )}
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsViewDialogOpen(false)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleUpdateStatus} 
              disabled={updateStatusMutation.isPending}
              data-testid="button-save-status"
            >
              {updateStatusMutation.isPending ? "جاري الحفظ..." : "حفظ التغييرات"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
