import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber } from "@/lib/formatters";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import {
  Plus,
  Search,
  Eye,
  Edit,
  Trash2,
  Users,
  Phone,
  Mail,
  MapPin,
  Star,
  Gift,
} from "lucide-react";

type Customer = {
  id: number;
  nameAr: string;
  phone: string | null;
  phoneCountryCode: string | null;
  email: string | null;
  cityAr: string | null;
  branchId: number | null;
  address: string | null;
  profileImageUrl: string | null;
  loyaltyPointsBalance: number;
  isActive: boolean;
  notes: string | null;
  createdAt: string;
  updatedAt: string;
};

const customerFormSchema = z.object({
  nameAr: z.string().min(1, "اسم العميل مطلوب"),
  phone: z.string().optional().nullable(),
  phoneCountryCode: z.string().default("966").optional(),
  email: z.string().email("البريد الإلكتروني غير صالح").optional().nullable().or(z.literal("")),
  cityAr: z.string().optional().nullable(),
  branchId: z.number().optional().nullable(),
  address: z.string().optional().nullable(),
  profileImageUrl: z.string().url("رابط الصورة غير صالح").optional().nullable().or(z.literal("")),
  password: z.string().min(6, "كلمة المرور يجب أن تكون 6 أحرف على الأقل").optional().nullable().or(z.literal("")),
  notes: z.string().optional().nullable(),
});

type CustomerFormData = z.infer<typeof customerFormSchema>;

export default function CustomersPage() {
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [isFormOpen, setIsFormOpen] = useState(false);
  const [editingCustomer, setEditingCustomer] = useState<Customer | null>(null);
  const [deleteConfirmId, setDeleteConfirmId] = useState<number | null>(null);

  const { data: customers, isLoading } = useQuery<Customer[]>({
    queryKey: ["/api/customers"],
  });

  const form = useForm<CustomerFormData>({
    resolver: zodResolver(customerFormSchema),
    defaultValues: {
      nameAr: "",
      phone: "",
      phoneCountryCode: "966",
      email: "",
      cityAr: "",
      branchId: null,
      address: "",
      profileImageUrl: "",
      password: "",
      notes: "",
    },
  });

  const createMutation = useMutation({
    mutationFn: (data: CustomerFormData) => apiRequest("POST", "/api/customers", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      toast({ title: "تم إنشاء العميل بنجاح" });
      closeForm();
    },
    onError: (error: any) => {
      toast({ title: "فشل في إنشاء العميل", description: error.message, variant: "destructive" });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: CustomerFormData }) => 
      apiRequest("PATCH", `/api/customers/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      toast({ title: "تم تحديث العميل بنجاح" });
      closeForm();
    },
    onError: (error: any) => {
      toast({ title: "فشل في تحديث العميل", description: error.message, variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/customers/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      toast({ title: "تم حذف العميل بنجاح" });
      setDeleteConfirmId(null);
    },
    onError: (error: any) => {
      toast({ title: "فشل في حذف العميل", description: error.message, variant: "destructive" });
    },
  });

  const closeForm = () => {
    setIsFormOpen(false);
    setEditingCustomer(null);
    form.reset({
      nameAr: "",
      phone: "",
      phoneCountryCode: "966",
      email: "",
      cityAr: "",
      branchId: null,
      address: "",
      profileImageUrl: "",
      password: "",
      notes: "",
    });
  };

  const openEditForm = (customer: Customer) => {
    setEditingCustomer(customer);
    form.reset({
      nameAr: customer.nameAr,
      phone: customer.phone || "",
      phoneCountryCode: customer.phoneCountryCode || "966",
      email: customer.email || "",
      cityAr: customer.cityAr || "",
      branchId: customer.branchId,
      address: customer.address || "",
      profileImageUrl: customer.profileImageUrl || "",
      password: "",
      notes: customer.notes || "",
    });
    setIsFormOpen(true);
  };

  const onSubmit = (data: CustomerFormData) => {
    const cleanData = {
      ...data,
      email: data.email || null,
      phone: data.phone || null,
      profileImageUrl: data.profileImageUrl || null,
      password: data.password || null,
    };

    if (editingCustomer) {
      updateMutation.mutate({ id: editingCustomer.id, data: cleanData });
    } else {
      createMutation.mutate(cleanData);
    }
  };

  const filteredCustomers = customers?.filter((c) => {
    if (!searchQuery) return true;
    const query = searchQuery.toLowerCase();
    return (
      c.nameAr?.toLowerCase().includes(query) ||
      c.phone?.toLowerCase().includes(query) ||
      c.email?.toLowerCase().includes(query) ||
      c.cityAr?.toLowerCase().includes(query)
    );
  });

  const activeCustomers = customers?.filter((c) => c.isActive) || [];
  const totalPoints = customers?.reduce((sum, c) => sum + c.loyaltyPointsBalance, 0) || 0;

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-3xl font-bold" data-testid="page-title">العملاء</h1>
          <p className="text-muted-foreground">إدارة بيانات العملاء ونقاط الولاء</p>
        </div>
        <div className="flex items-center gap-2 flex-wrap">
          <Link href="/loyalty-settings">
            <Button variant="outline" data-testid="link-loyalty-settings">
              إعدادات الولاء
              <Gift className="mr-2 h-4 w-4" />
            </Button>
          </Link>
          <Button onClick={() => setIsFormOpen(true)} data-testid="button-add-customer">
            إضافة عميل
            <Plus className="mr-2 h-4 w-4" />
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="hover-elevate">
          <CardContent className="p-4 text-center">
            <Users className="h-8 w-8 mx-auto mb-2 text-primary" />
            <div className="text-2xl font-bold" data-testid="stat-total">{formatNumber(customers?.length || 0)}</div>
            <div className="text-sm text-muted-foreground">إجمالي العملاء</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate">
          <CardContent className="p-4 text-center">
            <Users className="h-8 w-8 mx-auto mb-2 text-green-600" />
            <div className="text-2xl font-bold text-green-600" data-testid="stat-active">{formatNumber(activeCustomers.length)}</div>
            <div className="text-sm text-muted-foreground">العملاء النشطين</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate">
          <CardContent className="p-4 text-center">
            <Star className="h-8 w-8 mx-auto mb-2 text-amber-500" />
            <div className="text-2xl font-bold text-amber-500" data-testid="stat-points">{formatNumber(totalPoints)}</div>
            <div className="text-sm text-muted-foreground">إجمالي نقاط الولاء</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate">
          <CardContent className="p-4 text-center">
            <Gift className="h-8 w-8 mx-auto mb-2 text-purple-600" />
            <div className="text-2xl font-bold text-purple-600" data-testid="stat-avg-points">
              {formatNumber(customers?.length ? Math.round(totalPoints / customers.length) : 0)}
            </div>
            <div className="text-sm text-muted-foreground">متوسط النقاط</div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between gap-2 flex-wrap">
            <CardTitle>قائمة العملاء</CardTitle>
            <div className="relative w-full max-w-sm">
              <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="البحث عن عميل..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pr-10"
                data-testid="input-search"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8 text-muted-foreground">جاري التحميل...</div>
          ) : filteredCustomers?.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">لا يوجد عملاء</div>
          ) : (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">الاسم</TableHead>
                    <TableHead className="text-right">الهاتف</TableHead>
                    <TableHead className="text-right">البريد</TableHead>
                    <TableHead className="text-right">المدينة</TableHead>
                    <TableHead className="text-right">نقاط الولاء</TableHead>
                    <TableHead className="text-right">الحالة</TableHead>
                    <TableHead className="text-center">الإجراءات</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredCustomers?.map((customer) => (
                    <TableRow key={customer.id} data-testid={`row-customer-${customer.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          {customer.profileImageUrl ? (
                            <img
                              src={customer.profileImageUrl}
                              alt={customer.nameAr}
                              className="h-8 w-8 rounded-full object-cover"
                            />
                          ) : (
                            <div className="h-8 w-8 rounded-full bg-primary/10 flex items-center justify-center">
                              <Users className="h-4 w-4 text-primary" />
                            </div>
                          )}
                          <span data-testid={`text-customer-name-${customer.id}`}>{customer.nameAr}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {customer.phone ? (
                          <div className="flex items-center gap-1">
                            <Phone className="h-3 w-3" />
                            <span dir="ltr">{customer.phoneCountryCode && `+${customer.phoneCountryCode}`} {customer.phone}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {customer.email ? (
                          <div className="flex items-center gap-1">
                            <Mail className="h-3 w-3" />
                            <span>{customer.email}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {customer.cityAr ? (
                          <div className="flex items-center gap-1">
                            <MapPin className="h-3 w-3" />
                            <span>{customer.cityAr}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <Badge variant="secondary" className="font-bold">
                          <Star className="h-3 w-3 ml-1" />
                          {formatNumber(customer.loyaltyPointsBalance)}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        {customer.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400">نشط</Badge>
                        ) : (
                          <Badge variant="secondary">غير نشط</Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center justify-center gap-1">
                          <Link href={`/customers/${customer.id}`}>
                            <Button size="icon" variant="ghost" data-testid={`button-view-${customer.id}`}>
                              <Eye className="h-4 w-4" />
                            </Button>
                          </Link>
                          <Button
                            size="icon"
                            variant="ghost"
                            onClick={() => openEditForm(customer)}
                            data-testid={`button-edit-${customer.id}`}
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                          <Button
                            size="icon"
                            variant="ghost"
                            className="text-red-600 hover:text-red-700"
                            onClick={() => setDeleteConfirmId(customer.id)}
                            data-testid={`button-delete-${customer.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={isFormOpen} onOpenChange={(open) => !open && closeForm()}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto" dir="rtl">
          <DialogHeader>
            <DialogTitle>{editingCustomer ? "تعديل العميل" : "إضافة عميل جديد"}</DialogTitle>
            <DialogDescription>
              {editingCustomer ? "قم بتحديث بيانات العميل" : "أدخل بيانات العميل الجديد"}
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="nameAr"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الاسم *</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder="أدخل اسم العميل" data-testid="input-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <div className="flex gap-2">
                  <FormField
                    control={form.control}
                    name="phoneCountryCode"
                    render={({ field }) => (
                      <FormItem className="w-24">
                        <FormLabel>الرمز</FormLabel>
                        <Select value={field.value || "966"} onValueChange={field.onChange}>
                          <FormControl>
                            <SelectTrigger data-testid="select-country-code">
                              <SelectValue />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="966">+966</SelectItem>
                            <SelectItem value="971">+971</SelectItem>
                            <SelectItem value="965">+965</SelectItem>
                            <SelectItem value="973">+973</SelectItem>
                            <SelectItem value="968">+968</SelectItem>
                            <SelectItem value="974">+974</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="phone"
                    render={({ field }) => (
                      <FormItem className="flex-1">
                        <FormLabel>الهاتف</FormLabel>
                        <FormControl>
                          <Input {...field} value={field.value || ""} placeholder="0500000000" dir="ltr" data-testid="input-phone" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <FormField
                  control={form.control}
                  name="email"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>البريد الإلكتروني</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value || ""} type="email" placeholder="email@example.com" dir="ltr" data-testid="input-email" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="password"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>كلمة المرور (للتطبيق)</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value || ""} type="password" placeholder="******" data-testid="input-password" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="cityAr"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>المدينة</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value || ""} placeholder="الرياض" data-testid="input-city" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="address"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>العنوان</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value || ""} placeholder="أدخل العنوان" data-testid="input-address" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="profileImageUrl"
                  render={({ field }) => (
                    <FormItem className="md:col-span-2">
                      <FormLabel>رابط صورة الملف الشخصي</FormLabel>
                      <FormControl>
                        <Input {...field} value={field.value || ""} placeholder="https://..." dir="ltr" data-testid="input-image-url" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="notes"
                  render={({ field }) => (
                    <FormItem className="md:col-span-2">
                      <FormLabel>ملاحظات</FormLabel>
                      <FormControl>
                        <Textarea {...field} value={field.value || ""} placeholder="ملاحظات إضافية" data-testid="input-notes" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="flex justify-end gap-2 pt-4">
                <Button type="button" variant="outline" onClick={closeForm} data-testid="button-cancel">
                  إلغاء
                </Button>
                <Button type="submit" disabled={createMutation.isPending || updateMutation.isPending} data-testid="button-submit">
                  {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : editingCustomer ? "تحديث" : "إضافة"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      <Dialog open={!!deleteConfirmId} onOpenChange={() => setDeleteConfirmId(null)}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تأكيد الحذف</DialogTitle>
            <DialogDescription>
              هل أنت متأكد من حذف هذا العميل؟ لن يتم حذف البيانات نهائياً وسيتم تعطيل الحساب فقط.
            </DialogDescription>
          </DialogHeader>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setDeleteConfirmId(null)} data-testid="button-cancel-delete">
              إلغاء
            </Button>
            <Button
              variant="destructive"
              onClick={() => deleteConfirmId && deleteMutation.mutate(deleteConfirmId)}
              disabled={deleteMutation.isPending}
              data-testid="button-confirm-delete"
            >
              {deleteMutation.isPending ? "جاري الحذف..." : "حذف"}
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
