import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Calendar,
  Wallet,
  CreditCard,
  ArrowDownRight,
  ArrowUpRight,
  Banknote,
  DollarSign,
  Receipt,
  Home,
  Undo2,
  Package,
  Calculator,
} from "lucide-react";
import { format } from "date-fns";
import type { SalonDailyReport } from "@shared/schema";
import { formatCurrency, formatDateLong } from "@/lib/formatters";

export default function DailyReportPage() {
  const [selectedDate, setSelectedDate] = useState(
    format(new Date(), "yyyy-MM-dd")
  );

  const { data: report, isLoading } = useQuery<SalonDailyReport>({
    queryKey: ["/api/reports/daily", selectedDate],
    queryFn: async () => {
      const response = await fetch(`/api/reports/daily?date=${selectedDate}`);
      if (!response.ok) throw new Error("Failed to fetch daily report");
      return response.json();
    },
  });

  return (
    <div className="space-y-6">
      <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-4">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">
            تقرير اليومية
          </h1>
          <p className="text-muted-foreground">
            تقرير تفصيلي لإيرادات ومتحصلات اليوم
          </p>
        </div>

        <div className="flex items-center gap-3">
          <Label htmlFor="date-picker" className="whitespace-nowrap">
            التاريخ:
          </Label>
          <Input
            id="date-picker"
            type="date"
            value={selectedDate}
            onChange={(e) => setSelectedDate(e.target.value)}
            className="w-44"
            data-testid="input-date"
          />
        </div>
      </div>

      {isLoading ? (
        <DailyReportSkeleton />
      ) : report ? (
        <>
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5" />
                {formatDateLong(report.date)}
              </CardTitle>
            </CardHeader>
          </Card>

          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  إجمالي الإيرادات
                </CardTitle>
                <DollarSign className="h-4 w-4 text-green-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-green-600"
                  data-testid="text-grand-total"
                >
                  {formatCurrency(report.grandTotal)}
                </div>
                <p className="text-xs text-muted-foreground">شامل الضريبة</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  صافي قبل الضريبة
                </CardTitle>
                <Calculator className="h-4 w-4 text-blue-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-blue-600"
                  data-testid="text-revenue-before-vat"
                >
                  {formatCurrency(report.totalRevenueBeforeVat)}
                </div>
                <p className="text-xs text-muted-foreground">
                  قبل ضريبة القيمة المضافة
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  ضريبة القيمة المضافة
                </CardTitle>
                <Receipt className="h-4 w-4 text-amber-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-amber-600"
                  data-testid="text-total-vat"
                >
                  {formatCurrency(report.totalVat)}
                </div>
                <p className="text-xs text-muted-foreground">VAT 15%</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  صافي المتحصلات
                </CardTitle>
                <Wallet className="h-4 w-4 text-purple-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-purple-600"
                  data-testid="text-net-payments"
                >
                  {formatCurrency(report.netPayments)}
                </div>
                <p className="text-xs text-muted-foreground">
                  بعد خصم المرتجعات
                </p>
              </CardContent>
            </Card>
          </div>

          <div className="grid gap-6 lg:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <ArrowUpRight className="h-5 w-5 text-green-500" />
                  تفاصيل الإيرادات
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">البند</TableHead>
                      <TableHead className="text-left">المبلغ قبل VAT</TableHead>
                      <TableHead className="text-left">VAT</TableHead>
                      <TableHead className="text-left">الإجمالي</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    <TableRow data-testid="row-in-salon-services">
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Receipt className="h-4 w-4 text-primary" />
                          {report.inSalonServicesRevenue.label}
                        </div>
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.inSalonServicesRevenue.subtotal)}
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.inSalonServicesRevenue.vat)}
                      </TableCell>
                      <TableCell className="font-semibold text-green-600">
                        {formatCurrency(report.inSalonServicesRevenue.total)}
                      </TableCell>
                    </TableRow>

                    <TableRow data-testid="row-home-services">
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Home className="h-4 w-4 text-primary" />
                          {report.homeServicesRevenue.label}
                        </div>
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.homeServicesRevenue.subtotal)}
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.homeServicesRevenue.vat)}
                      </TableCell>
                      <TableCell className="font-semibold text-green-600">
                        {formatCurrency(report.homeServicesRevenue.total)}
                      </TableCell>
                    </TableRow>

                    <TableRow data-testid="row-product-sales">
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Package className="h-4 w-4 text-primary" />
                          {report.productSalesRevenue.label}
                        </div>
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.productSalesRevenue.subtotal)}
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.productSalesRevenue.vat)}
                      </TableCell>
                      <TableCell className="font-semibold text-green-600">
                        {formatCurrency(report.productSalesRevenue.total)}
                      </TableCell>
                    </TableRow>
                  </TableBody>
                </Table>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <ArrowDownRight className="h-5 w-5 text-red-500" />
                  المرتجعات والخصومات
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">البند</TableHead>
                      <TableHead className="text-left">المبلغ قبل VAT</TableHead>
                      <TableHead className="text-left">VAT</TableHead>
                      <TableHead className="text-left">الإجمالي</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    <TableRow data-testid="row-returned-services">
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Undo2 className="h-4 w-4 text-red-500" />
                          {report.returnedServicesAmount.label}
                        </div>
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.returnedServicesAmount.subtotal)}
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.returnedServicesAmount.vat)}
                      </TableCell>
                      <TableCell className="font-semibold text-red-600">
                        -{formatCurrency(report.returnedServicesAmount.total)}
                      </TableCell>
                    </TableRow>

                    <TableRow data-testid="row-returned-products">
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Undo2 className="h-4 w-4 text-red-500" />
                          {report.returnedProductsAmount.label}
                        </div>
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.returnedProductsAmount.subtotal)}
                      </TableCell>
                      <TableCell>
                        {formatCurrency(report.returnedProductsAmount.vat)}
                      </TableCell>
                      <TableCell className="font-semibold text-red-600">
                        -{formatCurrency(report.returnedProductsAmount.total)}
                      </TableCell>
                    </TableRow>

                    {report.loyaltyPointsRedemption > 0 && (
                      <TableRow data-testid="row-loyalty-redemption">
                        <TableCell className="font-medium">
                          <div className="flex items-center gap-2">
                            <Undo2 className="h-4 w-4 text-orange-500" />
                            استرداد نقاط الولاء
                          </div>
                        </TableCell>
                        <TableCell colSpan={2}></TableCell>
                        <TableCell className="font-semibold text-orange-600">
                          -{formatCurrency(report.loyaltyPointsRedemption)}
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CreditCard className="h-5 w-5" />
                تفصيل الإيرادات حسب طرق الدفع
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">طريقة الدفع</TableHead>
                    <TableHead className="text-left">الإيرادات</TableHead>
                    <TableHead className="text-left">المرتجعات</TableHead>
                    <TableHead className="text-left">الصافي</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {report.paymentMethodBreakdown.map((pm) => (
                    <TableRow
                      key={pm.paymentMethodId}
                      data-testid={`row-payment-method-${pm.paymentMethodId}`}
                    >
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          {pm.paymentMethodName.includes("نقد") ||
                          pm.paymentMethodName.includes("كاش") ? (
                            <Banknote className="h-4 w-4 text-green-600" />
                          ) : pm.paymentMethodName.includes("شبكة") ||
                            pm.paymentMethodName.includes("بطاقة") ? (
                            <CreditCard className="h-4 w-4 text-blue-600" />
                          ) : (
                            <Wallet className="h-4 w-4 text-purple-600" />
                          )}
                          {pm.paymentMethodName}
                        </div>
                      </TableCell>
                      <TableCell className="text-green-600">
                        {formatCurrency(pm.revenue)}
                      </TableCell>
                      <TableCell className="text-red-600">
                        {pm.refunds > 0
                          ? `-${formatCurrency(pm.refunds)}`
                          : formatCurrency(0)}
                      </TableCell>
                      <TableCell
                        className={`font-semibold ${pm.net >= 0 ? "text-green-600" : "text-red-600"}`}
                      >
                        {formatCurrency(pm.net)}
                      </TableCell>
                    </TableRow>
                  ))}

                  {report.paymentMethodBreakdown.length === 0 && (
                    <TableRow>
                      <TableCell
                        colSpan={4}
                        className="text-center text-muted-foreground py-8"
                      >
                        لا توجد مدفوعات في هذا اليوم
                      </TableCell>
                    </TableRow>
                  )}

                  <TableRow className="bg-muted/50 font-bold">
                    <TableCell>الإجمالي</TableCell>
                    <TableCell className="text-green-600">
                      {formatCurrency(report.totalPaymentsReceived)}
                    </TableCell>
                    <TableCell className="text-red-600">
                      {report.totalRefunds > 0
                        ? `-${formatCurrency(report.totalRefunds)}`
                        : formatCurrency(0)}
                    </TableCell>
                    <TableCell
                      className={
                        report.netPayments >= 0
                          ? "text-green-600"
                          : "text-red-600"
                      }
                    >
                      {formatCurrency(report.netPayments)}
                    </TableCell>
                  </TableRow>
                </TableBody>
              </Table>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Banknote className="h-5 w-5 text-green-600" />
                ملخص النقدي
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4 md:grid-cols-3">
                <div className="p-4 rounded-lg bg-muted/50">
                  <div className="text-sm text-muted-foreground mb-1">
                    إيرادات اليوم (نقداً)
                  </div>
                  <div
                    className="text-xl font-bold text-green-600"
                    data-testid="text-cash-today"
                  >
                    {formatCurrency(report.cashSummary.todaysRevenue)}
                  </div>
                </div>
                <div className="p-4 rounded-lg bg-muted/50">
                  <div className="text-sm text-muted-foreground mb-1">
                    الرصيد السابق
                  </div>
                  <div
                    className="text-xl font-bold"
                    data-testid="text-previous-balance"
                  >
                    {formatCurrency(report.cashSummary.previousBalance)}
                  </div>
                </div>
                <div className="p-4 rounded-lg bg-primary/10">
                  <div className="text-sm text-muted-foreground mb-1">
                    صافي النقدي
                  </div>
                  <div
                    className="text-xl font-bold text-primary"
                    data-testid="text-net-cash"
                  >
                    {formatCurrency(report.cashSummary.netCash)}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </>
      ) : (
        <Card>
          <CardContent className="flex items-center justify-center py-12">
            <div className="text-center text-muted-foreground">
              <Calendar className="h-12 w-12 mx-auto mb-2 opacity-50" />
              <p>اختر تاريخاً لعرض التقرير</p>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}

function DailyReportSkeleton() {
  return (
    <>
      <Card>
        <CardHeader className="pb-2">
          <Skeleton className="h-6 w-48" />
        </CardHeader>
      </Card>

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {[1, 2, 3, 4].map((i) => (
          <Card key={i}>
            <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
              <Skeleton className="h-4 w-24" />
              <Skeleton className="h-4 w-4" />
            </CardHeader>
            <CardContent>
              <Skeleton className="h-8 w-32 mb-1" />
              <Skeleton className="h-3 w-20" />
            </CardContent>
          </Card>
        ))}
      </div>

      <div className="grid gap-6 lg:grid-cols-2">
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <Skeleton className="h-40 w-full" />
          </CardContent>
        </Card>
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <Skeleton className="h-40 w-full" />
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-48" />
        </CardHeader>
        <CardContent>
          <Skeleton className="h-48 w-full" />
        </CardContent>
      </Card>
    </>
  );
}
