import { useQuery } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Link } from "wouter";
import {
  ShoppingCart,
  Users,
  Calendar,
  XCircle,
  TrendingUp,
  Package,
  AlertTriangle,
  RefreshCw,
  Filter,
  UserPlus,
  DollarSign,
  Percent,
  Crown,
} from "lucide-react";
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
  BarChart,
  Bar,
  Legend,
} from "recharts";
import type { SalonDashboardSummary } from "@shared/schema";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";

const statusColors: Record<string, string> = {
  completed: "#22c55e",
  inProgress: "#3b82f6",
  cancelled: "#ef4444",
  scheduled: "#f59e0b",
};

const statusLabelsAr: Record<string, string> = {
  completed: "مكتمل",
  inProgress: "قيد التنفيذ",
  cancelled: "ملغي",
  scheduled: "مجدول",
};

function StatCard({
  title,
  value,
  subtitle,
  icon: Icon,
  color = "primary",
  loading,
}: {
  title: string;
  value: string | number;
  subtitle?: string;
  icon: typeof ShoppingCart;
  color?: "primary" | "green" | "red" | "amber" | "blue";
  loading?: boolean;
}) {
  const colorClasses = {
    primary: "bg-primary/10 text-primary",
    green: "bg-green-500/10 text-green-600",
    red: "bg-red-500/10 text-red-600",
    amber: "bg-amber-500/10 text-amber-600",
    blue: "bg-blue-500/10 text-blue-600",
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-start justify-between gap-4">
            <div className="space-y-2">
              <Skeleton className="h-4 w-24" />
              <Skeleton className="h-8 w-20" />
              <Skeleton className="h-3 w-32" />
            </div>
            <Skeleton className="h-12 w-12 rounded-lg" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className="hover-elevate">
      <CardContent className="p-6">
        <div className="flex items-start justify-between gap-4">
          <div className="space-y-1">
            <p className="text-sm text-muted-foreground">{title}</p>
            <p className="text-2xl font-bold tabular-nums" data-testid={`stat-${title.replace(/\s/g, "-")}`}>
              {value}
            </p>
            {subtitle && (
              <p className="text-xs text-muted-foreground">{subtitle}</p>
            )}
          </div>
          <div className={`flex h-12 w-12 items-center justify-center rounded-lg ${colorClasses[color]}`}>
            <Icon className="h-6 w-6" />
          </div>
        </div>
      </CardContent>
    </Card>
  );
}

function SalesChart({ data, loading }: { data?: { date: string; amount: number }[]; loading: boolean }) {
  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">المبيعات خلال الفترة</CardTitle>
        </CardHeader>
        <CardContent>
          <Skeleton className="h-64 w-full" />
        </CardContent>
      </Card>
    );
  }

  const chartData = data?.map(d => ({
    date: new Date(d.date).toLocaleDateString("ar-SA", { day: "numeric", month: "short" }),
    amount: d.amount,
  })) || [];

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-lg flex items-center gap-2">
          <TrendingUp className="h-5 w-5" />
          المبيعات خلال الفترة
        </CardTitle>
      </CardHeader>
      <CardContent>
        {chartData.length === 0 ? (
          <div className="flex items-center justify-center h-64 text-muted-foreground">
            لا توجد بيانات مبيعات للفترة المحددة
          </div>
        ) : (
          <ResponsiveContainer width="100%" height={280}>
            <LineChart data={chartData} margin={{ top: 5, right: 20, left: 20, bottom: 5 }}>
              <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
              <XAxis 
                dataKey="date" 
                tick={{ fill: "hsl(var(--muted-foreground))", fontSize: 12 }}
                axisLine={{ stroke: "hsl(var(--border))" }}
              />
              <YAxis 
                tick={{ fill: "hsl(var(--muted-foreground))", fontSize: 12 }}
                axisLine={{ stroke: "hsl(var(--border))" }}
                tickFormatter={(value) => formatNumber(value)}
              />
              <Tooltip 
                formatter={(value: number) => [formatCurrency(value), "المبيعات"]}
                contentStyle={{ 
                  backgroundColor: "hsl(var(--card))",
                  border: "1px solid hsl(var(--border))",
                  borderRadius: "8px",
                  direction: "rtl",
                }}
              />
              <Line 
                type="monotone" 
                dataKey="amount" 
                stroke="hsl(var(--primary))" 
                strokeWidth={2}
                dot={{ fill: "hsl(var(--primary))", r: 4 }}
              />
            </LineChart>
          </ResponsiveContainer>
        )}
      </CardContent>
    </Card>
  );
}

function OrdersStatusChart({ data, loading }: { data?: SalonDashboardSummary["ordersStatusDistribution"]; loading: boolean }) {
  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">توزيع حالات الطلبات</CardTitle>
        </CardHeader>
        <CardContent>
          <Skeleton className="h-64 w-full" />
        </CardContent>
      </Card>
    );
  }

  const chartData = data ? [
    { name: statusLabelsAr.completed, value: data.completed, color: statusColors.completed },
    { name: statusLabelsAr.inProgress, value: data.inProgress, color: statusColors.inProgress },
    { name: statusLabelsAr.cancelled, value: data.cancelled, color: statusColors.cancelled },
    { name: statusLabelsAr.scheduled, value: data.scheduled, color: statusColors.scheduled },
  ].filter(d => d.value > 0) : [];

  const total = chartData.reduce((sum, d) => sum + d.value, 0);

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-lg">توزيع حالات الطلبات</CardTitle>
      </CardHeader>
      <CardContent>
        {total === 0 ? (
          <div className="flex items-center justify-center h-64 text-muted-foreground">
            لا توجد طلبات للفترة المحددة
          </div>
        ) : (
          <div className="flex items-center gap-4">
            <ResponsiveContainer width="100%" height={200}>
              <PieChart>
                <Pie
                  data={chartData}
                  cx="50%"
                  cy="50%"
                  innerRadius={50}
                  outerRadius={80}
                  paddingAngle={2}
                  dataKey="value"
                >
                  {chartData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={entry.color} />
                  ))}
                </Pie>
                <Tooltip 
                  formatter={(value: number) => [value, "عدد"]}
                  contentStyle={{ 
                    backgroundColor: "hsl(var(--card))",
                    border: "1px solid hsl(var(--border))",
                    borderRadius: "8px",
                    direction: "rtl",
                  }}
                />
              </PieChart>
            </ResponsiveContainer>
            <div className="space-y-2">
              {chartData.map((entry) => (
                <div key={entry.name} className="flex items-center gap-2">
                  <div className="w-3 h-3 rounded-full" style={{ backgroundColor: entry.color }} />
                  <span className="text-sm">{entry.name}: {entry.value}</span>
                </div>
              ))}
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  );
}

function TopServicesChart({ data, loading }: { data?: SalonDashboardSummary["topServices"]; loading: boolean }) {
  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">أفضل الخدمات مبيعاً</CardTitle>
        </CardHeader>
        <CardContent>
          <Skeleton className="h-64 w-full" />
        </CardContent>
      </Card>
    );
  }

  const chartData = data?.map(s => ({
    name: s.nameAr.length > 15 ? s.nameAr.substring(0, 15) + "..." : s.nameAr,
    fullName: s.nameAr,
    amount: s.totalAmount,
    count: s.count,
  })) || [];

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-lg flex items-center gap-2">
          <Crown className="h-5 w-5 text-amber-500" />
          أفضل 5 خدمات مبيعاً
        </CardTitle>
      </CardHeader>
      <CardContent>
        {chartData.length === 0 ? (
          <div className="flex items-center justify-center h-64 text-muted-foreground">
            لا توجد بيانات خدمات
          </div>
        ) : (
          <ResponsiveContainer width="100%" height={280}>
            <BarChart data={chartData} layout="vertical" margin={{ top: 5, right: 30, left: 100, bottom: 5 }}>
              <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
              <XAxis 
                type="number"
                tick={{ fill: "hsl(var(--muted-foreground))", fontSize: 12 }}
                tickFormatter={(value) => formatNumber(value)}
              />
              <YAxis 
                dataKey="name" 
                type="category"
                tick={{ fill: "hsl(var(--muted-foreground))", fontSize: 12 }}
                width={100}
              />
              <Tooltip 
                formatter={(value: number, name: string) => [formatCurrency(value), "المبيعات"]}
                contentStyle={{ 
                  backgroundColor: "hsl(var(--card))",
                  border: "1px solid hsl(var(--border))",
                  borderRadius: "8px",
                  direction: "rtl",
                }}
              />
              <Bar dataKey="amount" fill="hsl(var(--primary))" radius={[0, 4, 4, 0]} />
            </BarChart>
          </ResponsiveContainer>
        )}
      </CardContent>
    </Card>
  );
}

function LatestOrdersTable({ data, loading }: { data?: SalonDashboardSummary["latestOrders"]; loading: boolean }) {
  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">أحدث الطلبات</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-2">
            {[1, 2, 3, 4, 5].map((i) => (
              <Skeleton key={i} className="h-12 w-full" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  const statusColors: Record<string, string> = {
    "جديد": "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200",
    "قيد التنفيذ": "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-200",
    "مكتمل": "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200",
    "ملغي": "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
    "مجدول": "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200",
    "مؤجل": "bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200",
    "مرتجع": "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200",
  };

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between gap-2">
        <CardTitle className="text-lg">أحدث 10 طلبات</CardTitle>
        <Button variant="ghost" size="sm" asChild>
          <Link href="/orders" data-testid="view-all-orders">عرض الكل</Link>
        </Button>
      </CardHeader>
      <CardContent>
        {!data || data.length === 0 ? (
          <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
            <ShoppingCart className="h-12 w-12 mb-4 opacity-50" />
            <p>لا توجد طلبات</p>
          </div>
        ) : (
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="text-right">رقم الطلب</TableHead>
                <TableHead className="text-right">التاريخ</TableHead>
                <TableHead className="text-right">العميل</TableHead>
                <TableHead className="text-right">الحالة</TableHead>
                <TableHead className="text-right">الإجمالي</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {data.map((order) => (
                <TableRow key={order.id} data-testid={`order-row-${order.id}`}>
                  <TableCell>
                    <Link href={`/orders/${order.id}`} className="font-medium text-primary hover:underline">
                      {order.orderNumber}
                    </Link>
                  </TableCell>
                  <TableCell className="text-sm text-muted-foreground">{formatDate(order.date)}</TableCell>
                  <TableCell>{order.customer}</TableCell>
                  <TableCell>
                    <Badge className={statusColors[order.status] || ""}>{order.status}</Badge>
                  </TableCell>
                  <TableCell className="font-semibold tabular-nums">{formatCurrency(order.total)}</TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        )}
      </CardContent>
    </Card>
  );
}

function TopEmployeesTable({ data, loading }: { data?: SalonDashboardSummary["topEmployees"]; loading: boolean }) {
  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">أفضل الموظفات</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-2">
            {[1, 2, 3].map((i) => (
              <Skeleton key={i} className="h-12 w-full" />
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between gap-2">
        <CardTitle className="text-lg flex items-center gap-2">
          <Users className="h-5 w-5" />
          أفضل 5 موظفات
        </CardTitle>
        <Button variant="ghost" size="sm" asChild>
          <Link href="/employees" data-testid="view-all-employees">عرض الكل</Link>
        </Button>
      </CardHeader>
      <CardContent>
        {!data || data.length === 0 ? (
          <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
            <Users className="h-12 w-12 mb-4 opacity-50" />
            <p>لا توجد بيانات موظفات</p>
          </div>
        ) : (
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="text-right">الموظفة</TableHead>
                <TableHead className="text-right">عدد الخدمات</TableHead>
                <TableHead className="text-right">إجمالي المبيعات</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {data.map((emp, index) => (
                <TableRow key={emp.employeeId} data-testid={`employee-row-${emp.employeeId}`}>
                  <TableCell className="flex items-center gap-2">
                    {index === 0 && <Crown className="h-4 w-4 text-amber-500" />}
                    <Link href={`/employees/${emp.employeeId}`} className="font-medium text-primary hover:underline">
                      {emp.name}
                    </Link>
                  </TableCell>
                  <TableCell className="tabular-nums">{emp.servicesCount}</TableCell>
                  <TableCell className="font-semibold tabular-nums">{formatCurrency(emp.totalSales)}</TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        )}
      </CardContent>
    </Card>
  );
}

function InventoryAlerts({ data, loading }: { data?: SalonDashboardSummary["inventory"]; loading: boolean }) {
  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">تنبيهات المخزون</CardTitle>
        </CardHeader>
        <CardContent>
          <Skeleton className="h-24 w-full" />
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-lg flex items-center gap-2">
          <AlertTriangle className="h-5 w-5 text-amber-500" />
          تنبيهات المخزون
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="grid grid-cols-3 gap-4 text-center">
          <div className="space-y-1">
            <div className="text-2xl font-bold text-red-600" data-testid="low-stock-count">
              {data?.lowStockCount || 0}
            </div>
            <p className="text-xs text-muted-foreground">مخزون منخفض</p>
          </div>
          <div className="space-y-1">
            <div className="text-2xl font-bold text-amber-600" data-testid="near-expiry-count">
              {data?.nearExpiryCount || 0}
            </div>
            <p className="text-xs text-muted-foreground">قارب على الانتهاء</p>
          </div>
          <div className="space-y-1">
            <div className="text-2xl font-bold text-primary" data-testid="inventory-value">
              {formatCurrency(data?.estimatedValue || 0)}
            </div>
            <p className="text-xs text-muted-foreground">قيمة المخزون</p>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}

export default function Dashboard() {
  const [filters, setFilters] = useState({
    from: new Date(new Date().getFullYear(), new Date().getMonth(), 1).toISOString().split('T')[0],
    to: new Date().toISOString().split('T')[0],
    source: "all" as "all" | "pos" | "app",
  });
  
  const [appliedFilters, setAppliedFilters] = useState(filters);

  const queryString = new URLSearchParams({
    from: appliedFilters.from,
    to: appliedFilters.to,
    source: appliedFilters.source,
  }).toString();

  const { data, isLoading, refetch } = useQuery<SalonDashboardSummary>({
    queryKey: ["/api/dashboard/summary", queryString],
    queryFn: async () => {
      const response = await fetch(`/api/dashboard/summary?${queryString}`);
      if (!response.ok) throw new Error("Failed to fetch dashboard data");
      return response.json();
    },
  });

  const handleApplyFilters = () => {
    setAppliedFilters(filters);
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">لوحة التحكم</h1>
          <p className="text-muted-foreground">نظرة عامة على مبيعات وحجوزات الصالون</p>
        </div>
        <Button onClick={() => refetch()} variant="outline" size="icon" data-testid="button-refresh">
          <RefreshCw className="h-4 w-4" />
        </Button>
      </div>

      <Card>
        <CardHeader className="pb-3">
          <CardTitle className="text-base flex items-center gap-2">
            <Filter className="h-4 w-4" />
            عوامل التصفية
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap items-end gap-4">
            <div className="space-y-2">
              <Label htmlFor="fromDate">من تاريخ</Label>
              <Input
                id="fromDate"
                type="date"
                value={filters.from}
                onChange={(e) => setFilters({ ...filters, from: e.target.value })}
                className="w-40"
                data-testid="input-from-date"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="toDate">إلى تاريخ</Label>
              <Input
                id="toDate"
                type="date"
                value={filters.to}
                onChange={(e) => setFilters({ ...filters, to: e.target.value })}
                className="w-40"
                data-testid="input-to-date"
              />
            </div>
            <div className="space-y-2">
              <Label>مصدر الطلب</Label>
              <Select 
                value={filters.source} 
                onValueChange={(value) => setFilters({ ...filters, source: value as "all" | "pos" | "app" })}
              >
                <SelectTrigger className="w-40" data-testid="select-source">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">كل المصادر</SelectItem>
                  <SelectItem value="pos">طلبات الصالون</SelectItem>
                  <SelectItem value="app">طلبات التطبيق</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <Button onClick={handleApplyFilters} data-testid="button-apply-filters">
              تطبيق
            </Button>
          </div>
        </CardContent>
      </Card>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <StatCard
          title="مبيعات اليوم"
          value={formatCurrency(data?.salesToday.totalAmount || 0)}
          subtitle={`${data?.salesToday.ordersCount || 0} طلب - متوسط ${formatCurrency(data?.salesToday.avgOrderValue || 0)}`}
          icon={DollarSign}
          color="green"
          loading={isLoading}
        />
        <StatCard
          title="حجوزات اليوم"
          value={data?.bookingsToday.total || 0}
          subtitle={`${data?.bookingsToday.upcoming || 0} قادمة - ${data?.bookingsToday.cancelled || 0} ملغاة`}
          icon={Calendar}
          color="blue"
          loading={isLoading}
        />
        <StatCard
          title="الطلبات الملغاة"
          value={data?.cancelledOrders.count || 0}
          subtitle={`مبلغ مفقود: ${formatCurrency(data?.cancelledOrders.lostAmount || 0)}`}
          icon={XCircle}
          color="red"
          loading={isLoading}
        />
        <StatCard
          title="العملاء الجدد (30 يوم)"
          value={data?.customers.newLast30Days || 0}
          subtitle={`إجمالي العملاء: ${data?.customers.totalCustomers || 0}`}
          icon={UserPlus}
          color="primary"
          loading={isLoading}
        />
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <StatCard
          title="إجمالي المبيعات"
          value={formatCurrency(data?.salesRevenue.totalSales || 0)}
          subtitle="خلال الفترة المحددة"
          icon={ShoppingCart}
          color="green"
          loading={isLoading}
        />
        <StatCard
          title="ضريبة القيمة المضافة"
          value={formatCurrency(data?.salesRevenue.vatCollected || 0)}
          subtitle="إجمالي الضريبة المحصلة"
          icon={Percent}
          color="amber"
          loading={isLoading}
        />
        <StatCard
          title="إجمالي الخصومات"
          value={formatCurrency(data?.salesRevenue.totalDiscounts || 0)}
          subtitle="خصومات الكوبونات"
          icon={TrendingUp}
          color="blue"
          loading={isLoading}
        />
      </div>

      <InventoryAlerts data={data?.inventory} loading={isLoading} />

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <SalesChart data={data?.salesOverTime} loading={isLoading} />
        <OrdersStatusChart data={data?.ordersStatusDistribution} loading={isLoading} />
      </div>

      <TopServicesChart data={data?.topServices} loading={isLoading} />

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <LatestOrdersTable data={data?.latestOrders} loading={isLoading} />
        <TopEmployeesTable data={data?.topEmployees} loading={isLoading} />
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Users className="h-5 w-5" />
            أكثر العملاء إنفاقاً
          </CardTitle>
        </CardHeader>
        <CardContent>
          {!data?.topCustomers || data.topCustomers.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
              <Users className="h-12 w-12 mb-4 opacity-50" />
              <p>لا توجد بيانات عملاء</p>
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">العميل</TableHead>
                  <TableHead className="text-right">عدد الطلبات</TableHead>
                  <TableHead className="text-right">إجمالي الإنفاق</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {data.topCustomers.map((customer, index) => (
                  <TableRow key={customer.customerId} data-testid={`customer-row-${customer.customerId}`}>
                    <TableCell className="flex items-center gap-2">
                      {index === 0 && <Crown className="h-4 w-4 text-amber-500" />}
                      <Link href={`/customers/${customer.customerId}`} className="font-medium text-primary hover:underline">
                        {customer.name}
                      </Link>
                    </TableCell>
                    <TableCell className="tabular-nums">{customer.ordersCount}</TableCell>
                    <TableCell className="font-semibold tabular-nums">{formatCurrency(customer.totalSpent)}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
