import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { 
  Car, 
  Plus, 
  Pencil, 
  Trash2, 
  CheckCircle2, 
  XCircle, 
  Search, 
  Phone, 
  Mail, 
  Calendar, 
  MapPin,
  Route,
  Settings,
  Save,
  User
} from "lucide-react";
import { format } from "date-fns";

interface Driver {
  id: number;
  nameAr: string | null;
  code: string | null;
  phone: string | null;
  fullNameAr: string | null;
  phoneCountryCode: string | null;
  phoneNumber: string | null;
  email: string | null;
  fingerprintEmployeeNo: string | null;
  profileImageUrl: string | null;
  jobTitleAr: string | null;
  cityNameAr: string | null;
  branchId: number | null;
  nationalityNameAr: string | null;
  nationalityId: number | null;
  birthDate: string | null;
  hireDate: string | null;
  iqamaNumber: string | null;
  iqamaExpiryDate: string | null;
  insuranceCompany: string | null;
  insuranceNumber: string | null;
  insuranceExpiryDate: string | null;
  insuranceCardExpiryDate: string | null;
  healthNumber: string | null;
  passportNumber: string | null;
  passportExpiryDate: string | null;
  licenseNumber: string | null;
  licenseExpiryDate: string | null;
  vehiclePlateNumber: string | null;
  vehicleType: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface Appointment {
  id: number;
  appointmentNumber: string;
  appointmentType: string;
  status: string;
  scheduledAt: string;
  locationLat: number | null;
  locationLng: number | null;
  addressText: string | null;
  clientId: number;
  notes: string | null;
}

interface Trip {
  id: number;
  tripNumber: string;
  driverId: number;
  tripType: string;
  appointmentId: number | null;
  startLocationLat: number | null;
  startLocationLng: number | null;
  startAddressText: string | null;
  endLocationLat: number | null;
  endLocationLng: number | null;
  endAddressText: string | null;
  status: string;
  notes: string | null;
  scheduledAt: string | null;
  startedAt: string | null;
  completedAt: string | null;
  driver?: Driver;
  appointment?: Appointment;
  createdAt: string;
  updatedAt: string;
}

interface DriverFormData {
  fullNameAr: string;
  phoneCountryCode: string;
  phoneNumber: string;
  email: string;
  jobTitleAr: string;
  nationalityNameAr: string;
  cityNameAr: string;
  birthDate: string;
  hireDate: string;
  iqamaNumber: string;
  iqamaExpiryDate: string;
  passportNumber: string;
  passportExpiryDate: string;
  licenseNumber: string;
  licenseExpiryDate: string;
  vehiclePlateNumber: string;
  vehicleType: string;
  isActive: boolean;
}

interface TripFormData {
  driverId: number | null;
  tripType: string;
  appointmentId: number | null;
  startAddressText: string;
  endAddressText: string;
  startLocationLat: number | null;
  startLocationLng: number | null;
  endLocationLat: number | null;
  endLocationLng: number | null;
  notes: string;
  scheduledAt: string;
  status: string;
}

const tripStatusOptions = [
  { value: "pending", label: "قيد الانتظار" },
  { value: "in_progress", label: "جاري التنفيذ" },
  { value: "completed", label: "مكتمل" },
  { value: "cancelled", label: "ملغي" },
];

const tripTypeOptions = [
  { value: "appointment", label: "حجز" },
  { value: "general", label: "عام" },
];

const vehicleTypeOptions = [
  { value: "sedan", label: "سيدان" },
  { value: "suv", label: "SUV" },
  { value: "van", label: "فان" },
  { value: "motorcycle", label: "دراجة نارية" },
  { value: "other", label: "أخرى" },
];

export default function DriversPage() {
  const [activeTab, setActiveTab] = useState("drivers");
  const [isDriverDialogOpen, setIsDriverDialogOpen] = useState(false);
  const [isTripDialogOpen, setIsTripDialogOpen] = useState(false);
  const [editingDriver, setEditingDriver] = useState<Driver | null>(null);
  const [editingTrip, setEditingTrip] = useState<Trip | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [deliveryFee, setDeliveryFee] = useState("");
  const [isSavingFee, setIsSavingFee] = useState(false);
  const { toast } = useToast();

  const [driverFormData, setDriverFormData] = useState<DriverFormData>({
    fullNameAr: "",
    phoneCountryCode: "966",
    phoneNumber: "",
    email: "",
    jobTitleAr: "سائق",
    nationalityNameAr: "",
    cityNameAr: "",
    birthDate: "",
    hireDate: "",
    iqamaNumber: "",
    iqamaExpiryDate: "",
    passportNumber: "",
    passportExpiryDate: "",
    licenseNumber: "",
    licenseExpiryDate: "",
    vehiclePlateNumber: "",
    vehicleType: "",
    isActive: true,
  });

  const [tripFormData, setTripFormData] = useState<TripFormData>({
    driverId: null,
    tripType: "general",
    appointmentId: null,
    startAddressText: "",
    endAddressText: "",
    startLocationLat: null,
    startLocationLng: null,
    endLocationLat: null,
    endLocationLng: null,
    notes: "",
    scheduledAt: "",
    status: "pending",
  });

  const { data: drivers, isLoading: isLoadingDrivers } = useQuery<Driver[]>({
    queryKey: ["/api/drivers"],
  });

  const { data: trips, isLoading: isLoadingTrips } = useQuery<Trip[]>({
    queryKey: ["/api/trips"],
  });

  const { data: appointments } = useQuery<Appointment[]>({
    queryKey: ["/api/appointments"],
  });

  const { data: deliveryFeeSetting } = useQuery<{ key: string; value: string }>({
    queryKey: ["/api/settings", "home_service_delivery_fee"],
    queryFn: async () => {
      const response = await fetch("/api/settings/home_service_delivery_fee");
      if (!response.ok) throw new Error("Failed to fetch setting");
      return response.json();
    },
  });

  useState(() => {
    if (deliveryFeeSetting?.value) {
      setDeliveryFee(deliveryFeeSetting.value);
    }
  });

  const createDriverMutation = useMutation({
    mutationFn: (data: DriverFormData) =>
      apiRequest("POST", "/api/drivers", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/drivers"] });
      toast({ title: "تم إنشاء السائق بنجاح" });
      resetDriverForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء السائق",
        variant: "destructive",
      });
    },
  });

  const updateDriverMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<DriverFormData> }) =>
      apiRequest("PUT", `/api/drivers/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/drivers"] });
      toast({ title: "تم تحديث السائق بنجاح" });
      resetDriverForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث السائق",
        variant: "destructive",
      });
    },
  });

  const deleteDriverMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/drivers/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/drivers"] });
      toast({ title: "تم حذف السائق بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف السائق",
        variant: "destructive",
      });
    },
  });

  const createTripMutation = useMutation({
    mutationFn: (data: TripFormData) =>
      apiRequest("POST", "/api/trips", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/trips"] });
      toast({ title: "تم إنشاء الرحلة بنجاح" });
      resetTripForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الرحلة",
        variant: "destructive",
      });
    },
  });

  const updateTripMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<TripFormData> }) =>
      apiRequest("PUT", `/api/trips/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/trips"] });
      toast({ title: "تم تحديث الرحلة بنجاح" });
      resetTripForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الرحلة",
        variant: "destructive",
      });
    },
  });

  const deleteTripMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/trips/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/trips"] });
      toast({ title: "تم حذف الرحلة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الرحلة",
        variant: "destructive",
      });
    },
  });

  const resetDriverForm = () => {
    setDriverFormData({
      fullNameAr: "",
      phoneCountryCode: "966",
      phoneNumber: "",
      email: "",
      jobTitleAr: "سائق",
      nationalityNameAr: "",
      cityNameAr: "",
      birthDate: "",
      hireDate: "",
      iqamaNumber: "",
      iqamaExpiryDate: "",
      passportNumber: "",
      passportExpiryDate: "",
      licenseNumber: "",
      licenseExpiryDate: "",
      vehiclePlateNumber: "",
      vehicleType: "",
      isActive: true,
    });
    setEditingDriver(null);
    setIsDriverDialogOpen(false);
  };

  const resetTripForm = () => {
    setTripFormData({
      driverId: null,
      tripType: "general",
      appointmentId: null,
      startAddressText: "",
      endAddressText: "",
      startLocationLat: null,
      startLocationLng: null,
      endLocationLat: null,
      endLocationLng: null,
      notes: "",
      scheduledAt: "",
      status: "pending",
    });
    setEditingTrip(null);
    setIsTripDialogOpen(false);
  };

  const handleOpenDriverDialog = (driver?: Driver) => {
    if (driver) {
      setEditingDriver(driver);
      setDriverFormData({
        fullNameAr: driver.fullNameAr || driver.nameAr || "",
        phoneCountryCode: driver.phoneCountryCode || "966",
        phoneNumber: driver.phoneNumber || driver.phone || "",
        email: driver.email || "",
        jobTitleAr: driver.jobTitleAr || "سائق",
        nationalityNameAr: driver.nationalityNameAr || "",
        cityNameAr: driver.cityNameAr || "",
        birthDate: driver.birthDate || "",
        hireDate: driver.hireDate || "",
        iqamaNumber: driver.iqamaNumber || "",
        iqamaExpiryDate: driver.iqamaExpiryDate || "",
        passportNumber: driver.passportNumber || "",
        passportExpiryDate: driver.passportExpiryDate || "",
        licenseNumber: driver.licenseNumber || "",
        licenseExpiryDate: driver.licenseExpiryDate || "",
        vehiclePlateNumber: driver.vehiclePlateNumber || "",
        vehicleType: driver.vehicleType || "",
        isActive: driver.isActive,
      });
    } else {
      resetDriverForm();
    }
    setIsDriverDialogOpen(true);
  };

  const handleOpenTripDialog = (trip?: Trip) => {
    if (trip) {
      setEditingTrip(trip);
      setTripFormData({
        driverId: trip.driverId,
        tripType: trip.tripType,
        appointmentId: trip.appointmentId,
        startAddressText: trip.startAddressText || "",
        endAddressText: trip.endAddressText || "",
        startLocationLat: trip.startLocationLat,
        startLocationLng: trip.startLocationLng,
        endLocationLat: trip.endLocationLat,
        endLocationLng: trip.endLocationLng,
        notes: trip.notes || "",
        scheduledAt: trip.scheduledAt ? format(new Date(trip.scheduledAt), "yyyy-MM-dd'T'HH:mm") : "",
        status: trip.status,
      });
    } else {
      resetTripForm();
    }
    setIsTripDialogOpen(true);
  };

  const handleDriverSubmit = () => {
    if (!driverFormData.fullNameAr.trim()) {
      toast({ title: "اسم السائق مطلوب", variant: "destructive" });
      return;
    }

    if (editingDriver) {
      updateDriverMutation.mutate({ id: editingDriver.id, data: driverFormData });
    } else {
      createDriverMutation.mutate(driverFormData);
    }
  };

  const handleTripSubmit = () => {
    if (!tripFormData.driverId) {
      toast({ title: "يجب اختيار السائق", variant: "destructive" });
      return;
    }

    if (tripFormData.tripType === "appointment" && !tripFormData.appointmentId) {
      toast({ title: "يجب اختيار الحجز", variant: "destructive" });
      return;
    }

    if (tripFormData.tripType === "general" && (!tripFormData.startAddressText || !tripFormData.endAddressText)) {
      toast({ title: "يجب تحديد نقطة الانطلاق والوصول", variant: "destructive" });
      return;
    }

    if (editingTrip) {
      updateTripMutation.mutate({ id: editingTrip.id, data: tripFormData });
    } else {
      createTripMutation.mutate(tripFormData);
    }
  };

  const handleAppointmentSelect = (appointmentId: number) => {
    const appointment = appointments?.find(a => a.id === appointmentId);
    if (appointment) {
      setTripFormData(prev => ({
        ...prev,
        appointmentId,
        endAddressText: appointment.addressText || "",
        endLocationLat: appointment.locationLat,
        endLocationLng: appointment.locationLng,
      }));
    }
  };

  const handleSaveDeliveryFee = async () => {
    setIsSavingFee(true);
    try {
      await apiRequest("PUT", "/api/settings/home_service_delivery_fee", { value: deliveryFee });
      queryClient.invalidateQueries({ queryKey: ["/api/settings", "home_service_delivery_fee"] });
      toast({ title: "تم حفظ رسوم التوصيل بنجاح" });
    } catch (error: any) {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حفظ رسوم التوصيل",
        variant: "destructive",
      });
    } finally {
      setIsSavingFee(false);
    }
  };

  const filteredDrivers = drivers?.filter(driver => {
    if (!searchTerm) return true;
    const search = searchTerm.toLowerCase();
    return (
      (driver.fullNameAr?.toLowerCase() || "").includes(search) ||
      (driver.nameAr?.toLowerCase() || "").includes(search) ||
      (driver.phoneNumber || "").includes(search) ||
      (driver.vehiclePlateNumber?.toLowerCase() || "").includes(search)
    );
  }) || [];

  const activeDrivers = drivers?.filter(d => d.isActive) || [];
  const inactiveDrivers = drivers?.filter(d => !d.isActive) || [];

  const getDriverName = (driver: Driver) => driver.fullNameAr || driver.nameAr || "-";
  const getDriverPhone = (driver: Driver) => {
    if (driver.phoneNumber) return `+${driver.phoneCountryCode || "966"} ${driver.phoneNumber}`;
    if (driver.phone) return driver.phone;
    return "-";
  };

  const getTripStatusBadge = (status: string) => {
    const statusMap: Record<string, { label: string; className: string }> = {
      pending: { label: "قيد الانتظار", className: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200" },
      in_progress: { label: "جاري التنفيذ", className: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200" },
      completed: { label: "مكتمل", className: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" },
      cancelled: { label: "ملغي", className: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200" },
    };
    const s = statusMap[status] || { label: status, className: "" };
    return <Badge className={s.className}>{s.label}</Badge>;
  };

  const homeServiceAppointments = appointments?.filter(a => a.appointmentType === "home_service" && a.status !== "completed" && a.status !== "cancelled") || [];

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة السائقين</h1>
          <p className="text-muted-foreground">السائقين والرحلات وإعدادات التوصيل</p>
        </div>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
        <TabsList className="flex-wrap gap-1">
          <TabsTrigger value="drivers" data-testid="tab-drivers">
            <Car className="ml-2 h-4 w-4" />
            السائقون
          </TabsTrigger>
          <TabsTrigger value="trips" data-testid="tab-trips">
            <Route className="ml-2 h-4 w-4" />
            الرحلات
          </TabsTrigger>
          <TabsTrigger value="settings" data-testid="tab-delivery-settings">
            <Settings className="ml-2 h-4 w-4" />
            إعدادات رسوم التوصيل
          </TabsTrigger>
        </TabsList>

        <TabsContent value="drivers" className="space-y-4">
          <div className="flex items-center justify-between gap-4 flex-wrap">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 flex-1">
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-3">
                    <div className="p-2 rounded-lg bg-primary/10 text-primary">
                      <Car className="h-5 w-5" />
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">إجمالي السائقين</p>
                      <p className="text-xl font-bold" data-testid="text-total-drivers">
                        {drivers?.length || 0}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-3">
                    <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                      <CheckCircle2 className="h-5 w-5" />
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">سائقين نشطين</p>
                      <p className="text-xl font-bold" data-testid="text-active-drivers">
                        {activeDrivers.length}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center gap-3">
                    <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                      <XCircle className="h-5 w-5" />
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">سائقين غير نشطين</p>
                      <p className="text-xl font-bold" data-testid="text-inactive-drivers">
                        {inactiveDrivers.length}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
            <Button onClick={() => handleOpenDriverDialog()} data-testid="button-add-driver">
              <Plus className="h-4 w-4 ml-2" />
              إضافة سائق جديد
            </Button>
          </div>

          <Card>
            <CardHeader>
              <div className="flex items-center justify-between gap-4 flex-wrap">
                <CardTitle className="flex items-center gap-2">
                  <Car className="h-5 w-5" />
                  قائمة السائقين
                </CardTitle>
                <div className="relative">
                  <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                  <Input
                    placeholder="بحث بالاسم أو الهاتف أو رقم اللوحة..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pr-9 w-64"
                    data-testid="input-search-driver"
                  />
                </div>
              </div>
            </CardHeader>
            <CardContent>
              {isLoadingDrivers ? (
                <div className="space-y-2">
                  {[1, 2, 3].map((i) => (
                    <Skeleton key={i} className="h-12 w-full" />
                  ))}
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">الاسم</TableHead>
                      <TableHead className="text-right">الهاتف</TableHead>
                      <TableHead className="text-right">رقم الرخصة</TableHead>
                      <TableHead className="text-right">رقم اللوحة</TableHead>
                      <TableHead className="text-right">نوع المركبة</TableHead>
                      <TableHead className="text-right">الحالة</TableHead>
                      <TableHead className="text-left">الإجراءات</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredDrivers.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                          لا يوجد سائقين مسجلين
                        </TableCell>
                      </TableRow>
                    ) : (
                      filteredDrivers.map((driver) => (
                        <TableRow key={driver.id} data-testid={`row-driver-${driver.id}`}>
                          <TableCell className="font-medium">
                            <div className="flex items-center gap-2">
                              <div className="h-8 w-8 rounded-full bg-primary/10 flex items-center justify-center">
                                <User className="h-4 w-4 text-primary" />
                              </div>
                              <div>
                                <p>{getDriverName(driver)}</p>
                                {driver.jobTitleAr && (
                                  <p className="text-xs text-muted-foreground">{driver.jobTitleAr}</p>
                                )}
                              </div>
                            </div>
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-1 text-sm">
                              <Phone className="h-3 w-3" />
                              {getDriverPhone(driver)}
                            </div>
                          </TableCell>
                          <TableCell>{driver.licenseNumber || "-"}</TableCell>
                          <TableCell>{driver.vehiclePlateNumber || "-"}</TableCell>
                          <TableCell>
                            {driver.vehicleType ? vehicleTypeOptions.find(v => v.value === driver.vehicleType)?.label || driver.vehicleType : "-"}
                          </TableCell>
                          <TableCell>
                            {driver.isActive ? (
                              <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                                نشط
                              </Badge>
                            ) : (
                              <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                                غير نشط
                              </Badge>
                            )}
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-2">
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => handleOpenDriverDialog(driver)}
                                data-testid={`button-edit-driver-${driver.id}`}
                              >
                                <Pencil className="h-4 w-4" />
                              </Button>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => {
                                  if (confirm("هل أنت متأكد من حذف هذا السائق؟")) {
                                    deleteDriverMutation.mutate(driver.id);
                                  }
                                }}
                                data-testid={`button-delete-driver-${driver.id}`}
                              >
                                <Trash2 className="h-4 w-4 text-destructive" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="trips" className="space-y-4">
          <div className="flex items-center justify-end">
            <Button onClick={() => handleOpenTripDialog()} data-testid="button-add-trip">
              <Plus className="h-4 w-4 ml-2" />
              إنشاء رحلة جديدة
            </Button>
          </div>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Route className="h-5 w-5" />
                قائمة الرحلات
              </CardTitle>
            </CardHeader>
            <CardContent>
              {isLoadingTrips ? (
                <div className="space-y-2">
                  {[1, 2, 3].map((i) => (
                    <Skeleton key={i} className="h-12 w-full" />
                  ))}
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">رقم الرحلة</TableHead>
                      <TableHead className="text-right">السائق</TableHead>
                      <TableHead className="text-right">النوع</TableHead>
                      <TableHead className="text-right">من</TableHead>
                      <TableHead className="text-right">إلى</TableHead>
                      <TableHead className="text-right">الحالة</TableHead>
                      <TableHead className="text-left">الإجراءات</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {(trips?.length || 0) === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                          لا يوجد رحلات مسجلة
                        </TableCell>
                      </TableRow>
                    ) : (
                      trips?.map((trip) => (
                        <TableRow key={trip.id} data-testid={`row-trip-${trip.id}`}>
                          <TableCell className="font-medium">{trip.tripNumber}</TableCell>
                          <TableCell>
                            {trip.driver ? getDriverName(trip.driver) : "-"}
                          </TableCell>
                          <TableCell>
                            <Badge variant="outline">
                              {trip.tripType === "appointment" ? "حجز" : "عام"}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-1 text-sm max-w-32 truncate">
                              <MapPin className="h-3 w-3 flex-shrink-0" />
                              {trip.startAddressText || "-"}
                            </div>
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-1 text-sm max-w-32 truncate">
                              <MapPin className="h-3 w-3 flex-shrink-0" />
                              {trip.endAddressText || (trip.appointment?.addressText) || "-"}
                            </div>
                          </TableCell>
                          <TableCell>{getTripStatusBadge(trip.status)}</TableCell>
                          <TableCell>
                            <div className="flex items-center gap-2">
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => handleOpenTripDialog(trip)}
                                data-testid={`button-edit-trip-${trip.id}`}
                              >
                                <Pencil className="h-4 w-4" />
                              </Button>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => {
                                  if (confirm("هل أنت متأكد من حذف هذه الرحلة؟")) {
                                    deleteTripMutation.mutate(trip.id);
                                  }
                                }}
                                data-testid={`button-delete-trip-${trip.id}`}
                              >
                                <Trash2 className="h-4 w-4 text-destructive" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="settings" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Settings className="h-5 w-5" />
                إعدادات رسوم التوصيل
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="max-w-md">
                <Label htmlFor="delivery-fee">رسوم التوصيل للخدمة المنزلية (ريال)</Label>
                <p className="text-sm text-muted-foreground mb-2">
                  سيتم تطبيق هذه الرسوم تلقائياً على أي حجز جديد للخدمة المنزلية
                </p>
                <div className="flex items-center gap-2">
                  <Input
                    id="delivery-fee"
                    type="number"
                    step="0.01"
                    min="0"
                    value={deliveryFee}
                    onChange={(e) => setDeliveryFee(e.target.value)}
                    placeholder="0.00"
                    className="w-48"
                    data-testid="input-delivery-fee"
                  />
                  <Button 
                    onClick={handleSaveDeliveryFee} 
                    disabled={isSavingFee}
                    data-testid="button-save-delivery-fee"
                  >
                    <Save className="h-4 w-4 ml-2" />
                    {isSavingFee ? "جاري الحفظ..." : "حفظ"}
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      <Dialog open={isDriverDialogOpen} onOpenChange={setIsDriverDialogOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingDriver ? "تعديل بيانات السائق" : "إضافة سائق جديد"}
            </DialogTitle>
          </DialogHeader>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="fullNameAr">الاسم الكامل *</Label>
              <Input
                id="fullNameAr"
                value={driverFormData.fullNameAr}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, fullNameAr: e.target.value }))}
                placeholder="أدخل الاسم الكامل"
                data-testid="input-driver-name"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="phoneNumber">رقم الهاتف</Label>
              <div className="flex gap-2">
                <Select
                  value={driverFormData.phoneCountryCode}
                  onValueChange={(value) => setDriverFormData(prev => ({ ...prev, phoneCountryCode: value }))}
                >
                  <SelectTrigger className="w-24">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="966">+966</SelectItem>
                    <SelectItem value="971">+971</SelectItem>
                    <SelectItem value="965">+965</SelectItem>
                    <SelectItem value="973">+973</SelectItem>
                  </SelectContent>
                </Select>
                <Input
                  id="phoneNumber"
                  value={driverFormData.phoneNumber}
                  onChange={(e) => setDriverFormData(prev => ({ ...prev, phoneNumber: e.target.value }))}
                  placeholder="رقم الهاتف"
                  className="flex-1"
                  data-testid="input-driver-phone"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="email">البريد الإلكتروني</Label>
              <Input
                id="email"
                type="email"
                value={driverFormData.email}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, email: e.target.value }))}
                placeholder="example@email.com"
                data-testid="input-driver-email"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="nationalityNameAr">الجنسية</Label>
              <Input
                id="nationalityNameAr"
                value={driverFormData.nationalityNameAr}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, nationalityNameAr: e.target.value }))}
                placeholder="الجنسية"
                data-testid="input-driver-nationality"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="cityNameAr">المدينة</Label>
              <Input
                id="cityNameAr"
                value={driverFormData.cityNameAr}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, cityNameAr: e.target.value }))}
                placeholder="المدينة"
                data-testid="input-driver-city"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="birthDate">تاريخ الميلاد</Label>
              <Input
                id="birthDate"
                type="date"
                value={driverFormData.birthDate}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, birthDate: e.target.value }))}
                data-testid="input-driver-birthdate"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="hireDate">تاريخ التعيين</Label>
              <Input
                id="hireDate"
                type="date"
                value={driverFormData.hireDate}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, hireDate: e.target.value }))}
                data-testid="input-driver-hiredate"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="iqamaNumber">رقم الإقامة</Label>
              <Input
                id="iqamaNumber"
                value={driverFormData.iqamaNumber}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, iqamaNumber: e.target.value }))}
                placeholder="رقم الإقامة"
                data-testid="input-driver-iqama"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="iqamaExpiryDate">تاريخ انتهاء الإقامة</Label>
              <Input
                id="iqamaExpiryDate"
                type="date"
                value={driverFormData.iqamaExpiryDate}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, iqamaExpiryDate: e.target.value }))}
                data-testid="input-driver-iqama-expiry"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="passportNumber">رقم الجواز</Label>
              <Input
                id="passportNumber"
                value={driverFormData.passportNumber}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, passportNumber: e.target.value }))}
                placeholder="رقم الجواز"
                data-testid="input-driver-passport"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="passportExpiryDate">تاريخ انتهاء الجواز</Label>
              <Input
                id="passportExpiryDate"
                type="date"
                value={driverFormData.passportExpiryDate}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, passportExpiryDate: e.target.value }))}
                data-testid="input-driver-passport-expiry"
              />
            </div>

            <div className="col-span-2 border-t pt-4 mt-2">
              <h3 className="font-semibold mb-3">بيانات المركبة والرخصة</h3>
            </div>

            <div className="space-y-2">
              <Label htmlFor="licenseNumber">رقم رخصة القيادة</Label>
              <Input
                id="licenseNumber"
                value={driverFormData.licenseNumber}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, licenseNumber: e.target.value }))}
                placeholder="رقم الرخصة"
                data-testid="input-driver-license"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="licenseExpiryDate">تاريخ انتهاء الرخصة</Label>
              <Input
                id="licenseExpiryDate"
                type="date"
                value={driverFormData.licenseExpiryDate}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, licenseExpiryDate: e.target.value }))}
                data-testid="input-driver-license-expiry"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="vehiclePlateNumber">رقم لوحة المركبة</Label>
              <Input
                id="vehiclePlateNumber"
                value={driverFormData.vehiclePlateNumber}
                onChange={(e) => setDriverFormData(prev => ({ ...prev, vehiclePlateNumber: e.target.value }))}
                placeholder="رقم اللوحة"
                data-testid="input-driver-plate"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="vehicleType">نوع المركبة</Label>
              <Select
                value={driverFormData.vehicleType}
                onValueChange={(value) => setDriverFormData(prev => ({ ...prev, vehicleType: value }))}
              >
                <SelectTrigger data-testid="select-driver-vehicle-type">
                  <SelectValue placeholder="اختر نوع المركبة" />
                </SelectTrigger>
                <SelectContent>
                  {vehicleTypeOptions.map(option => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="col-span-2 flex items-center gap-2">
              <Switch
                id="isActive"
                checked={driverFormData.isActive}
                onCheckedChange={(checked) => setDriverFormData(prev => ({ ...prev, isActive: checked }))}
                data-testid="switch-driver-active"
              />
              <Label htmlFor="isActive">السائق نشط</Label>
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={resetDriverForm} data-testid="button-cancel-driver">
              إلغاء
            </Button>
            <Button
              onClick={handleDriverSubmit}
              disabled={createDriverMutation.isPending || updateDriverMutation.isPending}
              data-testid="button-save-driver"
            >
              {createDriverMutation.isPending || updateDriverMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isTripDialogOpen} onOpenChange={setIsTripDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingTrip ? "تعديل الرحلة" : "إنشاء رحلة جديدة"}
            </DialogTitle>
          </DialogHeader>
          
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="driverId">السائق *</Label>
              <Select
                value={tripFormData.driverId?.toString() || ""}
                onValueChange={(value) => setTripFormData(prev => ({ ...prev, driverId: parseInt(value) }))}
              >
                <SelectTrigger data-testid="select-trip-driver">
                  <SelectValue placeholder="اختر السائق" />
                </SelectTrigger>
                <SelectContent>
                  {activeDrivers.map(driver => (
                    <SelectItem key={driver.id} value={driver.id.toString()}>
                      {getDriverName(driver)}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="tripType">نوع الرحلة *</Label>
              <Select
                value={tripFormData.tripType}
                onValueChange={(value) => setTripFormData(prev => ({ 
                  ...prev, 
                  tripType: value,
                  appointmentId: value === "general" ? null : prev.appointmentId,
                }))}
              >
                <SelectTrigger data-testid="select-trip-type">
                  <SelectValue placeholder="اختر نوع الرحلة" />
                </SelectTrigger>
                <SelectContent>
                  {tripTypeOptions.map(option => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {tripFormData.tripType === "appointment" && (
              <div className="space-y-2">
                <Label htmlFor="appointmentId">الحجز *</Label>
                <Select
                  value={tripFormData.appointmentId?.toString() || ""}
                  onValueChange={(value) => handleAppointmentSelect(parseInt(value))}
                >
                  <SelectTrigger data-testid="select-trip-appointment">
                    <SelectValue placeholder="اختر الحجز" />
                  </SelectTrigger>
                  <SelectContent>
                    {homeServiceAppointments.map(apt => (
                      <SelectItem key={apt.id} value={apt.id.toString()}>
                        {apt.appointmentNumber} - {apt.addressText || "بدون عنوان"}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {tripFormData.appointmentId && (
                  <div className="p-3 bg-muted rounded-lg text-sm">
                    <p className="font-medium">بيانات الحجز:</p>
                    {(() => {
                      const apt = appointments?.find(a => a.id === tripFormData.appointmentId);
                      return apt ? (
                        <div className="mt-1 space-y-1">
                          <p>العنوان: {apt.addressText || "-"}</p>
                          {apt.locationLat && apt.locationLng && (
                            <p>الإحداثيات: {apt.locationLat}, {apt.locationLng}</p>
                          )}
                        </div>
                      ) : null;
                    })()}
                  </div>
                )}
              </div>
            )}

            {tripFormData.tripType === "general" && (
              <>
                <div className="space-y-2">
                  <Label htmlFor="startAddressText">نقطة الانطلاق *</Label>
                  <Input
                    id="startAddressText"
                    value={tripFormData.startAddressText}
                    onChange={(e) => setTripFormData(prev => ({ ...prev, startAddressText: e.target.value }))}
                    placeholder="أدخل عنوان نقطة الانطلاق"
                    data-testid="input-trip-start-address"
                  />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="startLocationLat">خط العرض (Lat)</Label>
                    <Input
                      id="startLocationLat"
                      type="number"
                      step="any"
                      value={tripFormData.startLocationLat || ""}
                      onChange={(e) => setTripFormData(prev => ({ ...prev, startLocationLat: e.target.value ? parseFloat(e.target.value) : null }))}
                      placeholder="24.7136"
                      data-testid="input-trip-start-lat"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="startLocationLng">خط الطول (Lng)</Label>
                    <Input
                      id="startLocationLng"
                      type="number"
                      step="any"
                      value={tripFormData.startLocationLng || ""}
                      onChange={(e) => setTripFormData(prev => ({ ...prev, startLocationLng: e.target.value ? parseFloat(e.target.value) : null }))}
                      placeholder="46.6753"
                      data-testid="input-trip-start-lng"
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="endAddressText">نقطة الوصول *</Label>
                  <Input
                    id="endAddressText"
                    value={tripFormData.endAddressText}
                    onChange={(e) => setTripFormData(prev => ({ ...prev, endAddressText: e.target.value }))}
                    placeholder="أدخل عنوان نقطة الوصول"
                    data-testid="input-trip-end-address"
                  />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="endLocationLat">خط العرض (Lat)</Label>
                    <Input
                      id="endLocationLat"
                      type="number"
                      step="any"
                      value={tripFormData.endLocationLat || ""}
                      onChange={(e) => setTripFormData(prev => ({ ...prev, endLocationLat: e.target.value ? parseFloat(e.target.value) : null }))}
                      placeholder="24.7136"
                      data-testid="input-trip-end-lat"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="endLocationLng">خط الطول (Lng)</Label>
                    <Input
                      id="endLocationLng"
                      type="number"
                      step="any"
                      value={tripFormData.endLocationLng || ""}
                      onChange={(e) => setTripFormData(prev => ({ ...prev, endLocationLng: e.target.value ? parseFloat(e.target.value) : null }))}
                      placeholder="46.6753"
                      data-testid="input-trip-end-lng"
                    />
                  </div>
                </div>
              </>
            )}

            <div className="space-y-2">
              <Label htmlFor="scheduledAt">موعد الرحلة</Label>
              <Input
                id="scheduledAt"
                type="datetime-local"
                value={tripFormData.scheduledAt}
                onChange={(e) => setTripFormData(prev => ({ ...prev, scheduledAt: e.target.value }))}
                data-testid="input-trip-scheduled"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="status">حالة الرحلة</Label>
              <Select
                value={tripFormData.status}
                onValueChange={(value) => setTripFormData(prev => ({ ...prev, status: value }))}
              >
                <SelectTrigger data-testid="select-trip-status">
                  <SelectValue placeholder="اختر الحالة" />
                </SelectTrigger>
                <SelectContent>
                  {tripStatusOptions.map(option => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="notes">ملاحظات</Label>
              <Textarea
                id="notes"
                value={tripFormData.notes}
                onChange={(e) => setTripFormData(prev => ({ ...prev, notes: e.target.value }))}
                placeholder="أي ملاحظات إضافية..."
                data-testid="input-trip-notes"
              />
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={resetTripForm} data-testid="button-cancel-trip">
              إلغاء
            </Button>
            <Button
              onClick={handleTripSubmit}
              disabled={createTripMutation.isPending || updateTripMutation.isPending}
              data-testid="button-save-trip"
            >
              {createTripMutation.isPending || updateTripMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
