import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { format } from "date-fns";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Textarea } from "@/components/ui/textarea";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
  FormDescription,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
} from "@/components/ui/dialog";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Plus, Package, Users, Calendar, Loader2, ChevronDown, AlertTriangle, CheckCircle, XCircle, RotateCcw, Trash2, Eye } from "lucide-react";
import type { EmployeeCustodyWithDetails, Employee, Product } from "@shared/schema";

const custodyStatuses = {
  open: { label: "مفتوحة", variant: "default" as const },
  returned: { label: "مرتجعة", variant: "secondary" as const },
  consumed: { label: "مستهلكة", variant: "outline" as const },
  partially_returned: { label: "مرتجعة جزئياً", variant: "secondary" as const },
  lost: { label: "مفقودة", variant: "destructive" as const },
  damaged: { label: "تالفة", variant: "destructive" as const },
};

const custodyFormSchema = z.object({
  employeeId: z.number({ required_error: "الموظف مطلوب" }),
  productId: z.number({ required_error: "المنتج مطلوب" }),
  quantity: z.number().min(0.01, "الكمية يجب أن تكون أكبر من صفر"),
  custodyDate: z.string().min(1, "تاريخ العهدة مطلوب"),
  notes: z.string().optional(),
});

const closureFormSchema = z.object({
  status: z.enum(["returned", "consumed", "partially_returned", "lost", "damaged"]),
  closedDate: z.string().min(1, "تاريخ الإغلاق مطلوب"),
  closureReason: z.string().min(1, "سبب الإغلاق مطلوب"),
  closureNotes: z.string().optional(),
  returnedQuantity: z.number().optional(),
}).refine(
  (data) => {
    if (data.status === "partially_returned") {
      return data.returnedQuantity !== undefined && data.returnedQuantity > 0;
    }
    return true;
  },
  {
    message: "الكمية المرتجعة مطلوبة عند الإرجاع الجزئي",
    path: ["returnedQuantity"],
  }
);

type CustodyFormValues = z.infer<typeof custodyFormSchema>;
type ClosureFormValues = z.infer<typeof closureFormSchema>;

const closureReasonLabels: Record<string, string> = {
  returned: "إرجاع العهدة بالكامل",
  consumed: "استهلاك العهدة بالكامل",
  partially_returned: "إرجاع جزئي للعهدة",
  lost: "فقدان العهدة",
  damaged: "تلف العهدة",
};

export default function EmployeeCustodiesPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [isClosureDialogOpen, setIsClosureDialogOpen] = useState(false);
  const [selectedCustody, setSelectedCustody] = useState<EmployeeCustodyWithDetails | null>(null);
  const [filterEmployeeId, setFilterEmployeeId] = useState<string>("all");
  const [filterStatus, setFilterStatus] = useState<string>("all");
  const [expandedRows, setExpandedRows] = useState<Set<number>>(new Set());

  const { data: custodies, isLoading } = useQuery<EmployeeCustodyWithDetails[]>({
    queryKey: ["/api/employee-custodies"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: products } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const form = useForm<CustodyFormValues>({
    resolver: zodResolver(custodyFormSchema),
    defaultValues: {
      employeeId: undefined,
      productId: undefined,
      quantity: 1,
      custodyDate: new Date().toISOString().split("T")[0],
      notes: "",
    },
  });

  const closureForm = useForm<ClosureFormValues>({
    resolver: zodResolver(closureFormSchema),
    defaultValues: {
      status: "returned",
      closedDate: new Date().toISOString().split("T")[0],
      closureReason: "",
      closureNotes: "",
      returnedQuantity: undefined,
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: CustodyFormValues) => {
      const response = await apiRequest("POST", "/api/employee-custodies", data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employee-custodies"] });
      queryClient.invalidateQueries({ queryKey: ["/api/stock-batches"] });
      queryClient.invalidateQueries({ queryKey: ["/api/inventory"] });
      toast({
        title: "تم الإنشاء بنجاح",
        description: "تم إضافة العهدة بنجاح",
      });
      setIsDialogOpen(false);
      form.reset();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء إنشاء العهدة",
        variant: "destructive",
      });
    },
  });

  const closeCustodyMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: ClosureFormValues }) => {
      const response = await apiRequest("PUT", `/api/employee-custodies/${id}`, {
        status: data.status,
        closedDate: data.closedDate,
        closureReason: data.closureReason,
        closureNotes: data.closureNotes,
        returnedQuantity: data.returnedQuantity,
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employee-custodies"] });
      queryClient.invalidateQueries({ queryKey: ["/api/stock-batches"] });
      queryClient.invalidateQueries({ queryKey: ["/api/inventory"] });
      toast({
        title: "تم إغلاق العهدة بنجاح",
        description: "تم تحديث حالة العهدة وحفظ التفاصيل",
      });
      setIsClosureDialogOpen(false);
      setSelectedCustody(null);
      closureForm.reset();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء إغلاق العهدة",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: CustodyFormValues) => {
    createMutation.mutate(data);
  };

  const onClosureSubmit = (data: ClosureFormValues) => {
    if (selectedCustody) {
      closeCustodyMutation.mutate({ id: selectedCustody.id, data });
    }
  };

  const openClosureDialog = (custody: EmployeeCustodyWithDetails) => {
    setSelectedCustody(custody);
    closureForm.reset({
      status: "returned",
      closedDate: new Date().toISOString().split("T")[0],
      closureReason: "",
      closureNotes: "",
      returnedQuantity: custody.quantity,
    });
    setIsClosureDialogOpen(true);
  };

  const toggleRowExpanded = (id: number) => {
    const newExpanded = new Set(expandedRows);
    if (newExpanded.has(id)) {
      newExpanded.delete(id);
    } else {
      newExpanded.add(id);
    }
    setExpandedRows(newExpanded);
  };

  const consumableProducts = products?.filter(
    (p) => (p as any).usageType === "consume" || (p as any).usageType === "sale_and_consume"
  );

  const filteredCustodies = custodies?.filter((custody) => {
    if (filterEmployeeId !== "all" && custody.employeeId !== parseInt(filterEmployeeId)) {
      return false;
    }
    if (filterStatus !== "all" && custody.status !== filterStatus) {
      return false;
    }
    return true;
  });

  const totalOpenCustodyCost = filteredCustodies
    ?.filter((c) => c.status === "open")
    .reduce((sum, c) => sum + c.totalCost, 0) || 0;

  if (isLoading) {
    return (
      <div className="space-y-6">
        <Skeleton className="h-8 w-48" />
        <div className="grid gap-4 md:grid-cols-3">
          {[1, 2, 3].map((i) => (
            <Skeleton key={i} className="h-32" />
          ))}
        </div>
        <Skeleton className="h-96" />
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">عهدة الموظفين</h1>
          <p className="text-muted-foreground">
            إدارة عهدة المنتجات المسلمة للموظفين
          </p>
        </div>
        <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
          <DialogTrigger asChild>
            <Button data-testid="button-add-custody">
              <Plus className="h-4 w-4 ml-2" />
              إضافة عهدة جديدة
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>إضافة عهدة جديدة</DialogTitle>
            </DialogHeader>
            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                <FormField
                  control={form.control}
                  name="employeeId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الموظف *</FormLabel>
                      <Select
                        onValueChange={(value) => field.onChange(parseInt(value))}
                        value={field.value?.toString()}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-employee">
                            <SelectValue placeholder="اختر الموظف" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {employees?.filter(e => e.isActive).map((employee) => (
                            <SelectItem key={employee.id} value={employee.id.toString()}>
                              {employee.fullNameAr || employee.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="productId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>المنتج *</FormLabel>
                      <Select
                        onValueChange={(value) => field.onChange(parseInt(value))}
                        value={field.value?.toString()}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-product">
                            <SelectValue placeholder="اختر المنتج" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {consumableProducts?.map((product) => (
                            <SelectItem key={product.id} value={product.id.toString()}>
                              {product.nameAr}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="quantity"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الكمية *</FormLabel>
                      <FormControl>
                        <Input
                          type="number"
                          min={0.01}
                          step={0.01}
                          {...field}
                          onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                          data-testid="input-quantity"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="custodyDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>تاريخ العهدة *</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} data-testid="input-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="notes"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>ملاحظات</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-notes" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <div className="flex justify-end gap-2 pt-4">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setIsDialogOpen(false)}
                    data-testid="button-cancel"
                  >
                    إلغاء
                  </Button>
                  <Button type="submit" disabled={createMutation.isPending} data-testid="button-submit">
                    {createMutation.isPending && <Loader2 className="h-4 w-4 ml-2 animate-spin" />}
                    إضافة العهدة
                  </Button>
                </div>
              </form>
            </Form>
          </DialogContent>
        </Dialog>
      </div>

      <div className="grid gap-4 md:grid-cols-3">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي العهد المفتوحة</CardTitle>
            <Package className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-open-count">
              {custodies?.filter((c) => c.status === "open").length || 0}
            </div>
            <p className="text-xs text-muted-foreground">عهدة مفتوحة</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">قيمة العهد المفتوحة</CardTitle>
            <Package className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-open-value">
              {totalOpenCustodyCost.toFixed(2)} ر.س
            </div>
            <p className="text-xs text-muted-foreground">قيمة التكلفة</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">الموظفين بعهدة</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-employee-count">
              {new Set(custodies?.filter((c) => c.status === "open").map((c) => c.employeeId)).size}
            </div>
            <p className="text-xs text-muted-foreground">موظف لديه عهدة مفتوحة</p>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-2">
          <CardTitle>قائمة العهد</CardTitle>
          <div className="flex gap-2">
            <Select value={filterEmployeeId} onValueChange={setFilterEmployeeId}>
              <SelectTrigger className="w-48" data-testid="filter-employee">
                <SelectValue placeholder="جميع الموظفين" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">جميع الموظفين</SelectItem>
                {employees?.filter(e => e.isActive).map((employee) => (
                  <SelectItem key={employee.id} value={employee.id.toString()}>
                    {employee.fullNameAr || employee.nameAr}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            <Select value={filterStatus} onValueChange={setFilterStatus}>
              <SelectTrigger className="w-40" data-testid="filter-status">
                <SelectValue placeholder="جميع الحالات" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">جميع الحالات</SelectItem>
                {Object.entries(custodyStatuses).map(([value, { label }]) => (
                  <SelectItem key={value} value={value}>
                    {label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>الموظف</TableHead>
                <TableHead>المنتج</TableHead>
                <TableHead>الكمية</TableHead>
                <TableHead>تكلفة الوحدة</TableHead>
                <TableHead>التكلفة الإجمالية</TableHead>
                <TableHead>التاريخ</TableHead>
                <TableHead>الحالة</TableHead>
                <TableHead>الإجراءات</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredCustodies?.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={8} className="text-center text-muted-foreground py-8">
                    لا توجد عهد
                  </TableCell>
                </TableRow>
              ) : (
                filteredCustodies?.map((custody) => {
                  const isExpanded = expandedRows.has(custody.id);
                  const isClosed = custody.status !== "open";
                  return (
                    <>
                      <TableRow key={custody.id} data-testid={`row-custody-${custody.id}`}>
                        <TableCell>
                          {custody.employee?.fullNameAr || custody.employee?.nameAr || "-"}
                        </TableCell>
                        <TableCell>{custody.product?.nameAr || "-"}</TableCell>
                        <TableCell>{custody.quantity}</TableCell>
                        <TableCell>{custody.unitCost.toFixed(2)} ر.س</TableCell>
                        <TableCell>{custody.totalCost.toFixed(2)} ر.س</TableCell>
                        <TableCell>
                          {format(new Date(custody.custodyDate), "yyyy-MM-dd")}
                        </TableCell>
                        <TableCell>
                          <Badge variant={custodyStatuses[custody.status as keyof typeof custodyStatuses]?.variant || "default"}>
                            {custodyStatuses[custody.status as keyof typeof custodyStatuses]?.label || custody.status}
                          </Badge>
                        </TableCell>
                        <TableCell>
                          <div className="flex gap-2">
                            {custody.status === "open" ? (
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => openClosureDialog(custody)}
                                data-testid={`button-close-custody-${custody.id}`}
                              >
                                <XCircle className="h-4 w-4 ml-1" />
                                إغلاق العهدة
                              </Button>
                            ) : (
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => toggleRowExpanded(custody.id)}
                                data-testid={`button-view-details-${custody.id}`}
                              >
                                <Eye className="h-4 w-4 ml-1" />
                                {isExpanded ? "إخفاء" : "تفاصيل"}
                              </Button>
                            )}
                          </div>
                        </TableCell>
                      </TableRow>
                      {isExpanded && isClosed && (
                        <TableRow key={`details-${custody.id}`} className="bg-muted/30">
                          <TableCell colSpan={8}>
                            <div className="p-4 space-y-3">
                              <h4 className="font-semibold text-sm">تفاصيل إغلاق العهدة</h4>
                              <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                                <div>
                                  <span className="text-muted-foreground">تاريخ الإغلاق: </span>
                                  <span className="font-medium">
                                    {custody.closedDate ? format(new Date(custody.closedDate), "yyyy-MM-dd") : "-"}
                                  </span>
                                </div>
                                <div>
                                  <span className="text-muted-foreground">سبب الإغلاق: </span>
                                  <span className="font-medium">{custody.closureReason || "-"}</span>
                                </div>
                                {custody.status === "partially_returned" && (
                                  <div>
                                    <span className="text-muted-foreground">الكمية المرتجعة: </span>
                                    <span className="font-medium">{custody.returnedQuantity || 0}</span>
                                  </div>
                                )}
                                <div>
                                  <span className="text-muted-foreground">أغلق بواسطة: </span>
                                  <span className="font-medium">
                                    {custody.closedByEmployee?.fullNameAr || custody.closedByEmployee?.nameAr || "-"}
                                  </span>
                                </div>
                              </div>
                              {custody.closureNotes && (
                                <div>
                                  <span className="text-muted-foreground text-sm">ملاحظات الإغلاق: </span>
                                  <p className="text-sm mt-1">{custody.closureNotes}</p>
                                </div>
                              )}
                            </div>
                          </TableCell>
                        </TableRow>
                      )}
                    </>
                  );
                })
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Closure Dialog */}
      <Dialog open={isClosureDialogOpen} onOpenChange={setIsClosureDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>إغلاق العهدة</DialogTitle>
            <DialogDescription>
              {selectedCustody && (
                <span>
                  {selectedCustody.product?.nameAr} - {selectedCustody.employee?.fullNameAr || selectedCustody.employee?.nameAr}
                </span>
              )}
            </DialogDescription>
          </DialogHeader>
          <Form {...closureForm}>
            <form onSubmit={closureForm.handleSubmit(onClosureSubmit)} className="space-y-4">
              <FormField
                control={closureForm.control}
                name="status"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>نوع الإغلاق *</FormLabel>
                    <Select
                      onValueChange={field.onChange}
                      value={field.value}
                    >
                      <FormControl>
                        <SelectTrigger data-testid="select-closure-status">
                          <SelectValue placeholder="اختر نوع الإغلاق" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="returned">
                          <div className="flex items-center gap-2">
                            <RotateCcw className="h-4 w-4 text-green-600" />
                            مرتجعة بالكامل
                          </div>
                        </SelectItem>
                        <SelectItem value="consumed">
                          <div className="flex items-center gap-2">
                            <CheckCircle className="h-4 w-4 text-blue-600" />
                            مستهلكة بالكامل
                          </div>
                        </SelectItem>
                        <SelectItem value="partially_returned">
                          <div className="flex items-center gap-2">
                            <RotateCcw className="h-4 w-4 text-yellow-600" />
                            مرتجعة جزئياً
                          </div>
                        </SelectItem>
                        <SelectItem value="lost">
                          <div className="flex items-center gap-2">
                            <AlertTriangle className="h-4 w-4 text-red-600" />
                            مفقودة
                          </div>
                        </SelectItem>
                        <SelectItem value="damaged">
                          <div className="flex items-center gap-2">
                            <Trash2 className="h-4 w-4 text-red-600" />
                            تالفة
                          </div>
                        </SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              {closureForm.watch("status") === "partially_returned" && (
                <FormField
                  control={closureForm.control}
                  name="returnedQuantity"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>الكمية المرتجعة *</FormLabel>
                      <FormControl>
                        <Input
                          type="number"
                          min={0}
                          max={selectedCustody?.quantity || 0}
                          step={0.01}
                          {...field}
                          onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                          data-testid="input-returned-quantity"
                        />
                      </FormControl>
                      <FormDescription>
                        الحد الأقصى: {selectedCustody?.quantity || 0}
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              )}

              <FormField
                control={closureForm.control}
                name="closedDate"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>تاريخ الإغلاق *</FormLabel>
                    <FormControl>
                      <Input type="date" {...field} data-testid="input-closed-date" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={closureForm.control}
                name="closureReason"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>سبب الإغلاق *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="اذكر سبب إغلاق العهدة" data-testid="input-closure-reason" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={closureForm.control}
                name="closureNotes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>ملاحظات إضافية</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="أي ملاحظات إضافية..." data-testid="input-closure-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2 pt-4">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsClosureDialogOpen(false)}
                  data-testid="button-cancel-closure"
                >
                  إلغاء
                </Button>
                <Button type="submit" disabled={closeCustodyMutation.isPending} data-testid="button-submit-closure">
                  {closeCustodyMutation.isPending && <Loader2 className="h-4 w-4 ml-2 animate-spin" />}
                  إغلاق العهدة
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
