import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { useParams, Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Skeleton } from "@/components/ui/skeleton";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { 
  ArrowRight, User, Phone, Mail, Calendar, Building, 
  IdCard, Wallet, Clock, CalendarCheck, CalendarX, 
  Package, DollarSign, Briefcase, MapPin, FileText,
  TrendingUp, AlertCircle, Percent, Download, FileSpreadsheet,
  Coins, Gift, CircleMinus
} from "lucide-react";
import { format, startOfMonth, endOfMonth } from "date-fns";
import { formatCurrency, formatDate } from "@/lib/formatters";

interface Role {
  id: number;
  nameAr: string;
}

interface Employee {
  id: number;
  nameAr: string | null;
  code: string | null;
  phone: string | null;
  fullNameAr: string | null;
  phoneCountryCode: string | null;
  phoneNumber: string | null;
  email: string | null;
  fingerprintEmployeeNo: string | null;
  profileImageUrl: string | null;
  jobTitleAr: string | null;
  roleId: number | null;
  cityNameAr: string | null;
  branchId: number | null;
  nationalityNameAr: string | null;
  birthDate: string | null;
  hireDate: string | null;
  iqamaNumber: string | null;
  iqamaExpiryDate: string | null;
  insuranceCompany: string | null;
  insuranceNumber: string | null;
  insuranceExpiryDate: string | null;
  insuranceCardExpiryDate: string | null;
  healthNumber: string | null;
  passportNumber: string | null;
  passportExpiryDate: string | null;
  annualLeaveDays: number | null;
  weeklyOffDay: string | null;
  isActive: boolean;
  role?: Role;
  createdAt: string;
  updatedAt: string;
}

interface SalaryProfile {
  id: number;
  employeeId: number;
  basicSalary: number;
  housingAllowance: number;
  transportAllowance: number;
  otherFixedAllowances: number;
  targetAmount: number;
  commissionPercentServices: number;
  commissionPercentProducts: number;
  commissionPercentCafeteria: number;
  isActive: boolean;
}

interface PayrollRun {
  id: number;
  periodName: string;
  periodStartDate: string;
  periodEndDate: string;
  status: string;
}

interface PayrollLine {
  id: number;
  payrollRunId: number;
  employeeId: number;
  basicSalary: number;
  fixedAllowances: number;
  overtimeAmount: number;
  commissionAmount: number;
  otherEarnings: number;
  deductionLate: number;
  deductionOther: number;
  grossSalary: number;
  totalDeductions: number;
  netSalary: number;
  payrollRun?: PayrollRun;
}

interface Leave {
  id: number;
  employeeId: number;
  leaveType: string;
  startDate: string;
  endDate: string;
  durationDays: number;
  durationHours: number;
  status: string;
  reason: string | null;
}

interface WorkShift {
  id: number;
  nameAr: string;
  shiftType: string;
  startTime: string;
  endTime: string;
}

interface ShiftAssignment {
  id: number;
  employeeId: number;
  workShiftId: number;
  startDate: string;
  endDate: string;
  workShift?: WorkShift;
}

interface AttendanceLog {
  id: number;
  employeeId: number;
  date: string;
  checkInTime: string | null;
  checkOutTime: string | null;
  totalWorkMinutes: number;
  status: string;
  notes: string | null;
}

interface Product {
  id: number;
  nameAr: string;
}

interface ClosedByEmployee {
  id: number;
  nameAr: string | null;
  fullNameAr: string | null;
}

interface Custody {
  id: number;
  employeeId: number;
  productId: number;
  quantity: number;
  unitCost: number;
  totalCost: number;
  custodyDate: string;
  status: string;
  notes: string | null;
  closedDate: string | null;
  closedByEmployeeId: number | null;
  closureReason: string | null;
  returnedQuantity: number | null;
  closureNotes: string | null;
  product?: Product;
  closedByEmployee?: ClosedByEmployee;
}

interface CommissionTargetSource {
  id: number;
  profileId: number;
  sourceType: string;
  isEnabled: boolean;
}

interface CommissionProfile {
  id: number;
  employeeId: number;
  monthlyTarget: number;
  commissionRate: number;
  isActive: boolean;
  targetSources: CommissionTargetSource[];
}

interface EmployeeServiceActivity {
  id: number;
  date: string;
  type: 'order' | 'appointment' | 'return';
  referenceNumber: string;
  referenceId: number;
  serviceName: string;
  serviceId: number;
  quantity: number;
  amount: number;
  inventoryDeduction: number;
  netAmount: number;
  status: string;
  completedAt: string | null;
}

interface EmployeeOrderActivity {
  id: number;
  date: string;
  type: 'order' | 'appointment';
  referenceNumber: string;
  referenceId: number;
  totalAmount: number;
  returnsAmount: number;
  netAmount: number;
  status: string;
}

interface CafeteriaSaleItem {
  orderId: number;
  orderNumber: string;
  date: string;
  totalAmount: number;
  status: string;
}

interface ProductSaleItem {
  orderId: number;
  orderNumber: string;
  date: string;
  productName: string;
  quantity: number;
  amount: number;
  returnedAmount: number;
}

interface CommissionSummary {
  executedServicesTotal: number;
  inventoryDeductionTotal: number;
  createdOrdersTotal: number | null;
  ordersReturnsTotal: number;
  cafeteriaSalesTotal: number;
  cafeteriaReturnsTotal: number;
  productSalesTotal: number;
  productReturnsTotal: number;
  grossTotal: number | null;
  deductionsTotal: number;
  netTotal: number | null;
  targetAmount: number;
  achievementRate: number | null;
  commissionRate: number;
  commissionAmount: number;
}

interface EmployeeCommissionReport {
  employeeId: number;
  employeeName: string;
  periodMonth: number;
  periodYear: number;
  profile: CommissionProfile | null;
  executedServices: EmployeeServiceActivity[];
  createdOrders: EmployeeOrderActivity[];
  cafeteriaSales: CafeteriaSaleItem[];
  productSales: ProductSaleItem[];
  summary: CommissionSummary;
}

interface AdvanceInstallment {
  id: number;
  installmentNumber: number;
  amount: number;
  dueDate: string;
  status: string;
  deductedAt: string | null;
}

interface PaymentMethod {
  id: number;
  nameAr: string;
}

interface CashDisbursement {
  id: number;
  disbursementNumber: string;
  disbursementType: string;
  employeeId: number | null;
  driverId: number | null;
  paymentMethodId: number;
  amount: number;
  disbursementDate: string;
  notes: string | null;
  isInstallment: boolean;
  installmentMonths: number | null;
  installmentAmount: number | null;
  paymentMethod?: PaymentMethod;
  installments?: AdvanceInstallment[];
  createdAt: string;
}

const disbursementTypeLabels: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline"; icon: any }> = {
  advance: { label: "سلفة", variant: "default", icon: Wallet },
  gift: { label: "هدية", variant: "secondary", icon: Gift },
  deduction: { label: "خصم", variant: "destructive", icon: CircleMinus },
};

const installmentStatusLabels: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline" }> = {
  pending: { label: "قيد الانتظار", variant: "outline" },
  deducted: { label: "تم الخصم", variant: "default" },
};

const leaveTypeLabels: Record<string, string> = {
  annual: "سنوية",
  sick: "مرضية",
  unpaid: "بدون راتب",
  public_holiday: "إجازة رسمية",
  paid: "مدفوعة",
  other: "أخرى",
  hourly: "ساعية",
};

const leaveStatusLabels: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline" }> = {
  pending: { label: "معلقة", variant: "secondary" },
  approved: { label: "موافق عليها", variant: "default" },
  rejected: { label: "مرفوضة", variant: "destructive" },
  canceled: { label: "ملغاة", variant: "outline" },
};

const attendanceStatusLabels: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline" }> = {
  present: { label: "حاضر", variant: "default" },
  absent: { label: "غائب", variant: "destructive" },
  late: { label: "متأخر", variant: "secondary" },
  on_leave: { label: "إجازة", variant: "outline" },
  off: { label: "يوم إجازة", variant: "outline" },
};

const custodyStatusLabels: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline" }> = {
  open: { label: "مفتوحة", variant: "default" },
  returned: { label: "مرتجعة", variant: "secondary" },
  consumed: { label: "مستهلكة", variant: "outline" },
  partially_returned: { label: "مرتجعة جزئياً", variant: "secondary" },
  lost: { label: "مفقودة", variant: "destructive" },
  damaged: { label: "تالفة", variant: "destructive" },
};

const payrollStatusLabels: Record<string, { label: string; variant: "default" | "secondary" | "destructive" | "outline" }> = {
  draft: { label: "مسودة", variant: "secondary" },
  approved: { label: "معتمد", variant: "outline" },
  paid: { label: "مدفوع", variant: "default" },
  canceled: { label: "ملغي", variant: "destructive" },
};

const weeklyOffDayLabels: Record<string, string> = {
  friday: "الجمعة",
  saturday: "السبت",
  sunday: "الأحد",
  friday_saturday: "الجمعة والسبت",
  none: "لا يوجد",
};


export default function EmployeeDetailPage() {
  const params = useParams();
  const employeeId = parseInt(params.id || "0");
  
  const today = new Date();
  const defaultStartDate = format(startOfMonth(today), "yyyy-MM-dd");
  const defaultEndDate = format(endOfMonth(today), "yyyy-MM-dd");
  
  const [commissionStartDate, setCommissionStartDate] = useState(defaultStartDate);
  const [commissionEndDate, setCommissionEndDate] = useState(defaultEndDate);
  
  const [commissionMonth, setCommissionMonth] = useState(today.getMonth() + 1);
  const [commissionYear, setCommissionYear] = useState(today.getFullYear());
  const [activeCommissionSection, setActiveCommissionSection] = useState<'services' | 'orders' | 'cafeteria' | 'products'>('services');

  const { data: employee, isLoading: isLoadingEmployee } = useQuery<Employee>({
    queryKey: ["/api/employees", employeeId],
    enabled: employeeId > 0,
  });

  const { data: salaryProfiles } = useQuery<SalaryProfile[]>({
    queryKey: ["/api/salary-profiles"],
    enabled: employeeId > 0,
  });

  const { data: payrollHistory, isLoading: isLoadingPayroll } = useQuery<PayrollLine[]>({
    queryKey: [`/api/employees/${employeeId}/payroll-history`],
    enabled: employeeId > 0,
  });

  const { data: leaves, isLoading: isLoadingLeaves } = useQuery<Leave[]>({
    queryKey: [`/api/hr/leaves?employeeId=${employeeId}`],
    enabled: employeeId > 0,
  });

  const { data: shiftAssignments, isLoading: isLoadingShifts } = useQuery<ShiftAssignment[]>({
    queryKey: [`/api/hr/shift-assignments?employeeId=${employeeId}`],
    enabled: employeeId > 0,
  });

  const { data: attendanceLogs, isLoading: isLoadingAttendance } = useQuery<AttendanceLog[]>({
    queryKey: [`/api/hr/attendance-logs?employeeId=${employeeId}`],
    enabled: employeeId > 0,
  });

  const { data: custodies, isLoading: isLoadingCustodies } = useQuery<Custody[]>({
    queryKey: [`/api/employees/${employeeId}/custodies`],
    enabled: employeeId > 0,
  });

  const { data: commissionReport, isLoading: isLoadingCommissions } = useQuery<EmployeeCommissionReport>({
    queryKey: ["/api/employees", employeeId, "commission-report", { month: commissionMonth, year: commissionYear }],
    queryFn: async () => {
      const response = await fetch(`/api/employees/${employeeId}/commission-report?month=${commissionMonth}&year=${commissionYear}`);
      if (!response.ok) throw new Error("Failed to fetch commission report");
      return response.json();
    },
    enabled: employeeId > 0,
  });

  const { data: cashDisbursements, isLoading: isLoadingDisbursements } = useQuery<CashDisbursement[]>({
    queryKey: [`/api/cash-disbursements?employee_id=${employeeId}`],
    enabled: employeeId > 0,
  });

  const salaryProfile = salaryProfiles?.find(sp => sp.employeeId === employeeId);

  const approvedLeaves = leaves?.filter(l => l.status === "approved") || [];
  const usedLeaveDays = approvedLeaves.reduce((sum, l) => sum + (l.durationDays || 0), 0);
  const remainingLeaveDays = (employee?.annualLeaveDays || 0) - usedLeaveDays;

  if (isLoadingEmployee) {
    return (
      <div className="space-y-6">
        <Skeleton className="h-8 w-48" />
        <Skeleton className="h-64 w-full" />
      </div>
    );
  }

  if (!employee) {
    return (
      <div className="flex flex-col items-center justify-center py-12">
        <AlertCircle className="h-12 w-12 text-muted-foreground mb-4" />
        <h2 className="text-xl font-semibold mb-2">الموظف غير موجود</h2>
        <Link href="/employees">
          <Button variant="outline">
            <ArrowRight className="ml-2 h-4 w-4" />
            العودة للموظفين
          </Button>
        </Link>
      </div>
    );
  }

  const employeeName = employee.fullNameAr || employee.nameAr || "موظف";

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center gap-4">
        <Link href="/employees">
          <Button variant="ghost" size="icon" data-testid="button-back-employees">
            <ArrowRight className="h-5 w-5" />
          </Button>
        </Link>
        <h1 className="text-2xl font-bold" data-testid="text-employee-name">{employeeName}</h1>
        <Badge variant={employee.isActive ? "default" : "secondary"} data-testid="badge-employee-status">
          {employee.isActive ? "نشط" : "غير نشط"}
        </Badge>
      </div>

      <Tabs defaultValue="info" className="space-y-4">
        <TabsList className="flex-wrap gap-1">
          <TabsTrigger value="info" data-testid="tab-info">
            <User className="ml-2 h-4 w-4" />
            المعلومات الشخصية
          </TabsTrigger>
          <TabsTrigger value="salary" data-testid="tab-salary">
            <Wallet className="ml-2 h-4 w-4" />
            الراتب والبدلات
          </TabsTrigger>
          <TabsTrigger value="payroll" data-testid="tab-payroll">
            <DollarSign className="ml-2 h-4 w-4" />
            مسيرات الرواتب
          </TabsTrigger>
          <TabsTrigger value="leaves" data-testid="tab-leaves">
            <CalendarX className="ml-2 h-4 w-4" />
            الإجازات
          </TabsTrigger>
          <TabsTrigger value="shifts" data-testid="tab-shifts">
            <Clock className="ml-2 h-4 w-4" />
            الورديات
          </TabsTrigger>
          <TabsTrigger value="attendance" data-testid="tab-attendance">
            <CalendarCheck className="ml-2 h-4 w-4" />
            الحضور
          </TabsTrigger>
          <TabsTrigger value="custodies" data-testid="tab-custodies">
            <Package className="ml-2 h-4 w-4" />
            العهد
          </TabsTrigger>
          <TabsTrigger value="commissions" data-testid="tab-commissions">
            <Percent className="ml-2 h-4 w-4" />
            العمولات
          </TabsTrigger>
          <TabsTrigger value="disbursements" data-testid="tab-disbursements">
            <Coins className="ml-2 h-4 w-4" />
            الصرف النقدي
          </TabsTrigger>
        </TabsList>

        <TabsContent value="info" className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="h-5 w-5" />
                  البيانات الأساسية
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center gap-4 mb-6">
                  <Avatar className="h-20 w-20">
                    <AvatarImage src={employee.profileImageUrl || undefined} alt={employeeName} />
                    <AvatarFallback className="text-2xl">
                      {employeeName.charAt(0)}
                    </AvatarFallback>
                  </Avatar>
                  <div>
                    <h3 className="text-lg font-semibold">{employeeName}</h3>
                    <p className="text-muted-foreground">{employee.jobTitleAr || "-"}</p>
                    {employee.role && (
                      <Badge variant="outline" className="mt-1">
                        {employee.role.nameAr}
                      </Badge>
                    )}
                  </div>
                </div>

                <div className="grid gap-3">
                  <div className="flex items-center gap-2">
                    <IdCard className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">رقم الموظف:</span>
                    <span>{employee.code || employee.fingerprintEmployeeNo || "-"}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Phone className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">الجوال:</span>
                    <span dir="ltr">{employee.phoneNumber || employee.phone || "-"}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Mail className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">البريد:</span>
                    <span>{employee.email || "-"}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <MapPin className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">الجنسية:</span>
                    <span>{employee.nationalityNameAr || "-"}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Building className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">المدينة:</span>
                    <span>{employee.cityNameAr || "-"}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Calendar className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">تاريخ الميلاد:</span>
                    <span>{formatDate(employee.birthDate)}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Briefcase className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">تاريخ التعيين:</span>
                    <span>{formatDate(employee.hireDate)}</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Calendar className="h-4 w-4 text-muted-foreground" />
                    <span className="text-muted-foreground">يوم الإجازة الأسبوعية:</span>
                    <span>{weeklyOffDayLabels[employee.weeklyOffDay || "none"] || "-"}</span>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <FileText className="h-5 w-5" />
                  المستندات والتأمين
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-3">
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">رقم الإقامة:</span>
                    <span>{employee.iqamaNumber || "-"}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">تاريخ انتهاء الإقامة:</span>
                    <span>{formatDate(employee.iqamaExpiryDate)}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">رقم الجواز:</span>
                    <span>{employee.passportNumber || "-"}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">تاريخ انتهاء الجواز:</span>
                    <span>{formatDate(employee.passportExpiryDate)}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">شركة التأمين:</span>
                    <span>{employee.insuranceCompany || "-"}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">رقم التأمين:</span>
                    <span>{employee.insuranceNumber || "-"}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">تاريخ انتهاء التأمين:</span>
                    <span>{formatDate(employee.insuranceExpiryDate)}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">تاريخ انتهاء بطاقة التأمين:</span>
                    <span>{formatDate(employee.insuranceCardExpiryDate)}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">الرقم الصحي:</span>
                    <span>{employee.healthNumber || "-"}</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="salary" className="space-y-4">
          {salaryProfile ? (
            <div className="grid gap-4 md:grid-cols-2">
              <Card>
                <CardHeader>
                  <CardTitle>الراتب والبدلات</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid gap-3">
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">الراتب الأساسي:</span>
                      <span className="font-semibold">{formatCurrency(salaryProfile.basicSalary)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">بدل السكن:</span>
                      <span>{formatCurrency(salaryProfile.housingAllowance)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">بدل المواصلات:</span>
                      <span>{formatCurrency(salaryProfile.transportAllowance)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">بدلات أخرى:</span>
                      <span>{formatCurrency(salaryProfile.otherFixedAllowances)}</span>
                    </div>
                    <div className="flex justify-between border-t pt-2">
                      <span className="font-semibold">إجمالي الراتب:</span>
                      <span className="font-bold text-lg">
                        {formatCurrency(
                          salaryProfile.basicSalary +
                          salaryProfile.housingAllowance +
                          salaryProfile.transportAllowance +
                          salaryProfile.otherFixedAllowances
                        )}
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="h-5 w-5" />
                    العمولات والأهداف
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid gap-3">
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">الهدف الشهري:</span>
                      <span className="font-semibold">{formatCurrency(salaryProfile.targetAmount)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">عمولة الخدمات:</span>
                      <span>{salaryProfile.commissionPercentServices}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">عمولة المنتجات:</span>
                      <span>{salaryProfile.commissionPercentProducts}%</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">عمولة الكافتيريا:</span>
                      <span>{salaryProfile.commissionPercentCafeteria}%</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          ) : (
            <Card>
              <CardContent className="py-8 text-center">
                <Wallet className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                <p className="text-muted-foreground">لم يتم إعداد الملف الراتبي لهذا الموظف</p>
                <Link href="/salary-profiles">
                  <Button variant="outline" className="mt-4">
                    إعداد الملف الراتبي
                  </Button>
                </Link>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="payroll" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>سجل مسيرات الرواتب</CardTitle>
              <CardDescription>جميع مسيرات الرواتب المسجلة للموظف</CardDescription>
            </CardHeader>
            <CardContent>
              {isLoadingPayroll ? (
                <div className="space-y-2">
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                </div>
              ) : payrollHistory && payrollHistory.length > 0 ? (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>الفترة</TableHead>
                      <TableHead>الراتب الأساسي</TableHead>
                      <TableHead>البدلات</TableHead>
                      <TableHead>العمولات</TableHead>
                      <TableHead>الخصومات</TableHead>
                      <TableHead>صافي الراتب</TableHead>
                      <TableHead>الحالة</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {payrollHistory.map((line) => (
                      <TableRow key={line.id} data-testid={`row-payroll-${line.id}`}>
                        <TableCell>
                          {line.payrollRun?.periodName || "-"}
                        </TableCell>
                        <TableCell>{formatCurrency(line.basicSalary)}</TableCell>
                        <TableCell>{formatCurrency(line.fixedAllowances)}</TableCell>
                        <TableCell>{formatCurrency(line.commissionAmount)}</TableCell>
                        <TableCell className="text-destructive">
                          {formatCurrency(line.totalDeductions)}
                        </TableCell>
                        <TableCell className="font-semibold">
                          {formatCurrency(line.netSalary)}
                        </TableCell>
                        <TableCell>
                          <Badge variant={payrollStatusLabels[line.payrollRun?.status || "draft"]?.variant || "secondary"}>
                            {payrollStatusLabels[line.payrollRun?.status || "draft"]?.label || line.payrollRun?.status}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              ) : (
                <div className="text-center py-8">
                  <DollarSign className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد مسيرات رواتب مسجلة</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="leaves" className="space-y-4">
          <div className="grid gap-4 md:grid-cols-3">
            <Card>
              <CardContent className="pt-6">
                <div className="text-center">
                  <p className="text-sm text-muted-foreground">رصيد الإجازات السنوية</p>
                  <p className="text-3xl font-bold">{employee.annualLeaveDays || 0}</p>
                  <p className="text-xs text-muted-foreground">يوم</p>
                </div>
              </CardContent>
            </Card>
            <Card>
              <CardContent className="pt-6">
                <div className="text-center">
                  <p className="text-sm text-muted-foreground">الإجازات المستخدمة</p>
                  <p className="text-3xl font-bold text-destructive">{usedLeaveDays}</p>
                  <p className="text-xs text-muted-foreground">يوم</p>
                </div>
              </CardContent>
            </Card>
            <Card>
              <CardContent className="pt-6">
                <div className="text-center">
                  <p className="text-sm text-muted-foreground">الرصيد المتبقي</p>
                  <p className={`text-3xl font-bold ${remainingLeaveDays < 0 ? "text-destructive" : "text-green-600"}`}>
                    {remainingLeaveDays}
                  </p>
                  <p className="text-xs text-muted-foreground">يوم</p>
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>سجل الإجازات</CardTitle>
            </CardHeader>
            <CardContent>
              {isLoadingLeaves ? (
                <div className="space-y-2">
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                </div>
              ) : leaves && leaves.length > 0 ? (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>نوع الإجازة</TableHead>
                      <TableHead>من</TableHead>
                      <TableHead>إلى</TableHead>
                      <TableHead>المدة</TableHead>
                      <TableHead>الحالة</TableHead>
                      <TableHead>السبب</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {leaves.map((leave) => (
                      <TableRow key={leave.id} data-testid={`row-leave-${leave.id}`}>
                        <TableCell>{leaveTypeLabels[leave.leaveType] || leave.leaveType}</TableCell>
                        <TableCell>{formatDate(leave.startDate)}</TableCell>
                        <TableCell>{formatDate(leave.endDate)}</TableCell>
                        <TableCell>
                          {leave.durationDays > 0 ? `${leave.durationDays} يوم` : `${leave.durationHours} ساعة`}
                        </TableCell>
                        <TableCell>
                          <Badge variant={leaveStatusLabels[leave.status]?.variant || "secondary"}>
                            {leaveStatusLabels[leave.status]?.label || leave.status}
                          </Badge>
                        </TableCell>
                        <TableCell>{leave.reason || "-"}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              ) : (
                <div className="text-center py-8">
                  <CalendarX className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد إجازات مسجلة</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="shifts" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>الورديات المعينة</CardTitle>
              <CardDescription>جدول ورديات العمل المخصصة للموظف</CardDescription>
            </CardHeader>
            <CardContent>
              {isLoadingShifts ? (
                <div className="space-y-2">
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                </div>
              ) : shiftAssignments && shiftAssignments.length > 0 ? (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>اسم الوردية</TableHead>
                      <TableHead>النوع</TableHead>
                      <TableHead>وقت البدء</TableHead>
                      <TableHead>وقت الانتهاء</TableHead>
                      <TableHead>من تاريخ</TableHead>
                      <TableHead>إلى تاريخ</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {shiftAssignments.map((assignment) => (
                      <TableRow key={assignment.id} data-testid={`row-shift-${assignment.id}`}>
                        <TableCell>{assignment.workShift?.nameAr || "-"}</TableCell>
                        <TableCell>
                          <Badge variant="outline">
                            {assignment.workShift?.shiftType === "morning" ? "صباحية" :
                             assignment.workShift?.shiftType === "evening" ? "مسائية" : "مخصصة"}
                          </Badge>
                        </TableCell>
                        <TableCell>{assignment.workShift?.startTime || "-"}</TableCell>
                        <TableCell>{assignment.workShift?.endTime || "-"}</TableCell>
                        <TableCell>{formatDate(assignment.startDate)}</TableCell>
                        <TableCell>{formatDate(assignment.endDate)}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              ) : (
                <div className="text-center py-8">
                  <Clock className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد ورديات معينة</p>
                  <Link href="/hr/shift-assignments">
                    <Button variant="outline" className="mt-4">
                      تعيين وردية
                    </Button>
                  </Link>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="attendance" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>سجل الحضور</CardTitle>
              <CardDescription>آخر 30 سجل حضور للموظف</CardDescription>
            </CardHeader>
            <CardContent>
              {isLoadingAttendance ? (
                <div className="space-y-2">
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                </div>
              ) : attendanceLogs && attendanceLogs.length > 0 ? (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>التاريخ</TableHead>
                      <TableHead>وقت الحضور</TableHead>
                      <TableHead>وقت الانصراف</TableHead>
                      <TableHead>ساعات العمل</TableHead>
                      <TableHead>الحالة</TableHead>
                      <TableHead>ملاحظات</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {attendanceLogs.slice(0, 30).map((log) => (
                      <TableRow key={log.id} data-testid={`row-attendance-${log.id}`}>
                        <TableCell>{formatDate(log.date)}</TableCell>
                        <TableCell>{log.checkInTime || "-"}</TableCell>
                        <TableCell>{log.checkOutTime || "-"}</TableCell>
                        <TableCell>
                          {log.totalWorkMinutes ? `${Math.floor(log.totalWorkMinutes / 60)}:${String(log.totalWorkMinutes % 60).padStart(2, '0')}` : "-"}
                        </TableCell>
                        <TableCell>
                          <Badge variant={attendanceStatusLabels[log.status]?.variant || "secondary"}>
                            {attendanceStatusLabels[log.status]?.label || log.status}
                          </Badge>
                        </TableCell>
                        <TableCell>{log.notes || "-"}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              ) : (
                <div className="text-center py-8">
                  <CalendarCheck className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد سجلات حضور</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="custodies" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>عهدة الموظف من المخزون</CardTitle>
              <CardDescription>جميع العهد المسلمة للموظف - المفتوحة والمغلقة مع تفاصيل كاملة</CardDescription>
            </CardHeader>
            <CardContent>
              {isLoadingCustodies ? (
                <div className="space-y-2">
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                </div>
              ) : custodies && custodies.length > 0 ? (
                <div className="space-y-4">
                  {custodies.map((custody) => (
                    <div 
                      key={custody.id} 
                      className={`border rounded-lg p-4 ${custody.status === "open" ? "border-primary/50 bg-primary/5" : "border-muted"}`}
                      data-testid={`card-custody-${custody.id}`}
                    >
                      <div className="flex items-start justify-between mb-3">
                        <div className="flex items-center gap-3">
                          <Package className="h-5 w-5 text-muted-foreground" />
                          <div>
                            <h4 className="font-semibold">{custody.product?.nameAr || "-"}</h4>
                            <p className="text-sm text-muted-foreground">
                              الكمية: {custody.quantity} | التكلفة: {custody.totalCost?.toFixed(2) || "0.00"} ر.س
                            </p>
                          </div>
                        </div>
                        <Badge variant={custodyStatusLabels[custody.status]?.variant || "secondary"}>
                          {custodyStatusLabels[custody.status]?.label || custody.status}
                        </Badge>
                      </div>
                      
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                        <div>
                          <span className="text-muted-foreground">تاريخ التسليم: </span>
                          <span className="font-medium">{formatDate(custody.custodyDate)}</span>
                        </div>
                        <div>
                          <span className="text-muted-foreground">تكلفة الوحدة: </span>
                          <span className="font-medium">{custody.unitCost?.toFixed(2) || "0.00"} ر.س</span>
                        </div>
                        {custody.notes && (
                          <div className="col-span-2">
                            <span className="text-muted-foreground">ملاحظات: </span>
                            <span className="font-medium">{custody.notes}</span>
                          </div>
                        )}
                      </div>

                      {custody.status !== "open" && (
                        <div className="mt-4 pt-4 border-t border-dashed">
                          <h5 className="font-medium text-sm mb-2 flex items-center gap-2">
                            <CalendarX className="h-4 w-4" />
                            تفاصيل إغلاق العهدة
                          </h5>
                          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                            <div>
                              <span className="text-muted-foreground">تاريخ الإغلاق: </span>
                              <span className="font-medium">
                                {custody.closedDate ? formatDate(custody.closedDate) : "-"}
                              </span>
                            </div>
                            <div>
                              <span className="text-muted-foreground">سبب الإغلاق: </span>
                              <span className="font-medium">{custody.closureReason || "-"}</span>
                            </div>
                            {custody.status === "partially_returned" && (
                              <div>
                                <span className="text-muted-foreground">الكمية المرتجعة: </span>
                                <span className="font-medium">{custody.returnedQuantity || 0}</span>
                              </div>
                            )}
                            <div>
                              <span className="text-muted-foreground">أغلق بواسطة: </span>
                              <span className="font-medium">
                                {custody.closedByEmployee?.fullNameAr || custody.closedByEmployee?.nameAr || "-"}
                              </span>
                            </div>
                          </div>
                          {custody.closureNotes && (
                            <div className="mt-2">
                              <span className="text-muted-foreground text-sm">ملاحظات الإغلاق: </span>
                              <p className="text-sm mt-1">{custody.closureNotes}</p>
                            </div>
                          )}
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <Package className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد عهد مسجلة</p>
                  <Link href="/employee-custodies">
                    <Button variant="outline" className="mt-4">
                      إضافة عهدة
                    </Button>
                  </Link>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="commissions" className="space-y-4">
          {/* Period Selector */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Percent className="h-5 w-5" />
                تقرير العمولات
              </CardTitle>
              <CardDescription>
                تقرير مفصل بعمولات الموظف مع تتبع الهدف الشهري
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex flex-wrap items-end gap-4 p-4 bg-muted/50 rounded-lg">
                <div className="space-y-2">
                  <Label htmlFor="commission-month">الشهر</Label>
                  <select
                    id="commission-month"
                    value={commissionMonth}
                    onChange={(e) => setCommissionMonth(parseInt(e.target.value))}
                    className="flex h-9 w-32 rounded-md border border-input bg-background px-3 py-1 text-sm shadow-sm"
                    data-testid="select-commission-month"
                  >
                    <option value={1}>يناير</option>
                    <option value={2}>فبراير</option>
                    <option value={3}>مارس</option>
                    <option value={4}>أبريل</option>
                    <option value={5}>مايو</option>
                    <option value={6}>يونيو</option>
                    <option value={7}>يوليو</option>
                    <option value={8}>أغسطس</option>
                    <option value={9}>سبتمبر</option>
                    <option value={10}>أكتوبر</option>
                    <option value={11}>نوفمبر</option>
                    <option value={12}>ديسمبر</option>
                  </select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="commission-year">السنة</Label>
                  <select
                    id="commission-year"
                    value={commissionYear}
                    onChange={(e) => setCommissionYear(parseInt(e.target.value))}
                    className="flex h-9 w-28 rounded-md border border-input bg-background px-3 py-1 text-sm shadow-sm"
                    data-testid="select-commission-year"
                  >
                    {[2024, 2025, 2026].map(year => (
                      <option key={year} value={year}>{year}</option>
                    ))}
                  </select>
                </div>
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    onClick={() => console.log("Export to Excel")}
                    data-testid="button-export-excel"
                  >
                    <FileSpreadsheet className="ml-2 h-4 w-4" />
                    تصدير Excel
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => console.log("Export to PDF")}
                    data-testid="button-export-pdf"
                  >
                    <Download className="ml-2 h-4 w-4" />
                    تصدير PDF
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          {isLoadingCommissions ? (
            <div className="space-y-2">
              <Skeleton className="h-24 w-full" />
              <Skeleton className="h-32 w-full" />
              <Skeleton className="h-64 w-full" />
            </div>
          ) : commissionReport ? (
            <>
              {/* Profile Info & Target Progress */}
              {commissionReport.profile ? (
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <TrendingUp className="h-5 w-5" />
                      تتبع الهدف الشهري
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                      <div className="p-4 rounded-lg bg-blue-50 dark:bg-blue-950">
                        <div className="text-sm text-muted-foreground">الهدف الشهري</div>
                        <div className="text-xl font-bold">{formatCurrency(commissionReport.summary.targetAmount)}</div>
                      </div>
                      <div className="p-4 rounded-lg bg-green-50 dark:bg-green-950">
                        <div className="text-sm text-muted-foreground">الإجمالي المحقق</div>
                        <div className="text-xl font-bold text-green-600">{formatCurrency(commissionReport.summary.grossTotal)}</div>
                      </div>
                      <div className="p-4 rounded-lg bg-orange-50 dark:bg-orange-950">
                        <div className="text-sm text-muted-foreground">الخصومات</div>
                        <div className="text-xl font-bold text-orange-600">-{formatCurrency(commissionReport.summary.deductionsTotal)}</div>
                      </div>
                      <div className="p-4 rounded-lg bg-purple-50 dark:bg-purple-950">
                        <div className="text-sm text-muted-foreground">الصافي</div>
                        <div className="text-xl font-bold text-purple-600">{formatCurrency(commissionReport.summary.netTotal)}</div>
                      </div>
                    </div>

                    {/* Progress Bar */}
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>نسبة التحقيق</span>
                        <span className={(commissionReport.summary.achievementRate ?? 0) >= 100 ? "text-green-600 font-bold" : ""}>
                          {(commissionReport.summary.achievementRate ?? 0).toFixed(1)}%
                        </span>
                      </div>
                      <div className="w-full bg-muted rounded-full h-4">
                        <div
                          className={`h-4 rounded-full transition-all ${
                            (commissionReport.summary.achievementRate ?? 0) >= 100 
                              ? "bg-green-600" 
                              : (commissionReport.summary.achievementRate ?? 0) >= 75 
                                ? "bg-yellow-500" 
                                : "bg-red-500"
                          }`}
                          style={{ width: `${Math.min(commissionReport.summary.achievementRate ?? 0, 100)}%` }}
                        />
                      </div>
                    </div>

                    {/* Commission Calculation */}
                    <div className="flex items-center gap-4 p-4 bg-muted/50 rounded-lg">
                      <div>
                        <span className="text-muted-foreground ml-2">نسبة العمولة:</span>
                        <span className="font-bold">{commissionReport.summary.commissionRate}%</span>
                      </div>
                      <div className="border-r h-6 mx-2" />
                      <div>
                        <span className="text-muted-foreground ml-2">مبلغ العمولة المستحق:</span>
                        <span className={`font-bold text-xl ${(commissionReport.summary.achievementRate ?? 0) >= 100 ? "text-green-600" : "text-muted-foreground"}`}>
                          {formatCurrency(commissionReport.summary.commissionAmount)}
                        </span>
                      </div>
                      {(commissionReport.summary.achievementRate ?? 0) < 100 && (
                        <Badge variant="outline" className="mr-2">
                          لم يتحقق الهدف
                        </Badge>
                      )}
                    </div>

                    {/* Target Sources */}
                    <div className="flex flex-wrap gap-2">
                      <span className="text-sm text-muted-foreground ml-2">مصادر الهدف المفعلة:</span>
                      {commissionReport.profile.targetSources.filter(s => s.isEnabled).map(source => (
                        <Badge key={source.id} variant="secondary">
                          {source.sourceType === 'executed_services' && 'الخدمات المنفذة'}
                          {source.sourceType === 'created_orders' && 'الطلبات المنشأة'}
                          {source.sourceType === 'cafeteria_sales' && 'مبيعات الكافتيريا'}
                          {source.sourceType === 'product_sales' && 'مبيعات المنتجات'}
                        </Badge>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              ) : (
                <Card>
                  <CardContent className="py-8">
                    <div className="text-center">
                      <AlertCircle className="h-12 w-12 mx-auto text-yellow-500 mb-4" />
                      <h3 className="text-lg font-semibold mb-2">لم يتم إعداد ملف العمولات</h3>
                      <p className="text-muted-foreground mb-4">
                        يرجى إنشاء ملف عمولات لهذا الموظف لتتبع الأهداف والعمولات
                      </p>
                      <Link href="/commissions">
                        <Button variant="default" data-testid="button-setup-commission">
                          إعداد ملف العمولات
                        </Button>
                      </Link>
                    </div>
                  </CardContent>
                </Card>
              )}

              {/* Summary Breakdown */}
              <Card>
                <CardHeader>
                  <CardTitle>تفصيل المبالغ</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid gap-4">
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                      <div className="p-3 border rounded-lg">
                        <div className="text-muted-foreground">خدمات منفذة</div>
                        <div className="font-bold">{formatCurrency(commissionReport.summary.executedServicesTotal)}</div>
                        <div className="text-xs text-orange-600">-{formatCurrency(commissionReport.summary.inventoryDeductionTotal)} مواد</div>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-muted-foreground">طلبات منشأة</div>
                        <div className="font-bold">{formatCurrency(commissionReport.summary.createdOrdersTotal)}</div>
                        <div className="text-xs text-red-600">-{formatCurrency(commissionReport.summary.ordersReturnsTotal)} مرتجعات</div>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-muted-foreground">كافتيريا</div>
                        <div className="font-bold">{formatCurrency(commissionReport.summary.cafeteriaSalesTotal)}</div>
                        <div className="text-xs text-red-600">-{formatCurrency(commissionReport.summary.cafeteriaReturnsTotal)} مرتجعات</div>
                      </div>
                      <div className="p-3 border rounded-lg">
                        <div className="text-muted-foreground">مبيعات منتجات</div>
                        <div className="font-bold">{formatCurrency(commissionReport.summary.productSalesTotal)}</div>
                        <div className="text-xs text-red-600">-{formatCurrency(commissionReport.summary.productReturnsTotal)} مرتجعات</div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Section Selector */}
              <Card>
                <CardHeader>
                  <div className="flex flex-wrap items-center gap-2">
                    <Button
                      variant={activeCommissionSection === 'services' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setActiveCommissionSection('services')}
                      data-testid="button-section-services"
                    >
                      الخدمات المنفذة ({commissionReport.executedServices.length})
                    </Button>
                    <Button
                      variant={activeCommissionSection === 'orders' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setActiveCommissionSection('orders')}
                      data-testid="button-section-orders"
                    >
                      الطلبات المنشأة ({commissionReport.createdOrders.length})
                    </Button>
                    <Button
                      variant={activeCommissionSection === 'cafeteria' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setActiveCommissionSection('cafeteria')}
                      data-testid="button-section-cafeteria"
                    >
                      الكافتيريا ({commissionReport.cafeteriaSales.length})
                    </Button>
                    <Button
                      variant={activeCommissionSection === 'products' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setActiveCommissionSection('products')}
                      data-testid="button-section-products"
                    >
                      مبيعات المنتجات ({commissionReport.productSales.length})
                    </Button>
                  </div>
                </CardHeader>
                <CardContent>
                  {activeCommissionSection === 'services' && (
                    commissionReport.executedServices.length > 0 ? (
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead>التاريخ</TableHead>
                            <TableHead>النوع</TableHead>
                            <TableHead>رقم المرجع</TableHead>
                            <TableHead>الخدمة</TableHead>
                            <TableHead>الكمية</TableHead>
                            <TableHead>المبلغ</TableHead>
                            <TableHead>خصم المواد</TableHead>
                            <TableHead>الصافي</TableHead>
                            <TableHead>الحالة</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {commissionReport.executedServices.map((service, idx) => (
                            <TableRow key={`service-${idx}`} data-testid={`row-service-${service.id}`}>
                              <TableCell>{formatDate(service.date)}</TableCell>
                              <TableCell>
                                <Badge variant={service.type === 'return' ? 'destructive' : 'outline'}>
                                  {service.type === 'order' && 'طلب'}
                                  {service.type === 'appointment' && 'موعد'}
                                  {service.type === 'return' && 'مرتجع'}
                                </Badge>
                              </TableCell>
                              <TableCell>{service.referenceNumber}</TableCell>
                              <TableCell>{service.serviceName}</TableCell>
                              <TableCell>{service.quantity}</TableCell>
                              <TableCell className={service.type === 'return' ? 'text-red-600' : ''}>
                                {formatCurrency(service.amount)}
                              </TableCell>
                              <TableCell className="text-orange-600">
                                -{formatCurrency(service.inventoryDeduction)}
                              </TableCell>
                              <TableCell className={service.type === 'return' ? 'text-red-600 font-bold' : 'font-bold'}>
                                {formatCurrency(service.netAmount)}
                              </TableCell>
                              <TableCell>
                                <Badge variant={service.status === 'completed' ? 'default' : 'secondary'}>
                                  {service.status === 'completed' && 'مكتمل'}
                                  {service.status === 'returned' && 'مرتجع'}
                                  {service.status !== 'completed' && service.status !== 'returned' && service.status}
                                </Badge>
                              </TableCell>
                            </TableRow>
                          ))}
                        </TableBody>
                      </Table>
                    ) : (
                      <div className="text-center py-8 text-muted-foreground">
                        لا توجد خدمات منفذة في هذه الفترة
                      </div>
                    )
                  )}

                  {activeCommissionSection === 'orders' && (
                    commissionReport.createdOrders.length > 0 ? (
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead>التاريخ</TableHead>
                            <TableHead>النوع</TableHead>
                            <TableHead>رقم الطلب</TableHead>
                            <TableHead>الإجمالي</TableHead>
                            <TableHead>المرتجعات</TableHead>
                            <TableHead>الصافي</TableHead>
                            <TableHead>الحالة</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {commissionReport.createdOrders.map((order, idx) => (
                            <TableRow key={`order-${idx}`} data-testid={`row-order-${order.id}`}>
                              <TableCell>{formatDate(order.date)}</TableCell>
                              <TableCell>
                                <Badge variant="outline">
                                  {order.type === 'order' ? 'طلب' : 'موعد'}
                                </Badge>
                              </TableCell>
                              <TableCell>{order.referenceNumber}</TableCell>
                              <TableCell>{formatCurrency(order.totalAmount)}</TableCell>
                              <TableCell className="text-red-600">-{formatCurrency(order.returnsAmount)}</TableCell>
                              <TableCell className="font-bold">{formatCurrency(order.netAmount)}</TableCell>
                              <TableCell>
                                <Badge variant={order.status === 'completed' ? 'default' : 'secondary'}>
                                  {order.status}
                                </Badge>
                              </TableCell>
                            </TableRow>
                          ))}
                        </TableBody>
                      </Table>
                    ) : (
                      <div className="text-center py-8 text-muted-foreground">
                        لا توجد طلبات منشأة في هذه الفترة
                      </div>
                    )
                  )}

                  {activeCommissionSection === 'cafeteria' && (
                    commissionReport.cafeteriaSales.length > 0 ? (
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead>التاريخ</TableHead>
                            <TableHead>رقم الطلب</TableHead>
                            <TableHead>المبلغ</TableHead>
                            <TableHead>الحالة</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {commissionReport.cafeteriaSales.map((sale, idx) => (
                            <TableRow key={`cafe-${idx}`} data-testid={`row-cafe-${sale.orderId}`}>
                              <TableCell>{formatDate(sale.date)}</TableCell>
                              <TableCell>{sale.orderNumber}</TableCell>
                              <TableCell>{formatCurrency(sale.totalAmount)}</TableCell>
                              <TableCell>
                                <Badge variant={sale.status === 'completed' ? 'default' : 'secondary'}>
                                  {sale.status}
                                </Badge>
                              </TableCell>
                            </TableRow>
                          ))}
                        </TableBody>
                      </Table>
                    ) : (
                      <div className="text-center py-8 text-muted-foreground">
                        لا توجد مبيعات كافتيريا في هذه الفترة
                      </div>
                    )
                  )}

                  {activeCommissionSection === 'products' && (
                    commissionReport.productSales.length > 0 ? (
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead>التاريخ</TableHead>
                            <TableHead>رقم الطلب</TableHead>
                            <TableHead>المنتج</TableHead>
                            <TableHead>الكمية</TableHead>
                            <TableHead>المبلغ</TableHead>
                            <TableHead>المرتجع</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {commissionReport.productSales.map((sale, idx) => (
                            <TableRow key={`prod-${idx}`} data-testid={`row-product-${sale.orderId}`}>
                              <TableCell>{formatDate(sale.date)}</TableCell>
                              <TableCell>{sale.orderNumber}</TableCell>
                              <TableCell>{sale.productName}</TableCell>
                              <TableCell>{sale.quantity}</TableCell>
                              <TableCell>{formatCurrency(sale.amount)}</TableCell>
                              <TableCell className="text-red-600">-{formatCurrency(sale.returnedAmount)}</TableCell>
                            </TableRow>
                          ))}
                        </TableBody>
                      </Table>
                    ) : (
                      <div className="text-center py-8 text-muted-foreground">
                        لا توجد مبيعات منتجات في هذه الفترة
                      </div>
                    )
                  )}
                </CardContent>
              </Card>
            </>
          ) : (
            <Card>
              <CardContent className="py-8">
                <div className="text-center">
                  <Percent className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد بيانات عمولات للفترة المحددة</p>
                </div>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="disbursements" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Coins className="h-5 w-5" />
                السلف والصرف النقدي
              </CardTitle>
              <CardDescription>
                سجل السلف والهدايا والخصومات للموظف
              </CardDescription>
            </CardHeader>
            <CardContent>
              {isLoadingDisbursements ? (
                <div className="space-y-2">
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                  <Skeleton className="h-10 w-full" />
                </div>
              ) : cashDisbursements && cashDisbursements.length > 0 ? (
                <div className="space-y-6">
                  <div className="grid grid-cols-3 gap-4 mb-4">
                    <div className="p-4 rounded-lg bg-blue-50 dark:bg-blue-950">
                      <div className="text-sm text-muted-foreground">إجمالي السلف</div>
                      <div className="text-xl font-bold text-blue-600">
                        {formatCurrency(
                          cashDisbursements
                            .filter(d => d.disbursementType === "advance")
                            .reduce((sum, d) => sum + d.amount, 0)
                        )}
                      </div>
                    </div>
                    <div className="p-4 rounded-lg bg-green-50 dark:bg-green-950">
                      <div className="text-sm text-muted-foreground">إجمالي الهدايا</div>
                      <div className="text-xl font-bold text-green-600">
                        {formatCurrency(
                          cashDisbursements
                            .filter(d => d.disbursementType === "gift")
                            .reduce((sum, d) => sum + d.amount, 0)
                        )}
                      </div>
                    </div>
                    <div className="p-4 rounded-lg bg-red-50 dark:bg-red-950">
                      <div className="text-sm text-muted-foreground">إجمالي الخصومات</div>
                      <div className="text-xl font-bold text-red-600">
                        {formatCurrency(
                          cashDisbursements
                            .filter(d => d.disbursementType === "deduction")
                            .reduce((sum, d) => sum + d.amount, 0)
                        )}
                      </div>
                    </div>
                  </div>

                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>رقم الصرف</TableHead>
                        <TableHead>النوع</TableHead>
                        <TableHead>المبلغ</TableHead>
                        <TableHead>طريقة الدفع</TableHead>
                        <TableHead>التاريخ</TableHead>
                        <TableHead>الأقساط</TableHead>
                        <TableHead>ملاحظات</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {cashDisbursements.map((disbursement) => (
                        <TableRow key={disbursement.id} data-testid={`row-disbursement-${disbursement.id}`}>
                          <TableCell className="font-medium">{disbursement.disbursementNumber}</TableCell>
                          <TableCell>
                            <Badge variant={disbursementTypeLabels[disbursement.disbursementType]?.variant || "secondary"}>
                              {disbursementTypeLabels[disbursement.disbursementType]?.label || disbursement.disbursementType}
                            </Badge>
                          </TableCell>
                          <TableCell className="font-semibold">{formatCurrency(disbursement.amount)}</TableCell>
                          <TableCell>{disbursement.paymentMethod?.nameAr || "-"}</TableCell>
                          <TableCell>{formatDate(disbursement.disbursementDate)}</TableCell>
                          <TableCell>
                            {disbursement.isInstallment && disbursement.installmentMonths ? (
                              <Badge variant="outline">
                                {disbursement.installmentMonths} شهر
                              </Badge>
                            ) : "-"}
                          </TableCell>
                          <TableCell>{disbursement.notes || "-"}</TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>

                  {cashDisbursements.some(d => d.isInstallment && d.installments && d.installments.length > 0) && (
                    <div className="mt-6">
                      <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                        <Calendar className="h-5 w-5" />
                        جدول الأقساط
                      </h3>
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead>رقم السلفة</TableHead>
                            <TableHead>رقم القسط</TableHead>
                            <TableHead>المبلغ</TableHead>
                            <TableHead>تاريخ الاستحقاق</TableHead>
                            <TableHead>الحالة</TableHead>
                            <TableHead>تاريخ الخصم</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {cashDisbursements
                            .filter(d => d.isInstallment && d.installments && d.installments.length > 0)
                            .flatMap(d => 
                              d.installments!.map(inst => ({
                                ...inst,
                                disbursementNumber: d.disbursementNumber
                              }))
                            )
                            .map((inst) => (
                              <TableRow key={`inst-${inst.id}`} data-testid={`row-installment-${inst.id}`}>
                                <TableCell>{inst.disbursementNumber}</TableCell>
                                <TableCell>{inst.installmentNumber}</TableCell>
                                <TableCell>{formatCurrency(inst.amount)}</TableCell>
                                <TableCell>{formatDate(inst.dueDate)}</TableCell>
                                <TableCell>
                                  <Badge variant={installmentStatusLabels[inst.status]?.variant || "outline"}>
                                    {installmentStatusLabels[inst.status]?.label || inst.status}
                                  </Badge>
                                </TableCell>
                                <TableCell>{inst.deductedAt ? formatDate(inst.deductedAt) : "-"}</TableCell>
                              </TableRow>
                            ))}
                        </TableBody>
                      </Table>
                    </div>
                  )}
                </div>
              ) : (
                <div className="text-center py-8">
                  <Coins className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">لا توجد سجلات صرف نقدي</p>
                  <Link href="/cash-disbursements">
                    <Button variant="outline" className="mt-4">
                      إضافة سلفة أو صرف
                    </Button>
                  </Link>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
