import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Users, AlertTriangle, Globe, FileText, Calendar, Search } from "lucide-react";
import { format, differenceInDays } from "date-fns";
import { useState } from "react";
import type { Employee } from "@shared/schema";

export default function EmployeeReportsPage() {
  const [searchTerm, setSearchTerm] = useState("");
  const [expiryDaysFilter, setExpiryDaysFilter] = useState("30");
  
  const { data: employees = [], isLoading } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });
  
  const { data: nationalities = [] } = useQuery<{ id: number; nameAr: string; code: string }[]>({
    queryKey: ["/api/nationalities"],
  });
  
  const activeEmployees = employees.filter(e => e.isActive);
  
  const getEmployeeName = (e: Employee) => e.fullNameAr || e.nameAr || "غير محدد";
  
  const filteredEmployees = activeEmployees.filter(e => 
    getEmployeeName(e).toLowerCase().includes(searchTerm.toLowerCase()) ||
    e.iqamaNumber?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    e.passportNumber?.toLowerCase().includes(searchTerm.toLowerCase())
  );
  
  const employeesByNationality = nationalities.map(nat => ({
    nationality: nat,
    employees: filteredEmployees.filter(e => e.nationalityId === nat.id || e.nationalityNameAr === nat.nameAr),
  })).filter(g => g.employees.length > 0);
  
  const unknownNatEmployees = filteredEmployees.filter(e => !e.nationalityId && !e.nationalityNameAr);
  
  const daysThreshold = parseInt(expiryDaysFilter);
  const today = new Date();
  
  const getExpiryStatus = (expiryDate: string | null | undefined) => {
    if (!expiryDate) return null;
    const expiry = new Date(expiryDate);
    const daysRemaining = differenceInDays(expiry, today);
    
    if (daysRemaining < 0) return { status: "expired", label: "منتهي", daysRemaining, variant: "destructive" as const, className: "" };
    if (daysRemaining <= 7) return { status: "critical", label: `${daysRemaining} يوم`, daysRemaining, variant: "destructive" as const, className: "" };
    if (daysRemaining <= 30) return { status: "warning", label: `${daysRemaining} يوم`, daysRemaining, variant: "secondary" as const, className: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200" };
    if (daysRemaining <= daysThreshold) return { status: "soon", label: `${daysRemaining} يوم`, daysRemaining, variant: "secondary" as const, className: "" };
    return { status: "ok", label: `${daysRemaining} يوم`, daysRemaining, variant: "outline" as const, className: "" };
  };
  
  const expiringDocuments = filteredEmployees.flatMap(e => {
    const docs = [];
    
    const iqamaStatus = getExpiryStatus(e.iqamaExpiryDate);
    if (iqamaStatus && iqamaStatus.daysRemaining <= daysThreshold) {
      docs.push({
        employee: e,
        documentType: "الإقامة",
        documentNumber: e.iqamaNumber,
        expiryDate: e.iqamaExpiryDate,
        ...iqamaStatus,
      });
    }
    
    const passportStatus = getExpiryStatus(e.passportExpiryDate);
    if (passportStatus && passportStatus.daysRemaining <= daysThreshold) {
      docs.push({
        employee: e,
        documentType: "جواز السفر",
        documentNumber: e.passportNumber,
        expiryDate: e.passportExpiryDate,
        ...passportStatus,
      });
    }
    
    const insuranceStatus = getExpiryStatus(e.insuranceExpiryDate);
    if (insuranceStatus && insuranceStatus.daysRemaining <= daysThreshold) {
      docs.push({
        employee: e,
        documentType: "التأمين الطبي",
        documentNumber: e.insuranceNumber,
        expiryDate: e.insuranceExpiryDate,
        ...insuranceStatus,
      });
    }
    
    const insuranceCardStatus = getExpiryStatus(e.insuranceCardExpiryDate);
    if (insuranceCardStatus && insuranceCardStatus.daysRemaining <= daysThreshold) {
      docs.push({
        employee: e,
        documentType: "بطاقة التأمين",
        documentNumber: e.insuranceNumber,
        expiryDate: e.insuranceCardExpiryDate,
        ...insuranceCardStatus,
      });
    }
    
    return docs;
  }).sort((a, b) => a.daysRemaining - b.daysRemaining);
  
  const expiredCount = expiringDocuments.filter(d => d.status === "expired").length;
  const criticalCount = expiringDocuments.filter(d => d.status === "critical").length;
  const warningCount = expiringDocuments.filter(d => d.status === "warning").length;
  
  if (isLoading) {
    return (
      <div className="flex flex-col gap-4 p-6">
        <Skeleton className="h-8 w-48" />
        <div className="grid gap-4 md:grid-cols-4">
          {[1, 2, 3, 4].map(i => (
            <Skeleton key={i} className="h-24" />
          ))}
        </div>
        <Skeleton className="h-96" />
      </div>
    );
  }
  
  return (
    <div className="flex flex-col gap-6 p-6" dir="rtl">
      <div className="flex flex-wrap items-center justify-between gap-4">
        <h1 className="text-2xl font-bold">تقارير الموظفين</h1>
        <div className="flex items-center gap-2">
          <div className="relative">
            <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
            <Input
              placeholder="بحث بالاسم أو رقم الإقامة..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pr-10 w-64"
              data-testid="input-search-employees"
            />
          </div>
        </div>
      </div>
      
      <div className="grid gap-4 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي الموظفين</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-employees">{activeEmployees.length}</div>
            <p className="text-xs text-muted-foreground">موظف نشط</p>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">الجنسيات</CardTitle>
            <Globe className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-nationalities-count">{employeesByNationality.length}</div>
            <p className="text-xs text-muted-foreground">جنسية مختلفة</p>
          </CardContent>
        </Card>
        
        <Card className={expiredCount > 0 ? "border-destructive" : ""}>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">مستندات منتهية</CardTitle>
            <AlertTriangle className="h-4 w-4 text-destructive" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-destructive" data-testid="text-expired-docs">{expiredCount}</div>
            <p className="text-xs text-muted-foreground">تحتاج تجديد فوري</p>
          </CardContent>
        </Card>
        
        <Card className={criticalCount > 0 ? "border-yellow-500" : ""}>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">تنتهي قريباً</CardTitle>
            <Calendar className="h-4 w-4 text-yellow-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600" data-testid="text-expiring-soon">{criticalCount + warningCount}</div>
            <p className="text-xs text-muted-foreground">خلال 30 يوم</p>
          </CardContent>
        </Card>
      </div>
      
      <Tabs defaultValue="expiring" className="flex-1">
        <TabsList className="grid w-full grid-cols-2 max-w-md">
          <TabsTrigger value="expiring" className="flex items-center gap-2" data-testid="tab-expiring-docs">
            <FileText className="h-4 w-4" />
            المستندات المنتهية
          </TabsTrigger>
          <TabsTrigger value="nationality" className="flex items-center gap-2" data-testid="tab-by-nationality">
            <Globe className="h-4 w-4" />
            حسب الجنسية
          </TabsTrigger>
        </TabsList>
        
        <TabsContent value="expiring" className="mt-4">
          <Card>
            <CardHeader>
              <div className="flex flex-wrap items-center justify-between gap-4">
                <CardTitle>المستندات التي ستنتهي صلاحيتها</CardTitle>
                <Select value={expiryDaysFilter} onValueChange={setExpiryDaysFilter}>
                  <SelectTrigger className="w-40" data-testid="select-expiry-days">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="7">خلال 7 أيام</SelectItem>
                    <SelectItem value="15">خلال 15 يوم</SelectItem>
                    <SelectItem value="30">خلال 30 يوم</SelectItem>
                    <SelectItem value="60">خلال 60 يوم</SelectItem>
                    <SelectItem value="90">خلال 90 يوم</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardHeader>
            <CardContent>
              {expiringDocuments.length === 0 ? (
                <div className="flex flex-col items-center justify-center py-12 text-center">
                  <FileText className="h-12 w-12 text-muted-foreground mb-4" />
                  <p className="text-lg font-medium">لا توجد مستندات تنتهي</p>
                  <p className="text-muted-foreground">جميع مستندات الموظفين سارية</p>
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>الموظف</TableHead>
                      <TableHead>نوع المستند</TableHead>
                      <TableHead>رقم المستند</TableHead>
                      <TableHead>تاريخ الانتهاء</TableHead>
                      <TableHead>الحالة</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {expiringDocuments.map((doc, index) => (
                      <TableRow key={`${doc.employee.id}-${doc.documentType}-${index}`} data-testid={`row-expiring-doc-${index}`}>
                        <TableCell className="font-medium">
                          {getEmployeeName(doc.employee)}
                        </TableCell>
                        <TableCell>{doc.documentType}</TableCell>
                        <TableCell className="font-mono">{doc.documentNumber || "-"}</TableCell>
                        <TableCell>
                          {doc.expiryDate ? format(new Date(doc.expiryDate), "yyyy-MM-dd") : "-"}
                        </TableCell>
                        <TableCell>
                          <Badge variant={doc.variant} className={doc.className}>
                            {doc.label}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="nationality" className="mt-4">
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
            {employeesByNationality.map(group => (
              <Card key={group.nationality.id} data-testid={`card-nationality-${group.nationality.id}`}>
                <CardHeader className="pb-2">
                  <div className="flex items-center justify-between gap-2">
                    <CardTitle className="text-base">{group.nationality.nameAr}</CardTitle>
                    <Badge variant="secondary">{group.employees.length}</Badge>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="flex flex-col gap-2">
                    {group.employees.slice(0, 5).map(emp => (
                      <div key={emp.id} className="flex items-center justify-between text-sm">
                        <span>{getEmployeeName(emp)}</span>
                        <span className="text-muted-foreground">{emp.jobTitleAr || "-"}</span>
                      </div>
                    ))}
                    {group.employees.length > 5 && (
                      <p className="text-xs text-muted-foreground text-center">
                        و {group.employees.length - 5} آخرين...
                      </p>
                    )}
                  </div>
                </CardContent>
              </Card>
            ))}
            
            {unknownNatEmployees.length > 0 && (
              <Card className="border-dashed" data-testid="card-nationality-unknown">
                <CardHeader className="pb-2">
                  <div className="flex items-center justify-between gap-2">
                    <CardTitle className="text-base text-muted-foreground">غير محدد</CardTitle>
                    <Badge variant="outline">{unknownNatEmployees.length}</Badge>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="flex flex-col gap-2">
                    {unknownNatEmployees.slice(0, 5).map(emp => (
                      <div key={emp.id} className="flex items-center justify-between text-sm">
                        <span className="text-muted-foreground">{getEmployeeName(emp)}</span>
                        <span className="text-muted-foreground">{emp.jobTitleAr || "-"}</span>
                      </div>
                    ))}
                    {unknownNatEmployees.length > 5 && (
                      <p className="text-xs text-muted-foreground text-center">
                        و {unknownNatEmployees.length - 5} آخرين...
                      </p>
                    )}
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}
