import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ClipboardList, Plus, Pencil, Trash2, Clock, CheckCircle2, XCircle, AlertCircle, PlayCircle } from "lucide-react";
import { format } from "date-fns";

interface Employee {
  id: number;
  nameAr: string | null;
  fullNameAr: string | null;
}

interface EmployeeTask {
  id: number;
  titleAr: string;
  descriptionAr: string | null;
  assignedToEmployeeId: number;
  assignedByEmployeeId: number | null;
  status: string;
  priority: string;
  dueDate: string | null;
  completedAt: string | null;
  assignedToEmployee?: Employee;
  assignedByEmployee?: Employee;
  createdAt: string;
  updatedAt: string;
}

interface TaskFormData {
  titleAr: string;
  descriptionAr: string;
  assignedToEmployeeId: number | null;
  priority: string;
  dueDate: string;
  status: string;
}

const statusOptions = [
  { value: "new", label: "جديدة", icon: Clock, color: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200" },
  { value: "in_progress", label: "قيد التنفيذ", icon: PlayCircle, color: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200" },
  { value: "completed", label: "مكتملة", icon: CheckCircle2, color: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" },
  { value: "canceled", label: "ملغاة", icon: XCircle, color: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200" },
  { value: "overdue", label: "متأخرة", icon: AlertCircle, color: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200" },
];

const priorityOptions = [
  { value: "low", label: "منخفضة", color: "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200" },
  { value: "medium", label: "متوسطة", color: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200" },
  { value: "high", label: "عالية", color: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200" },
  { value: "urgent", label: "عاجلة", color: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200" },
];

export default function EmployeeTasksPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingTask, setEditingTask] = useState<EmployeeTask | null>(null);
  const [filterStatus, setFilterStatus] = useState<string>("_all");
  const [formData, setFormData] = useState<TaskFormData>({
    titleAr: "",
    descriptionAr: "",
    assignedToEmployeeId: null,
    priority: "medium",
    dueDate: "",
    status: "new",
  });
  const { toast } = useToast();

  const { data: tasks, isLoading } = useQuery<EmployeeTask[]>({
    queryKey: ["/api/employee-tasks"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const createMutation = useMutation({
    mutationFn: (data: TaskFormData) =>
      apiRequest("POST", "/api/employee-tasks", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employee-tasks"] });
      toast({ title: "تم إنشاء المهمة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء المهمة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<TaskFormData> }) =>
      apiRequest("PUT", `/api/employee-tasks/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employee-tasks"] });
      toast({ title: "تم تحديث المهمة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث المهمة",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/employee-tasks/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employee-tasks"] });
      toast({ title: "تم حذف المهمة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف المهمة",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      titleAr: "",
      descriptionAr: "",
      assignedToEmployeeId: null,
      priority: "medium",
      dueDate: "",
      status: "new",
    });
    setEditingTask(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (task?: EmployeeTask) => {
    if (task) {
      setEditingTask(task);
      setFormData({
        titleAr: task.titleAr,
        descriptionAr: task.descriptionAr || "",
        assignedToEmployeeId: task.assignedToEmployeeId,
        priority: task.priority,
        dueDate: task.dueDate || "",
        status: task.status,
      });
    } else {
      resetForm();
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.titleAr.trim()) {
      toast({ title: "عنوان المهمة مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.assignedToEmployeeId) {
      toast({ title: "يجب اختيار الموظف المكلف", variant: "destructive" });
      return;
    }

    if (editingTask) {
      updateMutation.mutate({ id: editingTask.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getStatusBadge = (status: string) => {
    const statusConfig = statusOptions.find(s => s.value === status);
    if (!statusConfig) return <Badge>{status}</Badge>;
    const Icon = statusConfig.icon;
    return (
      <Badge className={statusConfig.color}>
        <Icon className="h-3 w-3 ml-1" />
        {statusConfig.label}
      </Badge>
    );
  };

  const getPriorityBadge = (priority: string) => {
    const priorityConfig = priorityOptions.find(p => p.value === priority);
    if (!priorityConfig) return <Badge variant="outline">{priority}</Badge>;
    return <Badge className={priorityConfig.color}>{priorityConfig.label}</Badge>;
  };

  const getEmployeeName = (emp?: Employee) => emp?.fullNameAr || emp?.nameAr || "-";

  const filteredTasks = tasks?.filter(task => {
    if (filterStatus === "_all") return true;
    return task.status === filterStatus;
  }) || [];

  const taskStats = {
    total: tasks?.length || 0,
    new: tasks?.filter(t => t.status === "new").length || 0,
    inProgress: tasks?.filter(t => t.status === "in_progress").length || 0,
    completed: tasks?.filter(t => t.status === "completed").length || 0,
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">مهام الموظفين</h1>
          <p className="text-muted-foreground">إدارة وتتبع المهام المسندة للموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-task">
          <Plus className="h-4 w-4 ml-2" />
          إضافة مهمة جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <ClipboardList className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المهام</p>
                <p className="text-xl font-bold" data-testid="text-total-tasks">
                  {taskStats.total}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <Clock className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">مهام جديدة</p>
                <p className="text-xl font-bold" data-testid="text-new-tasks">
                  {taskStats.new}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">
                <PlayCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">قيد التنفيذ</p>
                <p className="text-xl font-bold" data-testid="text-in-progress-tasks">
                  {taskStats.inProgress}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">مكتملة</p>
                <p className="text-xl font-bold" data-testid="text-completed-tasks">
                  {taskStats.completed}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between gap-4 flex-wrap">
            <CardTitle className="flex items-center gap-2">
              <ClipboardList className="h-5 w-5" />
              قائمة المهام
            </CardTitle>
            <Select value={filterStatus} onValueChange={setFilterStatus}>
              <SelectTrigger className="w-40" data-testid="select-filter-status">
                <SelectValue placeholder="الحالة" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="_all">جميع الحالات</SelectItem>
                {statusOptions.map((status) => (
                  <SelectItem key={status.value} value={status.value}>
                    {status.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">عنوان المهمة</TableHead>
                  <TableHead className="text-right">الموظف المكلف</TableHead>
                  <TableHead className="text-right">الأولوية</TableHead>
                  <TableHead className="text-right">تاريخ الاستحقاق</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredTasks.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                      لا توجد مهام مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  filteredTasks.map((task) => (
                    <TableRow key={task.id} data-testid={`row-task-${task.id}`}>
                      <TableCell className="font-medium">
                        <div>
                          <p>{task.titleAr}</p>
                          {task.descriptionAr && (
                            <p className="text-xs text-muted-foreground truncate max-w-xs">
                              {task.descriptionAr}
                            </p>
                          )}
                        </div>
                      </TableCell>
                      <TableCell>{getEmployeeName(task.assignedToEmployee)}</TableCell>
                      <TableCell>{getPriorityBadge(task.priority)}</TableCell>
                      <TableCell>
                        {task.dueDate ? format(new Date(task.dueDate), "dd/MM/yyyy") : "-"}
                      </TableCell>
                      <TableCell>{getStatusBadge(task.status)}</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(task)}
                            data-testid={`button-edit-task-${task.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(task.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-task-${task.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingTask ? "تعديل المهمة" : "إضافة مهمة جديدة"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="titleAr">عنوان المهمة <span className="text-destructive">*</span></Label>
              <Input
                id="titleAr"
                value={formData.titleAr}
                onChange={(e) => setFormData({ ...formData, titleAr: e.target.value })}
                placeholder="عنوان المهمة"
                data-testid="input-task-title"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="descriptionAr">وصف المهمة</Label>
              <Textarea
                id="descriptionAr"
                value={formData.descriptionAr}
                onChange={(e) => setFormData({ ...formData, descriptionAr: e.target.value })}
                placeholder="تفاصيل المهمة"
                rows={3}
                data-testid="input-task-description"
              />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="assignedToEmployeeId">الموظف المكلف <span className="text-destructive">*</span></Label>
                <Select
                  value={formData.assignedToEmployeeId?.toString() || "_none"}
                  onValueChange={(value) => setFormData({ ...formData, assignedToEmployeeId: value === "_none" ? null : parseInt(value) })}
                >
                  <SelectTrigger data-testid="select-task-employee">
                    <SelectValue placeholder="اختر الموظف" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="_none">لم يحدد</SelectItem>
                    {employees?.map((emp) => (
                      <SelectItem key={emp.id} value={emp.id.toString()}>
                        {emp.fullNameAr || emp.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="priority">الأولوية</Label>
                <Select
                  value={formData.priority}
                  onValueChange={(value) => setFormData({ ...formData, priority: value })}
                >
                  <SelectTrigger data-testid="select-task-priority">
                    <SelectValue placeholder="اختر الأولوية" />
                  </SelectTrigger>
                  <SelectContent>
                    {priorityOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="dueDate">تاريخ الاستحقاق</Label>
                <Input
                  id="dueDate"
                  type="date"
                  value={formData.dueDate}
                  onChange={(e) => setFormData({ ...formData, dueDate: e.target.value })}
                  dir="ltr"
                  data-testid="input-task-due-date"
                />
              </div>
              {editingTask && (
                <div className="space-y-2">
                  <Label htmlFor="status">الحالة</Label>
                  <Select
                    value={formData.status}
                    onValueChange={(value) => setFormData({ ...formData, status: value })}
                  >
                    <SelectTrigger data-testid="select-task-status">
                      <SelectValue placeholder="اختر الحالة" />
                    </SelectTrigger>
                    <SelectContent>
                      {statusOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              )}
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-task"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-task"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
