import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Users, Plus, Pencil, Trash2, CheckCircle2, XCircle, Search, Phone, Mail, Calendar, CreditCard, Eye } from "lucide-react";
import { format } from "date-fns";
import { arSA } from "date-fns/locale";

interface Role {
  id: number;
  nameAr: string;
  code: string;
}

interface Employee {
  id: number;
  nameAr: string | null;
  code: string | null;
  phone: string | null;
  fullNameAr: string | null;
  phoneCountryCode: string | null;
  phoneNumber: string | null;
  email: string | null;
  fingerprintEmployeeNo: string | null;
  profileImageUrl: string | null;
  jobTitleAr: string | null;
  roleId: number | null;
  cityNameAr: string | null;
  branchId: number | null;
  nationalityNameAr: string | null;
  birthDate: string | null;
  hireDate: string | null;
  iqamaNumber: string | null;
  iqamaExpiryDate: string | null;
  insuranceCompany: string | null;
  insuranceNumber: string | null;
  insuranceExpiryDate: string | null;
  insuranceCardExpiryDate: string | null;
  healthNumber: string | null;
  passportNumber: string | null;
  passportExpiryDate: string | null;
  annualLeaveDays: number | null;
  weeklyOffDay: string | null;
  isActive: boolean;
  role?: Role;
  createdAt: string;
  updatedAt: string;
}

interface EmployeeFormData {
  fullNameAr: string;
  phoneCountryCode: string;
  phoneNumber: string;
  email: string;
  jobTitleAr: string;
  roleId: number | null;
  nationalityNameAr: string;
  cityNameAr: string;
  birthDate: string;
  hireDate: string;
  iqamaNumber: string;
  iqamaExpiryDate: string;
  passportNumber: string;
  passportExpiryDate: string;
  annualLeaveDays: number;
  weeklyOffDay: string;
  isActive: boolean;
}

const weeklyOffDayOptions = [
  { value: "friday", label: "الجمعة" },
  { value: "saturday", label: "السبت" },
  { value: "sunday", label: "الأحد" },
  { value: "friday_saturday", label: "الجمعة والسبت" },
  { value: "none", label: "لا يوجد" },
];

export default function EmployeesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingEmployee, setEditingEmployee] = useState<Employee | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [formData, setFormData] = useState<EmployeeFormData>({
    fullNameAr: "",
    phoneCountryCode: "966",
    phoneNumber: "",
    email: "",
    jobTitleAr: "",
    roleId: null,
    nationalityNameAr: "",
    cityNameAr: "",
    birthDate: "",
    hireDate: "",
    iqamaNumber: "",
    iqamaExpiryDate: "",
    passportNumber: "",
    passportExpiryDate: "",
    annualLeaveDays: 21,
    weeklyOffDay: "friday",
    isActive: true,
  });
  const { toast } = useToast();

  const { data: employees, isLoading } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: roles } = useQuery<Role[]>({
    queryKey: ["/api/roles"],
  });

  const createMutation = useMutation({
    mutationFn: (data: EmployeeFormData) =>
      apiRequest("POST", "/api/employees", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employees"] });
      toast({ title: "تم إنشاء الموظف بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الموظف",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<EmployeeFormData> }) =>
      apiRequest("PUT", `/api/employees/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employees"] });
      toast({ title: "تم تحديث الموظف بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الموظف",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/employees/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/employees"] });
      toast({ title: "تم تعطيل الموظف بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الموظف",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      fullNameAr: "",
      phoneCountryCode: "966",
      phoneNumber: "",
      email: "",
      jobTitleAr: "",
      roleId: null,
      nationalityNameAr: "",
      cityNameAr: "",
      birthDate: "",
      hireDate: "",
      iqamaNumber: "",
      iqamaExpiryDate: "",
      passportNumber: "",
      passportExpiryDate: "",
      annualLeaveDays: 21,
      weeklyOffDay: "friday",
      isActive: true,
    });
    setEditingEmployee(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (employee?: Employee) => {
    if (employee) {
      setEditingEmployee(employee);
      setFormData({
        fullNameAr: employee.fullNameAr || employee.nameAr || "",
        phoneCountryCode: employee.phoneCountryCode || "966",
        phoneNumber: employee.phoneNumber || employee.phone || "",
        email: employee.email || "",
        jobTitleAr: employee.jobTitleAr || "",
        roleId: employee.roleId,
        nationalityNameAr: employee.nationalityNameAr || "",
        cityNameAr: employee.cityNameAr || "",
        birthDate: employee.birthDate || "",
        hireDate: employee.hireDate || "",
        iqamaNumber: employee.iqamaNumber || "",
        iqamaExpiryDate: employee.iqamaExpiryDate || "",
        passportNumber: employee.passportNumber || "",
        passportExpiryDate: employee.passportExpiryDate || "",
        annualLeaveDays: employee.annualLeaveDays || 21,
        weeklyOffDay: employee.weeklyOffDay || "friday",
        isActive: employee.isActive,
      });
    } else {
      resetForm();
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.fullNameAr.trim()) {
      toast({ title: "اسم الموظف مطلوب", variant: "destructive" });
      return;
    }

    if (editingEmployee) {
      updateMutation.mutate({ id: editingEmployee.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const filteredEmployees = employees?.filter(emp => {
    if (!searchTerm) return true;
    const search = searchTerm.toLowerCase();
    return (
      (emp.fullNameAr?.toLowerCase() || "").includes(search) ||
      (emp.nameAr?.toLowerCase() || "").includes(search) ||
      (emp.email?.toLowerCase() || "").includes(search) ||
      (emp.phoneNumber || "").includes(search)
    );
  }) || [];

  const activeEmployees = employees?.filter(e => e.isActive) || [];
  const inactiveEmployees = employees?.filter(e => !e.isActive) || [];

  const getEmployeeName = (emp: Employee) => emp.fullNameAr || emp.nameAr || "-";
  const getEmployeePhone = (emp: Employee) => {
    if (emp.phoneNumber) return `+${emp.phoneCountryCode || "966"} ${emp.phoneNumber}`;
    if (emp.phone) return emp.phone;
    return "-";
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الموظفين</h1>
          <p className="text-muted-foreground">بيانات الموظفين وملفاتهم الشخصية</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-employee">
          <Plus className="h-4 w-4 ml-2" />
          إضافة موظف جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Users className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الموظفين</p>
                <p className="text-xl font-bold" data-testid="text-total-employees">
                  {employees?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">موظفين نشطين</p>
                <p className="text-xl font-bold" data-testid="text-active-employees">
                  {activeEmployees.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">موظفين غير نشطين</p>
                <p className="text-xl font-bold" data-testid="text-inactive-employees">
                  {inactiveEmployees.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between gap-4 flex-wrap">
            <CardTitle className="flex items-center gap-2">
              <Users className="h-5 w-5" />
              قائمة الموظفين
            </CardTitle>
            <div className="relative">
              <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="بحث بالاسم أو البريد أو الهاتف..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pr-9 w-64"
                data-testid="input-search-employee"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الاسم</TableHead>
                  <TableHead className="text-right">الدور الوظيفي</TableHead>
                  <TableHead className="text-right">الهاتف</TableHead>
                  <TableHead className="text-right">البريد الإلكتروني</TableHead>
                  <TableHead className="text-right">تاريخ التعيين</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredEmployees.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                      لا يوجد موظفين مسجلين
                    </TableCell>
                  </TableRow>
                ) : (
                  filteredEmployees.map((employee) => (
                    <TableRow key={employee.id} data-testid={`row-employee-${employee.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <div className="h-8 w-8 rounded-full bg-primary/10 flex items-center justify-center">
                            <Users className="h-4 w-4 text-primary" />
                          </div>
                          <div>
                            <p>{getEmployeeName(employee)}</p>
                            {employee.jobTitleAr && (
                              <p className="text-xs text-muted-foreground">{employee.jobTitleAr}</p>
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        {employee.role ? (
                          <Badge variant="outline">{employee.role.nameAr}</Badge>
                        ) : "-"}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Phone className="h-3 w-3" />
                          {getEmployeePhone(employee)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Mail className="h-3 w-3" />
                          {employee.email || "-"}
                        </div>
                      </TableCell>
                      <TableCell>
                        {employee.hireDate ? (
                          <div className="flex items-center gap-1 text-sm">
                            <Calendar className="h-3 w-3" />
                            {format(new Date(employee.hireDate), "dd/MM/yyyy")}
                          </div>
                        ) : "-"}
                      </TableCell>
                      <TableCell>
                        {employee.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            نشط
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير نشط
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            asChild
                            data-testid={`button-view-employee-${employee.id}`}
                          >
                            <Link href={`/employees/${employee.id}`}>
                              <Eye className="h-4 w-4" />
                            </Link>
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(employee)}
                            data-testid={`button-edit-employee-${employee.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(employee.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-employee-${employee.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-2xl max-h-[90vh] overflow-y-auto" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingEmployee ? "تعديل بيانات الموظف" : "إضافة موظف جديد"}
            </DialogTitle>
          </DialogHeader>
          <Tabs defaultValue="basic" className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="basic">البيانات الأساسية</TabsTrigger>
              <TabsTrigger value="identity">الهوية والتأمين</TabsTrigger>
              <TabsTrigger value="work">بيانات العمل</TabsTrigger>
            </TabsList>
            <TabsContent value="basic" className="space-y-4 pt-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="fullNameAr">الاسم الكامل <span className="text-destructive">*</span></Label>
                  <Input
                    id="fullNameAr"
                    value={formData.fullNameAr}
                    onChange={(e) => setFormData({ ...formData, fullNameAr: e.target.value })}
                    placeholder="الاسم الكامل بالعربية"
                    data-testid="input-employee-name"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="email">البريد الإلكتروني</Label>
                  <Input
                    id="email"
                    type="email"
                    value={formData.email}
                    onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                    placeholder="example@salon.com"
                    dir="ltr"
                    data-testid="input-employee-email"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="phoneNumber">رقم الهاتف</Label>
                  <div className="flex gap-2">
                    <Input
                      value={formData.phoneCountryCode}
                      onChange={(e) => setFormData({ ...formData, phoneCountryCode: e.target.value })}
                      placeholder="966"
                      className="w-20 text-left"
                      dir="ltr"
                    />
                    <Input
                      id="phoneNumber"
                      value={formData.phoneNumber}
                      onChange={(e) => setFormData({ ...formData, phoneNumber: e.target.value })}
                      placeholder="0500000000"
                      dir="ltr"
                      data-testid="input-employee-phone"
                    />
                  </div>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="nationalityNameAr">الجنسية</Label>
                  <Input
                    id="nationalityNameAr"
                    value={formData.nationalityNameAr}
                    onChange={(e) => setFormData({ ...formData, nationalityNameAr: e.target.value })}
                    placeholder="سعودي"
                    data-testid="input-employee-nationality"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="cityNameAr">المدينة</Label>
                  <Input
                    id="cityNameAr"
                    value={formData.cityNameAr}
                    onChange={(e) => setFormData({ ...formData, cityNameAr: e.target.value })}
                    placeholder="الرياض"
                    data-testid="input-employee-city"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="birthDate">تاريخ الميلاد</Label>
                  <Input
                    id="birthDate"
                    type="date"
                    value={formData.birthDate}
                    onChange={(e) => setFormData({ ...formData, birthDate: e.target.value })}
                    dir="ltr"
                    data-testid="input-employee-birthdate"
                  />
                </div>
              </div>
            </TabsContent>
            <TabsContent value="identity" className="space-y-4 pt-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="iqamaNumber">رقم الإقامة</Label>
                  <Input
                    id="iqamaNumber"
                    value={formData.iqamaNumber}
                    onChange={(e) => setFormData({ ...formData, iqamaNumber: e.target.value })}
                    placeholder="رقم الإقامة"
                    dir="ltr"
                    data-testid="input-employee-iqama"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="iqamaExpiryDate">تاريخ انتهاء الإقامة</Label>
                  <Input
                    id="iqamaExpiryDate"
                    type="date"
                    value={formData.iqamaExpiryDate}
                    onChange={(e) => setFormData({ ...formData, iqamaExpiryDate: e.target.value })}
                    dir="ltr"
                    data-testid="input-employee-iqama-expiry"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="passportNumber">رقم جواز السفر</Label>
                  <Input
                    id="passportNumber"
                    value={formData.passportNumber}
                    onChange={(e) => setFormData({ ...formData, passportNumber: e.target.value })}
                    placeholder="رقم جواز السفر"
                    dir="ltr"
                    data-testid="input-employee-passport"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="passportExpiryDate">تاريخ انتهاء جواز السفر</Label>
                  <Input
                    id="passportExpiryDate"
                    type="date"
                    value={formData.passportExpiryDate}
                    onChange={(e) => setFormData({ ...formData, passportExpiryDate: e.target.value })}
                    dir="ltr"
                    data-testid="input-employee-passport-expiry"
                  />
                </div>
              </div>
            </TabsContent>
            <TabsContent value="work" className="space-y-4 pt-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="jobTitleAr">المسمى الوظيفي</Label>
                  <Input
                    id="jobTitleAr"
                    value={formData.jobTitleAr}
                    onChange={(e) => setFormData({ ...formData, jobTitleAr: e.target.value })}
                    placeholder="مثال: مدير الفرع"
                    data-testid="input-employee-job-title"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="roleId">الدور الوظيفي</Label>
                  <Select
                    value={formData.roleId?.toString() || "_none"}
                    onValueChange={(value) => setFormData({ ...formData, roleId: value === "_none" ? null : parseInt(value) })}
                  >
                    <SelectTrigger data-testid="select-employee-role">
                      <SelectValue placeholder="اختر الدور" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="_none">لم يحدد</SelectItem>
                      {roles?.map((role) => (
                        <SelectItem key={role.id} value={role.id.toString()}>
                          {role.nameAr}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="hireDate">تاريخ التعيين</Label>
                  <Input
                    id="hireDate"
                    type="date"
                    value={formData.hireDate}
                    onChange={(e) => setFormData({ ...formData, hireDate: e.target.value })}
                    dir="ltr"
                    data-testid="input-employee-hire-date"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="annualLeaveDays">أيام الإجازة السنوية</Label>
                  <Input
                    id="annualLeaveDays"
                    type="number"
                    value={formData.annualLeaveDays}
                    onChange={(e) => setFormData({ ...formData, annualLeaveDays: parseInt(e.target.value) || 0 })}
                    min={0}
                    data-testid="input-employee-leave-days"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="weeklyOffDay">يوم الإجازة الأسبوعية</Label>
                  <Select
                    value={formData.weeklyOffDay}
                    onValueChange={(value) => setFormData({ ...formData, weeklyOffDay: value })}
                  >
                    <SelectTrigger data-testid="select-employee-off-day">
                      <SelectValue placeholder="اختر يوم الإجازة" />
                    </SelectTrigger>
                    <SelectContent>
                      {weeklyOffDayOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="flex items-center justify-between">
                  <Label htmlFor="isActive">الحالة</Label>
                  <div className="flex items-center gap-2">
                    <span className="text-sm text-muted-foreground">
                      {formData.isActive ? "نشط" : "غير نشط"}
                    </span>
                    <Switch
                      id="isActive"
                      checked={formData.isActive}
                      onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                      data-testid="switch-employee-active"
                    />
                  </div>
                </div>
              </div>
            </TabsContent>
          </Tabs>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-employee"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-employee"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
