import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency } from "@/lib/formatters";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { Plus, Pencil, Trash2, Loader2 } from "lucide-react";
import { Badge } from "@/components/ui/badge";

interface ExpenseType {
  id: number;
  nameAr: string;
  descriptionAr: string | null;
  isFixedAmount: boolean;
  fixedAmount: number | null;
  isActive: boolean;
  createdAt: string;
}

export default function ExpenseTypesPage() {
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingType, setEditingType] = useState<ExpenseType | null>(null);
  const [formData, setFormData] = useState({
    nameAr: "",
    descriptionAr: "",
    isFixedAmount: false,
    fixedAmount: "",
    isActive: true,
  });

  const { data: expenseTypes = [], isLoading } = useQuery<ExpenseType[]>({
    queryKey: ["/api/expenses/types"],
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof formData) => {
      const response = await apiRequest("POST", "/api/expenses/types", {
        nameAr: data.nameAr,
        descriptionAr: data.descriptionAr || null,
        isFixedAmount: data.isFixedAmount,
        fixedAmount: data.isFixedAmount ? parseFloat(data.fixedAmount) : null,
        isActive: data.isActive,
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/expenses/types"] });
      toast({ title: "تم إنشاء نوع النفقة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "فشل في إنشاء نوع النفقة",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: typeof formData }) => {
      const response = await apiRequest("PUT", `/api/expenses/types/${id}`, {
        nameAr: data.nameAr,
        descriptionAr: data.descriptionAr || null,
        isFixedAmount: data.isFixedAmount,
        fixedAmount: data.isFixedAmount ? parseFloat(data.fixedAmount) : null,
        isActive: data.isActive,
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/expenses/types"] });
      toast({ title: "تم تحديث نوع النفقة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "فشل في تحديث نوع النفقة",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/expenses/types/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/expenses/types"] });
      toast({ title: "تم حذف نوع النفقة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "فشل في حذف نوع النفقة",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      nameAr: "",
      descriptionAr: "",
      isFixedAmount: false,
      fixedAmount: "",
      isActive: true,
    });
    setEditingType(null);
    setIsDialogOpen(false);
  };

  const handleEdit = (type: ExpenseType) => {
    setEditingType(type);
    setFormData({
      nameAr: type.nameAr,
      descriptionAr: type.descriptionAr || "",
      isFixedAmount: type.isFixedAmount,
      fixedAmount: type.fixedAmount?.toString() || "",
      isActive: type.isActive,
    });
    setIsDialogOpen(true);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (editingType) {
      updateMutation.mutate({ id: editingType.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const formatAmount = (amount: number | null) => {
    if (amount === null) return "-";
    return formatCurrency(amount);
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold">أنواع النفقات</h1>
        <Button onClick={() => setIsDialogOpen(true)} data-testid="button-add-expense-type">
          <Plus className="h-4 w-4 ml-2" />
          إضافة نوع نفقة
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>قائمة أنواع النفقات</CardTitle>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="text-right">الاسم</TableHead>
                <TableHead className="text-right">الوصف</TableHead>
                <TableHead className="text-right">مبلغ ثابت</TableHead>
                <TableHead className="text-right">المبلغ</TableHead>
                <TableHead className="text-right">الحالة</TableHead>
                <TableHead className="text-right">الإجراءات</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {expenseTypes.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={6} className="text-center text-muted-foreground">
                    لا توجد أنواع نفقات
                  </TableCell>
                </TableRow>
              ) : (
                expenseTypes.map((type) => (
                  <TableRow key={type.id} data-testid={`row-expense-type-${type.id}`}>
                    <TableCell className="font-medium">{type.nameAr}</TableCell>
                    <TableCell>{type.descriptionAr || "-"}</TableCell>
                    <TableCell>
                      <Badge variant={type.isFixedAmount ? "default" : "secondary"}>
                        {type.isFixedAmount ? "نعم" : "لا"}
                      </Badge>
                    </TableCell>
                    <TableCell>{formatAmount(type.fixedAmount)}</TableCell>
                    <TableCell>
                      <Badge variant={type.isActive ? "default" : "secondary"}>
                        {type.isActive ? "نشط" : "غير نشط"}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleEdit(type)}
                          data-testid={`button-edit-expense-type-${type.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => {
                            if (confirm("هل أنت متأكد من حذف هذا النوع؟")) {
                              deleteMutation.mutate(type.id);
                            }
                          }}
                          data-testid={`button-delete-expense-type-${type.id}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={(open) => !open && resetForm()}>
        <DialogContent className="max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingType ? "تعديل نوع النفقة" : "إضافة نوع نفقة جديد"}
            </DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSubmit}>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="nameAr">الاسم</Label>
                <Input
                  id="nameAr"
                  value={formData.nameAr}
                  onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                  placeholder="مثال: كهرباء، إيجار، رواتب"
                  required
                  data-testid="input-expense-type-name"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="descriptionAr">الوصف</Label>
                <Textarea
                  id="descriptionAr"
                  value={formData.descriptionAr}
                  onChange={(e) => setFormData({ ...formData, descriptionAr: e.target.value })}
                  placeholder="وصف إضافي لنوع النفقة"
                  data-testid="input-expense-type-description"
                />
              </div>

              <div className="flex items-center justify-between">
                <Label htmlFor="isFixedAmount">مبلغ ثابت</Label>
                <Switch
                  id="isFixedAmount"
                  checked={formData.isFixedAmount}
                  onCheckedChange={(checked) =>
                    setFormData({ ...formData, isFixedAmount: checked })
                  }
                  data-testid="switch-expense-type-fixed"
                />
              </div>

              {formData.isFixedAmount && (
                <div className="space-y-2">
                  <Label htmlFor="fixedAmount">المبلغ الثابت (ر.س)</Label>
                  <Input
                    id="fixedAmount"
                    type="number"
                    step="0.01"
                    min="0"
                    value={formData.fixedAmount}
                    onChange={(e) => setFormData({ ...formData, fixedAmount: e.target.value })}
                    placeholder="0.00"
                    required
                    data-testid="input-expense-type-amount"
                  />
                </div>
              )}

              <div className="flex items-center justify-between">
                <Label htmlFor="isActive">نشط</Label>
                <Switch
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) =>
                    setFormData({ ...formData, isActive: checked })
                  }
                  data-testid="switch-expense-type-active"
                />
              </div>
            </div>

            <DialogFooter className="mt-6">
              <Button type="button" variant="outline" onClick={resetForm}>
                إلغاء
              </Button>
              <Button
                type="submit"
                disabled={createMutation.isPending || updateMutation.isPending}
                data-testid="button-save-expense-type"
              >
                {(createMutation.isPending || updateMutation.isPending) && (
                  <Loader2 className="h-4 w-4 ml-2 animate-spin" />
                )}
                {editingType ? "تحديث" : "إضافة"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
