import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Plus, Pencil, Trash2, Loader2, Filter, ReceiptText } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

interface ExpenseType {
  id: number;
  nameAr: string;
  isFixedAmount: boolean;
  fixedAmount: number | null;
  isActive: boolean;
}

interface Supplier {
  id: number;
  nameAr: string;
}

interface PaymentMethod {
  id: number;
  nameAr: string;
}

interface Expense {
  id: number;
  expenseTypeId: number;
  expenseTypeName: string;
  supplierId: number | null;
  supplierName: string | null;
  invoiceReference: string | null;
  descriptionAr: string | null;
  amountWithoutVat: number;
  vatType: string;
  vatRate: number;
  vatAmount: number;
  totalAmount: number;
  paymentMethodId: number | null;
  paymentMethodName: string | null;
  paymentDate: string;
  isPaid: boolean;
  attachmentUrl: string | null;
  createdAt: string;
}

interface ExpensesSummary {
  totalAmountWithoutVat: number;
  totalVatAmount: number;
  totalAmount: number;
  expenseCount: number;
  byType: { expenseTypeId: number; expenseTypeName: string; totalAmount: number }[];
}

export default function ExpensesPage() {
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingExpense, setEditingExpense] = useState<Expense | null>(null);
  const [filters, setFilters] = useState({
    from_date: "",
    to_date: "",
    expense_type_id: "",
    supplier_id: "",
  });
  const [formData, setFormData] = useState({
    expense_type_id: "",
    supplier_id: "",
    invoice_reference: "",
    description_ar: "",
    amount: "",
    vat_type: "inclusive",
    vat_rate: "0.15",
    payment_method_id: "",
    payment_date: format(new Date(), "yyyy-MM-dd"),
  });

  const queryString = Object.entries(filters)
    .filter(([_, v]) => v)
    .map(([k, v]) => `${k}=${encodeURIComponent(v)}`)
    .join("&");

  const { data: expenses = [], isLoading } = useQuery<Expense[]>({
    queryKey: ["/api/expenses", queryString],
    queryFn: async () => {
      const res = await fetch(`/api/expenses${queryString ? `?${queryString}` : ""}`);
      if (!res.ok) throw new Error("Failed to fetch expenses");
      return res.json();
    },
  });

  const { data: expenseTypes = [] } = useQuery<ExpenseType[]>({
    queryKey: ["/api/expenses/types", "active"],
    queryFn: async () => {
      const res = await fetch("/api/expenses/types?isActive=true");
      if (!res.ok) throw new Error("Failed to fetch expense types");
      return res.json();
    },
  });

  const { data: suppliers = [] } = useQuery<Supplier[]>({
    queryKey: ["/api/suppliers"],
  });

  const { data: paymentMethods = [] } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const { data: summary } = useQuery<ExpensesSummary>({
    queryKey: ["/api/reports/expenses-summary", queryString],
    queryFn: async () => {
      const res = await fetch(`/api/reports/expenses-summary${queryString ? `?${queryString}` : ""}`);
      if (!res.ok) throw new Error("Failed to fetch summary");
      return res.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof formData) => {
      const response = await apiRequest("POST", "/api/expenses", {
        ...data,
        expense_type_id: parseInt(data.expense_type_id),
        supplier_id: data.supplier_id ? parseInt(data.supplier_id) : null,
        payment_method_id: data.payment_method_id ? parseInt(data.payment_method_id) : null,
        amount: parseFloat(data.amount),
        vat_rate: parseFloat(data.vat_rate),
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/expenses"] });
      queryClient.invalidateQueries({ queryKey: ["/api/reports/expenses-summary"] });
      toast({ title: "تم إنشاء النفقة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "فشل في إنشاء النفقة",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: typeof formData }) => {
      const response = await apiRequest("PUT", `/api/expenses/${id}`, {
        ...data,
        expense_type_id: parseInt(data.expense_type_id),
        supplier_id: data.supplier_id ? parseInt(data.supplier_id) : null,
        payment_method_id: data.payment_method_id ? parseInt(data.payment_method_id) : null,
        amount: parseFloat(data.amount),
        vat_rate: parseFloat(data.vat_rate),
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/expenses"] });
      queryClient.invalidateQueries({ queryKey: ["/api/reports/expenses-summary"] });
      toast({ title: "تم تحديث النفقة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "فشل في تحديث النفقة",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/expenses/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/expenses"] });
      queryClient.invalidateQueries({ queryKey: ["/api/reports/expenses-summary"] });
      toast({ title: "تم حذف النفقة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "فشل في حذف النفقة",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      expense_type_id: "",
      supplier_id: "",
      invoice_reference: "",
      description_ar: "",
      amount: "",
      vat_type: "inclusive",
      vat_rate: "0.15",
      payment_method_id: "",
      payment_date: format(new Date(), "yyyy-MM-dd"),
    });
    setEditingExpense(null);
    setIsDialogOpen(false);
  };

  const handleEdit = (expense: Expense) => {
    setEditingExpense(expense);
    setFormData({
      expense_type_id: expense.expenseTypeId.toString(),
      supplier_id: expense.supplierId?.toString() || "",
      invoice_reference: expense.invoiceReference || "",
      description_ar: expense.descriptionAr || "",
      amount: expense.totalAmount.toString(),
      vat_type: expense.vatType,
      vat_rate: expense.vatRate.toString(),
      payment_method_id: expense.paymentMethodId?.toString() || "",
      payment_date: expense.paymentDate,
    });
    setIsDialogOpen(true);
  };

  const handleTypeChange = (typeId: string) => {
    const type = expenseTypes.find((t) => t.id.toString() === typeId);
    if (type?.isFixedAmount && type.fixedAmount) {
      setFormData({
        ...formData,
        expense_type_id: typeId,
        amount: type.fixedAmount.toString(),
      });
    } else {
      setFormData({ ...formData, expense_type_id: typeId });
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (editingExpense) {
      updateMutation.mutate({ id: editingExpense.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const formatAmount = formatCurrency;

  const vatTypeLabels: Record<string, string> = {
    inclusive: "شامل الضريبة",
    exclusive: "غير شامل الضريبة",
    exempt: "معفى",
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="space-y-6" dir="rtl">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold">النفقات</h1>
        <Button onClick={() => setIsDialogOpen(true)} data-testid="button-add-expense">
          <Plus className="h-4 w-4 ml-2" />
          إضافة نفقة
        </Button>
      </div>

      {summary && (
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="text-sm text-muted-foreground">عدد النفقات</div>
              <div className="text-2xl font-bold">{summary.expenseCount}</div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="text-sm text-muted-foreground">المبلغ بدون ضريبة</div>
              <div className="text-2xl font-bold">{formatAmount(summary.totalAmountWithoutVat)}</div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="text-sm text-muted-foreground">ضريبة القيمة المضافة</div>
              <div className="text-2xl font-bold">{formatAmount(summary.totalVatAmount)}</div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="text-sm text-muted-foreground">الإجمالي</div>
              <div className="text-2xl font-bold text-primary">{formatAmount(summary.totalAmount)}</div>
            </CardContent>
          </Card>
        </div>
      )}

      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-4">
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            فلترة
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="space-y-2">
              <Label>من تاريخ</Label>
              <Input
                type="date"
                value={filters.from_date}
                onChange={(e) => setFilters({ ...filters, from_date: e.target.value })}
                data-testid="input-filter-from-date"
              />
            </div>
            <div className="space-y-2">
              <Label>إلى تاريخ</Label>
              <Input
                type="date"
                value={filters.to_date}
                onChange={(e) => setFilters({ ...filters, to_date: e.target.value })}
                data-testid="input-filter-to-date"
              />
            </div>
            <div className="space-y-2">
              <Label>نوع النفقة</Label>
              <Select
                value={filters.expense_type_id || "all"}
                onValueChange={(v) => setFilters({ ...filters, expense_type_id: v === "all" ? "" : v })}
              >
                <SelectTrigger data-testid="select-filter-expense-type">
                  <SelectValue placeholder="الكل" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  {expenseTypes.map((type) => (
                    <SelectItem key={type.id} value={type.id.toString()}>
                      {type.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>المورد</Label>
              <Select
                value={filters.supplier_id || "all"}
                onValueChange={(v) => setFilters({ ...filters, supplier_id: v === "all" ? "" : v })}
              >
                <SelectTrigger data-testid="select-filter-supplier">
                  <SelectValue placeholder="الكل" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">الكل</SelectItem>
                  {suppliers.map((s) => (
                    <SelectItem key={s.id} value={s.id.toString()}>
                      {s.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <ReceiptText className="h-5 w-5" />
            قائمة النفقات
          </CardTitle>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="text-right">التاريخ</TableHead>
                <TableHead className="text-right">نوع النفقة</TableHead>
                <TableHead className="text-right">المورد</TableHead>
                <TableHead className="text-right">المرجع</TableHead>
                <TableHead className="text-right">المبلغ</TableHead>
                <TableHead className="text-right">الضريبة</TableHead>
                <TableHead className="text-right">الإجمالي</TableHead>
                <TableHead className="text-right">طريقة الدفع</TableHead>
                <TableHead className="text-right">الإجراءات</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {expenses.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={9} className="text-center text-muted-foreground">
                    لا توجد نفقات
                  </TableCell>
                </TableRow>
              ) : (
                expenses.map((expense) => (
                  <TableRow key={expense.id} data-testid={`row-expense-${expense.id}`}>
                    <TableCell>
                      {format(new Date(expense.paymentDate), "dd/MM/yyyy", { locale: ar })}
                    </TableCell>
                    <TableCell className="font-medium">{expense.expenseTypeName}</TableCell>
                    <TableCell>{expense.supplierName || "-"}</TableCell>
                    <TableCell>{expense.invoiceReference || "-"}</TableCell>
                    <TableCell>{formatAmount(expense.amountWithoutVat)}</TableCell>
                    <TableCell>
                      <div className="flex flex-col gap-1">
                        <span>{formatAmount(expense.vatAmount)}</span>
                        <Badge variant="secondary" className="text-xs w-fit">
                          {vatTypeLabels[expense.vatType]}
                        </Badge>
                      </div>
                    </TableCell>
                    <TableCell className="font-bold">{formatAmount(expense.totalAmount)}</TableCell>
                    <TableCell>{expense.paymentMethodName || "-"}</TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleEdit(expense)}
                          data-testid={`button-edit-expense-${expense.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => {
                            if (confirm("هل أنت متأكد من حذف هذه النفقة؟")) {
                              deleteMutation.mutate(expense.id);
                            }
                          }}
                          data-testid={`button-delete-expense-${expense.id}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={(open) => !open && resetForm()}>
        <DialogContent className="max-w-2xl" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingExpense ? "تعديل النفقة" : "إضافة نفقة جديدة"}
            </DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSubmit}>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>نوع النفقة *</Label>
                <Select
                  value={formData.expense_type_id}
                  onValueChange={handleTypeChange}
                  required
                >
                  <SelectTrigger data-testid="select-expense-type">
                    <SelectValue placeholder="اختر نوع النفقة" />
                  </SelectTrigger>
                  <SelectContent>
                    {expenseTypes.map((type) => (
                      <SelectItem key={type.id} value={type.id.toString()}>
                        {type.nameAr}
                        {type.isFixedAmount && ` (${formatAmount(type.fixedAmount || 0)})`}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>المورد (اختياري)</Label>
                <Select
                  value={formData.supplier_id || "none"}
                  onValueChange={(v) => setFormData({ ...formData, supplier_id: v === "none" ? "" : v })}
                >
                  <SelectTrigger data-testid="select-expense-supplier">
                    <SelectValue placeholder="اختر المورد" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">بدون مورد</SelectItem>
                    {suppliers.map((s) => (
                      <SelectItem key={s.id} value={s.id.toString()}>
                        {s.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>تاريخ الدفع *</Label>
                <Input
                  type="date"
                  value={formData.payment_date}
                  onChange={(e) => setFormData({ ...formData, payment_date: e.target.value })}
                  required
                  data-testid="input-expense-date"
                />
              </div>

              <div className="space-y-2">
                <Label>رقم المرجع</Label>
                <Input
                  value={formData.invoice_reference}
                  onChange={(e) => setFormData({ ...formData, invoice_reference: e.target.value })}
                  placeholder="رقم الفاتورة أو المرجع"
                  data-testid="input-expense-reference"
                />
              </div>

              <div className="space-y-2">
                <Label>المبلغ *</Label>
                <Input
                  type="number"
                  step="0.01"
                  min="0"
                  value={formData.amount}
                  onChange={(e) => setFormData({ ...formData, amount: e.target.value })}
                  placeholder="0.00"
                  required
                  data-testid="input-expense-amount"
                />
              </div>

              <div className="space-y-2">
                <Label>نوع الضريبة</Label>
                <Select
                  value={formData.vat_type}
                  onValueChange={(v) => setFormData({ ...formData, vat_type: v })}
                >
                  <SelectTrigger data-testid="select-expense-vat-type">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="inclusive">شامل الضريبة</SelectItem>
                    <SelectItem value="exclusive">غير شامل الضريبة</SelectItem>
                    <SelectItem value="exempt">معفى</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>نسبة الضريبة</Label>
                <Select
                  value={formData.vat_rate}
                  onValueChange={(v) => setFormData({ ...formData, vat_rate: v })}
                >
                  <SelectTrigger data-testid="select-expense-vat-rate">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="0">0%</SelectItem>
                    <SelectItem value="0.05">5%</SelectItem>
                    <SelectItem value="0.15">15%</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>طريقة الدفع</Label>
                <Select
                  value={formData.payment_method_id || "unspecified"}
                  onValueChange={(v) => setFormData({ ...formData, payment_method_id: v === "unspecified" ? "" : v })}
                >
                  <SelectTrigger data-testid="select-expense-payment-method">
                    <SelectValue placeholder="اختر طريقة الدفع" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="unspecified">غير محدد</SelectItem>
                    {paymentMethods.map((m) => (
                      <SelectItem key={m.id} value={m.id.toString()}>
                        {m.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="col-span-2 space-y-2">
                <Label>الوصف</Label>
                <Textarea
                  value={formData.description_ar}
                  onChange={(e) => setFormData({ ...formData, description_ar: e.target.value })}
                  placeholder="وصف إضافي للنفقة"
                  data-testid="input-expense-description"
                />
              </div>
            </div>

            <DialogFooter className="mt-6">
              <Button type="button" variant="outline" onClick={resetForm}>
                إلغاء
              </Button>
              <Button
                type="submit"
                disabled={createMutation.isPending || updateMutation.isPending}
                data-testid="button-save-expense"
              >
                {(createMutation.isPending || updateMutation.isPending) && (
                  <Loader2 className="h-4 w-4 ml-2 animate-spin" />
                )}
                {editingExpense ? "تحديث" : "إضافة"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
