import { useQuery } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { BookOpen, Calendar, Filter, RefreshCw } from "lucide-react";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";

interface Account {
  id: number;
  code: string;
  nameAr: string;
  accountType: string;
}

interface LedgerEntry {
  date: string;
  entryNumber: string;
  description: string;
  accountCode: string;
  accountName: string;
  debit: number;
  credit: number;
  referenceType: string;
  referenceId: number;
}

const referenceTypeLabels: Record<string, string> = {
  PurchaseInvoice: "فاتورة مشتريات",
  Payment: "دفعة سداد",
  SalesInvoice: "فاتورة مبيعات",
  Manual: "قيد يدوي",
};

export default function GeneralLedgerPage() {
  const [accountCode, setAccountCode] = useState<string>("all");
  const [startDate, setStartDate] = useState<string>("");
  const [endDate, setEndDate] = useState<string>(
    new Date().toISOString().split("T")[0]
  );

  const { data: accounts } = useQuery<Account[]>({
    queryKey: ["/api/accounts"],
    queryFn: async () => {
      const res = await fetch("/api/accounts");
      if (!res.ok) throw new Error("Failed to fetch accounts");
      return res.json();
    },
  });

  const { data: ledger, isLoading, refetch } = useQuery<LedgerEntry[]>({
    queryKey: ["/api/reports/general-ledger", accountCode, startDate, endDate],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (accountCode && accountCode !== "all") params.set("accountCode", accountCode);
      if (startDate) params.set("startDate", startDate);
      if (endDate) params.set("endDate", endDate);
      const res = await fetch(`/api/reports/general-ledger?${params.toString()}`);
      if (!res.ok) throw new Error("Failed to fetch general ledger");
      return res.json();
    },
  });

  const postableAccounts = accounts?.filter(a => a.code.length >= 4) || [];

  const totalDebit = ledger?.reduce((sum, entry) => sum + entry.debit, 0) || 0;
  const totalCredit = ledger?.reduce((sum, entry) => sum + entry.credit, 0) || 0;

  const groupedByAccount = ledger?.reduce((acc, entry) => {
    if (!acc[entry.accountCode]) {
      acc[entry.accountCode] = {
        accountName: entry.accountName,
        entries: [],
        totalDebit: 0,
        totalCredit: 0,
      };
    }
    acc[entry.accountCode].entries.push(entry);
    acc[entry.accountCode].totalDebit += entry.debit;
    acc[entry.accountCode].totalCredit += entry.credit;
    return acc;
  }, {} as Record<string, { accountName: string; entries: LedgerEntry[]; totalDebit: number; totalCredit: number }>);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">دفتر الأستاذ العام</h1>
          <p className="text-muted-foreground">عرض تفاصيل حركة الحسابات</p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            عوامل التصفية
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div>
              <Label>الحساب</Label>
              <Select value={accountCode} onValueChange={setAccountCode}>
                <SelectTrigger data-testid="select-account">
                  <SelectValue placeholder="جميع الحسابات" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">جميع الحسابات</SelectItem>
                  {postableAccounts.map((acc) => (
                    <SelectItem key={acc.id} value={acc.code}>
                      {acc.code} - {acc.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>من تاريخ</Label>
              <Input
                type="date"
                value={startDate}
                onChange={(e) => setStartDate(e.target.value)}
                data-testid="input-start-date"
              />
            </div>
            <div>
              <Label>إلى تاريخ</Label>
              <Input
                type="date"
                value={endDate}
                onChange={(e) => setEndDate(e.target.value)}
                data-testid="input-end-date"
              />
            </div>
            <div className="flex items-end">
              <Button onClick={() => refetch()} className="w-full" data-testid="button-apply-filter">
                <RefreshCw className="h-4 w-4 ml-2" />
                تطبيق
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
                <BookOpen className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">عدد القيود</p>
                <p className="text-xl font-bold" data-testid="text-entries-count">
                  {ledger?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <BookOpen className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المدين</p>
                <p className="text-xl font-bold" data-testid="text-total-debit">
                  {formatNumber(totalDebit)} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <BookOpen className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الدائن</p>
                <p className="text-xl font-bold" data-testid="text-total-credit">
                  {formatNumber(totalCredit)} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {isLoading ? (
        <Card>
          <CardContent className="p-6">
            <div className="space-y-2">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          </CardContent>
        </Card>
      ) : ledger && ledger.length > 0 ? (
        accountCode && accountCode !== "all" ? (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BookOpen className="h-5 w-5" />
                حركة الحساب: {accountCode} - {ledger[0]?.accountName}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">رقم القيد</TableHead>
                    <TableHead className="text-right">البيان</TableHead>
                    <TableHead className="text-right">المرجع</TableHead>
                    <TableHead className="text-left">مدين</TableHead>
                    <TableHead className="text-left">دائن</TableHead>
                    <TableHead className="text-left">الرصيد</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {(() => {
                    let runningBalance = 0;
                    return ledger.map((entry, index) => {
                      runningBalance += entry.debit - entry.credit;
                      return (
                        <TableRow key={`${entry.entryNumber}-${index}`} data-testid={`row-entry-${entry.entryNumber}`}>
                          <TableCell>{formatDate(entry.date)}</TableCell>
                          <TableCell className="font-mono">{entry.entryNumber}</TableCell>
                          <TableCell>{entry.description}</TableCell>
                          <TableCell>
                            <Badge variant="outline">
                              {referenceTypeLabels[entry.referenceType] || entry.referenceType}
                              {entry.referenceId ? ` #${entry.referenceId}` : ""}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-left font-mono">
                            {entry.debit > 0 ? formatNumber(entry.debit) : "-"}
                          </TableCell>
                          <TableCell className="text-left font-mono">
                            {entry.credit > 0 ? formatNumber(entry.credit) : "-"}
                          </TableCell>
                          <TableCell className="text-left font-mono">
                            <span className={runningBalance >= 0 ? "text-green-600" : "text-red-600"}>
                              {formatNumber(Math.abs(runningBalance))} {runningBalance >= 0 ? "مدين" : "دائن"}
                            </span>
                          </TableCell>
                        </TableRow>
                      );
                    });
                  })()}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        ) : (
          Object.entries(groupedByAccount || {}).map(([code, data]) => (
            <Card key={code}>
              <CardHeader>
                <CardTitle className="flex items-center justify-between gap-2">
                  <div className="flex items-center gap-2">
                    <BookOpen className="h-5 w-5" />
                    <span className="font-mono">{code}</span> - {data.accountName}
                  </div>
                  <div className="flex items-center gap-4 text-sm font-normal">
                    <span className="text-green-600">مدين: {formatNumber(data.totalDebit)}</span>
                    <span className="text-red-600">دائن: {formatNumber(data.totalCredit)}</span>
                    <Badge variant={data.totalDebit >= data.totalCredit ? "default" : "secondary"}>
                      الرصيد: {formatNumber(Math.abs(data.totalDebit - data.totalCredit))}
                      {data.totalDebit >= data.totalCredit ? " مدين" : " دائن"}
                    </Badge>
                  </div>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right">التاريخ</TableHead>
                      <TableHead className="text-right">رقم القيد</TableHead>
                      <TableHead className="text-right">البيان</TableHead>
                      <TableHead className="text-right">المرجع</TableHead>
                      <TableHead className="text-left">مدين</TableHead>
                      <TableHead className="text-left">دائن</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {data.entries.map((entry, index) => (
                      <TableRow key={`${entry.entryNumber}-${index}`}>
                        <TableCell>{formatDate(entry.date)}</TableCell>
                        <TableCell className="font-mono">{entry.entryNumber}</TableCell>
                        <TableCell>{entry.description}</TableCell>
                        <TableCell>
                          <Badge variant="outline">
                            {referenceTypeLabels[entry.referenceType] || entry.referenceType}
                            {entry.referenceId ? ` #${entry.referenceId}` : ""}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-left font-mono">
                          {entry.debit > 0 ? formatNumber(entry.debit) : "-"}
                        </TableCell>
                        <TableCell className="text-left font-mono">
                          {entry.credit > 0 ? formatNumber(entry.credit) : "-"}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          ))
        )
      ) : (
        <Card>
          <CardContent className="p-8">
            <div className="text-center text-muted-foreground">
              <BookOpen className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد حركات محاسبية في الفترة المحددة</p>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
