import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Users, UserCheck, Clock, Calendar, DollarSign, Briefcase, Award, TrendingUp, Percent } from "lucide-react";
import { differenceInDays, parseISO } from "date-fns";
import { useState } from "react";
import type { Employee, Leave, SalaryProfile, PayrollRun, CommissionProfile, CommissionTransaction } from "@shared/schema";

export default function HRStatisticsPage() {
  const [period, setPeriod] = useState("all");
  
  const { data: employees = [], isLoading: loadingEmployees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });
  
  const { data: leaves = [] } = useQuery<Leave[]>({
    queryKey: ["/api/leaves"],
  });
  
  const { data: salaryProfiles = [] } = useQuery<SalaryProfile[]>({
    queryKey: ["/api/salary-profiles"],
  });
  
  const { data: payrollRuns = [] } = useQuery<PayrollRun[]>({
    queryKey: ["/api/payroll-runs"],
  });
  
  const { data: commissionProfiles = [] } = useQuery<CommissionProfile[]>({
    queryKey: ["/api/commission-profiles"],
  });
  
  const { data: commissionTransactions = [] } = useQuery<CommissionTransaction[]>({
    queryKey: ["/api/commission-transactions"],
  });
  
  const activeEmployees = employees.filter(e => e.isActive);
  const inactiveEmployees = employees.filter(e => !e.isActive);
  
  const getEmploymentDuration = (hireDate: string | null | undefined) => {
    if (!hireDate) return 0;
    return differenceInDays(new Date(), parseISO(hireDate));
  };
  
  const avgEmploymentDays = activeEmployees.length > 0
    ? Math.round(activeEmployees.reduce((sum, e) => sum + getEmploymentDuration(e.hireDate), 0) / activeEmployees.length)
    : 0;
  
  const avgEmploymentYears = Math.floor(avgEmploymentDays / 365);
  const avgEmploymentMonths = Math.floor((avgEmploymentDays % 365) / 30);
  
  const employeesByNationality = activeEmployees.reduce((acc, e) => {
    const nat = e.nationalityNameAr || "غير محدد";
    acc[nat] = (acc[nat] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);
  
  const employeesByJobTitle = activeEmployees.reduce((acc, e) => {
    const title = e.jobTitleAr || "غير محدد";
    acc[title] = (acc[title] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);
  
  const leavesByType = leaves.reduce((acc, l) => {
    const type = l.leaveType || "غير محدد";
    acc[type] = (acc[type] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);
  
  const pendingLeaves = leaves.filter(l => l.status === "pending");
  const approvedLeaves = leaves.filter(l => l.status === "approved");
  const rejectedLeaves = leaves.filter(l => l.status === "rejected");
  
  const leavesByTypeAndStatus = leaves.reduce((acc, l) => {
    const type = l.leaveType || "غير محدد";
    if (!acc[type]) {
      acc[type] = { pending: 0, approved: 0, rejected: 0, total: 0 };
    }
    acc[type][l.status as "pending" | "approved" | "rejected"]++;
    acc[type].total++;
    return acc;
  }, {} as Record<string, { pending: number; approved: number; rejected: number; total: number }>);
  
  const totalBasicSalary = salaryProfiles.reduce((sum, s) => sum + Number(s.basicSalary || 0), 0);
  const totalHousingAllowance = salaryProfiles.reduce((sum, s) => sum + Number(s.housingAllowance || 0), 0);
  const totalTransportAllowance = salaryProfiles.reduce((sum, s) => sum + Number(s.transportAllowance || 0), 0);
  const totalOtherAllowances = salaryProfiles.reduce((sum, s) => sum + Number(s.otherFixedAllowances || 0), 0);
  const totalGrossSalary = totalBasicSalary + totalHousingAllowance + totalTransportAllowance + totalOtherAllowances;
  
  const paidPayrolls = payrollRuns.filter(p => p.status === "paid");
  const pendingPayrolls = payrollRuns.filter(p => p.status === "pending");
  
  const activeCommissionProfiles = commissionProfiles.filter(c => c.isActive);
  const paymentTransactions = commissionTransactions.filter(c => c.transactionType === "payment");
  const totalPaidCommissions = Math.abs(paymentTransactions.reduce((sum, c) => sum + Number(c.amount || 0), 0));
  
  const formatCurrency = (amount: number) => 
    new Intl.NumberFormat("en-SA", { style: "currency", currency: "SAR" }).format(amount);
  
  if (loadingEmployees) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary"></div>
      </div>
    );
  }
  
  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold">إحصائيات الموارد البشرية</h1>
          <p className="text-muted-foreground">نظرة شاملة على موظفي المؤسسة</p>
        </div>
        <div className="flex gap-2 items-center">
          <span className="text-sm text-muted-foreground">الفترة:</span>
          <Select value={period} onValueChange={setPeriod} data-testid="select-period">
            <SelectTrigger className="w-[150px]">
              <SelectValue placeholder="اختر الفترة" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">الكل</SelectItem>
              <SelectItem value="year">هذا العام</SelectItem>
              <SelectItem value="month">هذا الشهر</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card data-testid="card-total-employees">
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">إجمالي الموظفين</CardTitle>
            <Users className="h-5 w-5 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-employees">{employees.length}</div>
            <p className="text-xs text-muted-foreground">
              نشط: {activeEmployees.length} | غير نشط: {inactiveEmployees.length}
            </p>
          </CardContent>
        </Card>
        
        <Card data-testid="card-salary-profiles">
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">ملفات الرواتب</CardTitle>
            <DollarSign className="h-5 w-5 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{salaryProfiles.length}</div>
            <p className="text-xs text-muted-foreground">
              ملف راتب نشط
            </p>
          </CardContent>
        </Card>
        
        <Card data-testid="card-avg-tenure">
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">متوسط مدة العمل</CardTitle>
            <Clock className="h-5 w-5 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-avg-tenure">
              {avgEmploymentYears > 0 ? `${avgEmploymentYears} سنة` : ""} 
              {avgEmploymentMonths > 0 ? ` ${avgEmploymentMonths} شهر` : ""}
              {avgEmploymentYears === 0 && avgEmploymentMonths === 0 ? "0" : ""}
            </div>
            <p className="text-xs text-muted-foreground">{avgEmploymentDays} يوم</p>
          </CardContent>
        </Card>
        
        <Card data-testid="card-nationalities">
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium">الجنسيات</CardTitle>
            <Briefcase className="h-5 w-5 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-nationalities-count">
              {Object.keys(employeesByNationality).length}
            </div>
            <p className="text-xs text-muted-foreground">جنسية مختلفة</p>
          </CardContent>
        </Card>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        <Card data-testid="card-nationality-breakdown">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Briefcase className="h-5 w-5" />
              توزيع الجنسيات
            </CardTitle>
          </CardHeader>
          <CardContent>
            {Object.keys(employeesByNationality).length > 0 ? (
              <div className="space-y-3">
                {Object.entries(employeesByNationality)
                  .sort((a, b) => b[1] - a[1])
                  .map(([nat, count]) => (
                    <div key={nat} className="flex justify-between items-center">
                      <span className="text-sm">{nat}</span>
                      <div className="flex items-center gap-2">
                        <div className="h-2 bg-primary/20 rounded-full w-24">
                          <div 
                            className="h-2 bg-primary rounded-full" 
                            style={{ width: `${(count / activeEmployees.length) * 100}%` }}
                          />
                        </div>
                        <Badge variant="secondary">{count}</Badge>
                      </div>
                    </div>
                  ))}
              </div>
            ) : (
              <p className="text-muted-foreground text-center py-4">لا توجد بيانات</p>
            )}
          </CardContent>
        </Card>
        
        <Card data-testid="card-job-titles">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Award className="h-5 w-5" />
              المسميات الوظيفية
            </CardTitle>
          </CardHeader>
          <CardContent>
            {Object.keys(employeesByJobTitle).length > 0 ? (
              <div className="space-y-3">
                {Object.entries(employeesByJobTitle)
                  .sort((a, b) => b[1] - a[1])
                  .slice(0, 8)
                  .map(([title, count]) => (
                    <div key={title} className="flex justify-between items-center">
                      <span className="text-sm">{title}</span>
                      <Badge variant="secondary">{count}</Badge>
                    </div>
                  ))}
              </div>
            ) : (
              <p className="text-muted-foreground text-center py-4">لا توجد بيانات</p>
            )}
          </CardContent>
        </Card>
        
        <Card data-testid="card-leaves-summary">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Calendar className="h-5 w-5" />
              ملخص الإجازات
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex justify-between items-center">
                <span className="text-sm">إجمالي الطلبات</span>
                <Badge variant="secondary">{leaves.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">قيد الانتظار</span>
                <Badge variant="default" className="bg-yellow-500">{pendingLeaves.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">موافق عليها</span>
                <Badge variant="default" className="bg-green-500">{approvedLeaves.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">مرفوضة</span>
                <Badge variant="destructive">{rejectedLeaves.length}</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card data-testid="card-salary-summary">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <DollarSign className="h-5 w-5" />
              ملخص الرواتب الشهرية
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex justify-between items-center border-b pb-2">
                <span className="text-sm">الراتب الأساسي الإجمالي</span>
                <span className="font-medium">{formatCurrency(totalBasicSalary)}</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">بدل السكن</span>
                <span className="font-medium">{formatCurrency(totalHousingAllowance)}</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">بدل النقل</span>
                <span className="font-medium">{formatCurrency(totalTransportAllowance)}</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">بدلات أخرى</span>
                <span className="font-medium">{formatCurrency(totalOtherAllowances)}</span>
              </div>
              <div className="flex justify-between items-center border-t pt-2 font-bold">
                <span>إجمالي الرواتب الشهرية</span>
                <span className="text-primary">{formatCurrency(totalGrossSalary)}</span>
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card data-testid="card-payroll-summary">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <TrendingUp className="h-5 w-5" />
              ملخص المسيرات
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex justify-between items-center">
                <span className="text-sm">إجمالي المسيرات</span>
                <Badge variant="secondary">{payrollRuns.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">مسيرات مدفوعة</span>
                <Badge variant="default" className="bg-green-500">{paidPayrolls.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">مسيرات معلقة</span>
                <Badge variant="default" className="bg-yellow-500">{pendingPayrolls.length}</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card data-testid="card-commission-summary">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Percent className="h-5 w-5" />
              ملخص العمولات
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex justify-between items-center">
                <span className="text-sm">ملفات العمولات النشطة</span>
                <Badge variant="secondary">{activeCommissionProfiles.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">إجمالي المعاملات</span>
                <Badge variant="secondary">{commissionTransactions.length}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm">الدفعات المسددة</span>
                <Badge variant="default" className="bg-green-500">{paymentTransactions.length}</Badge>
              </div>
              <div className="flex justify-between items-center font-bold">
                <span>إجمالي المبالغ المدفوعة</span>
                <span className="text-primary">{formatCurrency(totalPaidCommissions)}</span>
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card data-testid="card-leave-types">
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Calendar className="h-5 w-5" />
              الإجازات حسب النوع والحالة
            </CardTitle>
          </CardHeader>
          <CardContent>
            {Object.keys(leavesByTypeAndStatus).length > 0 ? (
              <div className="space-y-4">
                {Object.entries(leavesByTypeAndStatus)
                  .sort((a, b) => b[1].total - a[1].total)
                  .map(([type, counts]) => (
                    <div key={type} className="border-b pb-3 last:border-b-0">
                      <div className="flex justify-between items-center mb-2">
                        <span className="text-sm font-medium">{type}</span>
                        <Badge variant="secondary">{counts.total}</Badge>
                      </div>
                      <div className="flex gap-2 flex-wrap">
                        {counts.pending > 0 && (
                          <Badge variant="default" className="bg-yellow-500 text-xs">
                            قيد الانتظار: {counts.pending}
                          </Badge>
                        )}
                        {counts.approved > 0 && (
                          <Badge variant="default" className="bg-green-500 text-xs">
                            موافق: {counts.approved}
                          </Badge>
                        )}
                        {counts.rejected > 0 && (
                          <Badge variant="destructive" className="text-xs">
                            مرفوض: {counts.rejected}
                          </Badge>
                        )}
                      </div>
                    </div>
                  ))}
              </div>
            ) : (
              <p className="text-muted-foreground text-center py-4">لا توجد إجازات</p>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
