import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
  TableFooter,
} from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { 
  TrendingUp, 
  TrendingDown, 
  DollarSign, 
  Calendar, 
  RefreshCw,
  Download,
  Printer,
  ChevronLeft,
} from "lucide-react";
import { format, startOfMonth, endOfMonth, startOfYear, endOfYear } from "date-fns";
import { formatCurrency, formatNumber } from "@/lib/formatters";

interface IncomeStatementItem {
  accountCode: string;
  accountName: string;
  amount: number;
  level: number;
  isParent: boolean;
  isPostable: boolean;
}

interface IncomeStatementData {
  revenues: IncomeStatementItem[];
  expenses: IncomeStatementItem[];
  totalRevenue: number;
  totalExpenses: number;
  netIncome: number;
  startDate?: string;
  endDate?: string;
}

export default function IncomeStatementPage() {
  const today = new Date();
  const [startDate, setStartDate] = useState<string>(
    format(startOfMonth(today), "yyyy-MM-dd")
  );
  const [endDate, setEndDate] = useState<string>(
    format(endOfMonth(today), "yyyy-MM-dd")
  );
  const [hideZeroValues, setHideZeroValues] = useState<boolean>(false);

  const { data: incomeStatement, isLoading, refetch } = useQuery<IncomeStatementData>({
    queryKey: [`/api/reports/income-statement?startDate=${startDate}&endDate=${endDate}`],
  });

  const setThisMonth = () => {
    setStartDate(format(startOfMonth(today), "yyyy-MM-dd"));
    setEndDate(format(endOfMonth(today), "yyyy-MM-dd"));
  };

  const setThisYear = () => {
    setStartDate(format(startOfYear(today), "yyyy-MM-dd"));
    setEndDate(format(endOfYear(today), "yyyy-MM-dd"));
  };

  const handlePrint = () => {
    window.print();
  };

  const filterItems = (items: IncomeStatementItem[] | undefined): IncomeStatementItem[] => {
    if (!items) return [];
    if (!hideZeroValues) return items;
    return items.filter(item => item.amount !== 0);
  };

  const getIndentClass = (level: number): string => {
    switch (level) {
      case 1: return "";
      case 2: return "pr-4";
      case 3: return "pr-8";
      case 4: return "pr-12";
      default: return "pr-16";
    }
  };

  const filteredRevenues = filterItems(incomeStatement?.revenues);
  const filteredExpenses = filterItems(incomeStatement?.expenses);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap print:hidden">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">قائمة الدخل</h1>
          <p className="text-muted-foreground">تقرير الإيرادات والمصروفات</p>
        </div>
        <div className="flex items-center gap-4 flex-wrap">
          <div className="flex items-center gap-2">
            <Label htmlFor="startDate">من:</Label>
            <Input
              id="startDate"
              type="date"
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
              className="w-40"
              data-testid="input-start-date"
            />
          </div>
          <div className="flex items-center gap-2">
            <Label htmlFor="endDate">إلى:</Label>
            <Input
              id="endDate"
              type="date"
              value={endDate}
              onChange={(e) => setEndDate(e.target.value)}
              className="w-40"
              data-testid="input-end-date"
            />
          </div>
          <div className="flex gap-2">
            <Button variant="outline" size="sm" onClick={setThisMonth} data-testid="button-this-month">
              هذا الشهر
            </Button>
            <Button variant="outline" size="sm" onClick={setThisYear} data-testid="button-this-year">
              هذه السنة
            </Button>
          </div>
          <Button variant="outline" onClick={() => refetch()} data-testid="button-refresh">
            <RefreshCw className="h-4 w-4 ml-2" />
            تحديث
          </Button>
          <Button variant="outline" onClick={handlePrint} data-testid="button-print">
            <Printer className="h-4 w-4 ml-2" />
            طباعة
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 print:hidden">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <TrendingUp className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الإيرادات</p>
                <div className="text-xl font-bold text-green-600" data-testid="text-total-revenue">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(incomeStatement?.totalRevenue || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <TrendingDown className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المصروفات</p>
                <div className="text-xl font-bold text-red-600" data-testid="text-total-expenses">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(incomeStatement?.totalExpenses || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className={`p-2 rounded-lg ${(incomeStatement?.netIncome || 0) >= 0 ? 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200' : 'bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200'}`}>
                <DollarSign className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">صافي الدخل</p>
                <div className={`text-xl font-bold ${(incomeStatement?.netIncome || 0) >= 0 ? 'text-blue-600' : 'text-orange-600'}`} data-testid="text-net-income">
                  {isLoading ? <Skeleton className="h-6 w-24" /> : formatCurrency(incomeStatement?.netIncome || 0)}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card className="print:shadow-none print:border-0">
        <CardHeader className="print:pb-2">
          <div className="flex items-center justify-between flex-wrap gap-4">
            <CardTitle className="flex items-center gap-2 print:text-center print:block">
              <TrendingUp className="h-5 w-5 print:hidden" />
              <span>قائمة الدخل</span>
              <Badge variant="outline" className="mr-2">
                {startDate} - {endDate}
              </Badge>
            </CardTitle>
            <div className="flex items-center gap-2 print:hidden">
              <Checkbox
                id="hideZeroValues"
                checked={hideZeroValues}
                onCheckedChange={(checked) => setHideZeroValues(checked as boolean)}
                data-testid="checkbox-hide-zero"
              />
              <Label htmlFor="hideZeroValues" className="text-sm cursor-pointer">
                إخفاء القيم الصفرية
              </Label>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-10 w-full" />
              ))}
            </div>
          ) : (
            <div className="space-y-8">
              <div>
                <h3 className="text-lg font-semibold mb-4 text-green-700 dark:text-green-400">
                  الإيرادات (4)
                </h3>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right w-32">رقم الحساب</TableHead>
                      <TableHead className="text-right">اسم الحساب</TableHead>
                      <TableHead className="text-left w-40">المبلغ</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredRevenues.length > 0 ? (
                      filteredRevenues.map((item) => (
                        <TableRow 
                          key={item.accountCode} 
                          data-testid={`row-revenue-${item.accountCode}`}
                          className={item.isParent ? "bg-muted/30 font-medium" : ""}
                        >
                          <TableCell className="font-mono">
                            <div className={getIndentClass(item.level)}>
                              {item.level > 1 && <ChevronLeft className="inline h-3 w-3 ml-1 text-muted-foreground" />}
                              {item.accountCode}
                            </div>
                          </TableCell>
                          <TableCell className={item.isParent ? "font-semibold" : ""}>
                            <div className={getIndentClass(item.level)}>
                              {item.accountName}
                              {item.isParent && (
                                <Badge variant="outline" className="mr-2 text-xs">
                                  رئيسي
                                </Badge>
                              )}
                            </div>
                          </TableCell>
                          <TableCell className={`font-mono text-left ${item.amount > 0 ? 'text-green-600' : item.amount < 0 ? 'text-red-600' : 'text-muted-foreground'}`}>
                            {formatNumber(item.amount)}
                          </TableCell>
                        </TableRow>
                      ))
                    ) : (
                      <TableRow>
                        <TableCell colSpan={3} className="text-center text-muted-foreground">
                          {hideZeroValues ? "لا توجد إيرادات بقيم غير صفرية" : "لا توجد حسابات إيرادات"}
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                  <TableFooter>
                    <TableRow className="bg-green-50 dark:bg-green-950">
                      <TableCell colSpan={2} className="font-bold">إجمالي الإيرادات</TableCell>
                      <TableCell className="font-mono font-bold text-left text-green-600">
                        {formatCurrency(incomeStatement?.totalRevenue || 0)}
                      </TableCell>
                    </TableRow>
                  </TableFooter>
                </Table>
              </div>

              <div>
                <h3 className="text-lg font-semibold mb-4 text-red-700 dark:text-red-400">
                  المصروفات (5)
                </h3>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="text-right w-32">رقم الحساب</TableHead>
                      <TableHead className="text-right">اسم الحساب</TableHead>
                      <TableHead className="text-left w-40">المبلغ</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredExpenses.length > 0 ? (
                      filteredExpenses.map((item) => (
                        <TableRow 
                          key={item.accountCode} 
                          data-testid={`row-expense-${item.accountCode}`}
                          className={item.isParent ? "bg-muted/30 font-medium" : ""}
                        >
                          <TableCell className="font-mono">
                            <div className={getIndentClass(item.level)}>
                              {item.level > 1 && <ChevronLeft className="inline h-3 w-3 ml-1 text-muted-foreground" />}
                              {item.accountCode}
                            </div>
                          </TableCell>
                          <TableCell className={item.isParent ? "font-semibold" : ""}>
                            <div className={getIndentClass(item.level)}>
                              {item.accountName}
                              {item.isParent && (
                                <Badge variant="outline" className="mr-2 text-xs">
                                  رئيسي
                                </Badge>
                              )}
                            </div>
                          </TableCell>
                          <TableCell className={`font-mono text-left ${item.amount > 0 ? 'text-red-600' : item.amount < 0 ? 'text-green-600' : 'text-muted-foreground'}`}>
                            {formatNumber(item.amount)}
                          </TableCell>
                        </TableRow>
                      ))
                    ) : (
                      <TableRow>
                        <TableCell colSpan={3} className="text-center text-muted-foreground">
                          {hideZeroValues ? "لا توجد مصروفات بقيم غير صفرية" : "لا توجد حسابات مصروفات"}
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                  <TableFooter>
                    <TableRow className="bg-red-50 dark:bg-red-950">
                      <TableCell colSpan={2} className="font-bold">إجمالي المصروفات</TableCell>
                      <TableCell className="font-mono font-bold text-left text-red-600">
                        {formatCurrency(incomeStatement?.totalExpenses || 0)}
                      </TableCell>
                    </TableRow>
                  </TableFooter>
                </Table>
              </div>

              <Card className={`${(incomeStatement?.netIncome || 0) >= 0 ? 'bg-blue-50 dark:bg-blue-950 border-blue-200' : 'bg-orange-50 dark:bg-orange-950 border-orange-200'}`}>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <DollarSign className={`h-8 w-8 ${(incomeStatement?.netIncome || 0) >= 0 ? 'text-blue-600' : 'text-orange-600'}`} />
                      <div>
                        <p className="text-lg font-semibold">
                          {(incomeStatement?.netIncome || 0) >= 0 ? 'صافي الربح' : 'صافي الخسارة'}
                        </p>
                        <p className="text-sm text-muted-foreground">
                          إجمالي الإيرادات - إجمالي المصروفات
                        </p>
                      </div>
                    </div>
                    <p className={`text-3xl font-bold ${(incomeStatement?.netIncome || 0) >= 0 ? 'text-blue-600' : 'text-orange-600'}`}>
                      {formatCurrency(incomeStatement?.netIncome || 0)}
                    </p>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
