import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { Plus, Trash2, Check, X, Search, Package, Calendar, User, FileText } from "lucide-react";
import { format } from "date-fns";
import type { InventoryWithdrawalOrderWithDetails, Product, Employee } from "@shared/schema";

const reasonLabels: Record<string, string> = {
  internal_use: "استخدام داخلي",
  damage: "تلف",
  expiry: "انتهاء صلاحية",
  donation: "تبرع",
  other: "أخرى",
};

const statusLabels: Record<string, string> = {
  draft: "مسودة",
  confirmed: "مؤكد",
  cancelled: "ملغي",
};

const statusVariants: Record<string, "default" | "secondary" | "destructive"> = {
  draft: "secondary",
  confirmed: "default",
  cancelled: "destructive",
};

export default function InventoryWithdrawalsPage() {
  const { toast } = useToast();
  const [searchTerm, setSearchTerm] = useState("");
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedOrder, setSelectedOrder] = useState<InventoryWithdrawalOrderWithDetails | null>(null);
  const [confirmDeleteId, setConfirmDeleteId] = useState<number | null>(null);
  const [confirmActionId, setConfirmActionId] = useState<{ id: number; action: 'confirm' | 'cancel' } | null>(null);
  
  const [orderDate, setOrderDate] = useState(new Date().toISOString().split('T')[0]);
  const [reason, setReason] = useState<string>("internal_use");
  const [responsibleEmployeeId, setResponsibleEmployeeId] = useState<string>("");
  const [notes, setNotes] = useState("");
  const [lines, setLines] = useState<Array<{ productId: number; quantity: number; notes?: string }>>([]);
  const [newLine, setNewLine] = useState<{ productId: string; quantity: string; notes: string }>({
    productId: "",
    quantity: "1",
    notes: "",
  });

  const { data: withdrawals = [], isLoading } = useQuery<InventoryWithdrawalOrderWithDetails[]>({
    queryKey: ["/api/inventory-withdrawals"],
  });

  const { data: products = [] } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const { data: employees = [] } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const createMutation = useMutation({
    mutationFn: async (data: any) => {
      return apiRequest("POST", "/api/inventory-withdrawals", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/inventory-withdrawals"] });
      queryClient.invalidateQueries({ queryKey: ["/api/inventory"] });
      toast({ title: "تم إنشاء أمر السحب بنجاح" });
      resetForm();
      setIsDialogOpen(false);
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في إنشاء أمر السحب", variant: "destructive" });
    },
  });

  const confirmMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("POST", `/api/inventory-withdrawals/${id}/confirm`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/inventory-withdrawals"] });
      queryClient.invalidateQueries({ queryKey: ["/api/inventory"] });
      queryClient.invalidateQueries({ queryKey: ["/api/journal-entries"] });
      toast({ title: "تم تأكيد أمر السحب بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في تأكيد أمر السحب", variant: "destructive" });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("POST", `/api/inventory-withdrawals/${id}/cancel`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/inventory-withdrawals"] });
      toast({ title: "تم إلغاء أمر السحب بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في إلغاء أمر السحب", variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/inventory-withdrawals/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/inventory-withdrawals"] });
      toast({ title: "تم حذف أمر السحب بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "خطأ", description: error.message || "فشل في حذف أمر السحب", variant: "destructive" });
    },
  });

  const resetForm = () => {
    setOrderDate(new Date().toISOString().split('T')[0]);
    setReason("internal_use");
    setResponsibleEmployeeId("");
    setNotes("");
    setLines([]);
    setNewLine({ productId: "", quantity: "1", notes: "" });
    setSelectedOrder(null);
  };

  const handleAddLine = () => {
    if (!newLine.productId || !newLine.quantity) return;
    const productId = parseInt(newLine.productId);
    const quantity = parseFloat(newLine.quantity);
    if (isNaN(productId) || isNaN(quantity) || quantity <= 0) return;
    
    setLines([...lines, { productId, quantity, notes: newLine.notes || undefined }]);
    setNewLine({ productId: "", quantity: "1", notes: "" });
  };

  const handleRemoveLine = (index: number) => {
    setLines(lines.filter((_, i) => i !== index));
  };

  const handleSubmit = () => {
    if (lines.length === 0) {
      toast({ title: "خطأ", description: "يجب إضافة منتج واحد على الأقل", variant: "destructive" });
      return;
    }
    
    createMutation.mutate({
      orderDate,
      reason,
      responsibleEmployeeId: responsibleEmployeeId ? parseInt(responsibleEmployeeId) : null,
      notes: notes || null,
      lines,
    });
  };

  const getProductName = (productId: number) => {
    const product = products.find(p => p.id === productId);
    return product?.nameAr || "غير معروف";
  };

  const getEmployeeName = (employeeId: number | null | undefined) => {
    if (!employeeId) return "-";
    const employee = employees.find(e => e.id === employeeId);
    return employee?.fullNameAr || employee?.nameAr || "-";
  };

  const formatCurrency = (amount: number) => 
    new Intl.NumberFormat("en-SA", { style: "currency", currency: "SAR" }).format(amount);

  const filteredWithdrawals = withdrawals.filter(w =>
    w.orderNumber?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    reasonLabels[w.reason]?.includes(searchTerm) ||
    statusLabels[w.status]?.includes(searchTerm)
  );

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary"></div>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold">أوامر سحب المخزون</h1>
          <p className="text-muted-foreground">إدارة سحب المنتجات من المخزون للاستخدام الداخلي أو التلف أو التبرع</p>
        </div>
        <Button onClick={() => { resetForm(); setIsDialogOpen(true); }} data-testid="button-new-withdrawal">
          <Plus className="h-4 w-4 ml-2" />
          أمر سحب جديد
        </Button>
      </div>

      <Card>
        <CardHeader>
          <div className="flex items-center gap-4">
            <div className="relative flex-1">
              <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="بحث برقم الأمر أو السبب..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pr-10"
                data-testid="input-search"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>رقم الأمر</TableHead>
                <TableHead>التاريخ</TableHead>
                <TableHead>السبب</TableHead>
                <TableHead>المسؤول</TableHead>
                <TableHead>إجمالي التكلفة</TableHead>
                <TableHead>الحالة</TableHead>
                <TableHead>الإجراءات</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredWithdrawals.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={7} className="text-center text-muted-foreground py-8">
                    لا توجد أوامر سحب
                  </TableCell>
                </TableRow>
              ) : (
                filteredWithdrawals.map((w) => (
                  <TableRow key={w.id} data-testid={`row-withdrawal-${w.id}`}>
                    <TableCell className="font-medium">{w.orderNumber}</TableCell>
                    <TableCell>{format(new Date(w.orderDate), "yyyy-MM-dd")}</TableCell>
                    <TableCell>{reasonLabels[w.reason] || w.reason}</TableCell>
                    <TableCell>{getEmployeeName(w.responsibleEmployeeId)}</TableCell>
                    <TableCell>{formatCurrency(w.totalCost)}</TableCell>
                    <TableCell>
                      <Badge variant={statusVariants[w.status]}>{statusLabels[w.status]}</Badge>
                    </TableCell>
                    <TableCell>
                      <div className="flex gap-1">
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => { setSelectedOrder(w); }}
                          data-testid={`button-view-${w.id}`}
                        >
                          <FileText className="h-4 w-4" />
                        </Button>
                        {w.status === 'draft' && (
                          <>
                            <Button
                              size="icon"
                              variant="ghost"
                              className="text-green-600"
                              onClick={() => setConfirmActionId({ id: w.id, action: 'confirm' })}
                              data-testid={`button-confirm-${w.id}`}
                            >
                              <Check className="h-4 w-4" />
                            </Button>
                            <Button
                              size="icon"
                              variant="ghost"
                              className="text-red-600"
                              onClick={() => setConfirmDeleteId(w.id)}
                              data-testid={`button-delete-${w.id}`}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </>
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl" dir="rtl">
          <DialogHeader>
            <DialogTitle>أمر سحب جديد</DialogTitle>
            <DialogDescription>
              إنشاء أمر سحب منتجات من المخزون
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>التاريخ</Label>
                <Input
                  type="date"
                  value={orderDate}
                  onChange={(e) => setOrderDate(e.target.value)}
                  data-testid="input-order-date"
                />
              </div>
              <div className="space-y-2">
                <Label>سبب السحب</Label>
                <Select value={reason} onValueChange={setReason} data-testid="select-reason">
                  <SelectTrigger>
                    <SelectValue placeholder="اختر السبب" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="internal_use">استخدام داخلي</SelectItem>
                    <SelectItem value="damage">تلف</SelectItem>
                    <SelectItem value="expiry">انتهاء صلاحية</SelectItem>
                    <SelectItem value="donation">تبرع</SelectItem>
                    <SelectItem value="other">أخرى</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
            
            <div className="space-y-2">
              <Label>الموظف المسؤول (اختياري)</Label>
              <Select value={responsibleEmployeeId} onValueChange={setResponsibleEmployeeId}>
                <SelectTrigger data-testid="select-employee">
                  <SelectValue placeholder="اختر الموظف" />
                </SelectTrigger>
                <SelectContent>
                  {employees.filter(e => e.isActive).map((e) => (
                    <SelectItem key={e.id} value={e.id.toString()}>
                      {e.fullNameAr || e.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label>ملاحظات</Label>
              <Textarea
                value={notes}
                onChange={(e) => setNotes(e.target.value)}
                placeholder="ملاحظات إضافية..."
                data-testid="input-notes"
              />
            </div>
            
            <div className="border rounded-lg p-4 space-y-4">
              <h4 className="font-medium">المنتجات</h4>
              <div className="grid grid-cols-12 gap-2">
                <div className="col-span-5">
                  <Select value={newLine.productId} onValueChange={(v) => setNewLine({ ...newLine, productId: v })}>
                    <SelectTrigger data-testid="select-product">
                      <SelectValue placeholder="اختر المنتج" />
                    </SelectTrigger>
                    <SelectContent>
                      {products.map((p) => (
                        <SelectItem key={p.id} value={p.id.toString()}>
                          {p.nameAr}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="col-span-3">
                  <Input
                    type="number"
                    placeholder="الكمية"
                    value={newLine.quantity}
                    onChange={(e) => setNewLine({ ...newLine, quantity: e.target.value })}
                    min="0.01"
                    step="0.01"
                    data-testid="input-quantity"
                  />
                </div>
                <div className="col-span-3">
                  <Input
                    placeholder="ملاحظة"
                    value={newLine.notes}
                    onChange={(e) => setNewLine({ ...newLine, notes: e.target.value })}
                    data-testid="input-line-notes"
                  />
                </div>
                <div className="col-span-1">
                  <Button
                    type="button"
                    size="icon"
                    onClick={handleAddLine}
                    disabled={!newLine.productId || !newLine.quantity}
                    data-testid="button-add-line"
                  >
                    <Plus className="h-4 w-4" />
                  </Button>
                </div>
              </div>
              
              {lines.length > 0 && (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>المنتج</TableHead>
                      <TableHead>الكمية</TableHead>
                      <TableHead>ملاحظة</TableHead>
                      <TableHead className="w-[50px]"></TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {lines.map((line, index) => (
                      <TableRow key={index}>
                        <TableCell>{getProductName(line.productId)}</TableCell>
                        <TableCell>{line.quantity}</TableCell>
                        <TableCell>{line.notes || "-"}</TableCell>
                        <TableCell>
                          <Button
                            size="icon"
                            variant="ghost"
                            className="text-red-600"
                            onClick={() => handleRemoveLine(index)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              إلغاء
            </Button>
            <Button onClick={handleSubmit} disabled={createMutation.isPending || lines.length === 0} data-testid="button-submit">
              {createMutation.isPending ? "جاري الإنشاء..." : "إنشاء أمر السحب"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={!!selectedOrder} onOpenChange={(open) => !open && setSelectedOrder(null)}>
        <DialogContent className="max-w-2xl" dir="rtl">
          <DialogHeader>
            <DialogTitle>تفاصيل أمر السحب - {selectedOrder?.orderNumber}</DialogTitle>
          </DialogHeader>
          {selectedOrder && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="flex items-center gap-2">
                  <Calendar className="h-4 w-4 text-muted-foreground" />
                  <span className="text-sm text-muted-foreground">التاريخ:</span>
                  <span>{format(new Date(selectedOrder.orderDate), "yyyy-MM-dd")}</span>
                </div>
                <div className="flex items-center gap-2">
                  <Package className="h-4 w-4 text-muted-foreground" />
                  <span className="text-sm text-muted-foreground">السبب:</span>
                  <span>{reasonLabels[selectedOrder.reason] || selectedOrder.reason}</span>
                </div>
                <div className="flex items-center gap-2">
                  <User className="h-4 w-4 text-muted-foreground" />
                  <span className="text-sm text-muted-foreground">المسؤول:</span>
                  <span>{getEmployeeName(selectedOrder.responsibleEmployeeId)}</span>
                </div>
                <div>
                  <Badge variant={statusVariants[selectedOrder.status]}>
                    {statusLabels[selectedOrder.status]}
                  </Badge>
                </div>
              </div>
              
              {selectedOrder.notes && (
                <div className="p-3 bg-muted rounded-lg">
                  <p className="text-sm">{selectedOrder.notes}</p>
                </div>
              )}
              
              <div className="border rounded-lg">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>المنتج</TableHead>
                      <TableHead>الكمية</TableHead>
                      <TableHead>تكلفة الوحدة</TableHead>
                      <TableHead>إجمالي التكلفة</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {selectedOrder.lines.map((line) => (
                      <TableRow key={line.id}>
                        <TableCell>{line.product?.nameAr || "غير معروف"}</TableCell>
                        <TableCell>{line.quantity}</TableCell>
                        <TableCell>{formatCurrency(line.unitCost)}</TableCell>
                        <TableCell>{formatCurrency(line.totalCost)}</TableCell>
                      </TableRow>
                    ))}
                    <TableRow className="font-bold">
                      <TableCell colSpan={3}>الإجمالي</TableCell>
                      <TableCell>{formatCurrency(selectedOrder.totalCost)}</TableCell>
                    </TableRow>
                  </TableBody>
                </Table>
              </div>
              
              {selectedOrder.journalEntry && (
                <div className="p-3 bg-green-50 dark:bg-green-900/20 rounded-lg">
                  <p className="text-sm text-green-800 dark:text-green-200">
                    تم إنشاء قيد محاسبي رقم: {selectedOrder.journalEntry.id}
                  </p>
                </div>
              )}
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => setSelectedOrder(null)}>
              إغلاق
            </Button>
            {selectedOrder?.status === 'draft' && (
              <Button
                onClick={() => {
                  setSelectedOrder(null);
                  setConfirmActionId({ id: selectedOrder.id, action: 'confirm' });
                }}
                data-testid="button-confirm-from-dialog"
              >
                <Check className="h-4 w-4 ml-2" />
                تأكيد الأمر
              </Button>
            )}
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <AlertDialog open={confirmDeleteId !== null} onOpenChange={(open) => !open && setConfirmDeleteId(null)}>
        <AlertDialogContent dir="rtl">
          <AlertDialogHeader>
            <AlertDialogTitle>هل أنت متأكد من حذف أمر السحب؟</AlertDialogTitle>
            <AlertDialogDescription>
              لا يمكن التراجع عن هذا الإجراء.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>إلغاء</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => {
                if (confirmDeleteId) {
                  deleteMutation.mutate(confirmDeleteId);
                  setConfirmDeleteId(null);
                }
              }}
            >
              حذف
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <AlertDialog open={confirmActionId !== null} onOpenChange={(open) => !open && setConfirmActionId(null)}>
        <AlertDialogContent dir="rtl">
          <AlertDialogHeader>
            <AlertDialogTitle>
              {confirmActionId?.action === 'confirm' ? 'تأكيد أمر السحب' : 'إلغاء أمر السحب'}
            </AlertDialogTitle>
            <AlertDialogDescription>
              {confirmActionId?.action === 'confirm'
                ? 'سيتم إنشاء قيد محاسبي وخصم الكميات من المخزون. لا يمكن التراجع عن هذا الإجراء.'
                : 'سيتم إلغاء أمر السحب. لا يمكن التراجع عن هذا الإجراء.'}
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>رجوع</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => {
                if (confirmActionId) {
                  if (confirmActionId.action === 'confirm') {
                    confirmMutation.mutate(confirmActionId.id);
                  } else {
                    cancelMutation.mutate(confirmActionId.id);
                  }
                  setConfirmActionId(null);
                }
              }}
            >
              {confirmActionId?.action === 'confirm' ? 'تأكيد' : 'إلغاء'}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
