import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Progress } from "@/components/ui/progress";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Tabs,
  TabsContent,
  TabsList,
  TabsTrigger,
} from "@/components/ui/tabs";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Search, Package, Warehouse, AlertTriangle, Clock, ChevronDown, ArrowUpDown, ShoppingCart, Scissors, PackageOpen } from "lucide-react";
import type { InventorySummaryItem, StockBatch, Product } from "@shared/schema";

type StockBatchWithProduct = StockBatch & { product?: Product };

function InventorySkeleton() {
  return (
    <TableRow>
      <TableCell><Skeleton className="h-4 w-32" /></TableCell>
      <TableCell><Skeleton className="h-4 w-20" /></TableCell>
      <TableCell><Skeleton className="h-4 w-16" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
      <TableCell><Skeleton className="h-4 w-24" /></TableCell>
    </TableRow>
  );
}

function EmptyState() {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <Warehouse className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا يوجد مخزون</h3>
      <p className="text-muted-foreground mb-4">ابدأ بإنشاء فواتير شراء لإضافة مخزون</p>
      <Button asChild>
        <Link href="/purchase-invoices/new">إنشاء فاتورة شراء</Link>
      </Button>
    </div>
  );
}

const usageTypeLabels: Record<string, { label: string; color: string; icon: typeof Package }> = {
  sale: { label: "للبيع", color: "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400", icon: ShoppingCart },
  consume: { label: "مستهلك", color: "bg-blue-100 text-blue-700 dark:bg-blue-900/30 dark:text-blue-400", icon: Scissors },
  sale_and_consume: { label: "للبيع والاستهلاك", color: "bg-purple-100 text-purple-700 dark:bg-purple-900/30 dark:text-purple-400", icon: PackageOpen },
};

function InventorySummaryTable({
  items,
  isLoading,
  searchQuery,
  usageTypeFilter,
}: {
  items?: InventorySummaryItem[];
  isLoading: boolean;
  searchQuery: string;
  usageTypeFilter: string;
}) {
  const filteredItems = items ? items.filter((item) => {
    const matchesSearch = item.productNameAr.includes(searchQuery) ||
      item.sku?.includes(searchQuery) ||
      item.brandNameAr?.includes(searchQuery);
    const matchesType = usageTypeFilter === "all" || item.usageType === usageTypeFilter;
    return matchesSearch && matchesType;
  }) : [];

  if (isLoading) {
    return (
      <div className="rounded-lg border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead className="text-right">المنتج</TableHead>
              <TableHead className="text-right">النوع</TableHead>
              <TableHead className="text-right">الكمية</TableHead>
              <TableHead className="text-right">إجمالي التكلفة</TableHead>
              <TableHead className="text-right">متوسط التكلفة</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {[1, 2, 3, 4, 5].map((i) => (
              <InventorySkeleton key={i} />
            ))}
          </TableBody>
        </Table>
      </div>
    );
  }

  if (filteredItems.length === 0 && !isLoading) {
    return <EmptyState />;
  }

  return (
    <div className="rounded-lg border">
      <Table>
        <TableHeader>
          <TableRow>
            <TableHead className="text-right">المنتج</TableHead>
            <TableHead className="text-right">النوع</TableHead>
            <TableHead className="text-right">الكمية</TableHead>
            <TableHead className="text-right">إجمالي التكلفة</TableHead>
            <TableHead className="text-right">متوسط التكلفة</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {filteredItems.map((item) => {
            const typeInfo = usageTypeLabels[item.usageType] || usageTypeLabels.sale;
            const TypeIcon = typeInfo.icon;
            return (
              <TableRow key={item.productId} data-testid={`inventory-row-${item.productId}`}>
                <TableCell>
                  <div className="flex items-center gap-3">
                    <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-primary/10 text-primary">
                      <Package className="h-5 w-5" />
                    </div>
                    <div>
                      <Link
                        href={`/products/${item.productId}`}
                        className="font-medium text-primary hover:underline"
                      >
                        {item.productNameAr}
                      </Link>
                      <div className="flex items-center gap-2">
                        {item.sku && (
                          <span className="text-xs text-muted-foreground font-mono">{item.sku}</span>
                        )}
                        {item.brandNameAr && (
                          <Badge variant="outline" className="text-xs">{item.brandNameAr}</Badge>
                        )}
                      </div>
                    </div>
                  </div>
                </TableCell>
                <TableCell>
                  <Badge className={`${typeInfo.color} border-0`} data-testid={`badge-usage-type-${item.productId}`}>
                    <TypeIcon className="h-3 w-3 ml-1" />
                    {typeInfo.label}
                  </Badge>
                </TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <span className="font-mono font-medium tabular-nums">
                      {item.quantityOnHand.toLocaleString("ar-SA")}
                    </span>
                    <span className="text-muted-foreground text-sm">
                      {item.sellUnitNameAr}
                    </span>
                    {item.quantityOnHand <= 10 && (
                      <Badge variant="destructive" className="text-xs">
                        <AlertTriangle className="h-3 w-3 ml-1" />
                        منخفض
                      </Badge>
                    )}
                  </div>
                </TableCell>
                <TableCell className="font-mono tabular-nums">
                  {item.totalCost.toLocaleString("ar-SA", { minimumFractionDigits: 2 })} ر.س
                </TableCell>
                <TableCell className="font-mono tabular-nums">
                  {item.quantityOnHand > 0
                    ? (item.totalCost / item.quantityOnHand).toLocaleString("ar-SA", {
                        minimumFractionDigits: 2,
                      })
                    : "0.00"}{" "}
                  ر.س
                </TableCell>
              </TableRow>
            );
          })}
        </TableBody>
      </Table>
    </div>
  );
}

function BatchesTable({
  batches,
  isLoading,
  searchQuery,
}: {
  batches?: StockBatchWithProduct[];
  isLoading: boolean;
  searchQuery: string;
}) {
  const filteredBatches = batches?.filter((batch) =>
    batch.product?.nameAr?.includes(searchQuery)
  );

  const now = new Date();
  const thirtyDaysFromNow = new Date(now.getTime() + 30 * 24 * 60 * 60 * 1000);

  const getBatchStatus = (expiryDate?: string) => {
    if (!expiryDate) return null;
    const expiry = new Date(expiryDate);
    if (expiry < now) return "expired";
    if (expiry < thirtyDaysFromNow) return "expiring";
    return "ok";
  };

  if (isLoading) {
    return (
      <div className="rounded-lg border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead className="text-right">المنتج</TableHead>
              <TableHead className="text-right">الكمية</TableHead>
              <TableHead className="text-right">تكلفة الوحدة</TableHead>
              <TableHead className="text-right">تاريخ الانتهاء</TableHead>
              <TableHead className="text-right">الحالة</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {[1, 2, 3, 4].map((i) => (
              <InventorySkeleton key={i} />
            ))}
          </TableBody>
        </Table>
      </div>
    );
  }

  if (!filteredBatches || filteredBatches.length === 0) {
    return (
      <div className="flex flex-col items-center justify-center py-12 text-center text-muted-foreground">
        <Package className="h-12 w-12 mb-4 opacity-50" />
        <p>لا توجد دفعات مخزون</p>
      </div>
    );
  }

  return (
    <div className="rounded-lg border">
      <Table>
        <TableHeader>
          <TableRow>
            <TableHead className="text-right">المنتج</TableHead>
            <TableHead className="text-right">الكمية</TableHead>
            <TableHead className="text-right">تكلفة الوحدة</TableHead>
            <TableHead className="text-right">تاريخ الانتهاء</TableHead>
            <TableHead className="text-right">الحالة</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {filteredBatches.map((batch) => {
            const status = getBatchStatus(batch.expiryDate ?? undefined);
            return (
              <TableRow key={batch.id} data-testid={`batch-row-${batch.id}`}>
                <TableCell>
                  <Link
                    href={`/products/${batch.productId}`}
                    className="font-medium text-primary hover:underline"
                  >
                    {batch.product?.nameAr}
                  </Link>
                </TableCell>
                <TableCell className="font-mono tabular-nums">
                  {batch.quantityOnHand.toLocaleString("ar-SA")}
                </TableCell>
                <TableCell className="font-mono tabular-nums">
                  {batch.unitCost.toLocaleString("ar-SA", { minimumFractionDigits: 2 })} ر.س
                </TableCell>
                <TableCell>
                  {batch.expiryDate
                    ? new Date(batch.expiryDate).toLocaleDateString("ar-SA")
                    : "-"}
                </TableCell>
                <TableCell>
                  {status === "expired" && (
                    <Badge variant="destructive">منتهي الصلاحية</Badge>
                  )}
                  {status === "expiring" && (
                    <Badge variant="secondary">
                      <Clock className="h-3 w-3 ml-1" />
                      قارب على الانتهاء
                    </Badge>
                  )}
                  {status === "ok" && <Badge variant="outline">صالح</Badge>}
                  {!status && <span className="text-muted-foreground">-</span>}
                </TableCell>
              </TableRow>
            );
          })}
        </TableBody>
      </Table>
    </div>
  );
}

export default function InventoryPage() {
  const [searchQuery, setSearchQuery] = useState("");
  const [activeTab, setActiveTab] = useState("summary");
  const [usageTypeFilter, setUsageTypeFilter] = useState("all");

  const { data: summary, isLoading: isLoadingSummary } = useQuery<InventorySummaryItem[]>({
    queryKey: ["/api/inventory"],
  });

  const { data: batches, isLoading: isLoadingBatches } = useQuery<StockBatchWithProduct[]>({
    queryKey: ["/api/stock-batches"],
  });

  const totalValue = summary?.reduce((sum, item) => sum + item.totalCost, 0) || 0;
  const totalItems = summary?.reduce((sum, item) => sum + item.quantityOnHand, 0) || 0;
  const lowStockCount = summary?.filter((item) => item.quantityOnHand <= 10).length || 0;

  // Calculate stats by usage type
  const saleProducts = summary?.filter(item => item.usageType === "sale") || [];
  const consumeProducts = summary?.filter(item => item.usageType === "consume") || [];
  const saleAndConsumeProducts = summary?.filter(item => item.usageType === "sale_and_consume") || [];

  const saleValue = saleProducts.reduce((sum, item) => sum + item.totalCost, 0);
  const consumeValue = consumeProducts.reduce((sum, item) => sum + item.totalCost, 0);
  const saleAndConsumeValue = saleAndConsumeProducts.reduce((sum, item) => sum + item.totalCost, 0);

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">المخزون</h1>
          <p className="text-muted-foreground">إدارة ومتابعة المخزون والدفعات</p>
        </div>
      </div>

      {/* Main Stats */}
      <div className="grid gap-4 md:grid-cols-3">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10">
                <Warehouse className="h-6 w-6 text-primary" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">قيمة المخزون الإجمالية</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-value">
                  {totalValue.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-green-500/10">
                <Package className="h-6 w-6 text-green-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الوحدات</p>
                <p className="text-2xl font-bold tabular-nums">
                  {totalItems.toLocaleString("ar-SA")}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-amber-500/10">
                <AlertTriangle className="h-6 w-6 text-amber-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">مخزون منخفض</p>
                <p className="text-2xl font-bold tabular-nums">
                  {lowStockCount} منتج
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Product Type Breakdown */}
      <Card>
        <CardHeader className="pb-4">
          <CardTitle className="text-lg flex items-center gap-2">
            <PackageOpen className="h-5 w-5" />
            تفصيل المخزون حسب نوع المنتج
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-3">
            <div className="flex items-center justify-between p-4 rounded-lg bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800" data-testid="card-sale-products">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-green-100 dark:bg-green-900/40">
                  <ShoppingCart className="h-5 w-5 text-green-600 dark:text-green-400" />
                </div>
                <div>
                  <p className="text-sm text-green-700 dark:text-green-300">منتجات للبيع</p>
                  <p className="text-lg font-bold text-green-800 dark:text-green-200" data-testid="text-sale-products-count">
                    {saleProducts.length} منتج
                  </p>
                </div>
              </div>
              <div className="text-left">
                <p className="text-sm text-muted-foreground">القيمة</p>
                <p className="font-bold tabular-nums text-green-700 dark:text-green-300" data-testid="text-sale-products-value">
                  {saleValue.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>

            <div className="flex items-center justify-between p-4 rounded-lg bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800" data-testid="card-consume-products">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-blue-100 dark:bg-blue-900/40">
                  <Scissors className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                </div>
                <div>
                  <p className="text-sm text-blue-700 dark:text-blue-300">منتجات مستهلكة</p>
                  <p className="text-lg font-bold text-blue-800 dark:text-blue-200" data-testid="text-consume-products-count">
                    {consumeProducts.length} منتج
                  </p>
                </div>
              </div>
              <div className="text-left">
                <p className="text-sm text-muted-foreground">القيمة</p>
                <p className="font-bold tabular-nums text-blue-700 dark:text-blue-300" data-testid="text-consume-products-value">
                  {consumeValue.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>

            <div className="flex items-center justify-between p-4 rounded-lg bg-purple-50 dark:bg-purple-900/20 border border-purple-200 dark:border-purple-800" data-testid="card-both-products">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-purple-100 dark:bg-purple-900/40">
                  <PackageOpen className="h-5 w-5 text-purple-600 dark:text-purple-400" />
                </div>
                <div>
                  <p className="text-sm text-purple-700 dark:text-purple-300">للبيع والاستهلاك</p>
                  <p className="text-lg font-bold text-purple-800 dark:text-purple-200" data-testid="text-both-products-count">
                    {saleAndConsumeProducts.length} منتج
                  </p>
                </div>
              </div>
              <div className="text-left">
                <p className="text-sm text-muted-foreground">القيمة</p>
                <p className="font-bold tabular-nums text-purple-700 dark:text-purple-300" data-testid="text-both-products-value">
                  {saleAndConsumeValue.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">تفاصيل المخزون</CardTitle>
            <div className="flex flex-col sm:flex-row gap-3">
              <Select value={usageTypeFilter} onValueChange={setUsageTypeFilter}>
                <SelectTrigger className="w-full sm:w-44" data-testid="select-usage-type">
                  <SelectValue placeholder="نوع المنتج" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">جميع الأنواع</SelectItem>
                  <SelectItem value="sale">للبيع فقط</SelectItem>
                  <SelectItem value="consume">مستهلك فقط</SelectItem>
                  <SelectItem value="sale_and_consume">للبيع والاستهلاك</SelectItem>
                </SelectContent>
              </Select>
              <div className="relative w-full sm:w-72">
                <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                  placeholder="بحث بالمنتج أو العلامة التجارية..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pr-10"
                  data-testid="input-search-inventory"
                />
              </div>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="mb-4">
              <TabsTrigger value="summary" data-testid="tab-summary">
                <Warehouse className="h-4 w-4 ml-2" />
                ملخص المخزون
              </TabsTrigger>
              <TabsTrigger value="batches" data-testid="tab-batches">
                <Package className="h-4 w-4 ml-2" />
                الدفعات
              </TabsTrigger>
            </TabsList>
            <TabsContent value="summary">
              <InventorySummaryTable
                items={summary}
                isLoading={isLoadingSummary}
                searchQuery={searchQuery}
                usageTypeFilter={usageTypeFilter}
              />
            </TabsContent>
            <TabsContent value="batches">
              <BatchesTable
                batches={batches}
                isLoading={isLoadingBatches}
                searchQuery={searchQuery}
              />
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>
    </div>
  );
}
