import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  Building2,
  Phone,
  MapPin,
  FileText,
  Save,
  Loader2,
  AlertCircle,
  CheckCircle2,
} from "lucide-react";

interface SystemSetting {
  id: number;
  key: string;
  value: string;
  description: string | null;
  valueType: string;
}

const invoiceSettingsKeys = [
  { key: "company_name_ar", label: "اسم الشركة (عربي)", placeholder: "مثال: صالون قمر سمايا النسائي", required: true },
  { key: "company_phone", label: "رقم الجوال", placeholder: "05xxxxxxxx", required: false },
  { key: "company_branch_name", label: "اسم الفرع", placeholder: "مثال: الفرع الرئيسي", required: false },
  { key: "company_vat_number", label: "الرقم الضريبي", placeholder: "3xxxxxxxxxxxxxxx", required: true },
  { key: "company_cr_number", label: "السجل التجاري", placeholder: "رقم السجل التجاري", required: true },
  { key: "company_address", label: "العنوان", placeholder: "مثال: شارع الملك فهد", required: false },
  { key: "company_city", label: "المدينة", placeholder: "مثال: الرياض", required: false },
  { key: "invoice_extra_note", label: "ملاحظة أسفل الفاتورة", placeholder: "ملاحظة اختيارية تظهر أسفل الفاتورة", required: false },
];

export default function InvoiceSettingsPage() {
  const { toast } = useToast();
  const [formData, setFormData] = useState<Record<string, string>>({});
  const [isSaving, setIsSaving] = useState(false);

  const { data: settings, isLoading } = useQuery<SystemSetting[]>({
    queryKey: ["/api/settings"],
  });

  useEffect(() => {
    if (settings) {
      const initialData: Record<string, string> = {};
      for (const setting of invoiceSettingsKeys) {
        const found = settings.find(s => s.key === setting.key);
        initialData[setting.key] = found?.value || "";
      }
      setFormData(initialData);
    }
  }, [settings]);

  const saveMutation = useMutation({
    mutationFn: async ({ key, value }: { key: string; value: string }) => {
      const response = await apiRequest("PUT", `/api/settings/${key}`, {
        value,
        description: invoiceSettingsKeys.find(k => k.key === key)?.label || key,
        valueType: "string",
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/settings"] });
    },
  });

  const handleSave = async () => {
    setIsSaving(true);
    try {
      for (const setting of invoiceSettingsKeys) {
        const value = formData[setting.key] || "";
        await saveMutation.mutateAsync({ key: setting.key, value });
      }
      toast({
        title: "تم الحفظ بنجاح",
        description: "تم حفظ إعدادات الفاتورة",
      });
    } catch (error) {
      toast({
        title: "خطأ",
        description: "حدث خطأ أثناء حفظ الإعدادات",
        variant: "destructive",
      });
    } finally {
      setIsSaving(false);
    }
  };

  const handleChange = (key: string, value: string) => {
    setFormData(prev => ({ ...prev, [key]: value }));
  };

  const requiredFields = invoiceSettingsKeys.filter(k => k.required);
  const optionalFields = invoiceSettingsKeys.filter(k => !k.required);
  
  const missingRequired = requiredFields.filter(
    field => !formData[field.key]?.trim()
  );

  if (isLoading) {
    return (
      <div className="container mx-auto p-4 max-w-4xl" dir="rtl">
        <Skeleton className="h-8 w-64 mb-6" />
        <Skeleton className="h-[400px]" />
      </div>
    );
  }

  return (
    <div className="container mx-auto p-4 max-w-4xl" dir="rtl">
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-bold flex items-center gap-2">
            <FileText className="h-6 w-6" />
            إعدادات الفاتورة
          </h1>
          <p className="text-muted-foreground mt-1">
            معلومات المنشأة التي ستظهر في الفاتورة الضريبية المبسطة
          </p>
        </div>
        <Button onClick={handleSave} disabled={isSaving} data-testid="button-save-settings">
          {isSaving ? (
            <Loader2 className="h-4 w-4 ml-2 animate-spin" />
          ) : (
            <Save className="h-4 w-4 ml-2" />
          )}
          حفظ الإعدادات
        </Button>
      </div>

      {missingRequired.length > 0 && (
        <Card className="mb-6 border-yellow-300 bg-yellow-50 dark:bg-yellow-950 dark:border-yellow-700">
          <CardContent className="p-4 flex items-center gap-3">
            <AlertCircle className="h-5 w-5 text-yellow-600 flex-shrink-0" />
            <div>
              <p className="font-medium text-yellow-800 dark:text-yellow-200">
                معلومات مطلوبة غير مكتملة
              </p>
              <p className="text-sm text-yellow-700 dark:text-yellow-300">
                يرجى إكمال الحقول التالية للامتثال لمتطلبات هيئة الزكاة والضريبة والجمارك:
                {" "}
                {missingRequired.map(f => f.label).join("، ")}
              </p>
            </div>
          </CardContent>
        </Card>
      )}

      {missingRequired.length === 0 && (
        <Card className="mb-6 border-green-300 bg-green-50 dark:bg-green-950 dark:border-green-700">
          <CardContent className="p-4 flex items-center gap-3">
            <CheckCircle2 className="h-5 w-5 text-green-600 flex-shrink-0" />
            <div>
              <p className="font-medium text-green-800 dark:text-green-200">
                جميع المعلومات المطلوبة مكتملة
              </p>
              <p className="text-sm text-green-700 dark:text-green-300">
                إعدادات الفاتورة متوافقة مع متطلبات ZATCA المرحلة الأولى
              </p>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="grid gap-6">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Building2 className="h-5 w-5" />
              معلومات المنشأة (مطلوبة)
            </CardTitle>
            <CardDescription>
              هذه المعلومات مطلوبة للفاتورة الضريبية المبسطة
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {requiredFields.map((field) => (
              <div key={field.key} className="space-y-2">
                <Label htmlFor={field.key} className="flex items-center gap-1">
                  {field.label}
                  <span className="text-destructive">*</span>
                </Label>
                <Input
                  id={field.key}
                  value={formData[field.key] || ""}
                  onChange={(e) => handleChange(field.key, e.target.value)}
                  placeholder={field.placeholder}
                  data-testid={`input-${field.key}`}
                  className={!formData[field.key]?.trim() ? "border-yellow-400" : ""}
                />
              </div>
            ))}
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Phone className="h-5 w-5" />
              معلومات إضافية (اختيارية)
            </CardTitle>
            <CardDescription>
              معلومات إضافية تظهر في الفاتورة
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {optionalFields.slice(0, 4).map((field) => (
              <div key={field.key} className="space-y-2">
                <Label htmlFor={field.key}>{field.label}</Label>
                <Input
                  id={field.key}
                  value={formData[field.key] || ""}
                  onChange={(e) => handleChange(field.key, e.target.value)}
                  placeholder={field.placeholder}
                  data-testid={`input-${field.key}`}
                />
              </div>
            ))}
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <MapPin className="h-5 w-5" />
              ملاحظات الفاتورة
            </CardTitle>
            <CardDescription>
              نص يظهر أسفل الفاتورة
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Label htmlFor="invoice_extra_note">ملاحظة أسفل الفاتورة</Label>
              <Textarea
                id="invoice_extra_note"
                value={formData["invoice_extra_note"] || ""}
                onChange={(e) => handleChange("invoice_extra_note", e.target.value)}
                placeholder="مثال: شكراً لزيارتكم - نتمنى لكم تجربة سعيدة"
                rows={3}
                data-testid="input-invoice_extra_note"
              />
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
