import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Briefcase, Plus, Pencil, Trash2, CheckCircle2, XCircle, Code } from "lucide-react";

interface JobPosition {
  id: number;
  nameAr: string;
  code: string | null;
  descriptionAr: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface JobPositionFormData {
  nameAr: string;
  code: string;
  descriptionAr: string;
  isActive: boolean;
}

export default function JobPositionsPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingPosition, setEditingPosition] = useState<JobPosition | null>(null);
  const [formData, setFormData] = useState<JobPositionFormData>({
    nameAr: "",
    code: "",
    descriptionAr: "",
    isActive: true,
  });
  const { toast } = useToast();

  const { data: positions, isLoading } = useQuery<JobPosition[]>({
    queryKey: ["/api/hr/job-positions"],
  });

  const createMutation = useMutation({
    mutationFn: (data: JobPositionFormData) =>
      apiRequest("POST", "/api/hr/job-positions", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/job-positions"] });
      toast({ title: "تم إنشاء الوظيفة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الوظيفة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<JobPositionFormData> }) =>
      apiRequest("PUT", `/api/hr/job-positions/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/job-positions"] });
      toast({ title: "تم تحديث الوظيفة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الوظيفة",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/job-positions/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/job-positions"] });
      toast({ title: "تم حذف الوظيفة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الوظيفة",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({ nameAr: "", code: "", descriptionAr: "", isActive: true });
    setEditingPosition(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (position?: JobPosition) => {
    if (position) {
      setEditingPosition(position);
      setFormData({
        nameAr: position.nameAr,
        code: position.code || "",
        descriptionAr: position.descriptionAr || "",
        isActive: position.isActive,
      });
    } else {
      setFormData({ nameAr: "", code: "", descriptionAr: "", isActive: true });
      setEditingPosition(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الوظيفة مطلوب", variant: "destructive" });
      return;
    }

    if (editingPosition) {
      updateMutation.mutate({ id: editingPosition.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const activePositions = positions?.filter(p => p.isActive) || [];
  const inactivePositions = positions?.filter(p => !p.isActive) || [];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الوظائف</h1>
          <p className="text-muted-foreground">تعريف المسميات الوظيفية للموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-job-position">
          <Plus className="h-4 w-4 ml-2" />
          إضافة وظيفة جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Briefcase className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الوظائف</p>
                <p className="text-xl font-bold" data-testid="text-total-positions">
                  {positions?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">وظائف مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-active-positions">
                  {activePositions.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">وظائف غير مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-inactive-positions">
                  {inactivePositions.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Briefcase className="h-5 w-5" />
            قائمة الوظائف
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">اسم الوظيفة</TableHead>
                  <TableHead className="text-right">الرمز</TableHead>
                  <TableHead className="text-right">الوصف</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {positions?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={5} className="text-center py-8 text-muted-foreground">
                      لا توجد وظائف مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  positions?.map((position) => (
                    <TableRow key={position.id} data-testid={`row-job-position-${position.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Briefcase className="h-4 w-4 text-muted-foreground" />
                          {position.nameAr}
                        </div>
                      </TableCell>
                      <TableCell>
                        {position.code ? (
                          <Badge variant="outline" className="font-mono">
                            <Code className="h-3 w-3 ml-1" />
                            {position.code}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className="max-w-xs truncate">
                        {position.descriptionAr || "-"}
                      </TableCell>
                      <TableCell>
                        {position.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            مفعّل
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير مفعّل
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(position)}
                            data-testid={`button-edit-job-position-${position.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(position.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-job-position-${position.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingPosition ? "تعديل الوظيفة" : "إضافة وظيفة جديدة"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="nameAr">اسم الوظيفة <span className="text-destructive">*</span></Label>
              <Input
                id="nameAr"
                value={formData.nameAr}
                onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                placeholder="مثال: مدير الصالون"
                data-testid="input-job-position-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="code">رمز الوظيفة</Label>
              <Input
                id="code"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                placeholder="مثال: MGR"
                className="font-mono text-left"
                dir="ltr"
                data-testid="input-job-position-code"
              />
              <p className="text-xs text-muted-foreground">
                رمز اختياري للاستخدام البرمجي
              </p>
            </div>
            <div className="space-y-2">
              <Label htmlFor="descriptionAr">الوصف</Label>
              <Textarea
                id="descriptionAr"
                value={formData.descriptionAr}
                onChange={(e) => setFormData({ ...formData, descriptionAr: e.target.value })}
                placeholder="وصف مختصر للوظيفة ومسؤولياتها"
                rows={3}
                data-testid="input-job-position-description"
              />
            </div>
            <div className="flex items-center justify-between">
              <Label htmlFor="isActive">الحالة</Label>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">
                  {formData.isActive ? "مفعّل" : "غير مفعّل"}
                </span>
                <Switch
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                  data-testid="switch-job-position-active"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-job-position"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-job-position"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
