import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import { 
  Search, 
  BookOpen, 
  ChevronDown, 
  ChevronUp, 
  FileText, 
  CreditCard, 
  Plus, 
  Trash2,
  AlertCircle,
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { format } from "date-fns";
import type { JournalEntryWithLines, Account } from "@shared/schema";

interface JournalLine {
  accountId: number;
  description: string;
  debit: number;
  credit: number;
}

function JournalEntrySkeleton() {
  return (
    <div className="rounded-lg border p-4 space-y-4">
      <div className="flex items-center justify-between">
        <Skeleton className="h-5 w-32" />
        <Skeleton className="h-5 w-24" />
      </div>
      <Skeleton className="h-20" />
    </div>
  );
}

function EmptyState({ onAddClick }: { onAddClick: () => void }) {
  return (
    <div className="flex flex-col items-center justify-center py-16 text-center">
      <div className="flex h-16 w-16 items-center justify-center rounded-full bg-muted mb-4">
        <BookOpen className="h-8 w-8 text-muted-foreground" />
      </div>
      <h3 className="text-lg font-semibold mb-2">لا توجد قيود محاسبية</h3>
      <p className="text-muted-foreground mb-4">يمكنك إضافة قيد يدوي أو انتظار القيود التلقائية</p>
      <Button onClick={onAddClick} data-testid="button-add-entry-empty">
        <Plus className="h-4 w-4 ml-2" />
        إضافة قيد يدوي
      </Button>
    </div>
  );
}

function JournalEntryCard({ entry }: { entry: JournalEntryWithLines }) {
  const [isOpen, setIsOpen] = useState(false);

  const totalDebit = entry.lines.reduce((sum, line) => sum + line.debit, 0);
  const totalCredit = entry.lines.reduce((sum, line) => sum + line.credit, 0);

  const getTypeIcon = () => {
    if (entry.referenceType === "PurchaseInvoice") {
      return <FileText className="h-4 w-4 text-primary" />;
    }
    if (entry.referenceType === "Payment") {
      return <CreditCard className="h-4 w-4 text-green-600" />;
    }
    if (entry.referenceType === "Manual") {
      return <BookOpen className="h-4 w-4 text-purple-600" />;
    }
    return <BookOpen className="h-4 w-4 text-muted-foreground" />;
  };

  const getTypeBadge = () => {
    if (entry.referenceType === "PurchaseInvoice") {
      return <Badge variant="outline">فاتورة شراء</Badge>;
    }
    if (entry.referenceType === "Payment") {
      return <Badge variant="secondary">دفعة</Badge>;
    }
    if (entry.referenceType === "Manual") {
      return <Badge className="bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">يدوي</Badge>;
    }
    return <Badge variant="outline">تلقائي</Badge>;
  };

  return (
    <Collapsible open={isOpen} onOpenChange={setIsOpen}>
      <div className="rounded-lg border hover-elevate" data-testid={`journal-entry-${entry.id}`}>
        <CollapsibleTrigger asChild>
          <Button
            variant="ghost"
            className="w-full p-4 h-auto justify-between hover:bg-transparent"
          >
            <div className="flex items-center gap-4">
              <div className="flex h-10 w-10 items-center justify-center rounded-lg bg-muted">
                {getTypeIcon()}
              </div>
              <div className="text-right">
                <div className="flex items-center gap-2">
                  <span className="font-mono font-bold">{entry.entryNumber}</span>
                  {getTypeBadge()}
                </div>
                <p className="text-sm text-muted-foreground">{entry.description}</p>
              </div>
            </div>
            <div className="flex items-center gap-4">
              <div className="text-left">
                <p className="text-sm text-muted-foreground">
                  {new Date(entry.date).toLocaleDateString("ar-SA")}
                </p>
                <p className="font-mono font-bold tabular-nums">
                  {totalDebit.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
              {isOpen ? (
                <ChevronUp className="h-4 w-4 text-muted-foreground" />
              ) : (
                <ChevronDown className="h-4 w-4 text-muted-foreground" />
              )}
            </div>
          </Button>
        </CollapsibleTrigger>
        <CollapsibleContent>
          <div className="px-4 pb-4">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رقم الحساب</TableHead>
                  <TableHead className="text-right">اسم الحساب</TableHead>
                  <TableHead className="text-right">البيان</TableHead>
                  <TableHead className="text-right">مدين</TableHead>
                  <TableHead className="text-right">دائن</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {entry.lines.map((line) => (
                  <TableRow key={line.id}>
                    <TableCell className="font-mono text-sm">{line.accountCode}</TableCell>
                    <TableCell>{line.accountName}</TableCell>
                    <TableCell className="text-sm text-muted-foreground">{line.description || "-"}</TableCell>
                    <TableCell className="font-mono tabular-nums">
                      {line.debit > 0 ? line.debit.toLocaleString("ar-SA") : "-"}
                    </TableCell>
                    <TableCell className="font-mono tabular-nums">
                      {line.credit > 0 ? line.credit.toLocaleString("ar-SA") : "-"}
                    </TableCell>
                  </TableRow>
                ))}
                <TableRow className="bg-muted/50 font-bold">
                  <TableCell colSpan={3}>الإجمالي</TableCell>
                  <TableCell className="font-mono tabular-nums">
                    {totalDebit.toLocaleString("ar-SA")}
                  </TableCell>
                  <TableCell className="font-mono tabular-nums">
                    {totalCredit.toLocaleString("ar-SA")}
                  </TableCell>
                </TableRow>
              </TableBody>
            </Table>
            {totalDebit !== totalCredit && (
              <div className="mt-2 rounded-lg bg-destructive/10 p-2 text-destructive text-sm text-center">
                تحذير: القيد غير متوازن - الفرق: {Math.abs(totalDebit - totalCredit).toLocaleString("ar-SA")} ر.س
              </div>
            )}
          </div>
        </CollapsibleContent>
      </div>
    </Collapsible>
  );
}

function ManualJournalEntryDialog({ 
  open, 
  onOpenChange 
}: { 
  open: boolean; 
  onOpenChange: (open: boolean) => void;
}) {
  const { toast } = useToast();
  const [date, setDate] = useState(format(new Date(), "yyyy-MM-dd"));
  const [description, setDescription] = useState("");
  const [lines, setLines] = useState<JournalLine[]>([
    { accountId: 0, description: "", debit: 0, credit: 0 },
    { accountId: 0, description: "", debit: 0, credit: 0 },
  ]);

  const { data: accounts } = useQuery<Account[]>({
    queryKey: ["/api/accounts"],
  });

  const postableAccounts = accounts?.filter(a => a.isPostable && a.isActive) || [];

  const createMutation = useMutation({
    mutationFn: async (data: { date: string; description: string; lines: JournalLine[] }) => {
      return apiRequest("/api/journal-entries/manual", {
        method: "POST",
        body: JSON.stringify(data),
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/journal-entries"] });
      toast({
        title: "تم إنشاء القيد",
        description: "تم إنشاء القيد اليدوي بنجاح",
      });
      resetForm();
      onOpenChange(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "فشل في إنشاء القيد",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setDate(format(new Date(), "yyyy-MM-dd"));
    setDescription("");
    setLines([
      { accountId: 0, description: "", debit: 0, credit: 0 },
      { accountId: 0, description: "", debit: 0, credit: 0 },
    ]);
  };

  const addLine = () => {
    setLines([...lines, { accountId: 0, description: "", debit: 0, credit: 0 }]);
  };

  const removeLine = (index: number) => {
    if (lines.length > 2) {
      setLines(lines.filter((_, i) => i !== index));
    }
  };

  const updateLine = (index: number, field: keyof JournalLine, value: number | string) => {
    const newLines = [...lines];
    if (field === "accountId") {
      newLines[index].accountId = value as number;
    } else if (field === "description") {
      newLines[index].description = value as string;
    } else if (field === "debit") {
      newLines[index].debit = Number(value) || 0;
      if (Number(value) > 0) {
        newLines[index].credit = 0;
      }
    } else if (field === "credit") {
      newLines[index].credit = Number(value) || 0;
      if (Number(value) > 0) {
        newLines[index].debit = 0;
      }
    }
    setLines(newLines);
  };

  const totalDebit = Math.round(lines.reduce((sum, l) => sum + l.debit, 0) * 100) / 100;
  const totalCredit = Math.round(lines.reduce((sum, l) => sum + l.credit, 0) * 100) / 100;
  const isBalanced = Math.abs(totalDebit - totalCredit) < 0.01;
  const hasValidLines = lines.filter(l => l.accountId > 0 && (l.debit > 0 || l.credit > 0)).length >= 2;

  const handleSubmit = () => {
    if (!description.trim()) {
      toast({
        title: "خطأ",
        description: "يرجى إدخال وصف القيد",
        variant: "destructive",
      });
      return;
    }

    if (!isBalanced) {
      toast({
        title: "خطأ",
        description: "القيد غير متوازن - مجموع المدين يجب أن يساوي مجموع الدائن",
        variant: "destructive",
      });
      return;
    }

    if (!hasValidLines) {
      toast({
        title: "خطأ",
        description: "يجب إضافة سطرين على الأقل بحسابات ومبالغ صحيحة",
        variant: "destructive",
      });
      return;
    }

    const validLines = lines.filter(l => l.accountId > 0 && (l.debit > 0 || l.credit > 0));
    createMutation.mutate({ date, description, lines: validLines });
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <BookOpen className="h-5 w-5" />
            إضافة قيد يدوي
          </DialogTitle>
          <DialogDescription>
            أدخل بيانات القيد المحاسبي مع التأكد من توازن المدين والدائن
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="date">التاريخ</Label>
              <Input
                id="date"
                type="date"
                value={date}
                onChange={(e) => setDate(e.target.value)}
                data-testid="input-entry-date"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="description">وصف القيد</Label>
              <Input
                id="description"
                placeholder="أدخل وصف القيد..."
                value={description}
                onChange={(e) => setDescription(e.target.value)}
                data-testid="input-entry-description"
              />
            </div>
          </div>

          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <Label>سطور القيد</Label>
              <Button type="button" variant="outline" size="sm" onClick={addLine} data-testid="button-add-line">
                <Plus className="h-4 w-4 ml-1" />
                إضافة سطر
              </Button>
            </div>

            <div className="rounded-lg border overflow-hidden">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right w-[200px]">الحساب</TableHead>
                    <TableHead className="text-right">البيان</TableHead>
                    <TableHead className="text-right w-[120px]">مدين</TableHead>
                    <TableHead className="text-right w-[120px]">دائن</TableHead>
                    <TableHead className="w-[50px]"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {lines.map((line, index) => (
                    <TableRow key={index} data-testid={`row-line-${index}`}>
                      <TableCell>
                        <Select
                          value={line.accountId > 0 ? String(line.accountId) : ""}
                          onValueChange={(value) => updateLine(index, "accountId", Number(value))}
                        >
                          <SelectTrigger data-testid={`select-account-${index}`}>
                            <SelectValue placeholder="اختر الحساب" />
                          </SelectTrigger>
                          <SelectContent>
                            {postableAccounts.map((account) => (
                              <SelectItem key={account.id} value={String(account.id)}>
                                {account.code} - {account.nameAr}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </TableCell>
                      <TableCell>
                        <Input
                          placeholder="البيان..."
                          value={line.description}
                          onChange={(e) => updateLine(index, "description", e.target.value)}
                          data-testid={`input-line-description-${index}`}
                        />
                      </TableCell>
                      <TableCell>
                        <Input
                          type="number"
                          min="0"
                          step="0.01"
                          placeholder="0.00"
                          value={line.debit || ""}
                          onChange={(e) => updateLine(index, "debit", e.target.value)}
                          className="text-left font-mono"
                          disabled={line.credit > 0}
                          data-testid={`input-line-debit-${index}`}
                        />
                      </TableCell>
                      <TableCell>
                        <Input
                          type="number"
                          min="0"
                          step="0.01"
                          placeholder="0.00"
                          value={line.credit || ""}
                          onChange={(e) => updateLine(index, "credit", e.target.value)}
                          className="text-left font-mono"
                          disabled={line.debit > 0}
                          data-testid={`input-line-credit-${index}`}
                        />
                      </TableCell>
                      <TableCell>
                        <Button
                          type="button"
                          variant="ghost"
                          size="icon"
                          onClick={() => removeLine(index)}
                          disabled={lines.length <= 2}
                          data-testid={`button-remove-line-${index}`}
                        >
                          <Trash2 className="h-4 w-4 text-muted-foreground" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                  <TableRow className="bg-muted/50 font-bold">
                    <TableCell colSpan={2}>الإجمالي</TableCell>
                    <TableCell className={`font-mono text-left ${!isBalanced ? 'text-destructive' : ''}`}>
                      {totalDebit.toLocaleString("ar-SA")}
                    </TableCell>
                    <TableCell className={`font-mono text-left ${!isBalanced ? 'text-destructive' : ''}`}>
                      {totalCredit.toLocaleString("ar-SA")}
                    </TableCell>
                    <TableCell></TableCell>
                  </TableRow>
                </TableBody>
              </Table>
            </div>

            {!isBalanced && totalDebit > 0 && totalCredit > 0 && (
              <div className="flex items-center gap-2 p-3 rounded-lg bg-destructive/10 text-destructive">
                <AlertCircle className="h-4 w-4" />
                <span className="text-sm">
                  القيد غير متوازن - الفرق: {Math.abs(totalDebit - totalCredit).toLocaleString("ar-SA")} ر.س
                </span>
              </div>
            )}

            {isBalanced && totalDebit > 0 && (
              <div className="flex items-center gap-2 p-3 rounded-lg bg-green-50 text-green-700 dark:bg-green-950 dark:text-green-300">
                <span className="text-sm">القيد متوازن</span>
              </div>
            )}
          </div>
        </div>

        <DialogFooter className="gap-2">
          <Button variant="outline" onClick={() => onOpenChange(false)} data-testid="button-cancel">
            إلغاء
          </Button>
          <Button 
            onClick={handleSubmit} 
            disabled={!isBalanced || !hasValidLines || !description.trim() || createMutation.isPending}
            data-testid="button-submit-entry"
          >
            {createMutation.isPending ? "جاري الحفظ..." : "حفظ القيد"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

export default function JournalEntriesPage() {
  const [searchQuery, setSearchQuery] = useState("");
  const [dialogOpen, setDialogOpen] = useState(false);

  const { data: entries, isLoading } = useQuery<JournalEntryWithLines[]>({
    queryKey: ["/api/journal-entries"],
  });

  const filteredEntries = entries?.filter(
    (entry) =>
      entry.entryNumber.includes(searchQuery) ||
      entry.description?.includes(searchQuery)
  );

  const totalDebit = entries?.reduce(
    (sum, entry) => sum + entry.lines.reduce((s, l) => s + l.debit, 0),
    0
  ) || 0;

  const totalCredit = entries?.reduce(
    (sum, entry) => sum + entry.lines.reduce((s, l) => s + l.credit, 0),
    0
  ) || 0;

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold" data-testid="page-title">القيود المحاسبية</h1>
          <p className="text-muted-foreground">سجل القيود اليومية والمحاسبية</p>
        </div>
        <Button onClick={() => setDialogOpen(true)} data-testid="button-add-entry">
          <Plus className="h-4 w-4 ml-2" />
          قيد يدوي جديد
        </Button>
      </div>

      <div className="grid gap-4 md:grid-cols-3">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10">
                <BookOpen className="h-6 w-6 text-primary" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي القيود</p>
                <p className="text-2xl font-bold tabular-nums">
                  {entries?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-green-500/10">
                <span className="text-lg font-bold text-green-600">مد</span>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي المدين</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-debit">
                  {totalDebit.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center gap-4">
              <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-blue-500/10">
                <span className="text-lg font-bold text-blue-600">دا</span>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الدائن</p>
                <p className="text-2xl font-bold tabular-nums" data-testid="total-credit">
                  {totalCredit.toLocaleString("ar-SA")} ر.س
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader className="pb-4">
          <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
            <CardTitle className="text-lg">سجل القيود</CardTitle>
            <div className="relative w-full sm:w-72">
              <Search className="absolute right-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="بحث برقم القيد أو الوصف..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pr-10"
                data-testid="input-search-entries"
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-4">
              {[1, 2, 3].map((i) => (
                <JournalEntrySkeleton key={i} />
              ))}
            </div>
          ) : !filteredEntries || filteredEntries.length === 0 ? (
            <EmptyState onAddClick={() => setDialogOpen(true)} />
          ) : (
            <div className="space-y-4">
              {filteredEntries.map((entry) => (
                <JournalEntryCard key={entry.id} entry={entry} />
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      <ManualJournalEntryDialog open={dialogOpen} onOpenChange={setDialogOpen} />
    </div>
  );
}
