import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatDate } from "@/lib/formatters";
import { Palmtree, Plus, Pencil, Trash2, Users, Calendar, CheckCircle2, XCircle, Clock } from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

interface Employee {
  id: number;
  nameAr: string;
}

interface Leave {
  id: number;
  employeeId: number;
  leaveType: string;
  startDate: string;
  endDate: string;
  startTime: string | null;
  endTime: string | null;
  durationDays: number;
  durationHours: number;
  reason: string | null;
  requestedByEmployeeId: number | null;
  attachmentUrl: string | null;
  status: string;
  createdAt: string;
  employee?: Employee;
  requestedBy?: Employee;
}

interface LeaveFormData {
  employeeId: number | null;
  leaveType: string;
  startDate: string;
  endDate: string;
  startTime: string;
  endTime: string;
  durationDays: number;
  durationHours: number;
  reason: string;
  requestedByEmployeeId: number | null;
  status: string;
}

const leaveTypes = [
  { value: "annual", label: "إجازة سنوية" },
  { value: "sick", label: "إجازة مرضية" },
  { value: "unpaid", label: "إجازة بدون راتب" },
  { value: "emergency", label: "إجازة طارئة" },
  { value: "maternity", label: "إجازة أمومة" },
  { value: "paternity", label: "إجازة أبوة" },
  { value: "marriage", label: "إجازة زواج" },
  { value: "bereavement", label: "إجازة وفاة" },
  { value: "hajj", label: "إجازة حج" },
  { value: "study", label: "إجازة دراسية" },
];

const statusOptions = [
  { value: "pending", label: "معلق", color: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200" },
  { value: "approved", label: "موافق عليه", color: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200" },
  { value: "rejected", label: "مرفوض", color: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200" },
  { value: "cancelled", label: "ملغي", color: "bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200" },
];

export default function LeavesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingLeave, setEditingLeave] = useState<Leave | null>(null);
  const [formData, setFormData] = useState<LeaveFormData>({
    employeeId: null,
    leaveType: "annual",
    startDate: "",
    endDate: "",
    startTime: "",
    endTime: "",
    durationDays: 1,
    durationHours: 0,
    reason: "",
    requestedByEmployeeId: null,
    status: "pending",
  });
  const { toast } = useToast();

  const { data: leaves, isLoading } = useQuery<Leave[]>({
    queryKey: ["/api/hr/leaves"],
  });

  const { data: employees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const createMutation = useMutation({
    mutationFn: (data: LeaveFormData) =>
      apiRequest("POST", "/api/hr/leaves", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/leaves"] });
      toast({ title: "تم إنشاء طلب الإجازة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء طلب الإجازة",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<LeaveFormData> }) =>
      apiRequest("PUT", `/api/hr/leaves/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/leaves"] });
      toast({ title: "تم تحديث طلب الإجازة بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث طلب الإجازة",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/leaves/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/leaves"] });
      toast({ title: "تم حذف طلب الإجازة بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف طلب الإجازة",
        variant: "destructive",
      });
    },
  });

  const approveMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("PUT", `/api/hr/leaves/${id}/approve`, {}),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/leaves"] });
      toast({ title: "تم قبول طلب الإجازة" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في قبول طلب الإجازة",
        variant: "destructive",
      });
    },
  });

  const rejectMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("PUT", `/api/hr/leaves/${id}/reject`, {}),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/leaves"] });
      toast({ title: "تم رفض طلب الإجازة" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في رفض طلب الإجازة",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      employeeId: null,
      leaveType: "annual",
      startDate: "",
      endDate: "",
      startTime: "",
      endTime: "",
      durationDays: 1,
      durationHours: 0,
      reason: "",
      requestedByEmployeeId: null,
      status: "pending",
    });
    setEditingLeave(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (leave?: Leave) => {
    if (leave) {
      setEditingLeave(leave);
      setFormData({
        employeeId: leave.employeeId,
        leaveType: leave.leaveType,
        startDate: leave.startDate,
        endDate: leave.endDate,
        startTime: leave.startTime || "",
        endTime: leave.endTime || "",
        durationDays: leave.durationDays,
        durationHours: leave.durationHours,
        reason: leave.reason || "",
        requestedByEmployeeId: leave.requestedByEmployeeId,
        status: leave.status,
      });
    } else {
      setFormData({
        employeeId: null,
        leaveType: "annual",
        startDate: "",
        endDate: "",
        startTime: "",
        endTime: "",
        durationDays: 1,
        durationHours: 0,
        reason: "",
        requestedByEmployeeId: null,
        status: "pending",
      });
      setEditingLeave(null);
    }
    setIsDialogOpen(true);
  };

  const calculateDuration = (start: string, end: string) => {
    if (!start || !end) return 0;
    const startDate = new Date(start);
    const endDate = new Date(end);
    const diffTime = Math.abs(endDate.getTime() - startDate.getTime());
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)) + 1;
    return diffDays;
  };

  const handleDateChange = (field: "startDate" | "endDate", value: string) => {
    const newFormData = { ...formData, [field]: value };
    if (field === "startDate" && !newFormData.endDate) {
      newFormData.endDate = value;
    }
    newFormData.durationDays = calculateDuration(
      field === "startDate" ? value : formData.startDate,
      field === "endDate" ? value : formData.endDate
    );
    setFormData(newFormData);
  };

  const handleSubmit = () => {
    if (!formData.employeeId) {
      toast({ title: "الموظف مطلوب", variant: "destructive" });
      return;
    }
    if (!formData.startDate || !formData.endDate) {
      toast({ title: "تاريخ البداية والنهاية مطلوبان", variant: "destructive" });
      return;
    }
    if (formData.startDate > formData.endDate) {
      toast({ title: "تاريخ البداية يجب أن يكون قبل تاريخ النهاية", variant: "destructive" });
      return;
    }

    if (editingLeave) {
      updateMutation.mutate({ id: editingLeave.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const getEmployeeName = (id: number) => {
    return employees?.find(e => e.id === id)?.nameAr || `موظف #${id}`;
  };

  const getLeaveTypeLabel = (type: string) => {
    return leaveTypes.find(t => t.value === type)?.label || type;
  };

  const getStatusBadge = (status: string) => {
    const statusOption = statusOptions.find(s => s.value === status);
    return statusOption || { label: status, color: "bg-gray-100 text-gray-800" };
  };

  const pendingCount = leaves?.filter(l => l.status === "pending").length || 0;
  const approvedCount = leaves?.filter(l => l.status === "approved").length || 0;
  const rejectedCount = leaves?.filter(l => l.status === "rejected").length || 0;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الإجازات</h1>
          <p className="text-muted-foreground">تسجيل ومتابعة طلبات إجازات الموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-leave">
          <Plus className="h-4 w-4 ml-2" />
          طلب إجازة جديد
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Palmtree className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الطلبات</p>
                <p className="text-xl font-bold" data-testid="text-total-leaves">
                  {leaves?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">
                <Clock className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">معلق</p>
                <p className="text-xl font-bold" data-testid="text-pending-leaves">
                  {pendingCount}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">موافق عليه</p>
                <p className="text-xl font-bold" data-testid="text-approved-leaves">
                  {approvedCount}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">مرفوض</p>
                <p className="text-xl font-bold" data-testid="text-rejected-leaves">
                  {rejectedCount}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Palmtree className="h-5 w-5" />
            طلبات الإجازات
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">الموظف</TableHead>
                  <TableHead className="text-right">نوع الإجازة</TableHead>
                  <TableHead className="text-right">من</TableHead>
                  <TableHead className="text-right">إلى</TableHead>
                  <TableHead className="text-right">المدة</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {leaves?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center py-8 text-muted-foreground">
                      لا توجد طلبات إجازات
                    </TableCell>
                  </TableRow>
                ) : (
                  leaves?.map((leave) => (
                    <TableRow key={leave.id} data-testid={`row-leave-${leave.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          {leave.employee?.nameAr || getEmployeeName(leave.employeeId)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">
                          {getLeaveTypeLabel(leave.leaveType)}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          {formatDate(leave.startDate)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          {formatDate(leave.endDate)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="secondary">
                          {leave.durationDays} يوم
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <Badge className={getStatusBadge(leave.status).color}>
                          {getStatusBadge(leave.status).label}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          {leave.status === "pending" && (
                            <>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => approveMutation.mutate(leave.id)}
                                disabled={approveMutation.isPending}
                                className="text-green-600 hover:text-green-700"
                                data-testid={`button-approve-leave-${leave.id}`}
                              >
                                <CheckCircle2 className="h-4 w-4" />
                              </Button>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => rejectMutation.mutate(leave.id)}
                                disabled={rejectMutation.isPending}
                                className="text-red-600 hover:text-red-700"
                                data-testid={`button-reject-leave-${leave.id}`}
                              >
                                <XCircle className="h-4 w-4" />
                              </Button>
                            </>
                          )}
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(leave)}
                            data-testid={`button-edit-leave-${leave.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(leave.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-leave-${leave.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-lg" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingLeave ? "تعديل طلب الإجازة" : "طلب إجازة جديد"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="employeeId">الموظف <span className="text-destructive">*</span></Label>
                <Select
                  value={formData.employeeId?.toString() || ""}
                  onValueChange={(value) => setFormData({ ...formData, employeeId: parseInt(value) })}
                >
                  <SelectTrigger data-testid="select-leave-employee">
                    <SelectValue placeholder="اختر الموظف" />
                  </SelectTrigger>
                  <SelectContent>
                    {employees?.map((employee) => (
                      <SelectItem key={employee.id} value={employee.id.toString()}>
                        {employee.nameAr}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="leaveType">نوع الإجازة <span className="text-destructive">*</span></Label>
                <Select
                  value={formData.leaveType}
                  onValueChange={(value) => setFormData({ ...formData, leaveType: value })}
                >
                  <SelectTrigger data-testid="select-leave-type">
                    <SelectValue placeholder="اختر النوع" />
                  </SelectTrigger>
                  <SelectContent>
                    {leaveTypes.map((type) => (
                      <SelectItem key={type.value} value={type.value}>
                        {type.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="grid grid-cols-3 gap-4">
              <div className="space-y-2">
                <Label htmlFor="startDate">تاريخ البداية <span className="text-destructive">*</span></Label>
                <Input
                  id="startDate"
                  type="date"
                  value={formData.startDate}
                  onChange={(e) => handleDateChange("startDate", e.target.value)}
                  data-testid="input-leave-start-date"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="endDate">تاريخ النهاية <span className="text-destructive">*</span></Label>
                <Input
                  id="endDate"
                  type="date"
                  value={formData.endDate}
                  onChange={(e) => handleDateChange("endDate", e.target.value)}
                  data-testid="input-leave-end-date"
                />
              </div>
              <div className="space-y-2">
                <Label>المدة (أيام)</Label>
                <div className="h-9 px-3 py-2 border rounded-md bg-muted text-sm">
                  {formData.durationDays}
                </div>
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="reason">السبب</Label>
              <Textarea
                id="reason"
                value={formData.reason}
                onChange={(e) => setFormData({ ...formData, reason: e.target.value })}
                placeholder="سبب طلب الإجازة"
                rows={2}
                data-testid="input-leave-reason"
              />
            </div>
            {editingLeave && (
              <div className="space-y-2">
                <Label htmlFor="status">الحالة</Label>
                <Select
                  value={formData.status}
                  onValueChange={(value) => setFormData({ ...formData, status: value })}
                >
                  <SelectTrigger data-testid="select-leave-status">
                    <SelectValue placeholder="اختر الحالة" />
                  </SelectTrigger>
                  <SelectContent>
                    {statusOptions.map((status) => (
                      <SelectItem key={status.value} value={status.value}>
                        {status.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            )}
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-leave"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-leave"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
