import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import {
  Coins,
  Search,
  Plus,
  Minus,
  TrendingUp,
  TrendingDown,
  Clock,
  Users,
  Gift,
  RefreshCw,
  ArrowUpDown,
  Filter,
} from "lucide-react";
import { format } from "date-fns";
import { ar } from "date-fns/locale";

type LoyaltyTransaction = {
  id: number;
  customerId: number;
  customerName?: string;
  transactionType: string;
  points: number;
  balanceAfter: number;
  referenceType: string | null;
  referenceId: number | null;
  notes: string | null;
  expiresAt: string | null;
  createdAt: string;
};

type LoyaltySummary = {
  totalCustomersWithPoints: number;
  totalActivePoints: number;
  totalPointsEarned: number;
  totalPointsRedeemed: number;
  totalPointsExpired: number;
  totalPointsAdjusted: number;
  topCustomers: { customerId: number; customerName: string; points: number }[];
};

type Customer = {
  id: number;
  nameAr: string;
  loyaltyPointsBalance: number;
};

export default function LoyaltyPointsPage() {
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [filterType, setFilterType] = useState<string>("all");
  const [filterCustomerId, setFilterCustomerId] = useState<string>("");
  const [startDate, setStartDate] = useState<string>("");
  const [endDate, setEndDate] = useState<string>("");
  const [isAdjustDialogOpen, setIsAdjustDialogOpen] = useState(false);
  const [selectedCustomerId, setSelectedCustomerId] = useState<string>("");
  const [adjustPoints, setAdjustPoints] = useState<string>("");
  const [adjustNotes, setAdjustNotes] = useState<string>("");
  const [adjustType, setAdjustType] = useState<"add" | "subtract">("add");

  const { data: summaryData, isLoading: summaryLoading } = useQuery<LoyaltySummary>({
    queryKey: ["/api/loyalty/summary"],
  });

  const { data: transactionsData, isLoading: transactionsLoading } = useQuery<{
    transactions: LoyaltyTransaction[];
    total: number;
  }>({
    queryKey: [
      "/api/loyalty/transactions",
      filterType !== "all" ? filterType : undefined,
      filterCustomerId || undefined,
      startDate || undefined,
      endDate || undefined,
    ],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (filterType !== "all") params.append("transactionType", filterType);
      if (filterCustomerId) params.append("customerId", filterCustomerId);
      if (startDate) params.append("startDate", startDate);
      if (endDate) params.append("endDate", endDate);
      const response = await fetch(`/api/loyalty/transactions?${params.toString()}`);
      if (!response.ok) throw new Error("Failed to fetch transactions");
      return response.json();
    },
  });

  const { data: customers } = useQuery<Customer[]>({
    queryKey: ["/api/customers"],
  });

  const adjustMutation = useMutation({
    mutationFn: (data: { customerId: number; points: number; notes?: string }) =>
      apiRequest("POST", "/api/loyalty/adjust", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/loyalty/summary"] });
      queryClient.invalidateQueries({ queryKey: ["/api/loyalty/transactions"] });
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      toast({ title: "تم تعديل النقاط بنجاح" });
      closeAdjustDialog();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ في تعديل النقاط",
        description: error.message || "حدث خطأ أثناء تعديل النقاط",
        variant: "destructive",
      });
    },
  });

  const expireMutation = useMutation({
    mutationFn: () => apiRequest("POST", "/api/loyalty/expire", {}),
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/loyalty/summary"] });
      queryClient.invalidateQueries({ queryKey: ["/api/loyalty/transactions"] });
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      toast({
        title: "تم إنهاء صلاحية النقاط",
        description: `تم إنهاء صلاحية ${data.totalPointsExpired} نقطة لـ ${data.expiredCount} عميل`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error.message || "حدث خطأ أثناء إنهاء صلاحية النقاط",
        variant: "destructive",
      });
    },
  });

  const closeAdjustDialog = () => {
    setIsAdjustDialogOpen(false);
    setSelectedCustomerId("");
    setAdjustPoints("");
    setAdjustNotes("");
    setAdjustType("add");
  };

  const handleAdjustSubmit = () => {
    if (!selectedCustomerId || !adjustPoints) {
      toast({
        title: "خطأ",
        description: "يرجى تحديد العميل وعدد النقاط",
        variant: "destructive",
      });
      return;
    }

    const points = parseInt(adjustPoints);
    if (isNaN(points) || points <= 0) {
      toast({
        title: "خطأ",
        description: "عدد النقاط يجب أن يكون رقم موجب",
        variant: "destructive",
      });
      return;
    }

    const finalPoints = adjustType === "subtract" ? -points : points;

    adjustMutation.mutate({
      customerId: parseInt(selectedCustomerId),
      points: finalPoints,
      notes: adjustNotes || undefined,
    });
  };

  const getTransactionTypeBadge = (type: string) => {
    switch (type) {
      case "earn":
        return <Badge className="bg-emerald-100 text-emerald-800 dark:bg-emerald-900 dark:text-emerald-200">اكتساب</Badge>;
      case "redeem":
        return <Badge className="bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">استبدال</Badge>;
      case "expire":
        return <Badge className="bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-200">منتهية</Badge>;
      case "adjust":
        return <Badge className="bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200">تعديل</Badge>;
      default:
        return <Badge variant="secondary">{type}</Badge>;
    }
  };

  const formatPoints = (points: number) => {
    if (points > 0) {
      return <span className="text-emerald-600 dark:text-emerald-400 font-medium">+{points.toLocaleString("ar-SA")}</span>;
    } else if (points < 0) {
      return <span className="text-red-600 dark:text-red-400 font-medium">{points.toLocaleString("ar-SA")}</span>;
    }
    return <span className="text-muted-foreground">0</span>;
  };

  const filteredTransactions = transactionsData?.transactions?.filter((tx) => {
    if (!searchQuery) return true;
    const searchLower = searchQuery.toLowerCase();
    return (
      tx.customerName?.toLowerCase().includes(searchLower) ||
      tx.notes?.toLowerCase().includes(searchLower)
    );
  }) || [];

  return (
    <div className="p-6 space-y-6" dir="rtl">
      <div className="flex flex-wrap items-center justify-between gap-4">
        <div>
          <h1 className="text-2xl font-bold text-foreground flex items-center gap-2">
            <Coins className="h-6 w-6 text-primary" />
            نقاط العملاء
          </h1>
          <p className="text-muted-foreground text-sm mt-1">
            إدارة ومتابعة نقاط الولاء للعملاء
          </p>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={() => expireMutation.mutate()}
            disabled={expireMutation.isPending}
            data-testid="button-expire-points"
          >
            <Clock className="ml-2 h-4 w-4" />
            إنهاء النقاط المنتهية
          </Button>
          <Button onClick={() => setIsAdjustDialogOpen(true)} data-testid="button-adjust-points">
            <ArrowUpDown className="ml-2 h-4 w-4" />
            تعديل النقاط
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">
              إجمالي النقاط النشطة
            </CardTitle>
            <Coins className="h-4 w-4 text-primary" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-foreground" data-testid="text-total-active-points">
              {summaryLoading ? "..." : summaryData?.totalActivePoints?.toLocaleString("ar-SA") || "0"}
            </div>
            <p className="text-xs text-muted-foreground">
              لدى {summaryData?.totalCustomersWithPoints || 0} عميل
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">
              إجمالي النقاط المكتسبة
            </CardTitle>
            <TrendingUp className="h-4 w-4 text-emerald-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-emerald-600 dark:text-emerald-400" data-testid="text-total-earned">
              {summaryLoading ? "..." : summaryData?.totalPointsEarned?.toLocaleString("ar-SA") || "0"}
            </div>
            <p className="text-xs text-muted-foreground">منذ البداية</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">
              إجمالي النقاط المستبدلة
            </CardTitle>
            <Gift className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600 dark:text-blue-400" data-testid="text-total-redeemed">
              {summaryLoading ? "..." : summaryData?.totalPointsRedeemed?.toLocaleString("ar-SA") || "0"}
            </div>
            <p className="text-xs text-muted-foreground">منذ البداية</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">
              النقاط المنتهية الصلاحية
            </CardTitle>
            <TrendingDown className="h-4 w-4 text-amber-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-amber-600 dark:text-amber-400" data-testid="text-total-expired">
              {summaryLoading ? "..." : summaryData?.totalPointsExpired?.toLocaleString("ar-SA") || "0"}
            </div>
            <p className="text-xs text-muted-foreground">منذ البداية</p>
          </CardContent>
        </Card>
      </div>

      {summaryData?.topCustomers && summaryData.topCustomers.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle className="text-lg flex items-center gap-2">
              <Users className="h-5 w-5 text-primary" />
              أفضل العملاء بالنقاط
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-wrap gap-3">
              {summaryData.topCustomers.slice(0, 5).map((customer, index) => (
                <div
                  key={customer.customerId}
                  className="flex items-center gap-2 px-3 py-2 bg-muted rounded-lg"
                  data-testid={`card-top-customer-${customer.customerId}`}
                >
                  <span className="text-sm font-medium text-muted-foreground">#{index + 1}</span>
                  <span className="text-sm font-medium">{customer.customerName}</span>
                  <Badge variant="secondary" className="bg-primary/10 text-primary">
                    {customer.points.toLocaleString("ar-SA")} نقطة
                  </Badge>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      <Card>
        <CardHeader>
          <CardTitle className="text-lg">سجل الحركات</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="flex flex-wrap gap-4">
            <div className="flex-1 min-w-[200px]">
              <div className="relative">
                <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="بحث بالاسم أو الملاحظات..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pr-10"
                  data-testid="input-search-transactions"
                />
              </div>
            </div>

            <Select value={filterType} onValueChange={setFilterType}>
              <SelectTrigger className="w-[150px]" data-testid="select-filter-type">
                <Filter className="ml-2 h-4 w-4" />
                <SelectValue placeholder="نوع الحركة" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">الكل</SelectItem>
                <SelectItem value="earn">اكتساب</SelectItem>
                <SelectItem value="redeem">استبدال</SelectItem>
                <SelectItem value="expire">منتهية</SelectItem>
                <SelectItem value="adjust">تعديل</SelectItem>
              </SelectContent>
            </Select>

            <Select value={filterCustomerId || "all"} onValueChange={(val) => setFilterCustomerId(val === "all" ? "" : val)}>
              <SelectTrigger className="w-[180px]" data-testid="select-filter-customer">
                <SelectValue placeholder="اختر العميل" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">جميع العملاء</SelectItem>
                {customers?.map((c) => (
                  <SelectItem key={c.id} value={c.id.toString()}>
                    {c.nameAr}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Input
              type="date"
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
              className="w-[150px]"
              placeholder="من تاريخ"
              data-testid="input-start-date"
            />

            <Input
              type="date"
              value={endDate}
              onChange={(e) => setEndDate(e.target.value)}
              className="w-[150px]"
              placeholder="إلى تاريخ"
              data-testid="input-end-date"
            />
          </div>

          {transactionsLoading ? (
            <div className="text-center py-8 text-muted-foreground">جاري التحميل...</div>
          ) : filteredTransactions.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">لا توجد حركات</div>
          ) : (
            <div className="border rounded-lg overflow-hidden">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="text-right">التاريخ</TableHead>
                    <TableHead className="text-right">العميل</TableHead>
                    <TableHead className="text-right">نوع الحركة</TableHead>
                    <TableHead className="text-right">النقاط</TableHead>
                    <TableHead className="text-right">الرصيد بعد</TableHead>
                    <TableHead className="text-right">المصدر</TableHead>
                    <TableHead className="text-right">ملاحظات</TableHead>
                    <TableHead className="text-right">تاريخ الانتهاء</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredTransactions.map((tx) => (
                    <TableRow key={tx.id} data-testid={`row-transaction-${tx.id}`}>
                      <TableCell className="text-muted-foreground text-sm">
                        {format(new Date(tx.createdAt), "dd/MM/yyyy HH:mm", { locale: ar })}
                      </TableCell>
                      <TableCell className="font-medium">{tx.customerName || `عميل #${tx.customerId}`}</TableCell>
                      <TableCell>{getTransactionTypeBadge(tx.transactionType)}</TableCell>
                      <TableCell>{formatPoints(tx.points)}</TableCell>
                      <TableCell className="font-medium">{tx.balanceAfter.toLocaleString("ar-SA")}</TableCell>
                      <TableCell className="text-muted-foreground text-sm">
                        {tx.referenceType ? (
                          <Badge variant="outline">
                            {tx.referenceType === "order" ? "طلب" : tx.referenceType === "manual" ? "يدوي" : tx.referenceType}
                            {tx.referenceId ? ` #${tx.referenceId}` : ""}
                          </Badge>
                        ) : (
                          "-"
                        )}
                      </TableCell>
                      <TableCell className="text-muted-foreground text-sm max-w-[200px] truncate">
                        {tx.notes || "-"}
                      </TableCell>
                      <TableCell className="text-muted-foreground text-sm">
                        {tx.expiresAt ? format(new Date(tx.expiresAt), "dd/MM/yyyy", { locale: ar }) : "-"}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}

          <div className="text-sm text-muted-foreground text-center">
            إجمالي الحركات: {transactionsData?.total || 0}
          </div>
        </CardContent>
      </Card>

      <Dialog open={isAdjustDialogOpen} onOpenChange={setIsAdjustDialogOpen}>
        <DialogContent className="max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>تعديل نقاط العميل</DialogTitle>
            <DialogDescription>
              اختر العميل وحدد عدد النقاط للإضافة أو الخصم
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div className="space-y-2">
              <Label>العميل</Label>
              <Select value={selectedCustomerId} onValueChange={setSelectedCustomerId}>
                <SelectTrigger data-testid="select-adjust-customer">
                  <SelectValue placeholder="اختر العميل" />
                </SelectTrigger>
                <SelectContent>
                  {customers?.map((c) => (
                    <SelectItem key={c.id} value={c.id.toString()}>
                      {c.nameAr} ({c.loyaltyPointsBalance} نقطة)
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>نوع التعديل</Label>
              <div className="flex gap-2">
                <Button
                  type="button"
                  variant={adjustType === "add" ? "default" : "outline"}
                  className="flex-1"
                  onClick={() => setAdjustType("add")}
                  data-testid="button-adjust-add"
                >
                  <Plus className="ml-2 h-4 w-4" />
                  إضافة
                </Button>
                <Button
                  type="button"
                  variant={adjustType === "subtract" ? "default" : "outline"}
                  className="flex-1"
                  onClick={() => setAdjustType("subtract")}
                  data-testid="button-adjust-subtract"
                >
                  <Minus className="ml-2 h-4 w-4" />
                  خصم
                </Button>
              </div>
            </div>

            <div className="space-y-2">
              <Label>عدد النقاط</Label>
              <Input
                type="number"
                value={adjustPoints}
                onChange={(e) => setAdjustPoints(e.target.value)}
                placeholder="0"
                min="1"
                data-testid="input-adjust-points"
              />
            </div>

            <div className="space-y-2">
              <Label>ملاحظات (اختياري)</Label>
              <Textarea
                value={adjustNotes}
                onChange={(e) => setAdjustNotes(e.target.value)}
                placeholder="سبب التعديل..."
                data-testid="input-adjust-notes"
              />
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={closeAdjustDialog} data-testid="button-cancel-adjust">
              إلغاء
            </Button>
            <Button
              onClick={handleAdjustSubmit}
              disabled={adjustMutation.isPending || !selectedCustomerId || !adjustPoints}
              data-testid="button-submit-adjust"
            >
              {adjustMutation.isPending ? "جاري الحفظ..." : "حفظ التعديل"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
