import { useQuery, useMutation } from "@tanstack/react-query";
import { Link } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber } from "@/lib/formatters";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
  FormDescription,
} from "@/components/ui/form";
import {
  ArrowRight,
  Gift,
  Settings,
  Coins,
  Calendar,
  Info,
} from "lucide-react";

type LoyaltySettings = {
  id: number;
  pointsPerCurrency: number;
  currencyPerPoint: number;
  expiryMonths: number;
  minRedeemPoints: number;
  isActive: boolean;
  updatedAt: string;
};

const settingsFormSchema = z.object({
  pointsPerCurrency: z.coerce.number().min(0.01, "يجب أن تكون القيمة أكبر من صفر"),
  currencyPerPoint: z.coerce.number().min(0.01, "يجب أن تكون القيمة أكبر من صفر"),
  expiryMonths: z.coerce.number().min(1, "يجب أن تكون المدة شهر واحد على الأقل").max(120, "الحد الأقصى 10 سنوات"),
  minRedeemPoints: z.coerce.number().min(1, "يجب أن يكون الحد الأدنى نقطة واحدة على الأقل"),
  isActive: z.boolean(),
});

type SettingsFormData = z.infer<typeof settingsFormSchema>;

export default function LoyaltySettingsPage() {
  const { toast } = useToast();

  const { data: settings, isLoading } = useQuery<LoyaltySettings>({
    queryKey: ["/api/loyalty/settings"],
  });

  const form = useForm<SettingsFormData>({
    resolver: zodResolver(settingsFormSchema),
    defaultValues: {
      pointsPerCurrency: 1,
      currencyPerPoint: 0.1,
      expiryMonths: 12,
      minRedeemPoints: 100,
      isActive: true,
    },
    values: settings ? {
      pointsPerCurrency: settings.pointsPerCurrency,
      currencyPerPoint: settings.currencyPerPoint,
      expiryMonths: settings.expiryMonths,
      minRedeemPoints: settings.minRedeemPoints,
      isActive: settings.isActive,
    } : undefined,
  });

  const updateMutation = useMutation({
    mutationFn: (data: SettingsFormData) =>
      apiRequest("PATCH", "/api/loyalty/settings", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/loyalty/settings"] });
      toast({ title: "تم حفظ الإعدادات بنجاح" });
    },
    onError: (error: any) => {
      toast({ title: "فشل في حفظ الإعدادات", description: error.message, variant: "destructive" });
    },
  });

  const onSubmit = (data: SettingsFormData) => {
    updateMutation.mutate(data);
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6" dir="rtl">
        <div className="text-center py-12 text-muted-foreground">جاري التحميل...</div>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center gap-2 text-muted-foreground mb-4">
        <Link href="/customers">
          <Button variant="ghost" size="sm" data-testid="link-back">
            العملاء
            <ArrowRight className="mr-2 h-4 w-4" />
          </Button>
        </Link>
        <span>/</span>
        <span>إعدادات نقاط الولاء</span>
      </div>

      <div className="flex items-center gap-4">
        <div className="h-12 w-12 rounded-lg bg-primary/10 flex items-center justify-center">
          <Gift className="h-6 w-6 text-primary" />
        </div>
        <div>
          <h1 className="text-3xl font-bold" data-testid="page-title">إعدادات نقاط الولاء</h1>
          <p className="text-muted-foreground">تحديد كيفية احتساب وانتهاء صلاحية نقاط الولاء</p>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Settings className="h-5 w-5" />
                إعدادات البرنامج
              </CardTitle>
              <CardDescription>
                حدد قواعد احتساب النقاط وشروط الاستبدال
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <FormField
                      control={form.control}
                      name="pointsPerCurrency"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="flex items-center gap-2">
                            <Coins className="h-4 w-4" />
                            نقاط لكل ريال
                          </FormLabel>
                          <FormControl>
                            <Input {...field} type="number" step="0.01" min="0.01" data-testid="input-points-per-currency" />
                          </FormControl>
                          <FormDescription>
                            عدد النقاط التي يحصل عليها العميل لكل ريال ينفقه
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="currencyPerPoint"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="flex items-center gap-2">
                            <Coins className="h-4 w-4" />
                            قيمة النقطة (ر.س)
                          </FormLabel>
                          <FormControl>
                            <Input {...field} type="number" step="0.01" min="0.01" data-testid="input-currency-per-point" />
                          </FormControl>
                          <FormDescription>
                            قيمة كل نقطة بالريال عند الاستبدال
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="expiryMonths"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="flex items-center gap-2">
                            <Calendar className="h-4 w-4" />
                            مدة صلاحية النقاط (بالأشهر)
                          </FormLabel>
                          <FormControl>
                            <Input {...field} type="number" min="1" max="120" data-testid="input-expiry-months" />
                          </FormControl>
                          <FormDescription>
                            عدد الأشهر قبل انتهاء صلاحية النقاط بعد اكتسابها
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="minRedeemPoints"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="flex items-center gap-2">
                            <Gift className="h-4 w-4" />
                            الحد الأدنى للاستبدال
                          </FormLabel>
                          <FormControl>
                            <Input {...field} type="number" min="1" data-testid="input-min-redeem" />
                          </FormControl>
                          <FormDescription>
                            الحد الأدنى من النقاط المطلوبة للاستبدال
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <FormField
                    control={form.control}
                    name="isActive"
                    render={({ field }) => (
                      <FormItem className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="space-y-0.5">
                          <FormLabel>تفعيل برنامج الولاء</FormLabel>
                          <FormDescription>
                            تفعيل أو تعطيل برنامج نقاط الولاء للعملاء
                          </FormDescription>
                        </div>
                        <FormControl>
                          <Switch
                            checked={field.value}
                            onCheckedChange={field.onChange}
                            data-testid="switch-active"
                          />
                        </FormControl>
                      </FormItem>
                    )}
                  />

                  <div className="flex justify-end">
                    <Button type="submit" disabled={updateMutation.isPending} data-testid="button-save">
                      {updateMutation.isPending ? "جاري الحفظ..." : "حفظ الإعدادات"}
                    </Button>
                  </div>
                </form>
              </Form>
            </CardContent>
          </Card>
        </div>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Info className="h-5 w-5" />
                ملخص الإعدادات
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="p-4 bg-muted/50 rounded-lg">
                <p className="text-sm text-muted-foreground mb-1">معادلة الاكتساب</p>
                <p className="font-medium">
                  كل {formatNumber(1)} ر.س = {formatNumber(form.watch("pointsPerCurrency") || 1)} نقطة
                </p>
              </div>
              <div className="p-4 bg-muted/50 rounded-lg">
                <p className="text-sm text-muted-foreground mb-1">معادلة الاستبدال</p>
                <p className="font-medium">
                  كل {formatNumber(1)} نقطة = {formatNumber(form.watch("currencyPerPoint") || 0.1)} ر.س
                </p>
              </div>
              <div className="p-4 bg-muted/50 rounded-lg">
                <p className="text-sm text-muted-foreground mb-1">مثال</p>
                <p className="text-sm">
                  إذا أنفق العميل {formatNumber(100)} ر.س سيحصل على {formatNumber((form.watch("pointsPerCurrency") || 1) * 100)} نقطة
                </p>
                <p className="text-sm mt-1">
                  يمكنه استبدال {formatNumber(100)} نقطة بـ {formatNumber((form.watch("currencyPerPoint") || 0.1) * 100)} ر.س
                </p>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5" />
                انتهاء الصلاحية
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                النقاط المكتسبة ستنتهي صلاحيتها بعد{" "}
                <span className="font-bold text-foreground">{form.watch("expiryMonths") || 12}</span>{" "}
                شهر من تاريخ اكتسابها.
              </p>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
