import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Skeleton } from "@/components/ui/skeleton";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Ticket,
  Package,
  Tag,
  Percent,
  TrendingUp,
  ShoppingBag,
  Scissors,
  Calendar,
  Activity,
} from "lucide-react";
import { format, subDays } from "date-fns";
import type { MarketingReportSummary } from "@shared/schema";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";

export default function MarketingReportsPage() {
  const [fromDate, setFromDate] = useState(
    format(subDays(new Date(), 30), "yyyy-MM-dd")
  );
  const [toDate, setToDate] = useState(format(new Date(), "yyyy-MM-dd"));

  const { data: report, isLoading, isError, error } = useQuery<MarketingReportSummary>({
    queryKey: ["/api/reports/marketing", fromDate, toDate],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (fromDate) params.set("from_date", fromDate);
      if (toDate) params.set("to_date", toDate);
      const response = await fetch(`/api/reports/marketing?${params.toString()}`);
      if (!response.ok) throw new Error("فشل في جلب تقرير التسويق");
      return response.json();
    },
  });

  return (
    <div className="space-y-6">
      <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-4">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">
            تقارير التسويق
          </h1>
          <p className="text-muted-foreground">
            تحليل استخدام الكوبونات والباقات
          </p>
        </div>

        <div className="flex items-center gap-3 flex-wrap">
          <div className="flex items-center gap-2">
            <Label htmlFor="from-date" className="whitespace-nowrap">
              من:
            </Label>
            <Input
              id="from-date"
              type="date"
              value={fromDate}
              onChange={(e) => setFromDate(e.target.value)}
              className="w-40"
              data-testid="input-from-date"
            />
          </div>
          <div className="flex items-center gap-2">
            <Label htmlFor="to-date" className="whitespace-nowrap">
              إلى:
            </Label>
            <Input
              id="to-date"
              type="date"
              value={toDate}
              onChange={(e) => setToDate(e.target.value)}
              className="w-40"
              data-testid="input-to-date"
            />
          </div>
        </div>
      </div>

      {isLoading ? (
        <MarketingReportSkeleton />
      ) : isError ? (
        <Card>
          <CardContent className="flex items-center justify-center py-12">
            <div className="text-center text-destructive">
              <Activity className="h-12 w-12 mx-auto mb-2 opacity-50" />
              <p>حدث خطأ أثناء جلب البيانات</p>
              <p className="text-sm text-muted-foreground mt-2">
                {error instanceof Error ? error.message : "خطأ غير معروف"}
              </p>
            </div>
          </CardContent>
        </Card>
      ) : report ? (
        <>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  إجمالي الكوبونات
                </CardTitle>
                <Ticket className="h-4 w-4 text-primary" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold"
                  data-testid="text-total-coupons"
                >
                  {formatNumber(report.couponStats.totalCoupons)}
                </div>
                <p className="text-xs text-muted-foreground">
                  {report.couponStats.activeCoupons} نشط
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  مرات استخدام الكوبونات
                </CardTitle>
                <Tag className="h-4 w-4 text-blue-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-blue-600"
                  data-testid="text-coupon-usage"
                >
                  {formatNumber(report.couponStats.totalUsageCount)}
                </div>
                <p className="text-xs text-muted-foreground">في الفترة المحددة</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  إجمالي الخصومات
                </CardTitle>
                <Percent className="h-4 w-4 text-red-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-red-600"
                  data-testid="text-total-discounts"
                >
                  {formatCurrency(report.couponStats.totalDiscountGiven)}
                </div>
                <p className="text-xs text-muted-foreground">خصومات ممنوحة</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">
                  إيرادات الباقات
                </CardTitle>
                <TrendingUp className="h-4 w-4 text-green-500" />
              </CardHeader>
              <CardContent>
                <div
                  className="text-2xl font-bold text-green-600"
                  data-testid="text-package-revenue"
                >
                  {formatCurrency(report.packageStats.totalPackageRevenue)}
                </div>
                <p className="text-xs text-muted-foreground">
                  {formatNumber(report.packageStats.totalPackagesSold)} مبيعات
                </p>
              </CardContent>
            </Card>
          </div>

          <Tabs defaultValue="coupons" className="w-full">
            <TabsList className="grid w-full max-w-md grid-cols-2 mb-4">
              <TabsTrigger value="coupons" data-testid="tab-coupons">
                <Ticket className="h-4 w-4 ml-2" />
                الكوبونات
              </TabsTrigger>
              <TabsTrigger value="packages" data-testid="tab-packages">
                <Package className="h-4 w-4 ml-2" />
                الباقات
              </TabsTrigger>
            </TabsList>

            <TabsContent value="coupons" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Ticket className="h-5 w-5 text-primary" />
                    تفاصيل استخدام الكوبونات
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="text-right">كود الكوبون</TableHead>
                        <TableHead className="text-right">اسم الكوبون</TableHead>
                        <TableHead className="text-right">نوع الخصم</TableHead>
                        <TableHead className="text-center">مرات الاستخدام</TableHead>
                        <TableHead className="text-left">إجمالي الخصم</TableHead>
                        <TableHead className="text-left">قيمة الطلبات</TableHead>
                        <TableHead className="text-left">متوسط الطلب</TableHead>
                        <TableHead className="text-right">آخر استخدام</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {report.couponUsageList.map((coupon) => (
                        <TableRow key={coupon.couponId} data-testid={`row-coupon-${coupon.couponId}`}>
                          <TableCell className="font-mono font-medium">
                            {coupon.couponCode}
                          </TableCell>
                          <TableCell>{coupon.couponNameAr || "-"}</TableCell>
                          <TableCell>
                            <Badge variant="outline">
                              {coupon.discountType === "percentage"
                                ? `${coupon.discountValue}%`
                                : formatCurrency(coupon.discountValue)}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-center">
                            <Badge>{formatNumber(coupon.usageCount)}</Badge>
                          </TableCell>
                          <TableCell className="text-left text-red-600">
                            {formatCurrency(coupon.totalDiscountGiven)}
                          </TableCell>
                          <TableCell className="text-left text-green-600">
                            {formatCurrency(coupon.totalOrdersValue)}
                          </TableCell>
                          <TableCell className="text-left">
                            {formatCurrency(coupon.avgOrderValue)}
                          </TableCell>
                          <TableCell className="text-right text-muted-foreground">
                            {formatDate(coupon.lastUsedDate)}
                          </TableCell>
                        </TableRow>
                      ))}

                      {report.couponUsageList.length === 0 && (
                        <TableRow>
                          <TableCell
                            colSpan={8}
                            className="text-center text-muted-foreground py-8"
                          >
                            <Ticket className="h-8 w-8 mx-auto mb-2 opacity-50" />
                            لا توجد بيانات استخدام كوبونات في الفترة المحددة
                          </TableCell>
                        </TableRow>
                      )}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="packages" className="space-y-6">
              <div className="grid gap-4 md:grid-cols-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">إجمالي الباقات</CardTitle>
                    <Package className="h-4 w-4 text-primary" />
                  </CardHeader>
                  <CardContent>
                    <div className="text-2xl font-bold" data-testid="text-total-packages">
                      {formatNumber(report.packageStats.totalPackages)}
                    </div>
                    <p className="text-xs text-muted-foreground">
                      {report.packageStats.activePackages} نشط
                    </p>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">مبيعات الباقات</CardTitle>
                    <ShoppingBag className="h-4 w-4 text-blue-500" />
                  </CardHeader>
                  <CardContent>
                    <div
                      className="text-2xl font-bold text-blue-600"
                      data-testid="text-packages-sold"
                    >
                      {formatNumber(report.packageStats.totalPackagesSold)}
                    </div>
                    <p className="text-xs text-muted-foreground">في الفترة المحددة</p>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">إجمالي الإيرادات</CardTitle>
                    <TrendingUp className="h-4 w-4 text-green-500" />
                  </CardHeader>
                  <CardContent>
                    <div
                      className="text-2xl font-bold text-green-600"
                      data-testid="text-packages-revenue"
                    >
                      {formatCurrency(report.packageStats.totalPackageRevenue)}
                    </div>
                    <p className="text-xs text-muted-foreground">إيرادات الباقات</p>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">متوسط قيمة الطلب</CardTitle>
                    <Activity className="h-4 w-4 text-amber-500" />
                  </CardHeader>
                  <CardContent>
                    <div
                      className="text-2xl font-bold text-amber-600"
                      data-testid="text-package-avg"
                    >
                      {report.packageStats.totalPackagesSold > 0
                        ? formatCurrency(
                            report.packageStats.totalPackageRevenue /
                              report.packageStats.totalPackagesSold
                          )
                        : formatCurrency(0)}
                    </div>
                    <p className="text-xs text-muted-foreground">لكل طلب</p>
                  </CardContent>
                </Card>
              </div>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Package className="h-5 w-5 text-primary" />
                    تفاصيل مبيعات الباقات
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="text-right">اسم الباقة</TableHead>
                        <TableHead className="text-center">عدد المبيعات</TableHead>
                        <TableHead className="text-left">إجمالي الإيرادات</TableHead>
                        <TableHead className="text-left">متوسط الطلب</TableHead>
                        <TableHead className="text-center">الخدمات</TableHead>
                        <TableHead className="text-center">المنتجات</TableHead>
                        <TableHead className="text-right">آخر بيع</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {report.packageUsageList.map((pkg) => (
                        <TableRow key={pkg.packageId} data-testid={`row-package-${pkg.packageId}`}>
                          <TableCell className="font-medium">
                            {pkg.packageNameAr || "-"}
                          </TableCell>
                          <TableCell className="text-center">
                            <Badge>{formatNumber(pkg.totalSales)}</Badge>
                          </TableCell>
                          <TableCell className="text-left text-green-600 font-semibold">
                            {formatCurrency(pkg.totalRevenue)}
                          </TableCell>
                          <TableCell className="text-left">
                            {formatCurrency(pkg.avgOrderValue)}
                          </TableCell>
                          <TableCell className="text-center">
                            <Badge variant="outline">
                              <Scissors className="h-3 w-3 ml-1" />
                              {formatNumber(pkg.servicesCount)}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-center">
                            <Badge variant="outline">
                              <ShoppingBag className="h-3 w-3 ml-1" />
                              {formatNumber(pkg.productsCount)}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-right text-muted-foreground">
                            {formatDate(pkg.lastSoldDate)}
                          </TableCell>
                        </TableRow>
                      ))}

                      {report.packageUsageList.length === 0 && (
                        <TableRow>
                          <TableCell
                            colSpan={7}
                            className="text-center text-muted-foreground py-8"
                          >
                            <Package className="h-8 w-8 mx-auto mb-2 opacity-50" />
                            لا توجد بيانات مبيعات باقات في الفترة المحددة
                          </TableCell>
                        </TableRow>
                      )}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>

              <div className="grid gap-6 lg:grid-cols-2">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Scissors className="h-5 w-5 text-purple-500" />
                      أكثر الخدمات مبيعاً في الباقات
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="text-right">الخدمة</TableHead>
                          <TableHead className="text-center">في باقات</TableHead>
                          <TableHead className="text-center">الكمية</TableHead>
                          <TableHead className="text-left">الإيرادات</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {report.topServicesInPackages.map((service) => (
                          <TableRow
                            key={service.serviceId}
                            data-testid={`row-top-service-${service.serviceId}`}
                          >
                            <TableCell className="font-medium">
                              {service.serviceNameAr}
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant="secondary">{service.packageCount}</Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              {formatNumber(service.totalQuantitySold)}
                            </TableCell>
                            <TableCell className="text-left text-green-600 font-semibold">
                              {formatCurrency(service.totalRevenue)}
                            </TableCell>
                          </TableRow>
                        ))}

                        {report.topServicesInPackages.length === 0 && (
                          <TableRow>
                            <TableCell
                              colSpan={4}
                              className="text-center text-muted-foreground py-8"
                            >
                              لا توجد بيانات
                            </TableCell>
                          </TableRow>
                        )}
                      </TableBody>
                    </Table>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <ShoppingBag className="h-5 w-5 text-orange-500" />
                      أكثر المنتجات مبيعاً في الباقات
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="text-right">المنتج</TableHead>
                          <TableHead className="text-center">في باقات</TableHead>
                          <TableHead className="text-center">الكمية</TableHead>
                          <TableHead className="text-left">الإيرادات</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {report.topProductsInPackages.map((product) => (
                          <TableRow
                            key={product.productId}
                            data-testid={`row-top-product-${product.productId}`}
                          >
                            <TableCell className="font-medium">
                              {product.productNameAr}
                            </TableCell>
                            <TableCell className="text-center">
                              <Badge variant="secondary">{product.packageCount}</Badge>
                            </TableCell>
                            <TableCell className="text-center">
                              {formatNumber(product.totalQuantitySold)}
                            </TableCell>
                            <TableCell className="text-left text-green-600 font-semibold">
                              {formatCurrency(product.totalRevenue)}
                            </TableCell>
                          </TableRow>
                        ))}

                        {report.topProductsInPackages.length === 0 && (
                          <TableRow>
                            <TableCell
                              colSpan={4}
                              className="text-center text-muted-foreground py-8"
                            >
                              لا توجد بيانات
                            </TableCell>
                          </TableRow>
                        )}
                      </TableBody>
                    </Table>
                  </CardContent>
                </Card>
              </div>
            </TabsContent>
          </Tabs>
        </>
      ) : (
        <Card>
          <CardContent className="flex items-center justify-center py-12">
            <div className="text-center text-muted-foreground">
              <Calendar className="h-12 w-12 mx-auto mb-2 opacity-50" />
              <p>حدد فترة التقرير لعرض البيانات</p>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}

function MarketingReportSkeleton() {
  return (
    <>
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {[1, 2, 3, 4].map((i) => (
          <Card key={i}>
            <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
              <Skeleton className="h-4 w-24" />
              <Skeleton className="h-4 w-4" />
            </CardHeader>
            <CardContent>
              <Skeleton className="h-8 w-32 mb-1" />
              <Skeleton className="h-3 w-20" />
            </CardContent>
          </Card>
        ))}
      </div>

      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-48" />
        </CardHeader>
        <CardContent>
          <Skeleton className="h-64 w-full" />
        </CardContent>
      </Card>

      <div className="grid gap-6 lg:grid-cols-2">
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <Skeleton className="h-40 w-full" />
          </CardContent>
        </Card>
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <Skeleton className="h-40 w-full" />
          </CardContent>
        </Card>
      </div>
    </>
  );
}
