import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Globe, Plus, Pencil, Trash2, CheckCircle2, XCircle, Code } from "lucide-react";

interface Nationality {
  id: number;
  nameAr: string;
  code: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface NationalityFormData {
  nameAr: string;
  code: string;
  isActive: boolean;
}

export default function NationalitiesPage() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingNationality, setEditingNationality] = useState<Nationality | null>(null);
  const [formData, setFormData] = useState<NationalityFormData>({
    nameAr: "",
    code: "",
    isActive: true,
  });
  const { toast } = useToast();

  const { data: nationalities, isLoading } = useQuery<Nationality[]>({
    queryKey: ["/api/hr/nationalities"],
  });

  const createMutation = useMutation({
    mutationFn: (data: NationalityFormData) =>
      apiRequest("POST", "/api/hr/nationalities", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/nationalities"] });
      toast({ title: "تم إنشاء الجنسية بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الجنسية",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: Partial<NationalityFormData> }) =>
      apiRequest("PUT", `/api/hr/nationalities/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/nationalities"] });
      toast({ title: "تم تحديث الجنسية بنجاح" });
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الجنسية",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) =>
      apiRequest("DELETE", `/api/hr/nationalities/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/hr/nationalities"] });
      toast({ title: "تم حذف الجنسية بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حذف الجنسية",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({ nameAr: "", code: "", isActive: true });
    setEditingNationality(null);
    setIsDialogOpen(false);
  };

  const handleOpenDialog = (nationality?: Nationality) => {
    if (nationality) {
      setEditingNationality(nationality);
      setFormData({
        nameAr: nationality.nameAr,
        code: nationality.code || "",
        isActive: nationality.isActive,
      });
    } else {
      setFormData({ nameAr: "", code: "", isActive: true });
      setEditingNationality(null);
    }
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.nameAr.trim()) {
      toast({ title: "اسم الجنسية مطلوب", variant: "destructive" });
      return;
    }

    if (editingNationality) {
      updateMutation.mutate({ id: editingNationality.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const activeNationalities = nationalities?.filter(n => n.isActive) || [];
  const inactiveNationalities = nationalities?.filter(n => !n.isActive) || [];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">إدارة الجنسيات</h1>
          <p className="text-muted-foreground">تعريف جنسيات الموظفين</p>
        </div>
        <Button onClick={() => handleOpenDialog()} data-testid="button-add-nationality">
          <Plus className="h-4 w-4 ml-2" />
          إضافة جنسية جديدة
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-primary/10 text-primary">
                <Globe className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">إجمالي الجنسيات</p>
                <p className="text-xl font-bold" data-testid="text-total-nationalities">
                  {nationalities?.length || 0}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                <CheckCircle2 className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">جنسيات مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-active-nationalities">
                  {activeNationalities.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 rounded-lg bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                <XCircle className="h-5 w-5" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">جنسيات غير مفعّلة</p>
                <p className="text-xl font-bold" data-testid="text-inactive-nationalities">
                  {inactiveNationalities.length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Globe className="h-5 w-5" />
            قائمة الجنسيات
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">اسم الجنسية</TableHead>
                  <TableHead className="text-right">الرمز</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-left">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {nationalities?.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={4} className="text-center py-8 text-muted-foreground">
                      لا توجد جنسيات مسجلة
                    </TableCell>
                  </TableRow>
                ) : (
                  nationalities?.map((nationality) => (
                    <TableRow key={nationality.id} data-testid={`row-nationality-${nationality.id}`}>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          <Globe className="h-4 w-4 text-muted-foreground" />
                          {nationality.nameAr}
                        </div>
                      </TableCell>
                      <TableCell>
                        {nationality.code ? (
                          <Badge variant="outline" className="font-mono">
                            <Code className="h-3 w-3 ml-1" />
                            {nationality.code}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {nationality.isActive ? (
                          <Badge className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                            مفعّل
                          </Badge>
                        ) : (
                          <Badge className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                            غير مفعّل
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => handleOpenDialog(nationality)}
                            data-testid={`button-edit-nationality-${nationality.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            onClick={() => deleteMutation.mutate(nationality.id)}
                            disabled={deleteMutation.isPending}
                            data-testid={`button-delete-nationality-${nationality.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-md" dir="rtl">
          <DialogHeader>
            <DialogTitle>
              {editingNationality ? "تعديل الجنسية" : "إضافة جنسية جديدة"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="nameAr">اسم الجنسية <span className="text-destructive">*</span></Label>
              <Input
                id="nameAr"
                value={formData.nameAr}
                onChange={(e) => setFormData({ ...formData, nameAr: e.target.value })}
                placeholder="مثال: سعودي"
                data-testid="input-nationality-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="code">رمز الدولة</Label>
              <Input
                id="code"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                placeholder="مثال: SA"
                className="font-mono text-left"
                dir="ltr"
                maxLength={3}
                data-testid="input-nationality-code"
              />
              <p className="text-xs text-muted-foreground">
                رمز الدولة المختصر (ISO 3166-1)
              </p>
            </div>
            <div className="flex items-center justify-between">
              <Label htmlFor="isActive">الحالة</Label>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">
                  {formData.isActive ? "مفعّل" : "غير مفعّل"}
                </span>
                <Switch
                  id="isActive"
                  checked={formData.isActive}
                  onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                  data-testid="switch-nationality-active"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:gap-0">
            <Button
              variant="outline"
              onClick={resetForm}
              data-testid="button-cancel-nationality"
            >
              إلغاء
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save-nationality"
            >
              {createMutation.isPending || updateMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
