import { useQuery, useMutation } from "@tanstack/react-query";
import { useState, useEffect } from "react";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber } from "@/lib/formatters";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Calendar as CalendarComponent } from "@/components/ui/calendar";
import { format } from "date-fns";
import { ar } from "date-fns/locale";
import { 
  Calendar as CalendarIcon, 
  Plus, 
  Minus,
  Trash2,
  User,
  ArrowRight,
  Scissors,
  MapPin,
  Home,
  Building2,
  Truck,
  Gift,
  Ticket,
  Check,
  X,
  Loader2,
  Star,
  UserPlus,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
  isActive: boolean;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
  loyaltyPointsBalance?: number | null;
}

interface ServiceCategory {
  id: number;
  nameAr: string;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  price: number;
  vatType: string;
  vatRate: number;
  categoryId: number;
  category?: ServiceCategory;
  isActive: boolean;
}

interface SystemSetting {
  id: number;
  key: string;
  value: string;
  description: string | null;
  valueType: string;
}

interface ServiceLine {
  serviceId: number;
  service?: Service;
  quantity: number;
  scheduledAt: string;
  preferredEmployeeId: number | null;
}

interface PackageServiceItem {
  id: number;
  serviceId: number;
  quantity: number;
  packagePrice: number;
  service?: Service;
}

interface PackageData {
  id: number;
  nameAr: string;
  startDate: string;
  endDate: string;
  descriptionAr: string | null;
  isActive: boolean;
  services: PackageServiceItem[];
  products: any[];
}

interface CouponData {
  id: number;
  nameAr: string;
  code: string;
  discountType: "percent" | "amount";
  discountValue: number;
  appliesToServices: boolean;
  startDate: string;
  endDate: string;
  isActive: boolean;
  allowedServices: any[];
}

const calculateServiceLineTotal = (service: Service, quantity: number) => {
  const basePrice = service.price;
  const vatType = service.vatType;
  const vatRate = service.vatRate;
  
  let lineSubtotal: number;
  let vatAmount: number;
  let lineTotal: number;
  
  if (vatType === "inclusive") {
    const gross = basePrice * quantity;
    lineSubtotal = gross / (1 + vatRate);
    vatAmount = gross - lineSubtotal;
    lineTotal = gross;
  } else if (vatType === "exclusive") {
    lineSubtotal = basePrice * quantity;
    vatAmount = lineSubtotal * vatRate;
    lineTotal = lineSubtotal + vatAmount;
  } else {
    lineSubtotal = basePrice * quantity;
    vatAmount = 0;
    lineTotal = lineSubtotal;
  }
  
  return { lineSubtotal, vatAmount, lineTotal };
};

export default function NewAppointmentPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const [appointmentType, setAppointmentType] = useState<string>("in_salon");
  const [clientId, setClientId] = useState<string>("");
  const [scheduledAt, setScheduledAt] = useState<string>("");
  const [locationLat, setLocationLat] = useState<string>("");
  const [locationLng, setLocationLng] = useState<string>("");
  const [addressText, setAddressText] = useState<string>("");
  const [notes, setNotes] = useState<string>("");
  
  // Inline customer creation
  const [showNewCustomerDialog, setShowNewCustomerDialog] = useState(false);
  const [newCustomerName, setNewCustomerName] = useState<string>("");
  const [newCustomerPhone, setNewCustomerPhone] = useState<string>("");
  
  const [serviceLines, setServiceLines] = useState<ServiceLine[]>([]);
  const [selectedServiceId, setSelectedServiceId] = useState<string>("");
  
  const [selectedPackageId, setSelectedPackageId] = useState<string>("");
  const [appliedPackage, setAppliedPackage] = useState<PackageData | null>(null);
  const [couponCode, setCouponCode] = useState<string>("");
  const [couponDiscountAmount, setCouponDiscountAmount] = useState<string>("0");
  const [appliedCoupon, setAppliedCoupon] = useState<CouponData | null>(null);
  const [validatingCoupon, setValidatingCoupon] = useState(false);

  const { data: employees, isLoading: loadingEmployees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: customers, isLoading: loadingCustomers } = useQuery<Customer[]>({
    queryKey: ["/api/customers"],
  });

  const { data: services, isLoading: loadingServices } = useQuery<Service[]>({
    queryKey: ["/api/services"],
  });

  const { data: packages, isLoading: loadingPackages } = useQuery<PackageData[]>({
    queryKey: ["/api/marketing/packages"],
  });

  const { data: deliveryFeeSetting } = useQuery<SystemSetting>({
    queryKey: ["/api/settings", "home_service_delivery_fee"],
    queryFn: async () => {
      try {
        const res = await fetch("/api/settings/home_service_delivery_fee");
        if (res.ok) return res.json();
        return null;
      } catch {
        return null;
      }
    },
  });

  const { data: coupons, isLoading: loadingCoupons } = useQuery<CouponData[]>({
    queryKey: ["/api/marketing/coupons"],
  });

  const deliveryFee = deliveryFeeSetting ? parseFloat(deliveryFeeSetting.value) || 0 : 0;

  const createCustomerMutation = useMutation({
    mutationFn: async (data: { nameAr: string; phone: string }) => {
      const response = await apiRequest("POST", "/api/customers", data);
      return response.json();
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      setClientId(data.id.toString());
      setShowNewCustomerDialog(false);
      setNewCustomerName("");
      setNewCustomerPhone("");
      toast({ title: "تم إنشاء العميل بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء العميل",
        variant: "destructive",
      });
    },
  });
  
  const createAppointmentMutation = useMutation({
    mutationFn: async (data: any) => {
      const response = await apiRequest("POST", "/api/appointments", data);
      return response.json();
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/appointments"] });
      toast({ title: "تم إنشاء الحجز بنجاح" });
      setLocation(`/appointments/${data.id}`);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الحجز",
        variant: "destructive",
      });
    },
  });

  const activeEmployees = employees?.filter(e => e.isActive) || [];
  const activeServices = services?.filter(s => s.isActive) || [];

  useEffect(() => {
    if (!scheduledAt) {
      const now = new Date();
      now.setHours(12, 0, 0, 0);
      setScheduledAt(now.toISOString());
    }
  }, []);

  const addServiceLine = () => {
    if (!selectedServiceId) return;
    
    const service = activeServices.find(s => s.id === parseInt(selectedServiceId));
    if (!service) return;
    
    if (serviceLines.some(sl => sl.serviceId === service.id)) {
      toast({ title: "الخدمة مضافة مسبقاً", variant: "destructive" });
      return;
    }
    
    setServiceLines([...serviceLines, {
      serviceId: service.id,
      service,
      quantity: 1,
      scheduledAt: "",
      preferredEmployeeId: null,
    }]);
    setSelectedServiceId("");
  };

  const removeServiceLine = (serviceId: number) => {
    setServiceLines(serviceLines.filter(sl => sl.serviceId !== serviceId));
  };

  const updateServiceLine = (serviceId: number, updates: Partial<ServiceLine>) => {
    setServiceLines(serviceLines.map(sl => 
      sl.serviceId === serviceId ? { ...sl, ...updates } : sl
    ));
  };

  const applyPackage = (pkg: PackageData) => {
    const newServiceLines: ServiceLine[] = pkg.services.map(ps => ({
      serviceId: ps.serviceId,
      service: ps.service || activeServices.find(s => s.id === ps.serviceId),
      quantity: ps.quantity,
      scheduledAt: "",
      preferredEmployeeId: null,
    }));
    setServiceLines(newServiceLines);
    setAppliedPackage(pkg);
    setSelectedPackageId("");
    toast({ title: `تم تطبيق الحزمة: ${pkg.nameAr}` });
  };

  const clearPackage = () => {
    setAppliedPackage(null);
    setServiceLines([]);
  };

  const validateCoupon = async (code: string) => {
    if (!code.trim()) return;
    setValidatingCoupon(true);
    try {
      const res = await fetch(`/api/marketing/coupons/validate?code=${encodeURIComponent(code.trim())}`);
      const data = await res.json();
      if (res.ok && data.valid) {
        setAppliedCoupon(data.coupon);
        const subtotal = serviceTotals.total;
        let discountAmount = 0;
        if (data.coupon.discountType === "percent") {
          discountAmount = subtotal * (data.coupon.discountValue / 100);
        } else {
          discountAmount = Math.min(data.coupon.discountValue, subtotal);
        }
        setCouponDiscountAmount(discountAmount.toFixed(2));
        toast({ title: "تم تطبيق الكوبون بنجاح" });
      } else {
        toast({ title: data.message || "كوبون غير صالح", variant: "destructive" });
        setAppliedCoupon(null);
        setCouponDiscountAmount("0");
      }
    } catch {
      toast({ title: "خطأ في التحقق من الكوبون", variant: "destructive" });
    }
    setValidatingCoupon(false);
  };

  const validPackages = packages?.filter(p => {
    const today = new Date().toISOString().split('T')[0];
    return p.isActive && p.startDate <= today && p.endDate >= today;
  }) || [];

  // Filter valid coupons (active and within date range)
  const validCoupons = coupons?.filter(c => {
    const today = new Date().toISOString().split('T')[0];
    return c.isActive && c.startDate <= today && c.endDate >= today;
  }) || [];

  const discount = parseFloat(couponDiscountAmount) || 0;

  const serviceTotals = serviceLines.reduce((acc, sl) => {
    if (sl.service) {
      const { lineSubtotal, vatAmount, lineTotal } = calculateServiceLineTotal(sl.service, sl.quantity);
      return {
        subtotal: acc.subtotal + lineSubtotal,
        vat: acc.vat + vatAmount,
        total: acc.total + lineTotal,
      };
    }
    return acc;
  }, { subtotal: 0, vat: 0, total: 0 });

  const effectiveDeliveryFee = appointmentType === "home_service" ? deliveryFee : 0;
  const grandTotal = Math.max(0, serviceTotals.total + effectiveDeliveryFee - discount);

  const handleSubmit = () => {
    if (serviceLines.length === 0) {
      toast({ title: "يجب إضافة خدمة واحدة على الأقل", variant: "destructive" });
      return;
    }
    
    if (!clientId) {
      toast({ title: "يجب اختيار العميل", variant: "destructive" });
      return;
    }

    if (!scheduledAt) {
      toast({ title: "يجب تحديد موعد الحجز", variant: "destructive" });
      return;
    }

    if (appointmentType === "home_service" && !addressText.trim()) {
      toast({ title: "يجب إدخال عنوان العميل للخدمة المنزلية", variant: "destructive" });
      return;
    }
    
    const appointmentData = {
      appointmentType,
      source: "app",
      clientId: parseInt(clientId),
      scheduledAt: new Date(scheduledAt).toISOString(),
      locationLat: locationLat ? parseFloat(locationLat) : null,
      locationLng: locationLng ? parseFloat(locationLng) : null,
      addressText: addressText.trim() || null,
      packageId: appliedPackage?.id || null,
      couponId: appliedCoupon?.id || null,
      couponCode: couponCode || null,
      couponDiscountAmount: discount,
      services: serviceLines.map(sl => ({
        serviceId: sl.serviceId,
        quantity: sl.quantity,
        scheduledAt: sl.scheduledAt || null,
        preferredEmployeeId: sl.preferredEmployeeId,
      })),
      notes: notes || null,
    };
    
    createAppointmentMutation.mutate(appointmentData);
  };

  const isLoading = loadingEmployees || loadingCustomers || loadingServices || loadingPackages;

  if (isLoading) {
    return (
      <div className="container mx-auto p-6 space-y-6" dir="rtl">
        <Skeleton className="h-10 w-48" />
        <Skeleton className="h-96 w-full" />
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => setLocation("/appointments")} data-testid="button-back">
          <ArrowRight className="h-5 w-5" />
        </Button>
        <div className="flex items-center gap-2">
          <CalendarIcon className="h-6 w-6" />
          <h1 className="text-2xl font-bold">حجز جديد</h1>
        </div>
      </div>

      <div className="grid lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="h-5 w-5" />
                معلومات الحجز
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>نوع الحجز</Label>
                  <RadioGroup 
                    value={appointmentType} 
                    onValueChange={setAppointmentType}
                    className="flex gap-4"
                    data-testid="radio-appointment-type"
                  >
                    <label 
                      className={`flex items-center gap-2 p-3 rounded-lg border cursor-pointer transition-colors ${appointmentType === "in_salon" ? "border-primary bg-primary/5" : "border-border hover-elevate"}`}
                      data-testid="radio-in-salon"
                    >
                      <RadioGroupItem value="in_salon" id="in_salon" />
                      <Building2 className="h-4 w-4" />
                      <span>في الصالون</span>
                    </label>
                    <label 
                      className={`flex items-center gap-2 p-3 rounded-lg border cursor-pointer transition-colors ${appointmentType === "home_service" ? "border-primary bg-primary/5" : "border-border hover-elevate"}`}
                      data-testid="radio-home-service"
                    >
                      <RadioGroupItem value="home_service" id="home_service" />
                      <Home className="h-4 w-4" />
                      <span>خدمة منزلية</span>
                    </label>
                  </RadioGroup>
                </div>
                <div className="space-y-2">
                  <Label>العميل *</Label>
                  <div className="flex gap-2">
                    <Select value={clientId || "_none"} onValueChange={(v) => setClientId(v === "_none" ? "" : v)} data-testid="select-client">
                      <SelectTrigger className="flex-1">
                        <SelectValue placeholder="اختر العميل" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="_none">-- اختر العميل --</SelectItem>
                        {customers?.map((customer) => (
                          <SelectItem key={customer.id} value={customer.id.toString()}>
                            <div className="flex items-center gap-2">
                              <span>{customer.nameAr}</span>
                              {customer.phone && <span className="text-muted-foreground">({customer.phone})</span>}
                              {customer.loyaltyPointsBalance ? (
                                <Badge variant="secondary" className="text-xs">
                                  <Star className="h-3 w-3 ml-1 text-yellow-500" />
                                  {customer.loyaltyPointsBalance}
                                </Badge>
                              ) : null}
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <Button 
                      variant="outline" 
                      size="icon" 
                      onClick={() => setShowNewCustomerDialog(true)}
                      title="إضافة عميل جديد"
                      data-testid="button-new-customer"
                    >
                      <UserPlus className="h-4 w-4" />
                    </Button>
                  </div>
                  {clientId && (() => {
                    const selectedCustomer = customers?.find(c => c.id === parseInt(clientId));
                    if (selectedCustomer?.loyaltyPointsBalance) {
                      return (
                        <div className="flex items-center gap-2 text-sm text-muted-foreground">
                          <Star className="h-4 w-4 text-yellow-500" />
                          <span>رصيد النقاط: {selectedCustomer.loyaltyPointsBalance} نقطة</span>
                        </div>
                      );
                    }
                    return null;
                  })()}
                </div>
              </div>
              <div className="space-y-2">
                <Label>موعد الحجز *</Label>
                <div className="flex gap-2">
                  <Popover>
                    <PopoverTrigger asChild>
                      <Button
                        variant="outline"
                        className="flex-1 justify-start text-right font-normal"
                        data-testid="input-scheduled-at"
                      >
                        <CalendarIcon className="ml-2 h-4 w-4" />
                        {scheduledAt ? format(new Date(scheduledAt), "PPP", { locale: ar }) : "اختر التاريخ"}
                      </Button>
                    </PopoverTrigger>
                    <PopoverContent className="w-auto p-0" align="start">
                      <CalendarComponent
                        mode="single"
                        selected={scheduledAt ? new Date(scheduledAt) : undefined}
                        onSelect={(date) => {
                          if (date) {
                            const existing = scheduledAt ? new Date(scheduledAt) : null;
                            if (existing) {
                              date.setHours(existing.getHours(), existing.getMinutes(), 0, 0);
                            } else {
                              date.setHours(12, 0, 0, 0);
                            }
                            setScheduledAt(date.toISOString());
                          }
                        }}
                        disabled={(date) => {
                          const today = new Date();
                          today.setHours(0, 0, 0, 0);
                          return date < today;
                        }}
                        initialFocus
                      />
                    </PopoverContent>
                  </Popover>
                  <Input
                    type="time"
                    value={scheduledAt ? format(new Date(scheduledAt), "HH:mm") : "12:00"}
                    onChange={(e) => {
                      if (scheduledAt && e.target.value) {
                        const [hours, minutes] = e.target.value.split(':').map(Number);
                        const date = new Date(scheduledAt);
                        date.setHours(hours, minutes, 0, 0);
                        setScheduledAt(date.toISOString());
                      }
                    }}
                    className="w-28"
                    data-testid="input-scheduled-time"
                  />
                </div>
              </div>
            </CardContent>
          </Card>

          {appointmentType === "home_service" && (
            <Card className="border-orange-200 bg-orange-50/50 dark:bg-orange-950/20 dark:border-orange-800">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-orange-700 dark:text-orange-400">
                  <MapPin className="h-5 w-5" />
                  معلومات الخدمة المنزلية
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label>العنوان *</Label>
                  <Textarea
                    value={addressText}
                    onChange={(e) => setAddressText(e.target.value)}
                    placeholder="أدخل عنوان العميل بالتفصيل (الحي، الشارع، رقم المبنى...)"
                    rows={2}
                    data-testid="input-address"
                  />
                </div>
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>خط العرض (اختياري)</Label>
                    <Input
                      type="number"
                      step="0.000001"
                      value={locationLat}
                      onChange={(e) => setLocationLat(e.target.value)}
                      placeholder="24.7136"
                      data-testid="input-lat"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>خط الطول (اختياري)</Label>
                    <Input
                      type="number"
                      step="0.000001"
                      value={locationLng}
                      onChange={(e) => setLocationLng(e.target.value)}
                      placeholder="46.6753"
                      data-testid="input-lng"
                    />
                  </div>
                </div>
                <div className="flex items-center gap-2 p-3 bg-orange-100 dark:bg-orange-900/30 rounded-lg">
                  <Truck className="h-5 w-5 text-orange-600 dark:text-orange-400" />
                  <div>
                    <div className="text-sm font-medium">رسوم التوصيل</div>
                    <div className="text-lg font-bold text-orange-700 dark:text-orange-400">
                      {formatNumber(deliveryFee)} ر.س
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          )}

          {validPackages.length > 0 && (
            <Card className="border-primary/20">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Gift className="h-5 w-5 text-primary" />
                  الحزم المتاحة
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {appliedPackage ? (
                  <div className="flex items-center justify-between p-3 bg-green-50 dark:bg-green-900/20 rounded-lg border border-green-200 dark:border-green-800">
                    <div className="flex items-center gap-2">
                      <Check className="h-5 w-5 text-green-600" />
                      <div>
                        <span className="font-medium text-green-700 dark:text-green-400">{appliedPackage.nameAr}</span>
                        {appliedPackage.descriptionAr && (
                          <p className="text-sm text-green-600 dark:text-green-500">{appliedPackage.descriptionAr}</p>
                        )}
                      </div>
                    </div>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={clearPackage}
                      className="text-destructive"
                      data-testid="button-clear-package"
                    >
                      <X className="h-4 w-4 ml-1" />
                      إزالة
                    </Button>
                  </div>
                ) : (
                  <div className="flex gap-2">
                    <Select value={selectedPackageId} onValueChange={setSelectedPackageId} data-testid="select-package">
                      <SelectTrigger className="flex-1">
                        <SelectValue placeholder="اختر حزمة لتطبيقها" />
                      </SelectTrigger>
                      <SelectContent>
                        {validPackages.map((pkg) => (
                          <SelectItem key={pkg.id} value={pkg.id.toString()}>
                            {pkg.nameAr} ({pkg.services.length} خدمة)
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <Button
                      onClick={() => {
                        const pkg = validPackages.find(p => p.id === parseInt(selectedPackageId));
                        if (pkg) applyPackage(pkg);
                      }}
                      disabled={!selectedPackageId}
                      data-testid="button-apply-package"
                    >
                      <Plus className="h-4 w-4 ml-1" />
                      تطبيق
                    </Button>
                  </div>
                )}
                {!appliedPackage && (
                  <p className="text-sm text-muted-foreground">
                    اختر حزمة لملء الحجز تلقائياً بالخدمات المتضمنة
                  </p>
                )}
              </CardContent>
            </Card>
          )}

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Scissors className="h-5 w-5" />
                الخدمات
                {appliedPackage && <Badge variant="secondary" className="mr-2">من الحزمة</Badge>}
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex gap-2">
                <Select value={selectedServiceId} onValueChange={setSelectedServiceId} data-testid="select-service">
                  <SelectTrigger className="flex-1">
                    <SelectValue placeholder="اختر خدمة لإضافتها" />
                  </SelectTrigger>
                  <SelectContent>
                    {activeServices.map((service) => (
                      <SelectItem key={service.id} value={service.id.toString()}>
                        {service.nameAr} - {formatNumber(service.price)} ر.س
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button onClick={addServiceLine} disabled={!selectedServiceId} data-testid="button-add-service">
                  <Plus className="h-4 w-4" />
                </Button>
              </div>

              {serviceLines.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Scissors className="h-10 w-10 mx-auto mb-2 opacity-50" />
                  <p>لم يتم إضافة خدمات بعد</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {serviceLines.map((sl) => (
                    <div 
                      key={sl.serviceId} 
                      className="border rounded-lg p-4 space-y-3"
                      data-testid={`service-line-${sl.serviceId}`}
                    >
                      <div className="flex items-center justify-between">
                        <div>
                          <span className="font-medium">{sl.service?.nameAr}</span>
                          <Badge variant="secondary" className="mr-2">
                            {formatNumber(sl.service?.price || 0)} ر.س
                          </Badge>
                        </div>
                        <Button 
                          variant="ghost" 
                          size="icon" 
                          onClick={() => removeServiceLine(sl.serviceId)}
                          data-testid={`button-remove-service-${sl.serviceId}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                      <div className="grid md:grid-cols-3 gap-3">
                        <div className="space-y-1">
                          <Label className="text-sm">الكمية</Label>
                          <div className="flex items-center gap-2">
                            <Button 
                              variant="outline" 
                              size="icon" 
                              className="h-8 w-8"
                              onClick={() => updateServiceLine(sl.serviceId, { quantity: Math.max(1, sl.quantity - 1) })}
                            >
                              <Minus className="h-3 w-3" />
                            </Button>
                            <span className="w-8 text-center">{sl.quantity}</span>
                            <Button 
                              variant="outline" 
                              size="icon" 
                              className="h-8 w-8"
                              onClick={() => updateServiceLine(sl.serviceId, { quantity: sl.quantity + 1 })}
                            >
                              <Plus className="h-3 w-3" />
                            </Button>
                          </div>
                        </div>
                        <div className="space-y-1">
                          <Label className="text-sm">الموعد (اختياري)</Label>
                          <div className="flex gap-1">
                            <Popover>
                              <PopoverTrigger asChild>
                                <Button
                                  variant="outline"
                                  className="flex-1 justify-start text-right font-normal text-xs"
                                  data-testid={`input-scheduled-${sl.serviceId}`}
                                >
                                  <CalendarIcon className="ml-1 h-3 w-3" />
                                  {sl.scheduledAt ? format(new Date(sl.scheduledAt), "d/M", { locale: ar }) : "تاريخ"}
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-auto p-0" align="start">
                                <CalendarComponent
                                  mode="single"
                                  selected={sl.scheduledAt ? new Date(sl.scheduledAt) : undefined}
                                  onSelect={(date) => {
                                    if (date) {
                                      const existing = sl.scheduledAt ? new Date(sl.scheduledAt) : null;
                                      if (existing) {
                                        date.setHours(existing.getHours(), existing.getMinutes(), 0, 0);
                                      } else {
                                        date.setHours(12, 0, 0, 0);
                                      }
                                      updateServiceLine(sl.serviceId, { scheduledAt: date.toISOString() });
                                    } else {
                                      updateServiceLine(sl.serviceId, { scheduledAt: "" });
                                    }
                                  }}
                                  initialFocus
                                />
                              </PopoverContent>
                            </Popover>
                            <Input
                              type="time"
                              value={sl.scheduledAt ? format(new Date(sl.scheduledAt), "HH:mm") : ""}
                              onChange={(e) => {
                                if (e.target.value) {
                                  const [hours, minutes] = e.target.value.split(':').map(Number);
                                  const date = sl.scheduledAt ? new Date(sl.scheduledAt) : new Date();
                                  date.setHours(hours, minutes, 0, 0);
                                  updateServiceLine(sl.serviceId, { scheduledAt: date.toISOString() });
                                }
                              }}
                              className="w-20 text-xs"
                              data-testid={`input-scheduled-time-${sl.serviceId}`}
                            />
                          </div>
                        </div>
                        <div className="space-y-1">
                          <Label className="text-sm">الموظف المفضل (اختياري)</Label>
                          <Select 
                            value={sl.preferredEmployeeId?.toString() || "_none"} 
                            onValueChange={(v) => updateServiceLine(sl.serviceId, { preferredEmployeeId: v && v !== "_none" ? parseInt(v) : null })}
                          >
                            <SelectTrigger>
                              <SelectValue placeholder="اختر" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="_none">غير محدد</SelectItem>
                              {activeEmployees.map((emp) => (
                                <SelectItem key={emp.id} value={emp.id.toString()}>
                                  {emp.nameAr}
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        </div>
                      </div>
                      {sl.service && (
                        <div className="text-left text-sm text-muted-foreground">
                          المجموع: {formatNumber(calculateServiceLineTotal(sl.service, sl.quantity).lineTotal)} ر.س
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Ticket className="h-5 w-5" />
                الكوبونات والملاحظات
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label>كود الكوبون (اختياري)</Label>
                <div className="flex gap-2">
                  <Input
                    value={couponCode}
                    onChange={(e) => {
                      setCouponCode(e.target.value.toUpperCase());
                      if (!e.target.value.trim()) {
                        setAppliedCoupon(null);
                        setCouponDiscountAmount("0");
                      }
                    }}
                    placeholder="أدخل كود الكوبون"
                    className="flex-1"
                    data-testid="input-coupon-code"
                  />
                  <Button 
                    variant="outline" 
                    onClick={() => validateCoupon(couponCode)}
                    disabled={validatingCoupon || !couponCode.trim()}
                    data-testid="button-apply-coupon"
                  >
                    {validatingCoupon ? (
                      <Loader2 className="h-4 w-4 animate-spin" />
                    ) : appliedCoupon ? (
                      <Check className="h-4 w-4 text-green-600" />
                    ) : (
                      "تطبيق"
                    )}
                  </Button>
                  {appliedCoupon && (
                    <Button 
                      variant="ghost" 
                      size="icon"
                      onClick={() => {
                        setCouponCode("");
                        setAppliedCoupon(null);
                        setCouponDiscountAmount("0");
                      }}
                      data-testid="button-clear-coupon"
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  )}
                </div>
                {appliedCoupon && (
                  <div className="flex items-center gap-2 p-2 bg-green-50 dark:bg-green-900/20 rounded-md border border-green-200 dark:border-green-800">
                    <Check className="h-4 w-4 text-green-600" />
                    <span className="text-sm text-green-700 dark:text-green-400">
                      {appliedCoupon.nameAr} - {appliedCoupon.discountType === "percent" ? `${appliedCoupon.discountValue}%` : `${appliedCoupon.discountValue} ر.س`}
                    </span>
                  </div>
                )}
              </div>
              <div className="space-y-2">
                <Label>ملاحظات (اختياري)</Label>
                <Textarea
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  placeholder="أضف ملاحظات للحجز"
                  data-testid="input-notes"
                />
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="lg:col-span-1">
          <Card className="sticky top-6">
            <CardHeader>
              <CardTitle>ملخص الحجز</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span>مجموع الخدمات:</span>
                  <span>{formatNumber(serviceTotals.total)} ر.س</span>
                </div>
                {serviceTotals.vat > 0 && (
                  <div className="flex justify-between text-muted-foreground">
                    <span className="text-xs">(شامل الضريبة: {formatNumber(serviceTotals.vat)} ر.س)</span>
                  </div>
                )}
                {appointmentType === "home_service" && (
                  <div className="flex justify-between text-orange-600 dark:text-orange-400">
                    <span>رسوم التوصيل:</span>
                    <span>{formatNumber(deliveryFee)} ر.س</span>
                  </div>
                )}
                {discount > 0 && (
                  <div className="flex justify-between text-green-600 dark:text-green-400">
                    <span>الخصم:</span>
                    <span>- {formatNumber(discount)} ر.س</span>
                  </div>
                )}
                <div className="border-t pt-2 flex justify-between font-bold text-lg">
                  <span>الإجمالي:</span>
                  <span data-testid="text-grand-total">{formatNumber(grandTotal)} ر.س</span>
                </div>
              </div>

              <div className="space-y-2 text-xs text-muted-foreground">
                <div className="flex justify-between">
                  <span>عدد الخدمات:</span>
                  <span>{serviceLines.length}</span>
                </div>
                <div className="flex justify-between">
                  <span>نوع الحجز:</span>
                  <span>{appointmentType === "home_service" ? "خدمة منزلية" : "في الصالون"}</span>
                </div>
              </div>

              <Button 
                className="w-full" 
                size="lg"
                onClick={handleSubmit}
                disabled={createAppointmentMutation.isPending || serviceLines.length === 0}
                data-testid="button-submit-appointment"
              >
                {createAppointmentMutation.isPending ? "جاري الإنشاء..." : "إنشاء الحجز"}
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
      
      {/* Inline Customer Creation Dialog */}
      <Dialog open={showNewCustomerDialog} onOpenChange={setShowNewCustomerDialog}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <UserPlus className="h-5 w-5" />
              إضافة عميل جديد
            </DialogTitle>
            <DialogDescription>
              أدخل بيانات العميل الجديد
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="newCustomerName">اسم العميل *</Label>
              <Input
                id="newCustomerName"
                value={newCustomerName}
                onChange={(e) => setNewCustomerName(e.target.value)}
                placeholder="أدخل اسم العميل"
                data-testid="input-new-customer-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="newCustomerPhone">رقم الهاتف</Label>
              <Input
                id="newCustomerPhone"
                value={newCustomerPhone}
                onChange={(e) => setNewCustomerPhone(e.target.value)}
                placeholder="أدخل رقم الهاتف"
                data-testid="input-new-customer-phone"
              />
            </div>
          </div>
          <DialogFooter className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => {
                setShowNewCustomerDialog(false);
                setNewCustomerName("");
                setNewCustomerPhone("");
              }}
              data-testid="button-cancel-new-customer"
            >
              إلغاء
            </Button>
            <Button
              onClick={() => {
                if (!newCustomerName.trim()) {
                  toast({ title: "يجب إدخال اسم العميل", variant: "destructive" });
                  return;
                }
                createCustomerMutation.mutate({
                  nameAr: newCustomerName.trim(),
                  phone: newCustomerPhone.trim() || "",
                });
              }}
              disabled={createCustomerMutation.isPending}
              data-testid="button-save-new-customer"
            >
              {createCustomerMutation.isPending ? (
                <>
                  <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                  جاري الإضافة...
                </>
              ) : (
                "إضافة العميل"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
