import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { useRoute, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  ShoppingCart, 
  ArrowRight,
  User,
  Calendar,
  Star,
  Clock,
  CheckCircle2,
  XCircle,
  Pause,
  Play,
  CalendarDays,
  Scissors,
  Package,
  Receipt,
  MessageSquare,
  RotateCcw,
  AlertCircle,
  UserCog,
  Ban,
  Printer,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface ServiceCategory {
  id: number;
  nameAr: string;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  category?: ServiceCategory;
}

interface Unit {
  id: number;
  nameAr: string;
}

interface PaymentMethod {
  id: number;
  nameAr: string;
  nameEn: string;
  type: string;
  isActive: boolean;
}

interface Product {
  id: number;
  nameAr: string;
  sellUnit?: Unit;
}

interface OrderService {
  id: number;
  orderId: number;
  serviceId: number;
  packageId: number | null;
  quantity: number;
  basePrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  status: string;
  scheduledAt: string | null;
  deferredAt: string | null;
  deferredReason: string | null;
  completedAt: string | null;
  executingEmployeeId: number | null;
  service?: Service;
  executingEmployee?: Employee;
}

interface PackageInfo {
  id: number;
  nameAr: string;
}

interface OrderPackage {
  id: number;
  orderId: number;
  packageId: number;
  package?: PackageInfo;
}

interface OrderProduct {
  id: number;
  orderId: number;
  productId: number;
  packageId: number | null;
  quantity: number;
  unitPrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  product?: Product;
}

interface OrderRating {
  id: number;
  orderId: number;
  clientId: number | null;
  rating: number;
  comment: string | null;
  createdAt: string;
}

interface Order {
  id: number;
  orderNumber: string;
  orderDate: string;
  source: string;
  orderType: string;
  clientId: number | null;
  createdByEmployeeId: number;
  branchId: number | null;
  couponCode: string | null;
  couponDiscountAmount: number;
  status: string;
  notes: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  deferredAt: string | null;
  deferredReason: string | null;
  cancelledAt: string | null;
  cancelledReason: string | null;
  returnedAt: string | null;
  returnedReason: string | null;
  createdAt: string;
  updatedAt: string;
  client?: Customer;
  createdByEmployee?: Employee;
  services: OrderService[];
  products: OrderProduct[];
  packages?: OrderPackage[];
  rating?: OrderRating;
}

const statusLabels: Record<string, string> = {
  new: "جديد",
  in_progress: "قيد التنفيذ",
  completed: "مكتمل",
  canceled: "ملغي",
  scheduled: "مجدول",
  postponed: "مؤجل",
  deferred: "مؤجل",
  returned: "مرتجع",
};

const statusColors: Record<string, string> = {
  new: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
  in_progress: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
  completed: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
  canceled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
  scheduled: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
  postponed: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
  deferred: "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300",
  returned: "bg-pink-100 text-pink-800 dark:bg-pink-900 dark:text-pink-300",
};

const serviceStatusLabels: Record<string, string> = {
  pending: "في الانتظار",
  in_progress: "قيد التنفيذ",
  completed: "مكتملة",
  deferred: "مؤجلة",
  cancelled: "ملغاة",
};

const serviceStatusColors: Record<string, string> = {
  pending: "bg-slate-100 text-slate-800 dark:bg-slate-800 dark:text-slate-200",
  in_progress: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
  completed: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
  deferred: "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300",
  cancelled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
};

const sourceLabels: Record<string, string> = {
  pos: "نقطة البيع",
  app: "التطبيق",
};

const orderTypeLabels: Record<string, string> = {
  normal: "طلب عادي",
  gift: "هدية",
};

const vatTypeLabels: Record<string, string> = {
  inclusive: "شامل الضريبة",
  exclusive: "غير شامل",
  exempt: "معفي",
};

const getStatusIcon = (status: string) => {
  switch (status) {
    case "new":
    case "pending":
      return <Clock className="h-4 w-4" />;
    case "in_progress":
      return <Play className="h-4 w-4" />;
    case "completed":
      return <CheckCircle2 className="h-4 w-4" />;
    case "canceled":
    case "cancelled":
      return <XCircle className="h-4 w-4" />;
    case "scheduled":
      return <CalendarDays className="h-4 w-4" />;
    case "postponed":
    case "deferred":
      return <Pause className="h-4 w-4" />;
    case "returned":
      return <RotateCcw className="h-4 w-4" />;
    default:
      return null;
  }
};

export default function OrderDetailPage() {
  const [, params] = useRoute("/orders/:id");
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const orderId = params?.id ? parseInt(params.id) : 0;

  const [isStatusDialogOpen, setIsStatusDialogOpen] = useState(false);
  const [isRatingDialogOpen, setIsRatingDialogOpen] = useState(false);
  const [isDeferDialogOpen, setIsDeferDialogOpen] = useState(false);
  const [isCancelDialogOpen, setIsCancelDialogOpen] = useState(false);
  const [isServiceDialogOpen, setIsServiceDialogOpen] = useState(false);
  const [selectedService, setSelectedService] = useState<OrderService | null>(null);
  const [newStatus, setNewStatus] = useState<string>("");
  const [ratingValue, setRatingValue] = useState<number>(5);
  const [ratingComment, setRatingComment] = useState<string>("");
  const [deferReason, setDeferReason] = useState<string>("");
  const [cancelReason, setCancelReason] = useState<string>("");
  const [refundPaymentMethodId, setRefundPaymentMethodId] = useState<string>("");
  const [serviceStatus, setServiceStatus] = useState<string>("");
  const [serviceEmployeeId, setServiceEmployeeId] = useState<string>("");

  const { data: order, isLoading, error } = useQuery<Order>({
    queryKey: ["/api/orders", orderId],
    queryFn: async () => {
      const response = await fetch(`/api/orders/${orderId}`);
      if (!response.ok) throw new Error("Failed to fetch order");
      return response.json();
    },
    enabled: orderId > 0,
  });

  const { data: employees = [] } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: paymentMethods = [] } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const updateStatusMutation = useMutation({
    mutationFn: (status: string) =>
      apiRequest("PUT", `/api/orders/${orderId}`, { status }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders", orderId] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم تحديث حالة الطلب بنجاح" });
      setIsStatusDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث حالة الطلب",
        variant: "destructive",
      });
    },
  });

  const submitRatingMutation = useMutation({
    mutationFn: (data: { rating: number; comment: string | null }) =>
      apiRequest("POST", `/api/orders/${orderId}/rate`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders", orderId] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم حفظ التقييم بنجاح" });
      setIsRatingDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في حفظ التقييم",
        variant: "destructive",
      });
    },
  });

  const deferOrderMutation = useMutation({
    mutationFn: (reason: string) =>
      apiRequest("POST", `/api/orders/${orderId}/defer`, { reason }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders", orderId] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم تأجيل الطلب بنجاح" });
      setIsDeferDialogOpen(false);
      setDeferReason("");
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تأجيل الطلب",
        variant: "destructive",
      });
    },
  });

  const cancelOrderMutation = useMutation({
    mutationFn: (data: { reason: string; refundPaymentMethodId?: number }) =>
      apiRequest("POST", `/api/orders/${orderId}/cancel`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders", orderId] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم إلغاء الطلب بنجاح وإنشاء إشعار الدائن" });
      setIsCancelDialogOpen(false);
      setCancelReason("");
      setRefundPaymentMethodId("");
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إلغاء الطلب",
        variant: "destructive",
      });
    },
  });

  const updateServiceMutation = useMutation({
    mutationFn: (data: { serviceId: number; status?: string; executingEmployeeId?: number | null }) =>
      apiRequest("PUT", `/api/order-services/${data.serviceId}`, {
        status: data.status,
        executingEmployeeId: data.executingEmployeeId,
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders", orderId] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم تحديث الخدمة بنجاح" });
      setIsServiceDialogOpen(false);
      setSelectedService(null);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث الخدمة",
        variant: "destructive",
      });
    },
  });

  const handleStatusChange = () => {
    if (newStatus) {
      updateStatusMutation.mutate(newStatus);
    }
  };

  const handleRatingSubmit = () => {
    submitRatingMutation.mutate({
      rating: ratingValue,
      comment: ratingComment || null,
    });
  };

  const openStatusDialog = () => {
    if (order) {
      setNewStatus(order.status);
      setIsStatusDialogOpen(true);
    }
  };

  const openRatingDialog = () => {
    if (order?.rating) {
      setRatingValue(order.rating.rating);
      setRatingComment(order.rating.comment || "");
    } else {
      setRatingValue(5);
      setRatingComment("");
    }
    setIsRatingDialogOpen(true);
  };

  const handleDeferOrder = () => {
    deferOrderMutation.mutate(deferReason);
  };

  const handleCancelOrder = () => {
    cancelOrderMutation.mutate({
      reason: cancelReason,
      refundPaymentMethodId: refundPaymentMethodId && refundPaymentMethodId !== "_none" ? parseInt(refundPaymentMethodId) : undefined,
    });
  };

  const openServiceDialog = (service: OrderService) => {
    setSelectedService(service);
    setServiceStatus(service.status);
    setServiceEmployeeId(service.executingEmployeeId?.toString() || "");
    setIsServiceDialogOpen(true);
  };

  const handleServiceUpdate = () => {
    if (selectedService) {
      updateServiceMutation.mutate({
        serviceId: selectedService.id,
        status: serviceStatus,
        executingEmployeeId: serviceEmployeeId ? parseInt(serviceEmployeeId) : null,
      });
    }
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6 space-y-6" dir="rtl">
        <Skeleton className="h-10 w-48" />
        <div className="grid lg:grid-cols-3 gap-6">
          <div className="lg:col-span-2">
            <Skeleton className="h-96 w-full" />
          </div>
          <Skeleton className="h-64 w-full" />
        </div>
      </div>
    );
  }

  if (error || !order) {
    return (
      <div className="container mx-auto p-6" dir="rtl">
        <Card>
          <CardContent className="p-8 text-center">
            <XCircle className="h-12 w-12 mx-auto mb-4 text-destructive" />
            <h2 className="text-xl font-bold mb-2">الطلب غير موجود</h2>
            <p className="text-muted-foreground mb-4">لم يتم العثور على الطلب المطلوب</p>
            <Button onClick={() => setLocation("/orders")}>العودة للطلبات</Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="icon" onClick={() => setLocation("/orders")} data-testid="button-back">
            <ArrowRight className="h-5 w-5" />
          </Button>
          <div className="flex items-center gap-2">
            <ShoppingCart className="h-6 w-6" />
            <h1 className="text-2xl font-bold" data-testid="text-order-number">{order.orderNumber}</h1>
          </div>
          <Badge 
            className={`${statusColors[order.status]} cursor-pointer`}
            onClick={openStatusDialog}
            data-testid="badge-order-status"
          >
            {getStatusIcon(order.status)}
            <span className="mr-1">{statusLabels[order.status]}</span>
          </Badge>
        </div>
        <div className="flex gap-2 flex-wrap">
          <Button variant="outline" onClick={() => setLocation(`/orders/${order.id}/print`)} data-testid="button-print-order">
            <Printer className="h-4 w-4 ml-2" />
            طباعة الفاتورة
          </Button>
          <Button variant="outline" onClick={openStatusDialog} data-testid="button-change-status">
            تغيير الحالة
          </Button>
          {!["completed", "canceled", "deferred", "returned"].includes(order.status) && (
            <Button variant="outline" onClick={() => setIsDeferDialogOpen(true)} data-testid="button-defer-order">
              تأجيل
              <Pause className="h-4 w-4 mr-2" />
            </Button>
          )}
          {!["completed", "canceled", "returned"].includes(order.status) && (
            <Button variant="destructive" onClick={() => setIsCancelDialogOpen(true)} data-testid="button-cancel-order">
              إلغاء
              <Ban className="h-4 w-4 mr-2" />
            </Button>
          )}
          {order.status === "completed" && (
            <Button onClick={openRatingDialog} data-testid="button-rate-order">
              {order.rating ? "تعديل التقييم" : "تقييم الطلب"}
              <Star className="h-4 w-4 mr-2" />
            </Button>
          )}
        </div>
      </div>

      <div className="grid lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Receipt className="h-5 w-5" />
                معلومات الطلب
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-2 gap-4 text-sm">
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">رقم الطلب:</span>
                    <span className="font-medium">{order.orderNumber}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">تاريخ الطلب:</span>
                    <span>{formatDate(order.orderDate)}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">نوع الطلب:</span>
                    <span>{orderTypeLabels[order.orderType]}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">المصدر:</span>
                    <span>{sourceLabels[order.source]}</span>
                  </div>
                </div>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">العميل:</span>
                    <span>{order.client?.nameAr || "زائر"}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-muted-foreground">أنشئ بواسطة:</span>
                    <span>{order.createdByEmployee?.nameAr || "-"}</span>
                  </div>
                  {order.couponCode && (
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">كود الخصم:</span>
                      <Badge variant="secondary">{order.couponCode}</Badge>
                    </div>
                  )}
                  {order.notes && (
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">ملاحظات:</span>
                      <span className="text-right max-w-48 truncate">{order.notes}</span>
                    </div>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Scissors className="h-5 w-5" />
                الخدمات ({order.services.length})
              </CardTitle>
            </CardHeader>
            <CardContent className="p-0">
              {(() => {
                const standaloneServices = order.services.filter(s => !s.packageId);
                const packageServices = order.services.filter(s => s.packageId);
                const packageGroups = order.packages?.reduce((acc, pkg) => {
                  acc[pkg.packageId] = {
                    package: pkg.package,
                    services: packageServices.filter(s => s.packageId === pkg.packageId)
                  };
                  return acc;
                }, {} as Record<number, { package?: PackageInfo; services: OrderService[] }>) || {};
                
                return (
                  <div className="space-y-4">
                    {standaloneServices.length > 0 && (
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead className="text-right">الخدمة</TableHead>
                            <TableHead className="text-right">الحالة</TableHead>
                            <TableHead className="text-right">الكمية</TableHead>
                            <TableHead className="text-right">السعر</TableHead>
                            <TableHead className="text-right">الضريبة</TableHead>
                            <TableHead className="text-right">المجموع</TableHead>
                            <TableHead className="text-right">المنفذ</TableHead>
                            <TableHead className="text-right">إجراء</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {standaloneServices.map((svc) => (
                            <TableRow key={svc.id}>
                              <TableCell>
                                <div>
                                  <span className="font-medium">{svc.service?.nameAr || `خدمة ${svc.serviceId}`}</span>
                                  {svc.service?.category && (
                                    <Badge variant="outline" className="mr-2 text-xs">
                                      {svc.service.category.nameAr}
                                    </Badge>
                                  )}
                                </div>
                                {svc.scheduledAt && (
                                  <div className="text-xs text-muted-foreground mt-1 flex items-center gap-1">
                                    <Calendar className="h-3 w-3" />
                                    {formatDate(svc.scheduledAt)}
                                  </div>
                                )}
                                {svc.deferredAt && (
                                  <div className="text-xs text-amber-600 mt-1 flex items-center gap-1">
                                    <AlertCircle className="h-3 w-3" />
                                    مؤجلة: {svc.deferredReason || "-"}
                                  </div>
                                )}
                              </TableCell>
                              <TableCell>
                                <Badge className={serviceStatusColors[svc.status] || serviceStatusColors.pending}>
                                  {getStatusIcon(svc.status)}
                                  <span className="mr-1">{serviceStatusLabels[svc.status] || "غير محددة"}</span>
                                </Badge>
                              </TableCell>
                              <TableCell>{svc.quantity}</TableCell>
                              <TableCell>{formatNumber(svc.basePrice)} ر.س</TableCell>
                              <TableCell>
                                <div className="text-xs">
                                  <div>{formatNumber(svc.vatAmount)} ر.س</div>
                                  <div className="text-muted-foreground">{vatTypeLabels[svc.vatType]}</div>
                                </div>
                              </TableCell>
                              <TableCell className="font-medium">{formatNumber(svc.lineTotal)} ر.س</TableCell>
                              <TableCell>
                                {svc.executingEmployee?.nameAr || "-"}
                              </TableCell>
                              <TableCell>
                                <Button 
                                  size="sm" 
                                  variant="ghost"
                                  onClick={() => openServiceDialog(svc)}
                                  data-testid={`button-edit-service-${svc.id}`}
                                >
                                  تعديل
                                  <UserCog className="h-4 w-4 mr-1" />
                                </Button>
                              </TableCell>
                            </TableRow>
                          ))}
                        </TableBody>
                      </Table>
                    )}
                    
                    {Object.entries(packageGroups).map(([pkgId, { package: pkg, services }]) => (
                      <div key={pkgId} className="border rounded-lg overflow-hidden">
                        <div className="bg-primary/10 px-4 py-2 flex items-center gap-2">
                          <Package className="h-4 w-4 text-primary" />
                          <span className="font-semibold text-primary">{pkg?.nameAr || `حزمة ${pkgId}`}</span>
                          <Badge variant="secondary" className="mr-auto">{services.length} خدمات</Badge>
                        </div>
                        <Table>
                          <TableHeader>
                            <TableRow>
                              <TableHead className="text-right">الخدمة</TableHead>
                              <TableHead className="text-right">الحالة</TableHead>
                              <TableHead className="text-right">الكمية</TableHead>
                              <TableHead className="text-right">المنفذ</TableHead>
                              <TableHead className="text-right">إجراء</TableHead>
                            </TableRow>
                          </TableHeader>
                          <TableBody>
                            {services.map((svc) => (
                              <TableRow key={svc.id}>
                                <TableCell>
                                  <div>
                                    <span className="font-medium">{svc.service?.nameAr || `خدمة ${svc.serviceId}`}</span>
                                    {svc.service?.category && (
                                      <Badge variant="outline" className="mr-2 text-xs">
                                        {svc.service.category.nameAr}
                                      </Badge>
                                    )}
                                  </div>
                                  {svc.deferredAt && (
                                    <div className="text-xs text-amber-600 mt-1 flex items-center gap-1">
                                      <AlertCircle className="h-3 w-3" />
                                      مؤجلة: {svc.deferredReason || "-"}
                                    </div>
                                  )}
                                </TableCell>
                                <TableCell>
                                  <Badge className={serviceStatusColors[svc.status] || serviceStatusColors.pending}>
                                    {getStatusIcon(svc.status)}
                                    <span className="mr-1">{serviceStatusLabels[svc.status] || "غير محددة"}</span>
                                  </Badge>
                                </TableCell>
                                <TableCell>{svc.quantity}</TableCell>
                                <TableCell>
                                  {svc.executingEmployee?.nameAr || "-"}
                                </TableCell>
                                <TableCell>
                                  <Button 
                                    size="sm" 
                                    variant="ghost"
                                    onClick={() => openServiceDialog(svc)}
                                    data-testid={`button-edit-service-${svc.id}`}
                                  >
                                    تعديل
                                    <UserCog className="h-4 w-4 mr-1" />
                                  </Button>
                                </TableCell>
                              </TableRow>
                            ))}
                          </TableBody>
                        </Table>
                      </div>
                    ))}
                  </div>
                );
              })()}
            </CardContent>
          </Card>

          {order.products.length > 0 && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  المنتجات ({order.products.length})
                </CardTitle>
              </CardHeader>
              <CardContent className="p-0">
                {(() => {
                  const standaloneProducts = order.products.filter(p => !p.packageId);
                  const packageProducts = order.products.filter(p => p.packageId);
                  const productPackageGroups = order.packages?.reduce((acc, pkg) => {
                    acc[pkg.packageId] = {
                      package: pkg.package,
                      products: packageProducts.filter(p => p.packageId === pkg.packageId)
                    };
                    return acc;
                  }, {} as Record<number, { package?: PackageInfo; products: OrderProduct[] }>) || {};
                  
                  return (
                    <div className="space-y-4">
                      {standaloneProducts.length > 0 && (
                        <Table>
                          <TableHeader>
                            <TableRow>
                              <TableHead className="text-right">المنتج</TableHead>
                              <TableHead className="text-right">الكمية</TableHead>
                              <TableHead className="text-right">سعر الوحدة</TableHead>
                              <TableHead className="text-right">الضريبة</TableHead>
                              <TableHead className="text-right">المجموع</TableHead>
                            </TableRow>
                          </TableHeader>
                          <TableBody>
                            {standaloneProducts.map((prod) => (
                              <TableRow key={prod.id}>
                                <TableCell className="font-medium">
                                  {prod.product?.nameAr || `منتج ${prod.productId}`}
                                </TableCell>
                                <TableCell>{prod.quantity}</TableCell>
                                <TableCell>{formatNumber(prod.unitPrice)} ر.س</TableCell>
                                <TableCell>{formatNumber(prod.vatAmount)} ر.س</TableCell>
                                <TableCell className="font-medium">{formatNumber(prod.lineTotal)} ر.س</TableCell>
                              </TableRow>
                            ))}
                          </TableBody>
                        </Table>
                      )}
                      
                      {Object.entries(productPackageGroups).map(([pkgId, { package: pkg, products }]) => (
                        products.length > 0 && (
                          <div key={pkgId} className="border rounded-lg overflow-hidden">
                            <div className="bg-primary/10 px-4 py-2 flex items-center gap-2">
                              <Package className="h-4 w-4 text-primary" />
                              <span className="font-semibold text-primary">{pkg?.nameAr || `حزمة ${pkgId}`}</span>
                              <Badge variant="secondary" className="mr-auto">{products.length} منتجات</Badge>
                            </div>
                            <Table>
                              <TableHeader>
                                <TableRow>
                                  <TableHead className="text-right">المنتج</TableHead>
                                  <TableHead className="text-right">الكمية</TableHead>
                                  <TableHead className="text-right">سعر الوحدة</TableHead>
                                </TableRow>
                              </TableHeader>
                              <TableBody>
                                {products.map((prod) => (
                                  <TableRow key={prod.id}>
                                    <TableCell className="font-medium">
                                      {prod.product?.nameAr || `منتج ${prod.productId}`}
                                    </TableCell>
                                    <TableCell>{prod.quantity}</TableCell>
                                    <TableCell>{formatNumber(prod.unitPrice)} ر.س</TableCell>
                                  </TableRow>
                                ))}
                              </TableBody>
                            </Table>
                          </div>
                        )
                      ))}
                    </div>
                  );
                })()}
              </CardContent>
            </Card>
          )}

          {order.rating && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Star className="h-5 w-5 fill-yellow-400 text-yellow-400" />
                  تقييم العميل
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex items-center gap-4">
                  <div className="flex">
                    {[1, 2, 3, 4, 5].map((star) => (
                      <Star 
                        key={star}
                        className={`h-6 w-6 ${star <= order.rating!.rating ? "fill-yellow-400 text-yellow-400" : "text-gray-300"}`}
                      />
                    ))}
                  </div>
                  <span className="text-lg font-bold">{order.rating.rating}/5</span>
                </div>
                {order.rating.comment && (
                  <div className="mt-4 p-4 bg-muted rounded-lg flex items-start gap-2">
                    <MessageSquare className="h-4 w-4 mt-0.5 text-muted-foreground" />
                    <p className="text-sm">{order.rating.comment}</p>
                  </div>
                )}
              </CardContent>
            </Card>
          )}
        </div>

        <div className="lg:col-span-1">
          <Card className="sticky top-6">
            <CardHeader>
              <CardTitle>ملخص الفاتورة</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">المجموع الفرعي:</span>
                  <span>{formatNumber(order.subtotalAmount)} ر.س</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">ضريبة القيمة المضافة:</span>
                  <span>{formatNumber(order.vatAmount)} ر.س</span>
                </div>
                {order.couponDiscountAmount > 0 && (
                  <div className="flex justify-between text-green-600">
                    <span>خصم الكوبون:</span>
                    <span>-{formatNumber(order.couponDiscountAmount)} ر.س</span>
                  </div>
                )}
                <div className="border-t pt-2 flex justify-between font-bold text-lg">
                  <span>الإجمالي:</span>
                  <span data-testid="text-total-amount">{formatNumber(order.totalAmount)} ر.س</span>
                </div>
              </div>

              <div className="border-t pt-4 space-y-2 text-xs text-muted-foreground">
                <div className="flex justify-between">
                  <span>تاريخ الإنشاء:</span>
                  <span>{formatDate(order.createdAt)}</span>
                </div>
                <div className="flex justify-between">
                  <span>آخر تحديث:</span>
                  <span>{formatDate(order.updatedAt)}</span>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      <Dialog open={isStatusDialogOpen} onOpenChange={setIsStatusDialogOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تغيير حالة الطلب</DialogTitle>
            <DialogDescription>اختر الحالة الجديدة للطلب {order.orderNumber}</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>الحالة الجديدة</Label>
              <Select value={newStatus} onValueChange={setNewStatus} data-testid="select-new-status">
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="new">جديد</SelectItem>
                  <SelectItem value="in_progress">قيد التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                  <SelectItem value="canceled">ملغي</SelectItem>
                  <SelectItem value="scheduled">مجدول</SelectItem>
                  <SelectItem value="postponed">مؤجل</SelectItem>
                </SelectContent>
              </Select>
            </div>
            {newStatus === "completed" && order.status !== "completed" && (
              <div className="p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg text-sm">
                <p className="text-yellow-800 dark:text-yellow-200">
                  سيتم خصم المنتجات المستهلكة من المخزون عند اكتمال الطلب
                </p>
              </div>
            )}
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsStatusDialogOpen(false)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleStatusChange} 
              disabled={updateStatusMutation.isPending}
              data-testid="button-confirm-status"
            >
              {updateStatusMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isRatingDialogOpen} onOpenChange={setIsRatingDialogOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تقييم الطلب</DialogTitle>
            <DialogDescription>أضف تقييم العميل للطلب {order.orderNumber}</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>التقييم</Label>
              <div className="flex items-center gap-2 justify-center py-4">
                {[1, 2, 3, 4, 5].map((star) => (
                  <button
                    key={star}
                    type="button"
                    onClick={() => setRatingValue(star)}
                    className="focus:outline-none"
                    data-testid={`button-star-${star}`}
                  >
                    <Star 
                      className={`h-8 w-8 cursor-pointer transition-colors ${
                        star <= ratingValue 
                          ? "fill-yellow-400 text-yellow-400" 
                          : "text-gray-300 hover:text-yellow-200"
                      }`}
                    />
                  </button>
                ))}
              </div>
              <p className="text-center text-lg font-bold">{ratingValue}/5</p>
            </div>
            <div className="space-y-2">
              <Label>تعليق (اختياري)</Label>
              <Textarea
                value={ratingComment}
                onChange={(e) => setRatingComment(e.target.value)}
                placeholder="أضف تعليق العميل..."
                data-testid="input-rating-comment"
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsRatingDialogOpen(false)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleRatingSubmit} 
              disabled={submitRatingMutation.isPending}
              data-testid="button-submit-rating"
            >
              {submitRatingMutation.isPending ? "جاري الحفظ..." : "حفظ التقييم"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isDeferDialogOpen} onOpenChange={setIsDeferDialogOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تأجيل الطلب</DialogTitle>
            <DialogDescription>أدخل سبب تأجيل الطلب {order.orderNumber}</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>سبب التأجيل</Label>
              <Textarea
                value={deferReason}
                onChange={(e) => setDeferReason(e.target.value)}
                placeholder="أدخل سبب تأجيل الطلب..."
                data-testid="input-defer-reason"
              />
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsDeferDialogOpen(false)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleDeferOrder} 
              disabled={deferOrderMutation.isPending}
              data-testid="button-confirm-defer"
            >
              {deferOrderMutation.isPending ? "جاري التأجيل..." : "تأجيل"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isCancelDialogOpen} onOpenChange={setIsCancelDialogOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>إلغاء الطلب</DialogTitle>
            <DialogDescription>أدخل سبب إلغاء الطلب {order.orderNumber} واختر طريقة استرداد المبلغ</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="p-3 bg-red-50 dark:bg-red-900/20 rounded-lg text-sm">
              <p className="text-red-800 dark:text-red-200">
                تحذير: لا يمكن التراجع عن إلغاء الطلب. سيتم إنشاء إشعار دائن تلقائياً.
              </p>
            </div>
            <div className="space-y-2">
              <Label>سبب الإلغاء</Label>
              <Textarea
                value={cancelReason}
                onChange={(e) => setCancelReason(e.target.value)}
                placeholder="أدخل سبب إلغاء الطلب..."
                data-testid="input-cancel-reason"
              />
            </div>
            <div className="space-y-2">
              <Label>طريقة استرداد المبلغ</Label>
              <Select value={refundPaymentMethodId} onValueChange={setRefundPaymentMethodId} data-testid="select-refund-payment-method">
                <SelectTrigger>
                  <SelectValue placeholder="اختر طريقة الاسترداد (اختياري)" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="_none">لم يتم تحديد</SelectItem>
                  {paymentMethods.filter(pm => pm.isActive).map((method) => (
                    <SelectItem key={method.id} value={method.id.toString()}>
                      {method.nameAr}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsCancelDialogOpen(false)}>
              تراجع
            </Button>
            <Button 
              variant="destructive"
              onClick={handleCancelOrder} 
              disabled={cancelOrderMutation.isPending}
              data-testid="button-confirm-cancel"
            >
              {cancelOrderMutation.isPending ? "جاري الإلغاء..." : "إلغاء الطلب"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={isServiceDialogOpen} onOpenChange={setIsServiceDialogOpen}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تعديل الخدمة</DialogTitle>
            <DialogDescription>
              تعديل حالة الخدمة {selectedService?.service?.nameAr || ""} والموظف المنفذ
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>حالة الخدمة</Label>
              <Select value={serviceStatus} onValueChange={setServiceStatus} data-testid="select-service-status">
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="pending">في الانتظار</SelectItem>
                  <SelectItem value="in_progress">قيد التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتملة</SelectItem>
                  <SelectItem value="deferred">مؤجلة</SelectItem>
                  <SelectItem value="cancelled">ملغاة</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>الموظف المنفذ</Label>
              <Select value={serviceEmployeeId || "_none"} onValueChange={(v) => setServiceEmployeeId(v === "_none" ? "" : v)} data-testid="select-service-employee">
                <SelectTrigger>
                  <SelectValue placeholder="اختر الموظف المنفذ" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="_none">بدون تعيين</SelectItem>
                  {employees.map((emp) => (
                    <SelectItem key={emp.id} value={emp.id.toString()}>
                      {emp.nameAr} {emp.code ? `(${emp.code})` : ""}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            {serviceStatus === "completed" && selectedService?.status !== "completed" && (
              <div className="p-3 bg-green-50 dark:bg-green-900/20 rounded-lg text-sm">
                <p className="text-green-800 dark:text-green-200">
                  عند اكتمال جميع الخدمات، سيتم تحديث حالة الطلب تلقائياً إلى مكتمل
                </p>
              </div>
            )}
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setIsServiceDialogOpen(false)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleServiceUpdate} 
              disabled={updateServiceMutation.isPending}
              data-testid="button-confirm-service-update"
            >
              {updateServiceMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
