import { useQuery, useMutation } from "@tanstack/react-query";
import { useState, useEffect } from "react";
import { useLocation } from "wouter";
import { format } from "date-fns";
import { ar } from "date-fns/locale";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Calendar } from "@/components/ui/calendar";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber } from "@/lib/formatters";
import { Checkbox } from "@/components/ui/checkbox";
import { 
  ShoppingCart, 
  Plus, 
  Minus,
  Trash2,
  User,
  Calendar as CalendarIcon,
  ArrowRight,
  Scissors,
  Package,
  Ticket,
  Gift,
  Check,
  X,
  Loader2,
  Star,
  UserPlus,
  Users,
  Wallet,
  CreditCard,
  Coins,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
  isActive: boolean;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
  loyaltyPointsBalance?: number;
  email?: string | null;
}

interface ServiceCategory {
  id: number;
  nameAr: string;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  price: number;
  vatType: string;
  vatRate: number;
  categoryId: number;
  category?: ServiceCategory;
  isActive: boolean;
}

interface Product {
  id: number;
  nameAr: string;
  sku: string | null;
  defaultSellPrice: number | null;
  isActive: boolean;
}

interface PaymentMethod {
  id: number;
  nameAr: string;
  code: string | null;
  type: string | null;
  isActive: boolean;
}

interface LoyaltySettings {
  id: number;
  pointsPerCurrency: number;
  currencyPerPoint: number;
  expiryMonths: number;
  minRedeemPoints: number;
  isActive: boolean;
}

interface PackageServiceItem {
  id: number;
  packageId: number;
  serviceId: number;
  quantity: number;
  packagePrice: number;
  service?: Service;
}

interface PackageProductItem {
  id: number;
  packageId: number;
  productId: number;
  quantity: number;
  packagePrice: number;
  product?: Product;
}

interface PackageData {
  id: number;
  nameAr: string;
  imageUrl: string | null;
  startDate: string;
  endDate: string;
  descriptionAr: string | null;
  isActive: boolean;
  services: PackageServiceItem[];
  products: PackageProductItem[];
}

interface CouponData {
  id: number;
  nameAr: string;
  code: string;
  descriptionAr: string | null;
  discountType: "percent" | "amount";
  discountValue: number;
  appliesToServices: boolean;
  appliesToProducts: boolean;
  startDate: string;
  endDate: string;
  isActive: boolean;
  allowedServices: { serviceId: number }[];
  allowedProducts: { productId: number }[];
}

interface ServiceLine {
  serviceId: number;
  service?: Service;
  quantity: number;
  scheduledAt: string;
  executingEmployeeId: number | null;
}

interface ProductLine {
  productId: number;
  product?: Product;
  quantity: number;
}

interface PackageServiceAssignment {
  packageServiceId: number;
  serviceId: number;
  service?: Service;
  quantity: number;
  packagePrice: number;
  executingEmployeeId: number | null;
}

interface AppliedPackage {
  package: PackageData;
  serviceAssignments: PackageServiceAssignment[];
}

interface PaymentLine {
  id: number; // Unique ID for React key
  paymentMethodId: number;
  amount: number;
}

const calculateServiceLineTotal = (service: Service, quantity: number) => {
  const basePrice = service.price;
  const vatType = service.vatType;
  const vatRate = service.vatRate;
  
  let lineSubtotal: number;
  let vatAmount: number;
  let lineTotal: number;
  
  if (vatType === "inclusive") {
    const gross = basePrice * quantity;
    lineSubtotal = gross / (1 + vatRate);
    vatAmount = gross - lineSubtotal;
    lineTotal = gross;
  } else if (vatType === "exclusive") {
    lineSubtotal = basePrice * quantity;
    vatAmount = lineSubtotal * vatRate;
    lineTotal = lineSubtotal + vatAmount;
  } else {
    lineSubtotal = basePrice * quantity;
    vatAmount = 0;
    lineTotal = lineSubtotal;
  }
  
  return { lineSubtotal, vatAmount, lineTotal };
};

export default function OrderNewPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const [orderType, setOrderType] = useState<string>("normal");
  const [clientId, setClientId] = useState<string>("");
  const [createdByEmployeeId, setCreatedByEmployeeId] = useState<string>("");
  const [couponCode, setCouponCode] = useState<string>("");
  const [couponDiscountAmount, setCouponDiscountAmount] = useState<string>("0");
  const [notes, setNotes] = useState<string>("");
  
  // Gift order fields
  const [giftFromType, setGiftFromType] = useState<string>("customer");
  const [giftFromCustomerId, setGiftFromCustomerId] = useState<string>("");
  const [giftFromSupervisorId, setGiftFromSupervisorId] = useState<string>("");
  const [giftToCustomerId, setGiftToCustomerId] = useState<string>("");
  const [giftExpiryDate, setGiftExpiryDate] = useState<string>("");
  
  // Customer search
  const [customerSearch, setCustomerSearch] = useState<string>("");
  const [showNewCustomerDialog, setShowNewCustomerDialog] = useState(false);
  const [newCustomerName, setNewCustomerName] = useState<string>("");
  const [newCustomerPhone, setNewCustomerPhone] = useState<string>("");
  
  const [serviceLines, setServiceLines] = useState<ServiceLine[]>([]);
  const [productLines, setProductLines] = useState<ProductLine[]>([]);
  
  const [selectedServiceId, setSelectedServiceId] = useState<string>("");
  const [selectedProductId, setSelectedProductId] = useState<string>("");
  const [selectedPackageId, setSelectedPackageId] = useState<string>("");
  const [appliedPackages, setAppliedPackages] = useState<AppliedPackage[]>([]);
  const [appliedCoupon, setAppliedCoupon] = useState<CouponData | null>(null);
  const [validatingCoupon, setValidatingCoupon] = useState(false);
  
  // Payment and Loyalty fields
  const [paymentLines, setPaymentLines] = useState<PaymentLine[]>([]);
  const [newPaymentMethodId, setNewPaymentMethodId] = useState<string>("");
  const [newPaymentAmount, setNewPaymentAmount] = useState<string>("");
  const [useLoyaltyPoints, setUseLoyaltyPoints] = useState<boolean>(false);
  const [loyaltyPointsToRedeem, setLoyaltyPointsToRedeem] = useState<number>(0);
  const [nextPaymentLineId, setNextPaymentLineId] = useState<number>(1);

  const { data: employees, isLoading: loadingEmployees } = useQuery<Employee[]>({
    queryKey: ["/api/employees"],
  });

  const { data: customers, isLoading: loadingCustomers } = useQuery<Customer[]>({
    queryKey: ["/api/customers"],
  });

  const { data: services, isLoading: loadingServices } = useQuery<Service[]>({
    queryKey: ["/api/services"],
  });

  const { data: products, isLoading: loadingProducts } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const { data: packages, isLoading: loadingPackages } = useQuery<PackageData[]>({
    queryKey: ["/api/marketing/packages"],
  });

  const { data: coupons, isLoading: loadingCoupons } = useQuery<CouponData[]>({
    queryKey: ["/api/marketing/coupons"],
  });

  const { data: paymentMethods } = useQuery<PaymentMethod[]>({
    queryKey: ["/api/payment-methods"],
  });

  const { data: loyaltySettings } = useQuery<LoyaltySettings | null>({
    queryKey: ["/api/loyalty/settings"],
  });
  
  // Adjust payments when grand total drops below total paid
  // This handles cases where loyalty points or coupons reduce the total after payments are added
  // Instead of clearing all payments, we proportionally reduce them
  useEffect(() => {
    if (paymentLines.length === 0) return;
    
    const totalPaid = paymentLines.reduce((sum, pl) => sum + pl.amount, 0);
    
    // Calculate current grand total for this effect
    const servicesTotalEffect = serviceLines.reduce((acc, sl) => {
      if (sl.service) {
        const { lineTotal } = calculateServiceLineTotal(sl.service, sl.quantity);
        return acc + lineTotal;
      }
      return acc;
    }, 0);
    
    const packagesTotalEffect = appliedPackages.reduce((acc, ap) => {
      return acc + ap.serviceAssignments.reduce((sum, sa) => sum + (sa.packagePrice * sa.quantity), 0);
    }, 0);
    
    const productsTotalEffect = productLines.reduce((acc, pl) => {
      if (pl.product) {
        const price = pl.product.defaultSellPrice || 0;
        const subtotal = price * pl.quantity;
        const vat = subtotal * 0.15;
        return acc + subtotal + vat;
      }
      return acc;
    }, 0);
    
    const discount = parseFloat(couponDiscountAmount) || 0;
    const orderTotalBeforeLoyaltyEffect = servicesTotalEffect + packagesTotalEffect + productsTotalEffect - discount;
    
    const loyaltyPointsValueEffect = useLoyaltyPoints && loyaltySettings?.isActive 
      ? loyaltyPointsToRedeem * (loyaltySettings?.currencyPerPoint || 0) 
      : 0;
    
    const grandTotalEffect = Math.max(0, orderTotalBeforeLoyaltyEffect - loyaltyPointsValueEffect);
    
    // Only adjust if total paid exceeds grand total by more than a small tolerance
    if (totalPaid > grandTotalEffect + 0.01) {
      if (grandTotalEffect <= 0) {
        // If order is free (fully covered by loyalty/coupon), clear all payments
        setPaymentLines([]);
      } else {
        // Proportionally reduce all payment amounts to fit the new total
        const ratio = grandTotalEffect / totalPaid;
        setPaymentLines(lines => 
          lines.map(pl => ({
            ...pl,
            amount: Math.round(pl.amount * ratio * 100) / 100
          })).filter(pl => pl.amount > 0)
        );
      }
    }
  }, [serviceLines, appliedPackages, productLines, couponDiscountAmount, useLoyaltyPoints, loyaltyPointsToRedeem, loyaltySettings]);

  const validateCoupon = async (code: string) => {
    if (!code.trim()) {
      setAppliedCoupon(null);
      setCouponDiscountAmount("0");
      return;
    }
    
    setValidatingCoupon(true);
    try {
      const response = await fetch(`/api/marketing/coupons/validate/${code}`);
      if (response.ok) {
        const coupon: CouponData = await response.json();
        setAppliedCoupon(coupon);
        
        // Calculate discount based on coupon type and current cart
        const applicableTotal = calculateApplicableTotal(coupon);
        let discount = 0;
        if (coupon.discountType === "percent") {
          discount = applicableTotal * (coupon.discountValue / 100);
        } else {
          discount = Math.min(coupon.discountValue, applicableTotal);
        }
        setCouponDiscountAmount(discount.toFixed(2));
        toast({ title: "تم تطبيق الكوبون بنجاح" });
      } else {
        const error = await response.json();
        setAppliedCoupon(null);
        setCouponDiscountAmount("0");
        toast({
          title: "خطأ",
          description: error.error || "الكوبون غير صالح",
          variant: "destructive",
        });
      }
    } catch (error) {
      setAppliedCoupon(null);
      setCouponDiscountAmount("0");
      toast({
        title: "خطأ",
        description: "فشل في التحقق من الكوبون",
        variant: "destructive",
      });
    } finally {
      setValidatingCoupon(false);
    }
  };

  const calculateApplicableTotal = (coupon: CouponData) => {
    let total = 0;
    
    if (coupon.appliesToServices) {
      for (const sl of serviceLines) {
        if (sl.service) {
          // Check if service is allowed (empty means all services)
          const isAllowed = coupon.allowedServices.length === 0 || 
            coupon.allowedServices.some(s => s.serviceId === sl.serviceId);
          if (isAllowed) {
            const { lineTotal } = calculateServiceLineTotal(sl.service, sl.quantity);
            total += lineTotal;
          }
        }
      }
    }
    
    if (coupon.appliesToProducts) {
      for (const pl of productLines) {
        if (pl.product) {
          const isAllowed = coupon.allowedProducts.length === 0 || 
            coupon.allowedProducts.some(p => p.productId === pl.productId);
          if (isAllowed) {
            const price = pl.product.defaultSellPrice || 0;
            const subtotal = price * pl.quantity;
            const vat = subtotal * 0.15;
            total += subtotal + vat;
          }
        }
      }
    }
    
    return total;
  };

  const applyPackage = (pkg: PackageData) => {
    // Check if package is already applied
    if (appliedPackages.some(ap => ap.package.id === pkg.id)) {
      toast({ title: "هذه الحزمة مضافة بالفعل", variant: "destructive" });
      return;
    }
    
    // Create service assignments for this package
    const serviceAssignments: PackageServiceAssignment[] = [];
    for (const pkgService of pkg.services) {
      const service = activeServices.find(s => s.id === pkgService.serviceId);
      if (service) {
        serviceAssignments.push({
          packageServiceId: pkgService.id,
          serviceId: pkgService.serviceId,
          service: { ...service, price: pkgService.packagePrice },
          quantity: pkgService.quantity,
          packagePrice: pkgService.packagePrice,
          executingEmployeeId: null,
        });
      }
    }
    
    // Package products stay with the package (not added to standalone productLines)
    // They will be counted in packageTotals
    
    setAppliedPackages(prev => [...prev, { package: pkg, serviceAssignments }]);
    setSelectedPackageId("");
    toast({ title: `تم إضافة الحزمة: ${pkg.nameAr}` });
  };

  const removePackage = (packageId: number) => {
    setAppliedPackages(prev => prev.filter(ap => ap.package.id !== packageId));
    toast({ title: "تم إزالة الحزمة" });
  };

  const updatePackageServiceEmployee = (packageId: number, packageServiceId: number, employeeId: number | null) => {
    setAppliedPackages(prev => prev.map(ap => {
      if (ap.package.id === packageId) {
        return {
          ...ap,
          serviceAssignments: ap.serviceAssignments.map(sa => {
            if (sa.packageServiceId === packageServiceId) {
              return { ...sa, executingEmployeeId: employeeId };
            }
            return sa;
          }),
        };
      }
      return ap;
    }));
  };

  const clearAllPackages = () => {
    setAppliedPackages([]);
  };

  const createCustomerMutation = useMutation({
    mutationFn: async (data: { nameAr: string; phone: string }) => {
      const response = await apiRequest("POST", "/api/customers", data);
      return response.json();
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/customers"] });
      setClientId(data.id.toString());
      setShowNewCustomerDialog(false);
      setNewCustomerName("");
      setNewCustomerPhone("");
      toast({ title: "تم إنشاء العميل بنجاح" });
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء العميل",
        variant: "destructive",
      });
    },
  });
  
  const createOrderMutation = useMutation({
    mutationFn: async (data: any) => {
      const response = await apiRequest("POST", "/api/orders", data);
      return response.json();
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم إنشاء الطلب بنجاح" });
      setLocation(`/orders/${data.id}`);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في إنشاء الطلب",
        variant: "destructive",
      });
    },
  });

  const activeEmployees = employees?.filter(e => e.isActive) || [];
  const activeServices = services?.filter(s => s.isActive) || [];
  const activeProducts = products?.filter(p => p.isActive) || [];

  const addServiceLine = () => {
    if (!selectedServiceId) return;
    
    const service = activeServices.find(s => s.id === parseInt(selectedServiceId));
    if (!service) return;
    
    // Check if already added
    if (serviceLines.some(sl => sl.serviceId === service.id)) {
      toast({ title: "الخدمة مضافة مسبقاً", variant: "destructive" });
      return;
    }
    
    setServiceLines([...serviceLines, {
      serviceId: service.id,
      service,
      quantity: 1,
      scheduledAt: "",
      executingEmployeeId: null,
    }]);
    setSelectedServiceId("");
  };

  const removeServiceLine = (serviceId: number) => {
    setServiceLines(serviceLines.filter(sl => sl.serviceId !== serviceId));
  };

  const updateServiceLine = (serviceId: number, updates: Partial<ServiceLine>) => {
    setServiceLines(serviceLines.map(sl => 
      sl.serviceId === serviceId ? { ...sl, ...updates } : sl
    ));
  };

  const addProductLine = () => {
    if (!selectedProductId) return;
    
    const product = activeProducts.find(p => p.id === parseInt(selectedProductId));
    if (!product) return;
    
    // Check if already added
    if (productLines.some(pl => pl.productId === product.id)) {
      toast({ title: "المنتج مضاف مسبقاً", variant: "destructive" });
      return;
    }
    
    setProductLines([...productLines, {
      productId: product.id,
      product,
      quantity: 1,
    }]);
    setSelectedProductId("");
  };

  const removeProductLine = (productId: number) => {
    setProductLines(productLines.filter(pl => pl.productId !== productId));
  };

  const updateProductLine = (productId: number, quantity: number) => {
    setProductLines(productLines.map(pl => 
      pl.productId === productId ? { ...pl, quantity } : pl
    ));
  };

  // Calculate totals for standalone services
  const serviceTotals = serviceLines.reduce((acc, sl) => {
    if (sl.service) {
      const { lineSubtotal, vatAmount, lineTotal } = calculateServiceLineTotal(sl.service, sl.quantity);
      return {
        subtotal: acc.subtotal + lineSubtotal,
        vat: acc.vat + vatAmount,
        total: acc.total + lineTotal,
      };
    }
    return acc;
  }, { subtotal: 0, vat: 0, total: 0 });

  // Calculate totals for packages using package prices (services + products)
  const packageTotals = appliedPackages.reduce((acc, ap) => {
    // Sum up package services using packagePrice from actual assignments
    const servicesTotal = ap.serviceAssignments.reduce((sum, sa) => sum + (sa.packagePrice * sa.quantity), 0);
    // Sum up package products using packagePrice from package definition
    const productsTotal = ap.package.products.reduce((sum, p) => sum + (p.packagePrice * p.quantity), 0);
    const packageTotal = servicesTotal + productsTotal;
    // Package prices are VAT-inclusive, calculate the breakdown
    const packageSubtotal = packageTotal / 1.15;
    const packageVat = packageTotal - packageSubtotal;
    return {
      subtotal: acc.subtotal + packageSubtotal,
      vat: acc.vat + packageVat,
      total: acc.total + packageTotal,
    };
  }, { subtotal: 0, vat: 0, total: 0 });

  const productTotals = productLines.reduce((acc, pl) => {
    if (pl.product) {
      const price = pl.product.defaultSellPrice || 0;
      const subtotal = price * pl.quantity;
      const vat = subtotal * 0.15;
      return {
        subtotal: acc.subtotal + subtotal,
        vat: acc.vat + vat,
        total: acc.total + subtotal + vat,
      };
    }
    return acc;
  }, { subtotal: 0, vat: 0, total: 0 });

  const discount = parseFloat(couponDiscountAmount) || 0;
  const allServicesTotals = {
    subtotal: serviceTotals.subtotal + packageTotals.subtotal,
    vat: serviceTotals.vat + packageTotals.vat,
    total: serviceTotals.total + packageTotals.total,
  };
  
  // Get selected customer's loyalty balance
  const selectedCustomer = clientId ? customers?.find(c => c.id === parseInt(clientId)) : null;
  const customerLoyaltyBalance = selectedCustomer?.loyaltyPointsBalance || 0;
  
  // Calculate loyalty points value (if using points)
  const loyaltyPointsValue = useLoyaltyPoints && loyaltySettings?.isActive 
    ? loyaltyPointsToRedeem * (loyaltySettings?.currencyPerPoint || 0) 
    : 0;
  
  // Check if customer can redeem points
  const canRedeemPoints = 
    clientId && 
    loyaltySettings?.isActive && 
    customerLoyaltyBalance >= (loyaltySettings?.minRedeemPoints || 0);
  
  // Maximum points that can be redeemed (limited by balance and order total)
  const orderTotalBeforeLoyalty = allServicesTotals.total + productTotals.total - discount;
  const maxRedeemablePoints = canRedeemPoints && loyaltySettings
    ? Math.min(
        customerLoyaltyBalance,
        Math.floor(orderTotalBeforeLoyalty / (loyaltySettings.currencyPerPoint || 1))
      )
    : 0;
  
  const grandTotal = orderTotalBeforeLoyalty - loyaltyPointsValue;

  const handleSubmit = () => {
    const totalServicesCount = serviceLines.length + appliedPackages.reduce((sum, ap) => sum + ap.serviceAssignments.length, 0);
    if (totalServicesCount === 0) {
      toast({ title: "يجب إضافة خدمة واحدة على الأقل", variant: "destructive" });
      return;
    }
    
    if (!createdByEmployeeId) {
      toast({ title: "يجب اختيار الموظف المنشئ", variant: "destructive" });
      return;
    }
    
    // Validate gift order fields
    if (orderType === "gift") {
      if (!giftFromType) {
        toast({ title: "يجب تحديد نوع المُهدي", variant: "destructive" });
        return;
      }
      if (!giftToCustomerId) {
        toast({ title: "يجب اختيار العميل المُهدى إليه", variant: "destructive" });
        return;
      }
      if (!giftExpiryDate) {
        toast({ title: "يجب تحديد تاريخ انتهاء الهدية", variant: "destructive" });
        return;
      }
      if (giftFromType === "customer" && !giftFromCustomerId) {
        toast({ title: "يجب اختيار العميل المُهدي", variant: "destructive" });
        return;
      }
      // Supervisor selection is optional when giftFromType is "supervisor"
    }
    
    // Combine service lines with package service assignments
    const allServices = [
      // Standalone services
      ...serviceLines.map(sl => ({
        serviceId: sl.serviceId,
        quantity: sl.quantity,
        scheduledAt: sl.scheduledAt || null,
        executingEmployeeId: sl.executingEmployeeId,
        packageId: null,
        packagePrice: null,
      })),
      // Package services - use packagePrice from serviceAssignments
      ...appliedPackages.flatMap(ap => 
        ap.serviceAssignments.map(sa => ({
          serviceId: sa.serviceId,
          quantity: sa.quantity,
          scheduledAt: null,
          executingEmployeeId: sa.executingEmployeeId,
          packageId: ap.package.id,
          packagePrice: sa.packagePrice, // VAT-inclusive package price
        }))
      ),
    ];
    
    // Combine standalone products with package products
    const allProducts = [
      // Standalone products
      ...productLines.map(pl => ({
        productId: pl.productId,
        quantity: pl.quantity,
        packageId: null,
        packagePrice: null,
      })),
      // Package products - use packagePrice from package definition
      ...appliedPackages.flatMap(ap => 
        ap.package.products.map(pp => ({
          productId: pp.productId,
          quantity: pp.quantity,
          packageId: ap.package.id,
          packagePrice: pp.packagePrice, // VAT-inclusive package price
        }))
      ),
    ];
    
    const orderData: any = {
      orderType,
      source: "pos",
      clientId: clientId ? parseInt(clientId) : null,
      createdByEmployeeId: parseInt(createdByEmployeeId),
      packageIds: appliedPackages.map(ap => ap.package.id),
      couponId: appliedCoupon?.id || null,
      couponCode: couponCode || null,
      couponDiscountAmount: discount,
      services: allServices,
      products: allProducts,
      notes: notes || null,
    };
    
    // Add gift order fields
    if (orderType === "gift") {
      orderData.giftFromType = giftFromType;
      orderData.giftToCustomerId = parseInt(giftToCustomerId);
      orderData.giftExpiryDate = giftExpiryDate;
      if (giftFromType === "customer" && giftFromCustomerId) {
        orderData.giftFromCustomerId = parseInt(giftFromCustomerId);
      } else if (giftFromType === "supervisor" && giftFromSupervisorId) {
        orderData.giftFromSupervisorId = parseInt(giftFromSupervisorId);
      }
    }
    
    // Add payments array if any payment lines exist
    if (paymentLines.length > 0) {
      orderData.payments = paymentLines.map(pl => ({
        paymentMethodId: pl.paymentMethodId,
        amount: pl.amount,
        notes: null,
      }));
    }
    
    // Add loyalty points redemption
    if (useLoyaltyPoints && loyaltyPointsToRedeem > 0 && loyaltyPointsValue > 0) {
      orderData.redeemLoyaltyPoints = loyaltyPointsToRedeem;
      orderData.loyaltyPointsValue = loyaltyPointsValue;
    }
    
    createOrderMutation.mutate(orderData);
  };

  const isLoading = loadingEmployees || loadingCustomers || loadingServices || loadingProducts || loadingPackages || loadingCoupons;
  
  // Filter valid packages (active and within date range)
  const validPackages = packages?.filter(p => {
    const today = new Date().toISOString().split('T')[0];
    return p.isActive && p.startDate <= today && p.endDate >= today;
  }) || [];

  // Filter valid coupons (active and within date range)
  const validCoupons = coupons?.filter(c => {
    const today = new Date().toISOString().split('T')[0];
    return c.isActive && c.startDate <= today && c.endDate >= today;
  }) || [];

  if (isLoading) {
    return (
      <div className="container mx-auto p-6 space-y-6" dir="rtl">
        <Skeleton className="h-10 w-48" />
        <Skeleton className="h-96 w-full" />
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => setLocation("/orders")} data-testid="button-back">
          <ArrowRight className="h-5 w-5" />
        </Button>
        <div className="flex items-center gap-2">
          <ShoppingCart className="h-6 w-6" />
          <h1 className="text-2xl font-bold">طلب جديد</h1>
        </div>
      </div>

      <div className="grid lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="h-5 w-5" />
                معلومات الطلب
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>نوع الطلب</Label>
                  <Select value={orderType} onValueChange={setOrderType} data-testid="select-order-type">
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="normal">طلب عادي</SelectItem>
                      <SelectItem value="gift">هدية</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>العميل (اختياري)</Label>
                  <div className="flex gap-2">
                    <Select value={clientId || "_none"} onValueChange={(v) => setClientId(v === "_none" ? "" : v)} data-testid="select-client">
                      <SelectTrigger className="flex-1">
                        <SelectValue placeholder="اختر العميل" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="_none">زائر</SelectItem>
                        {customers?.map((customer) => (
                          <SelectItem key={customer.id} value={customer.id.toString()}>
                            <div className="flex items-center gap-2">
                              <span>{customer.nameAr}</span>
                              {customer.phone && <span className="text-muted-foreground">({customer.phone})</span>}
                              {customer.loyaltyPointsBalance ? (
                                <Badge variant="secondary" className="text-xs">
                                  <Star className="h-3 w-3 ml-1 text-yellow-500" />
                                  {customer.loyaltyPointsBalance}
                                </Badge>
                              ) : null}
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <Button 
                      variant="outline" 
                      size="icon" 
                      onClick={() => setShowNewCustomerDialog(true)}
                      title="إضافة عميل جديد"
                      data-testid="button-new-customer"
                    >
                      <UserPlus className="h-4 w-4" />
                    </Button>
                  </div>
                  {clientId && (() => {
                    const selectedCustomer = customers?.find(c => c.id === parseInt(clientId));
                    if (selectedCustomer?.loyaltyPointsBalance) {
                      return (
                        <div className="flex items-center gap-2 text-sm text-muted-foreground">
                          <Star className="h-4 w-4 text-yellow-500" />
                          <span>رصيد النقاط: {selectedCustomer.loyaltyPointsBalance} نقطة</span>
                        </div>
                      );
                    }
                    return null;
                  })()}
                </div>
              </div>
              
              {/* Gift Order Fields */}
              {orderType === "gift" && (
                <div className="border border-primary/30 rounded-lg p-4 space-y-4 bg-primary/5">
                  <div className="flex items-center gap-2 text-primary">
                    <Gift className="h-5 w-5" />
                    <span className="font-medium">تفاصيل الهدية</span>
                  </div>
                  <div className="grid md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label>مصدر الهدية</Label>
                      <Select value={giftFromType} onValueChange={setGiftFromType} data-testid="select-gift-from-type">
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="customer">من عميل</SelectItem>
                          <SelectItem value="supervisor">من مشرف/مدير</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    {giftFromType === "customer" && (
                      <div className="space-y-2">
                        <Label>العميل المُهدي *</Label>
                        <Select value={giftFromCustomerId || ""} onValueChange={setGiftFromCustomerId} data-testid="select-gift-from-customer">
                          <SelectTrigger>
                            <SelectValue placeholder="اختر العميل المُهدي" />
                          </SelectTrigger>
                          <SelectContent>
                            {customers?.map((customer) => (
                              <SelectItem key={customer.id} value={customer.id.toString()}>
                                {customer.nameAr} {customer.phone && `(${customer.phone})`}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                    )}
                    {giftFromType === "supervisor" && (
                      <div className="space-y-2">
                        <Label>المشرف/المدير (اختياري)</Label>
                        <Select value={giftFromSupervisorId || ""} onValueChange={setGiftFromSupervisorId} data-testid="select-gift-from-supervisor">
                          <SelectTrigger>
                            <SelectValue placeholder="اختر المشرف/المدير" />
                          </SelectTrigger>
                          <SelectContent>
                            {activeEmployees.map((emp) => (
                              <SelectItem key={emp.id} value={emp.id.toString()}>
                                {emp.nameAr} {emp.code && `(${emp.code})`}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                    )}
                  </div>
                  <div className="grid md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label>العميل المُهدى إليه *</Label>
                      <Select value={giftToCustomerId || ""} onValueChange={setGiftToCustomerId} data-testid="select-gift-to-customer">
                        <SelectTrigger>
                          <SelectValue placeholder="اختر العميل المُهدى إليه" />
                        </SelectTrigger>
                        <SelectContent>
                          {customers?.map((customer) => (
                            <SelectItem key={customer.id} value={customer.id.toString()}>
                              {customer.nameAr} {customer.phone && `(${customer.phone})`}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div className="space-y-2">
                      <Label>تاريخ انتهاء الهدية *</Label>
                      <Popover>
                        <PopoverTrigger asChild>
                          <Button
                            variant="outline"
                            className="w-full justify-start text-right font-normal"
                            data-testid="input-gift-expiry"
                          >
                            <CalendarIcon className="ml-2 h-4 w-4" />
                            {giftExpiryDate ? format(new Date(giftExpiryDate + "T00:00:00"), "PPP", { locale: ar }) : "اختر التاريخ"}
                          </Button>
                        </PopoverTrigger>
                        <PopoverContent className="w-auto p-0" align="start">
                          <Calendar
                            mode="single"
                            selected={giftExpiryDate ? new Date(giftExpiryDate + "T00:00:00") : undefined}
                            onSelect={(date) => {
                              if (date) {
                                const year = date.getFullYear();
                                const month = String(date.getMonth() + 1).padStart(2, '0');
                                const day = String(date.getDate()).padStart(2, '0');
                                setGiftExpiryDate(`${year}-${month}-${day}`);
                              } else {
                                setGiftExpiryDate("");
                              }
                            }}
                            disabled={(date) => {
                              const today = new Date();
                              today.setHours(0, 0, 0, 0);
                              return date < today;
                            }}
                            initialFocus
                          />
                        </PopoverContent>
                      </Popover>
                    </div>
                  </div>
                </div>
              )}
              <div className="space-y-2">
                <Label>الموظف المنشئ *</Label>
                <Select value={createdByEmployeeId} onValueChange={setCreatedByEmployeeId} data-testid="select-employee">
                  <SelectTrigger>
                    <SelectValue placeholder="اختر الموظف" />
                  </SelectTrigger>
                  <SelectContent>
                    {activeEmployees.map((emp) => (
                      <SelectItem key={emp.id} value={emp.id.toString()}>
                        {emp.nameAr} {emp.code && `(${emp.code})`}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>

          {/* Coupons and Discounts - Simplified */}
          <Card className="border-primary/20">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Ticket className="h-5 w-5 text-primary" />
                الكوبونات والخصومات
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>اختر كوبون من القائمة</Label>
                  <Select 
                    value={appliedCoupon?.id?.toString() || "_none"} 
                    onValueChange={(v) => {
                      if (v === "_none") {
                        setAppliedCoupon(null);
                        setCouponCode("");
                        setCouponDiscountAmount("0");
                      } else {
                        const selectedCoupon = validCoupons.find(c => c.id === parseInt(v));
                        if (selectedCoupon) {
                          setAppliedCoupon(selectedCoupon);
                          setCouponCode(selectedCoupon.code);
                          const applicableTotal = calculateApplicableTotal(selectedCoupon);
                          let discount = 0;
                          if (selectedCoupon.discountType === "percent") {
                            discount = applicableTotal * (selectedCoupon.discountValue / 100);
                          } else {
                            discount = Math.min(selectedCoupon.discountValue, applicableTotal);
                          }
                          setCouponDiscountAmount(discount.toFixed(2));
                        }
                      }
                    }}
                    data-testid="select-coupon"
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="اختر كوبون" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="_none">بدون كوبون</SelectItem>
                      {validCoupons.map((coupon) => (
                        <SelectItem key={coupon.id} value={coupon.id.toString()}>
                          {coupon.nameAr} ({coupon.code}) - {coupon.discountType === "percent" ? `${coupon.discountValue}%` : `${coupon.discountValue} ر.س`}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>قيمة الخصم المحسوبة</Label>
                  <Input
                    type="number"
                    min="0"
                    step="0.01"
                    value={couponDiscountAmount}
                    readOnly
                    className="bg-muted cursor-not-allowed"
                    data-testid="input-discount-amount"
                  />
                </div>
              </div>
              {appliedCoupon && (
                <div className="flex items-center justify-between p-3 bg-green-50 dark:bg-green-900/20 rounded-lg border border-green-200 dark:border-green-800">
                  <div className="flex items-center gap-2">
                    <Check className="h-5 w-5 text-green-600" />
                    <div>
                      <span className="font-medium text-green-700 dark:text-green-400">{appliedCoupon.nameAr}</span>
                      <span className="text-sm text-green-600 dark:text-green-500 mr-2">
                        ({appliedCoupon.discountType === "percent" ? `${appliedCoupon.discountValue}%` : `${appliedCoupon.discountValue} ر.س`})
                      </span>
                    </div>
                  </div>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => {
                      setCouponCode("");
                      setAppliedCoupon(null);
                      setCouponDiscountAmount("0");
                    }}
                    className="text-destructive"
                    data-testid="button-clear-coupon"
                  >
                    <X className="h-4 w-4 ml-1" />
                    إزالة
                  </Button>
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Scissors className="h-5 w-5" />
                الخدمات
                {appliedPackages.length > 0 && <Badge variant="secondary" className="mr-2">+ {appliedPackages.length} حزمة</Badge>}
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex gap-2">
                <Select value={selectedServiceId} onValueChange={setSelectedServiceId} data-testid="select-service">
                  <SelectTrigger className="flex-1">
                    <SelectValue placeholder="اختر خدمة لإضافتها" />
                  </SelectTrigger>
                  <SelectContent>
                    {activeServices.map((service) => (
                      <SelectItem key={service.id} value={service.id.toString()}>
                        {service.nameAr} - {formatNumber(service.price)} ر.س
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button type="button" onClick={addServiceLine} disabled={!selectedServiceId} data-testid="button-add-service">
                  <Plus className="h-4 w-4" />
                </Button>
              </div>

              {serviceLines.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Scissors className="h-10 w-10 mx-auto mb-2 opacity-50" />
                  <p>لم يتم إضافة خدمات بعد</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {serviceLines.map((sl) => (
                    <div 
                      key={sl.serviceId} 
                      className="border rounded-lg p-4 space-y-3"
                      data-testid={`service-line-${sl.serviceId}`}
                    >
                      <div className="flex items-center justify-between">
                        <div>
                          <span className="font-medium">{sl.service?.nameAr}</span>
                          <Badge variant="secondary" className="mr-2">
                            {formatNumber(sl.service?.price || 0)} ر.س
                          </Badge>
                        </div>
                        <Button 
                          variant="ghost" 
                          size="icon" 
                          onClick={() => removeServiceLine(sl.serviceId)}
                          data-testid={`button-remove-service-${sl.serviceId}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                      <div className="grid md:grid-cols-3 gap-3">
                        <div className="space-y-1">
                          <Label className="text-sm">الكمية</Label>
                          <div className="flex items-center gap-2">
                            <Button 
                              variant="outline" 
                              size="icon" 
                              className="h-8 w-8"
                              onClick={() => updateServiceLine(sl.serviceId, { quantity: Math.max(1, sl.quantity - 1) })}
                            >
                              <Minus className="h-3 w-3" />
                            </Button>
                            <span className="w-8 text-center">{sl.quantity}</span>
                            <Button 
                              variant="outline" 
                              size="icon" 
                              className="h-8 w-8"
                              onClick={() => updateServiceLine(sl.serviceId, { quantity: sl.quantity + 1 })}
                            >
                              <Plus className="h-3 w-3" />
                            </Button>
                          </div>
                        </div>
                        <div className="space-y-1">
                          <Label className="text-sm">الموعد (اختياري)</Label>
                          <div className="flex gap-1">
                            <Popover>
                              <PopoverTrigger asChild>
                                <Button
                                  variant="outline"
                                  className="flex-1 justify-start text-right font-normal text-xs"
                                  data-testid={`input-scheduled-${sl.serviceId}`}
                                >
                                  <CalendarIcon className="ml-1 h-3 w-3" />
                                  {sl.scheduledAt ? format(new Date(sl.scheduledAt), "d/M", { locale: ar }) : "تاريخ"}
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-auto p-0" align="start">
                                <Calendar
                                  mode="single"
                                  selected={sl.scheduledAt ? new Date(sl.scheduledAt) : undefined}
                                  onSelect={(date) => {
                                    if (date) {
                                      const existing = sl.scheduledAt ? new Date(sl.scheduledAt) : null;
                                      if (existing) {
                                        date.setHours(existing.getHours(), existing.getMinutes(), 0, 0);
                                      } else {
                                        date.setHours(12, 0, 0, 0);
                                      }
                                      updateServiceLine(sl.serviceId, { scheduledAt: date.toISOString() });
                                    } else {
                                      updateServiceLine(sl.serviceId, { scheduledAt: "" });
                                    }
                                  }}
                                  initialFocus
                                />
                              </PopoverContent>
                            </Popover>
                            <Input
                              type="time"
                              value={sl.scheduledAt ? format(new Date(sl.scheduledAt), "HH:mm") : ""}
                              onChange={(e) => {
                                if (e.target.value) {
                                  const [hours, minutes] = e.target.value.split(':').map(Number);
                                  const date = sl.scheduledAt ? new Date(sl.scheduledAt) : new Date();
                                  date.setHours(hours, minutes, 0, 0);
                                  updateServiceLine(sl.serviceId, { scheduledAt: date.toISOString() });
                                }
                              }}
                              className="w-20 text-xs"
                              data-testid={`input-scheduled-time-${sl.serviceId}`}
                            />
                          </div>
                        </div>
                        <div className="space-y-1">
                          <Label className="text-sm">المنفذ (اختياري)</Label>
                          <Select 
                            value={sl.executingEmployeeId?.toString() || "_none"} 
                            onValueChange={(v) => updateServiceLine(sl.serviceId, { executingEmployeeId: v && v !== "_none" ? parseInt(v) : null })}
                          >
                            <SelectTrigger>
                              <SelectValue placeholder="اختر" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="_none">غير محدد</SelectItem>
                              {activeEmployees.map((emp) => (
                                <SelectItem key={emp.id} value={emp.id.toString()}>
                                  {emp.nameAr}
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        </div>
                      </div>
                      {sl.service && (
                        <div className="text-left text-sm text-muted-foreground">
                          المجموع: {formatNumber(calculateServiceLineTotal(sl.service, sl.quantity).lineTotal)} ر.س
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Package className="h-5 w-5" />
                منتجات إضافية (اختياري)
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex gap-2">
                <Select value={selectedProductId} onValueChange={setSelectedProductId} data-testid="select-product">
                  <SelectTrigger className="flex-1">
                    <SelectValue placeholder="اختر منتج لإضافته" />
                  </SelectTrigger>
                  <SelectContent>
                    {activeProducts.map((product) => (
                      <SelectItem key={product.id} value={product.id.toString()}>
                        {product.nameAr} - {formatNumber(product.defaultSellPrice || 0)} ر.س
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button type="button" onClick={addProductLine} disabled={!selectedProductId} data-testid="button-add-product">
                  <Plus className="h-4 w-4" />
                </Button>
              </div>

              {productLines.length > 0 && (
                <div className="space-y-2">
                  {productLines.map((pl) => (
                    <div 
                      key={pl.productId} 
                      className="flex items-center justify-between border rounded-lg p-3"
                      data-testid={`product-line-${pl.productId}`}
                    >
                      <div className="flex items-center gap-3">
                        <span>{pl.product?.nameAr}</span>
                        <Badge variant="secondary">
                          {formatNumber(pl.product?.defaultSellPrice || 0)} ر.س
                        </Badge>
                      </div>
                      <div className="flex items-center gap-3">
                        <div className="flex items-center gap-2">
                          <Button 
                            variant="outline" 
                            size="icon" 
                            className="h-8 w-8"
                            onClick={() => updateProductLine(pl.productId, Math.max(1, pl.quantity - 1))}
                          >
                            <Minus className="h-3 w-3" />
                          </Button>
                          <span className="w-8 text-center">{pl.quantity}</span>
                          <Button 
                            variant="outline" 
                            size="icon" 
                            className="h-8 w-8"
                            onClick={() => updateProductLine(pl.productId, pl.quantity + 1)}
                          >
                            <Plus className="h-3 w-3" />
                          </Button>
                        </div>
                        <Button 
                          variant="ghost" 
                          size="icon"
                          onClick={() => removeProductLine(pl.productId)}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>

          {/* Packages Section - After Services and Products */}
          {validPackages.length > 0 && (
            <Card className="border-primary/20">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Gift className="h-5 w-5 text-primary" />
                  الحزم المتاحة
                  {appliedPackages.length > 0 && (
                    <Badge variant="secondary">{appliedPackages.length} حزمة مختارة</Badge>
                  )}
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* Applied Packages List */}
                {appliedPackages.map((ap) => (
                  <div key={ap.package.id} className="border rounded-lg overflow-hidden" data-testid={`applied-package-${ap.package.id}`}>
                    <div className="flex items-center justify-between p-3 bg-green-50 dark:bg-green-900/20 border-b border-green-200 dark:border-green-800">
                      <div className="flex items-center gap-2">
                        <Package className="h-5 w-5 text-green-600" />
                        <div>
                          <span className="font-medium text-green-700 dark:text-green-400">{ap.package.nameAr}</span>
                          {ap.package.descriptionAr && (
                            <p className="text-sm text-green-600 dark:text-green-500">{ap.package.descriptionAr}</p>
                          )}
                        </div>
                      </div>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => removePackage(ap.package.id)}
                        className="text-destructive"
                        data-testid={`button-remove-package-${ap.package.id}`}
                      >
                        <X className="h-4 w-4 ml-1" />
                        إزالة
                      </Button>
                    </div>
                    
                    {/* Package Services with Employee Assignment */}
                    <div className="p-3 space-y-3 bg-muted/30">
                      <div className="text-sm font-medium text-muted-foreground mb-2">خدمات الحزمة:</div>
                      {ap.serviceAssignments.map((sa) => (
                        <div 
                          key={sa.packageServiceId} 
                          className="flex flex-col md:flex-row md:items-center gap-2 p-3 bg-background rounded-lg border"
                          data-testid={`package-service-${ap.package.id}-${sa.packageServiceId}`}
                        >
                          <div className="flex-1">
                            <div className="flex items-center gap-2">
                              <Scissors className="h-4 w-4 text-primary" />
                              <span className="font-medium">{sa.service?.nameAr}</span>
                              <Badge variant="outline" className="text-xs">
                                {sa.quantity} × {formatNumber(sa.packagePrice)} ر.س
                              </Badge>
                            </div>
                          </div>
                          <div className="flex items-center gap-2">
                            <Label className="text-sm whitespace-nowrap">الموظف المنفذ:</Label>
                            <Select 
                              value={sa.executingEmployeeId?.toString() || ""} 
                              onValueChange={(val) => updatePackageServiceEmployee(ap.package.id, sa.packageServiceId, val ? parseInt(val) : null)}
                              data-testid={`select-employee-package-${ap.package.id}-${sa.packageServiceId}`}
                            >
                              <SelectTrigger className="w-[180px]">
                                <SelectValue placeholder="اختر الموظف" />
                              </SelectTrigger>
                              <SelectContent>
                                {activeEmployees.map((emp) => (
                                  <SelectItem key={emp.id} value={emp.id.toString()}>
                                    {emp.nameAr}
                                  </SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                ))}
                
                {/* Add New Package */}
                <div className="flex gap-2">
                  <Select value={selectedPackageId} onValueChange={setSelectedPackageId} data-testid="select-package">
                    <SelectTrigger className="flex-1">
                      <SelectValue placeholder="اختر حزمة لإضافتها" />
                    </SelectTrigger>
                    <SelectContent>
                      {validPackages
                        .filter(pkg => !appliedPackages.some(ap => ap.package.id === pkg.id))
                        .map((pkg) => (
                          <SelectItem key={pkg.id} value={pkg.id.toString()}>
                            {pkg.nameAr} ({pkg.services.length} خدمة{pkg.products.length > 0 ? ` + ${pkg.products.length} منتج` : ""})
                          </SelectItem>
                        ))}
                    </SelectContent>
                  </Select>
                  <Button
                    onClick={() => {
                      const pkg = validPackages.find(p => p.id === parseInt(selectedPackageId));
                      if (pkg) applyPackage(pkg);
                    }}
                    disabled={!selectedPackageId}
                    data-testid="button-apply-package"
                  >
                    <Plus className="h-4 w-4 ml-1" />
                    إضافة
                  </Button>
                </div>
                
                {appliedPackages.length === 0 && (
                  <p className="text-sm text-muted-foreground">
                    اختر حزمة أو أكثر لملء الطلب بالخدمات والمنتجات المتضمنة بأسعار خاصة
                  </p>
                )}
                
                {appliedPackages.length > 0 && (
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={clearAllPackages}
                    className="w-full text-destructive"
                    data-testid="button-clear-all-packages"
                  >
                    <Trash2 className="h-4 w-4 ml-1" />
                    إزالة جميع الحزم
                  </Button>
                )}
              </CardContent>
            </Card>
          )}
        </div>

        <div className="lg:col-span-1">
          <Card className="sticky top-6">
            <CardHeader>
              <CardTitle>ملخص الطلب</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span>مجموع الخدمات:</span>
                  <span>{formatNumber(allServicesTotals.total)} ر.س</span>
                </div>
                {appliedPackages.length > 0 && (
                  <div className="flex justify-between text-muted-foreground text-xs mr-4">
                    <span>({appliedPackages.length} حزمة - {appliedPackages.reduce((sum, ap) => sum + ap.serviceAssignments.length, 0)} خدمة)</span>
                    <span>{formatNumber(packageTotals.total)} ر.س</span>
                  </div>
                )}
                {productLines.length > 0 && (
                  <div className="flex justify-between">
                    <span>مجموع المنتجات:</span>
                    <span>{formatNumber(productTotals.total)} ر.س</span>
                  </div>
                )}
                {discount > 0 && (
                  <div className="flex justify-between text-green-600">
                    <span>الخصم:</span>
                    <span>-{formatNumber(discount)} ر.س</span>
                  </div>
                )}
                {loyaltyPointsValue > 0 && (
                  <div className="flex justify-between text-yellow-600">
                    <span className="flex items-center gap-1">
                      <Coins className="h-4 w-4" />
                      خصم النقاط ({loyaltyPointsToRedeem} نقطة):
                    </span>
                    <span>-{formatNumber(loyaltyPointsValue)} ر.س</span>
                  </div>
                )}
                <div className="border-t pt-2 flex justify-between font-bold text-lg">
                  <span>الإجمالي:</span>
                  <span data-testid="text-grand-total">{formatNumber(grandTotal)} ر.س</span>
                </div>
              </div>
              
              {/* Loyalty Points Redemption Section */}
              {canRedeemPoints && (
                <div className="border rounded-lg p-3 space-y-3 bg-yellow-50 dark:bg-yellow-900/20">
                  <div className="flex items-center gap-2">
                    <Checkbox
                      id="use-loyalty"
                      checked={useLoyaltyPoints}
                      onCheckedChange={(checked) => {
                        setUseLoyaltyPoints(!!checked);
                        if (!checked) {
                          setLoyaltyPointsToRedeem(0);
                        }
                      }}
                      data-testid="checkbox-use-loyalty"
                    />
                    <Label htmlFor="use-loyalty" className="flex items-center gap-2 cursor-pointer">
                      <Star className="h-4 w-4 text-yellow-500" />
                      <span>استخدام نقاط الولاء</span>
                    </Label>
                  </div>
                  
                  {useLoyaltyPoints && (
                    <div className="space-y-2">
                      <div className="flex items-center justify-between text-sm">
                        <span>رصيدك: {customerLoyaltyBalance} نقطة</span>
                        <span className="text-muted-foreground">
                          (كل نقطة = {loyaltySettings?.currencyPerPoint || 0} ر.س)
                        </span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Label className="whitespace-nowrap">عدد النقاط:</Label>
                        <Input
                          type="number"
                          min={loyaltySettings?.minRedeemPoints || 0}
                          max={maxRedeemablePoints}
                          value={loyaltyPointsToRedeem || ""}
                          onChange={(e) => {
                            const val = parseInt(e.target.value) || 0;
                            setLoyaltyPointsToRedeem(Math.min(val, maxRedeemablePoints));
                          }}
                          className="w-24"
                          data-testid="input-loyalty-points"
                        />
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => setLoyaltyPointsToRedeem(maxRedeemablePoints)}
                          data-testid="button-use-max-points"
                        >
                          الحد الأقصى
                        </Button>
                      </div>
                      <div className="text-xs text-muted-foreground">
                        الحد الأدنى: {loyaltySettings?.minRedeemPoints || 0} نقطة | 
                        الحد الأقصى: {maxRedeemablePoints} نقطة
                      </div>
                    </div>
                  )}
                </div>
              )}
              
              {/* Multiple Payment Methods */}
              <div className="space-y-3">
                <Label className="flex items-center gap-2">
                  <CreditCard className="h-4 w-4" />
                  طرق الدفع (اختياري - يمكن استخدام أكثر من طريقة)
                </Label>
                
                {/* Existing payment lines */}
                {paymentLines.length > 0 && (
                  <div className="space-y-2">
                    {paymentLines.map((pl) => {
                      const method = paymentMethods?.find(pm => pm.id === pl.paymentMethodId);
                      return (
                        <div key={pl.id} className="flex items-center justify-between gap-2 p-2 bg-muted rounded-md" data-testid={`payment-line-${pl.id}`}>
                          <div className="flex items-center gap-2">
                            <Wallet className="h-4 w-4 text-muted-foreground" />
                            <span className="text-sm">{method?.nameAr || "غير معروف"}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <span className="text-sm font-medium">{formatNumber(pl.amount)} ر.س</span>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => {
                                setPaymentLines(lines => lines.filter(l => l.id !== pl.id));
                              }}
                              data-testid={`button-remove-payment-${pl.id}`}
                            >
                              <X className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      );
                    })}
                    <div className="flex justify-between text-sm font-medium pt-1 border-t">
                      <span>إجمالي المدفوع:</span>
                      <span className="text-green-600">{formatNumber(paymentLines.reduce((sum, pl) => sum + pl.amount, 0))} ر.س</span>
                    </div>
                    {grandTotal - paymentLines.reduce((sum, pl) => sum + pl.amount, 0) > 0.01 && (
                      <div className="flex justify-between text-sm font-medium text-orange-600">
                        <span>المتبقي:</span>
                        <span>{formatNumber(Math.max(0, grandTotal - paymentLines.reduce((sum, pl) => sum + pl.amount, 0)))} ر.س</span>
                      </div>
                    )}
                  </div>
                )}
                
                {/* Add new payment line */}
                {grandTotal - paymentLines.reduce((sum, pl) => sum + pl.amount, 0) > 0.01 && (
                  <div className="flex gap-2">
                    <Select 
                      value={newPaymentMethodId} 
                      onValueChange={setNewPaymentMethodId}
                      data-testid="select-payment-method"
                    >
                      <SelectTrigger className="flex-1">
                        <SelectValue placeholder="طريقة الدفع" />
                      </SelectTrigger>
                      <SelectContent>
                        {paymentMethods?.filter(pm => pm.isActive).map((pm) => (
                          <SelectItem key={pm.id} value={pm.id.toString()}>
                            <div className="flex items-center gap-2">
                              <Wallet className="h-4 w-4" />
                              {pm.nameAr}
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <Input
                      type="number"
                      placeholder="المبلغ"
                      value={newPaymentAmount}
                      onChange={(e) => setNewPaymentAmount(e.target.value)}
                      className="w-28"
                      data-testid="input-payment-amount"
                    />
                    <Button
                      variant="outline"
                      size="icon"
                      onClick={() => {
                        if (!newPaymentMethodId) {
                          toast({ title: "يرجى اختيار طريقة الدفع", variant: "destructive" });
                          return;
                        }
                        const amount = parseFloat(newPaymentAmount);
                        if (isNaN(amount) || amount <= 0) {
                          toast({ title: "يرجى إدخال مبلغ صحيح", variant: "destructive" });
                          return;
                        }
                        const totalPaid = paymentLines.reduce((sum, pl) => sum + pl.amount, 0);
                        const remaining = grandTotal - totalPaid;
                        if (amount > remaining + 0.01) {
                          toast({ title: "المبلغ أكبر من المتبقي", variant: "destructive" });
                          return;
                        }
                        setPaymentLines(lines => [...lines, {
                          id: nextPaymentLineId,
                          paymentMethodId: parseInt(newPaymentMethodId),
                          amount: Math.min(amount, remaining),
                        }]);
                        setNextPaymentLineId(id => id + 1);
                        setNewPaymentMethodId("");
                        setNewPaymentAmount("");
                      }}
                      data-testid="button-add-payment"
                    >
                      <Plus className="h-4 w-4" />
                    </Button>
                  </div>
                )}
                
                {/* Quick pay full amount button */}
                {paymentLines.length === 0 && grandTotal > 0 && (
                  <div className="flex gap-2 flex-wrap">
                    {paymentMethods?.filter(pm => pm.isActive).slice(0, 3).map((pm) => (
                      <Button
                        key={pm.id}
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          setPaymentLines([{
                            id: nextPaymentLineId,
                            paymentMethodId: pm.id,
                            amount: grandTotal,
                          }]);
                          setNextPaymentLineId(id => id + 1);
                        }}
                        data-testid={`button-quick-pay-${pm.code || pm.id}`}
                      >
                        <Wallet className="h-4 w-4 ml-1" />
                        {pm.nameAr}
                      </Button>
                    ))}
                  </div>
                )}
                
                {paymentLines.length > 0 && Math.abs(grandTotal - paymentLines.reduce((sum, pl) => sum + pl.amount, 0)) <= 0.01 && (
                  <div className="text-sm text-green-600 flex items-center gap-1">
                    <Check className="h-4 w-4" />
                    تم تسجيل الدفع بالكامل
                  </div>
                )}
              </div>

              <div className="space-y-2 text-xs text-muted-foreground">
                <div className="flex justify-between">
                  <span>الخدمات:</span>
                  <span>{serviceLines.length + appliedPackages.reduce((sum, ap) => sum + ap.serviceAssignments.length, 0)}</span>
                </div>
                <div className="flex justify-between">
                  <span>الحزم:</span>
                  <span>{appliedPackages.length}</span>
                </div>
                <div className="flex justify-between">
                  <span>المنتجات:</span>
                  <span>{productLines.length}</span>
                </div>
              </div>

              {/* Order Notes */}
              <div className="space-y-2">
                <Label className="text-sm">ملاحظات (اختياري)</Label>
                <Textarea
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  placeholder="أضف ملاحظات للطلب"
                  className="min-h-[60px]"
                  data-testid="input-notes"
                />
              </div>

              <Button 
                className="w-full" 
                size="lg"
                onClick={handleSubmit}
                disabled={createOrderMutation.isPending || (serviceLines.length === 0 && appliedPackages.length === 0)}
                data-testid="button-submit-order"
              >
                {createOrderMutation.isPending ? "جاري الإنشاء..." : "إنشاء الطلب"}
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
      
      {/* Inline Customer Creation Dialog */}
      <Dialog open={showNewCustomerDialog} onOpenChange={setShowNewCustomerDialog}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <UserPlus className="h-5 w-5" />
              إضافة عميل جديد
            </DialogTitle>
            <DialogDescription>
              أدخل بيانات العميل الجديد
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="newCustomerName">اسم العميل *</Label>
              <Input
                id="newCustomerName"
                value={newCustomerName}
                onChange={(e) => setNewCustomerName(e.target.value)}
                placeholder="أدخل اسم العميل"
                data-testid="input-new-customer-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="newCustomerPhone">رقم الهاتف</Label>
              <Input
                id="newCustomerPhone"
                value={newCustomerPhone}
                onChange={(e) => setNewCustomerPhone(e.target.value)}
                placeholder="أدخل رقم الهاتف"
                data-testid="input-new-customer-phone"
              />
            </div>
          </div>
          <DialogFooter className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => {
                setShowNewCustomerDialog(false);
                setNewCustomerName("");
                setNewCustomerPhone("");
              }}
              data-testid="button-cancel-new-customer"
            >
              إلغاء
            </Button>
            <Button
              onClick={() => {
                if (!newCustomerName.trim()) {
                  toast({ title: "يجب إدخال اسم العميل", variant: "destructive" });
                  return;
                }
                createCustomerMutation.mutate({
                  nameAr: newCustomerName.trim(),
                  phone: newCustomerPhone.trim() || "",
                });
              }}
              disabled={createCustomerMutation.isPending}
              data-testid="button-save-new-customer"
            >
              {createCustomerMutation.isPending ? (
                <>
                  <Loader2 className="ml-2 h-4 w-4 animate-spin" />
                  جاري الإضافة...
                </>
              ) : (
                "إضافة العميل"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
