import { useQuery } from "@tanstack/react-query";
import { useRoute, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { ThermalReceipt, PrintData } from "@/components/receipt/ThermalReceipt";
import { ServiceTicket } from "@/components/receipt/ServiceTicket";
import { Printer, FileText, ArrowRight, AlertCircle } from "lucide-react";

interface ServiceLine {
  type: "service" | "product";
  lineId: number;
  name: string;
  executionTime: string | null;
  employeeName: string | null;
  discount: number;
  price: number;
  quantity: number;
}

interface PrintDataWithLines extends PrintData {
  lines: ServiceLine[];
}

export default function OrderPrint() {
  const [, params] = useRoute("/orders/:id/print");
  const [, navigate] = useLocation();
  const orderId = params?.id ? parseInt(params.id) : 0;

  const { data: printData, isLoading, error } = useQuery<PrintDataWithLines>({
    queryKey: ["/api/orders", orderId, "print-data"],
    enabled: orderId > 0,
  });

  const handlePrintOrder = () => {
    window.print();
  };

  const handlePrintReceipt = () => {
    const printWindow = window.open("", "_blank");
    if (!printWindow) {
      alert("يرجى السماح بالنوافذ المنبثقة لطباعة الفاتورة");
      return;
    }
    
    const receiptElement = document.getElementById("invoice-section");
    if (!receiptElement) return;
    
    printWindow.document.write(`
      <!DOCTYPE html>
      <html dir="rtl" lang="ar">
      <head>
        <meta charset="UTF-8">
        <title>${printData?.isCancelled ? 'إشعار دائن' : 'فاتورة'} ${printData?.creditNoteInfo?.creditNoteNumber || printData?.invoiceNumber}</title>
        <style>
          @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap');
          
          * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
          }
          
          body {
            font-family: 'Cairo', 'Courier New', monospace;
            background: white;
            color: black;
          }
          
          @page {
            size: 58mm auto;
            margin: 0;
          }
          
          @media print {
            body {
              width: 58mm;
            }
          }
          
          ${getReceiptStyles()}
        </style>
      </head>
      <body>
        ${receiptElement.innerHTML}
      </body>
      </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    setTimeout(() => {
      printWindow.print();
      printWindow.close();
    }, 500);
  };

  const handlePrintTickets = () => {
    const printWindow = window.open("", "_blank");
    if (!printWindow) {
      alert("يرجى السماح بالنوافذ المنبثقة لطباعة القصاصات");
      return;
    }
    
    const ticketsElement = document.getElementById("service-tickets-section");
    if (!ticketsElement) return;
    
    printWindow.document.write(`
      <!DOCTYPE html>
      <html dir="rtl" lang="ar">
      <head>
        <meta charset="UTF-8">
        <title>قصاصات الخدمات - ${printData?.invoiceNumber}</title>
        <style>
          @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap');
          
          * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
          }
          
          body {
            font-family: 'Cairo', 'Courier New', monospace;
            background: white;
            color: black;
          }
          
          @page {
            size: 58mm auto;
            margin: 0;
          }
          
          @media print {
            body {
              width: 58mm;
            }
            .service-ticket {
              page-break-after: always;
            }
            .service-ticket:last-child {
              page-break-after: avoid;
            }
          }
          
          ${getReceiptStyles()}
        </style>
      </head>
      <body>
        ${ticketsElement.innerHTML}
      </body>
      </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    setTimeout(() => {
      printWindow.print();
      printWindow.close();
    }, 500);
  };

  const serviceLines = printData?.lines.filter(l => l.type === "service") || [];
  const hasCompanyInfo = printData?.companyInfo.companyNameAr && 
                         printData?.companyInfo.vatNumber && 
                         printData?.companyInfo.crNumber;

  if (isLoading) {
    return (
      <div className="container mx-auto p-4 max-w-4xl" dir="rtl">
        <div className="flex items-center gap-2 mb-6">
          <Skeleton className="h-8 w-8" />
          <Skeleton className="h-8 w-48" />
        </div>
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Skeleton className="h-[600px]" />
          <Skeleton className="h-[400px]" />
        </div>
      </div>
    );
  }

  if (error || !printData) {
    return (
      <div className="container mx-auto p-4 max-w-4xl" dir="rtl">
        <div className="flex items-center gap-2 mb-6">
          <Button 
            variant="ghost" 
            size="icon"
            onClick={() => navigate(`/orders/${orderId}`)}
            data-testid="button-back"
          >
            <ArrowRight className="h-4 w-4" />
          </Button>
          <h1 className="text-2xl font-bold">خطأ في تحميل البيانات</h1>
        </div>
        <Card>
          <CardContent className="p-6 text-center">
            <AlertCircle className="h-12 w-12 text-destructive mx-auto mb-4" />
            <p>تعذر تحميل بيانات الطباعة</p>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <>
      <style>{getPrintMediaStyles()}</style>
      
      <div className="container mx-auto p-4 max-w-6xl print-hide" dir="rtl">
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center gap-2">
            <Button 
              variant="ghost" 
              size="icon"
              onClick={() => navigate(`/orders/${orderId}`)}
              data-testid="button-back"
            >
              <ArrowRight className="h-4 w-4" />
            </Button>
            <h1 className="text-2xl font-bold">
              {printData.isCancelled ? "طباعة إشعار الدائن" : "طباعة الفاتورة"} - {printData.creditNoteInfo?.creditNoteNumber || printData.invoiceNumber}
            </h1>
          </div>
          <div className="flex gap-2 print-actions">
            <Button onClick={handlePrintOrder} data-testid="button-print-order">
              <Printer className="h-4 w-4 ml-2" />
              طباعة الطلب
            </Button>
            <Button onClick={handlePrintReceipt} variant="outline" data-testid="button-print-receipt">
              <FileText className="h-4 w-4 ml-2" />
              {printData.isCancelled ? "طباعة إشعار الدائن" : "طباعة الفاتورة"}
            </Button>
            {serviceLines.length > 0 && !printData.isCancelled && (
              <Button onClick={handlePrintTickets} variant="outline" data-testid="button-print-tickets">
                <FileText className="h-4 w-4 ml-2" />
                طباعة قصاصات الخدمات ({serviceLines.length})
              </Button>
            )}
          </div>
        </div>

        {!hasCompanyInfo && (
          <Card className="mb-6 border-yellow-300 bg-yellow-50 dark:bg-yellow-950 dark:border-yellow-700">
            <CardContent className="p-4 flex items-center gap-3">
              <AlertCircle className="h-5 w-5 text-yellow-600" />
              <div>
                <p className="font-medium text-yellow-800 dark:text-yellow-200">
                  معلومات المنشأة غير مكتملة
                </p>
                <p className="text-sm text-yellow-700 dark:text-yellow-300">
                  يرجى إكمال إعدادات الفاتورة (اسم الشركة، الرقم الضريبي، السجل التجاري) من صفحة الإعدادات.
                </p>
              </div>
            </CardContent>
          </Card>
        )}

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Card>
            <CardContent className="p-6">
              <h2 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <FileText className="h-5 w-5" />
                {printData.isCancelled ? "معاينة إشعار الدائن" : "معاينة الفاتورة"}
              </h2>
              <div className="bg-white rounded-lg overflow-auto max-h-[800px]">
                <ThermalReceipt data={printData} />
              </div>
            </CardContent>
          </Card>

          {serviceLines.length > 0 && !printData.isCancelled && (
            <Card>
              <CardContent className="p-6">
                <h2 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <FileText className="h-5 w-5" />
                  قصاصات الخدمات ({serviceLines.length})
                </h2>
                <div className="bg-white rounded-lg overflow-auto max-h-[800px] space-y-4">
                  {serviceLines.map((line, index) => (
                    <ServiceTicket
                      key={index}
                      orderId={orderId}
                      orderNumber={printData.invoiceNumber}
                      lineId={line.lineId}
                      serviceName={line.name}
                      customerName={printData.customerName}
                      employeeName={line.employeeName}
                      executionTime={line.executionTime}
                    />
                  ))}
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      <div id="print-root" className="print-only" dir="rtl">
        <div id="invoice-section">
          <ThermalReceipt data={printData} />
        </div>
        
        {serviceLines.length > 0 && !printData.isCancelled && (
          <div id="service-tickets-section">
            {serviceLines.map((line, index) => (
              <ServiceTicket
                key={index}
                orderId={orderId}
                orderNumber={printData.invoiceNumber}
                lineId={line.lineId}
                serviceName={line.name}
                customerName={printData.customerName}
                employeeName={line.employeeName}
                executionTime={line.executionTime}
              />
            ))}
          </div>
        )}
      </div>
    </>
  );
}

function getPrintMediaStyles(): string {
  return `
    @media screen {
      .print-only {
        display: none !important;
      }
    }
    
    @media print {
      @page {
        size: 58mm auto;
        margin: 0;
      }
      
      body {
        margin: 0;
        padding: 0;
        background: white !important;
        -webkit-print-color-adjust: exact;
        print-color-adjust: exact;
      }
      
      .print-hide,
      .navbar,
      .sidebar,
      .print-actions,
      header,
      nav,
      aside,
      [data-sidebar],
      .toaster {
        display: none !important;
      }
      
      .print-only {
        display: block !important;
      }
      
      #print-root {
        display: block;
        width: 58mm;
        margin: 0 auto;
      }
      
      #invoice-section {
        width: 58mm;
        margin: 0 auto;
        page-break-after: always;
      }
      
      #service-tickets-section {
        width: 58mm;
        margin: 0 auto;
      }
      
      .service-ticket {
        page-break-after: always;
      }
      
      .service-ticket:last-child {
        page-break-after: avoid;
      }
      
      ${getReceiptStyles()}
    }
  `;
}

function getReceiptStyles(): string {
  return `
    .thermal-receipt {
      width: 58mm;
      max-width: 230px;
      margin: 0 auto;
      padding: 4mm;
      background: white;
      color: #000;
      font-family: 'Cairo', 'Courier New', monospace;
      font-size: 10px;
      line-height: 1.4;
      text-align: right;
    }

    .receipt-header {
      text-align: center;
      margin-bottom: 3mm;
    }

    .receipt-title {
      font-size: 11px;
      font-weight: normal;
      margin-bottom: 2mm;
    }

    .company-name {
      font-size: 14px;
      font-weight: bold;
      margin-bottom: 2mm;
    }

    .company-phone,
    .branch-name {
      font-size: 10px;
      margin-bottom: 1mm;
    }

    .credit-note-subtitle {
      font-size: 9px;
      color: #666;
      margin-bottom: 2mm;
    }

    .receipt-divider {
      border-top: 1px dashed #000;
      margin: 2mm 0;
    }

    .receipt-info,
    .payment-info {
      margin-bottom: 2mm;
    }

    .info-row {
      display: flex;
      justify-content: space-between;
      margin-bottom: 1mm;
      font-size: 9px;
    }

    .info-label {
      font-weight: bold;
    }

    .info-value {
      text-align: left;
      direction: ltr;
    }

    .items-table {
      width: 100%;
      border-collapse: collapse;
      margin: 2mm 0;
      font-size: 8px;
    }

    .items-table th {
      border-bottom: 1px solid #000;
      padding: 1mm;
      text-align: center;
      font-weight: bold;
      font-size: 8px;
    }

    .items-table td {
      padding: 1mm;
      text-align: center;
      vertical-align: top;
      font-size: 8px;
      border-bottom: 1px dotted #ccc;
    }

    .col-service {
      text-align: right !important;
      width: 35%;
    }

    .col-price {
      width: 18%;
      direction: ltr;
    }

    .col-discount {
      width: 15%;
      direction: ltr;
    }

    .col-expert {
      width: 20%;
      font-size: 7px !important;
    }

    .col-time {
      width: 12%;
      direction: ltr;
    }

    .summary-section {
      margin: 2mm 0;
    }

    .summary-table {
      width: 100%;
      border-collapse: collapse;
    }

    .summary-table td {
      padding: 1mm 0;
      font-size: 9px;
    }

    .summary-label {
      text-align: right;
      font-weight: bold;
    }

    .summary-value {
      text-align: left;
      direction: ltr;
    }

    .summary-total {
      font-size: 11px !important;
      font-weight: bold;
    }

    .summary-total td {
      font-size: 11px !important;
      border-top: 1px solid #000;
      border-bottom: 1px solid #000;
      padding: 2mm 0;
    }

    .discount-value {
      color: #c00;
    }

    .refund-value {
      color: #080;
    }

    .receipt-footer {
      text-align: center;
      margin-top: 3mm;
    }

    .cashier-info {
      font-size: 9px;
      margin-bottom: 2mm;
    }

    .extra-note {
      font-size: 8px;
      margin-bottom: 2mm;
      font-style: italic;
    }

    .qr-container {
      display: flex;
      justify-content: center;
      margin: 3mm 0;
    }

    .zatca-note {
      font-size: 7px;
      color: #666;
      margin-top: 2mm;
    }

    .service-ticket {
      width: 58mm;
      max-width: 230px;
      margin: 0 auto;
      padding: 3mm;
      background: white;
      color: #000;
      font-family: 'Cairo', 'Courier New', monospace;
      font-size: 10px;
      text-align: center;
      border: 1px dashed #000;
    }

    .ticket-header {
      font-size: 10px;
      font-weight: bold;
      margin-bottom: 2mm;
      border-bottom: 1px solid #000;
      padding-bottom: 2mm;
    }

    .ticket-info {
      text-align: right;
      margin: 2mm 0;
    }

    .ticket-row {
      display: flex;
      justify-content: space-between;
      margin-bottom: 1mm;
      font-size: 9px;
    }

    .ticket-label {
      font-weight: bold;
    }

    .service-name {
      font-size: 12px;
      font-weight: bold;
      margin: 3mm 0;
      padding: 2mm;
      background: #f0f0f0;
      border-radius: 2mm;
    }

    .barcode-container {
      margin: 3mm 0;
      display: flex;
      flex-direction: column;
      align-items: center;
    }

    .barcode-container svg {
      max-width: 100%;
      height: auto;
    }

    .barcode-text {
      font-size: 8px;
      margin-top: 1mm;
    }
  `;
}
