import { useQuery, useMutation } from "@tanstack/react-query";
import { useState } from "react";
import { Link } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { formatCurrency, formatNumber, formatDate } from "@/lib/formatters";
import { 
  Smartphone, 
  Plus, 
  Eye, 
  Clock, 
  CheckCircle2, 
  XCircle,
  Calendar,
  User,
  Filter,
  Star,
  Pause,
  Play,
  CalendarDays,
  RefreshCw,
  RotateCcw,
  ArrowRight,
} from "lucide-react";

interface Employee {
  id: number;
  nameAr: string;
  code: string | null;
  role: string | null;
  isActive: boolean;
}

interface Customer {
  id: number;
  nameAr: string;
  phone: string | null;
}

interface Service {
  id: number;
  nameAr: string;
  code: string | null;
  category?: {
    id: number;
    nameAr: string;
  };
}

interface OrderService {
  id: number;
  orderId: number;
  serviceId: number;
  quantity: number;
  basePrice: number;
  vatType: string;
  vatRate: number;
  lineSubtotal: number;
  vatAmount: number;
  lineTotal: number;
  scheduledAt: string | null;
  executingEmployeeId: number | null;
  service?: Service;
  executingEmployee?: Employee;
}

interface Order {
  id: number;
  orderNumber: string;
  orderDate: string;
  source: string;
  orderType: string;
  clientId: number | null;
  createdByEmployeeId: number;
  branchId: number | null;
  couponCode: string | null;
  couponDiscountAmount: number;
  status: string;
  notes: string | null;
  subtotalAmount: number;
  vatAmount: number;
  totalAmount: number;
  createdAt: string;
  client?: Customer;
  createdByEmployee?: Employee;
  services: OrderService[];
  rating?: {
    id: number;
    rating: number;
    comment: string | null;
  };
}

const statusLabels: Record<string, string> = {
  new: "جديد",
  in_progress: "قيد التنفيذ",
  completed: "مكتمل",
  canceled: "ملغي",
  scheduled: "مجدول",
  postponed: "مؤجل",
  deferred: "مؤجل",
  returned: "مرتجع",
};

const statusColors: Record<string, string> = {
  new: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
  in_progress: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
  completed: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
  canceled: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
  scheduled: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
  postponed: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
  deferred: "bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300",
  returned: "bg-pink-100 text-pink-800 dark:bg-pink-900 dark:text-pink-300",
};

const orderTypeLabels: Record<string, string> = {
  normal: "طلب عادي",
  gift: "هدية",
};

export default function OrdersAppPage() {
  const [filterStatus, setFilterStatus] = useState<string>("all");
  const [statusChangeOrderId, setStatusChangeOrderId] = useState<number | null>(null);
  const [newStatus, setNewStatus] = useState<string>("");
  const { toast } = useToast();

  const buildQueryKey = () => {
    const params: string[] = ["source=app"];
    if (filterStatus !== "all") {
      params.push(`status=${filterStatus}`);
    }
    return `/api/orders?${params.join("&")}`;
  };

  const { data: orders, isLoading, refetch } = useQuery<Order[]>({
    queryKey: ["/api/orders", "app", filterStatus],
    queryFn: async () => {
      const response = await fetch(buildQueryKey());
      if (!response.ok) throw new Error("Failed to fetch app orders");
      return response.json();
    },
  });

  const updateStatusMutation = useMutation({
    mutationFn: ({ id, status }: { id: number; status: string }) =>
      apiRequest("PUT", `/api/orders/${id}`, { status }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({ title: "تم تحديث حالة الطلب بنجاح" });
      setStatusChangeOrderId(null);
    },
    onError: (error: any) => {
      toast({
        title: "خطأ",
        description: error?.message || "فشل في تحديث حالة الطلب",
        variant: "destructive",
      });
    },
  });

  const handleStatusChange = () => {
    if (statusChangeOrderId && newStatus) {
      updateStatusMutation.mutate({ id: statusChangeOrderId, status: newStatus });
    }
  };

  const openStatusChangeDialog = (order: Order) => {
    setStatusChangeOrderId(order.id);
    setNewStatus(order.status);
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "new":
        return <Clock className="h-4 w-4" />;
      case "in_progress":
        return <Play className="h-4 w-4" />;
      case "completed":
        return <CheckCircle2 className="h-4 w-4" />;
      case "canceled":
        return <XCircle className="h-4 w-4" />;
      case "scheduled":
        return <CalendarDays className="h-4 w-4" />;
      case "postponed":
      case "deferred":
        return <Pause className="h-4 w-4" />;
      case "returned":
        return <RotateCcw className="h-4 w-4" />;
      default:
        return null;
    }
  };

  const stats = {
    total: orders?.length || 0,
    new: orders?.filter(o => o.status === "new").length || 0,
    inProgress: orders?.filter(o => o.status === "in_progress").length || 0,
    completed: orders?.filter(o => o.status === "completed").length || 0,
    canceled: orders?.filter(o => o.status === "canceled").length || 0,
    deferred: orders?.filter(o => o.status === "deferred").length || 0,
    returned: orders?.filter(o => o.status === "returned").length || 0,
    totalRevenue: orders?.reduce((sum, o) => o.status === "completed" ? sum + o.totalAmount : sum, 0) || 0,
  };

  return (
    <div className="container mx-auto p-6 space-y-6" dir="rtl">
      <div className="flex items-center justify-between gap-4 flex-wrap">
        <div className="flex items-center gap-4">
          <Link href="/orders">
            <Button variant="ghost" size="icon" data-testid="button-back">
              <ArrowRight className="h-5 w-5" />
            </Button>
          </Link>
          <div className="flex items-center gap-2">
            <Smartphone className="h-6 w-6" />
            <h1 className="text-2xl font-bold">طلبات التطبيق</h1>
          </div>
        </div>
        <div className="flex gap-2 flex-wrap">
          <Button onClick={() => refetch()} variant="outline" size="icon" data-testid="button-refresh">
            <RefreshCw className="h-4 w-4" />
          </Button>
          <Link href="/orders/new">
            <Button data-testid="button-new-order">
              <Plus className="h-4 w-4 ml-2" />
              طلب جديد
            </Button>
          </Link>
        </div>
      </div>

      <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-8 gap-4">
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("all")} data-testid="card-filter-all">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-primary" data-testid="stat-total">{stats.total}</div>
            <div className="text-sm text-muted-foreground">إجمالي الطلبات</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("new")} data-testid="card-filter-new">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-blue-600" data-testid="stat-new">{stats.new}</div>
            <div className="text-sm text-muted-foreground">جديد</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("in_progress")} data-testid="card-filter-in-progress">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-yellow-600" data-testid="stat-in-progress">{stats.inProgress}</div>
            <div className="text-sm text-muted-foreground">قيد التنفيذ</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("completed")} data-testid="card-filter-completed">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-green-600" data-testid="stat-completed">{stats.completed}</div>
            <div className="text-sm text-muted-foreground">مكتمل</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("canceled")} data-testid="card-filter-canceled">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-red-600" data-testid="stat-canceled">{stats.canceled}</div>
            <div className="text-sm text-muted-foreground">ملغي</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("deferred")} data-testid="card-filter-deferred">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-amber-600" data-testid="stat-deferred">{stats.deferred}</div>
            <div className="text-sm text-muted-foreground">مؤجل</div>
          </CardContent>
        </Card>
        <Card className="hover-elevate cursor-pointer" onClick={() => setFilterStatus("returned")} data-testid="card-filter-returned">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-pink-600" data-testid="stat-returned">{stats.returned}</div>
            <div className="text-sm text-muted-foreground">مرتجع</div>
          </CardContent>
        </Card>
        <Card data-testid="card-revenue">
          <CardContent className="p-4 text-center">
            <div className="text-2xl font-bold text-primary" data-testid="stat-revenue">{formatNumber(stats.totalRevenue)}</div>
            <div className="text-sm text-muted-foreground">الإيرادات (ر.س)</div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between gap-2 flex-wrap">
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              تصفية الطلبات
            </CardTitle>
          </div>
        </CardHeader>
        <CardContent>
          <div className="space-y-2 max-w-xs">
            <Label>الحالة</Label>
            <Select value={filterStatus} onValueChange={setFilterStatus} data-testid="select-filter-status">
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">جميع الحالات</SelectItem>
                <SelectItem value="new">جديد</SelectItem>
                <SelectItem value="in_progress">قيد التنفيذ</SelectItem>
                <SelectItem value="completed">مكتمل</SelectItem>
                <SelectItem value="canceled">ملغي</SelectItem>
                <SelectItem value="scheduled">مجدول</SelectItem>
                <SelectItem value="postponed">مؤجل</SelectItem>
                <SelectItem value="deferred">مؤجل (تأجيل)</SelectItem>
                <SelectItem value="returned">مرتجع</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardContent className="p-0">
          {isLoading ? (
            <div className="p-4 space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : orders && orders.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="text-right">رقم الطلب</TableHead>
                  <TableHead className="text-right">التاريخ</TableHead>
                  <TableHead className="text-right">العميل</TableHead>
                  <TableHead className="text-right">الخدمات</TableHead>
                  <TableHead className="text-right">الإجمالي</TableHead>
                  <TableHead className="text-right">الحالة</TableHead>
                  <TableHead className="text-right">التقييم</TableHead>
                  <TableHead className="text-right">الإجراءات</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {orders.map((order) => (
                  <TableRow key={order.id} data-testid={`row-order-${order.id}`}>
                    <TableCell className="font-medium" data-testid={`text-order-number-${order.id}`}>
                      {order.orderNumber}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Calendar className="h-4 w-4 text-muted-foreground" />
                        <span className="text-sm">{formatDate(order.orderDate)}</span>
                      </div>
                    </TableCell>
                    <TableCell>
                      {order.client ? (
                        <div className="flex items-center gap-1">
                          <User className="h-4 w-4 text-muted-foreground" />
                          <span>{order.client.nameAr}</span>
                        </div>
                      ) : (
                        <span className="text-muted-foreground">زائر</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex flex-wrap gap-1">
                        {order.services.slice(0, 2).map((svc) => (
                          <Badge key={svc.id} variant="secondary" className="text-xs">
                            {svc.service?.nameAr || `خدمة ${svc.serviceId}`}
                          </Badge>
                        ))}
                        {order.services.length > 2 && (
                          <Badge variant="outline" className="text-xs">
                            +{order.services.length - 2}
                          </Badge>
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="font-semibold" data-testid={`text-order-total-${order.id}`}>
                      {formatNumber(order.totalAmount)} ر.س
                    </TableCell>
                    <TableCell>
                      <Badge 
                        className={`${statusColors[order.status]} cursor-pointer`}
                        onClick={() => openStatusChangeDialog(order)}
                        data-testid={`badge-status-${order.id}`}
                      >
                        {getStatusIcon(order.status)}
                        <span className="mr-1">{statusLabels[order.status]}</span>
                      </Badge>
                    </TableCell>
                    <TableCell>
                      {order.rating ? (
                        <div className="flex items-center gap-1">
                          <Star className="h-4 w-4 fill-yellow-400 text-yellow-400" />
                          <span>{order.rating.rating}/5</span>
                        </div>
                      ) : order.status === "completed" ? (
                        <span className="text-muted-foreground text-sm">بدون تقييم</span>
                      ) : (
                        <span className="text-muted-foreground text-sm">-</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Link href={`/orders/${order.id}`}>
                        <Button variant="ghost" size="icon" data-testid={`button-view-order-${order.id}`}>
                          <Eye className="h-4 w-4" />
                        </Button>
                      </Link>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="p-8 text-center text-muted-foreground">
              <Smartphone className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>لا توجد طلبات من التطبيق</p>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={statusChangeOrderId !== null} onOpenChange={() => setStatusChangeOrderId(null)}>
        <DialogContent dir="rtl">
          <DialogHeader>
            <DialogTitle>تغيير حالة الطلب</DialogTitle>
            <DialogDescription>اختر الحالة الجديدة للطلب</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>الحالة الجديدة</Label>
              <Select value={newStatus} onValueChange={setNewStatus} data-testid="select-new-status">
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="new">جديد</SelectItem>
                  <SelectItem value="in_progress">قيد التنفيذ</SelectItem>
                  <SelectItem value="completed">مكتمل</SelectItem>
                  <SelectItem value="canceled">ملغي</SelectItem>
                  <SelectItem value="scheduled">مجدول</SelectItem>
                  <SelectItem value="postponed">مؤجل</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={() => setStatusChangeOrderId(null)}>
              إلغاء
            </Button>
            <Button 
              onClick={handleStatusChange} 
              disabled={updateStatusMutation.isPending}
              data-testid="button-confirm-status"
            >
              {updateStatusMutation.isPending ? "جاري الحفظ..." : "حفظ"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
